#include "hook_action.h"

#include <drive/backend/chat_robots/configuration/config.h>
#include <drive/backend/data/complaint_tags/car_complaint.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/support_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/support_center/manager.h>
#include <drive/backend/support_center/categorizer/model.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/user_documents_check.h>
#include <drive/backend/user_document_photos/manager.h>
#include <drive/library/cpp/raw_text/phone_number.h>

#include <rtline/library/json/merge.h>

TProposeCarTagHookAction::TFactory::TRegistrator<TProposeCarTagHookAction> TProposeCarTagHookAction::Registrator(IHookAction::EType::ProposeTag);
TFeedbackOverallHookAction::TFactory::TRegistrator<TFeedbackOverallHookAction> TFeedbackOverallHookAction::Registrator(IHookAction::EType::FeedbackOverall);
TFeedbackDetailsHookAction::TFactory::TRegistrator<TFeedbackDetailsHookAction> TFeedbackDetailsHookAction::Registrator(IHookAction::EType::FeedbackDetails);
TSetDocumentStatusesHookAction::TFactory::TRegistrator<TSetDocumentStatusesHookAction> TSetDocumentStatusesHookAction::Registrator(IHookAction::EType::SetDocumentStatuses);
TStartDeviceVerifyHookAction::TFactory::TRegistrator<TStartDeviceVerifyHookAction> TStartDeviceVerifyHookAction::Registrator(IHookAction::EType::DeviceVerify);
TAddTagHookAction::TFactory::TRegistrator<TAddTagHookAction> TAddTagHookAction::Registrator(IHookAction::EType::AddTag);
TAlterFlagsHookAction::TFactory::TRegistrator<TAlterFlagsHookAction> TAlterFlagsHookAction::Registrator(IHookAction::EType::AlterFlag);
TClassifyHookAction::TFactory::TRegistrator<TClassifyHookAction> TClassifyHookAction::Registrator(IHookAction::EType::ClassifyRequest);
TEndpointCallHookAction::TFactory::TRegistrator<TEndpointCallHookAction> TEndpointCallHookAction::Registrator(IHookAction::EType::EndpointCall);
TEditDictionaryHookAction::TFactory::TRegistrator<TEditDictionaryHookAction> TEditDictionaryHookAction::Registrator(IHookAction::EType::EditDictionary);
TModifyTagsAction::TFactory::TRegistrator<TModifyTagsAction> TModifyTagsAction::Registrator(IHookAction::EType::ModifyTags);
TSendComplaintPhotos::TFactory::TRegistrator<TSendComplaintPhotos> TSendComplaintPhotos::Registrator(IHookAction::EType::SendComplaintPhotos);
TStartPhoneVerifyHookAction::TFactory::TRegistrator<TStartPhoneVerifyHookAction> TStartPhoneVerifyHookAction::Registrator(IHookAction::EType::PhoneVerify);


bool TProposeCarTagHookAction::Parse(const NJson::TJsonValue& raw, TMessagesCollector& /*errors*/) {
    JREAD_STRING(raw, "object", Object);
    // if (Object != "!current_car") {
    //     // only one possible option for a moment
    //     errors.AddMessage("propose_car_tag", "object is not !current_car");
    //     return false;
    // }
    JREAD_STRING(raw, "tag", Tag);
    return true;
}

IHookAction::TPtr TProposeCarTagHookAction::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TProposeCarTagHookAction> result(new TProposeCarTagHookAction());
    result->SetObject(NChatScript::GetMaybeParametrizedField(Object, params));
    result->SetTag(NChatScript::GetMaybeParametrizedField(Tag, params));
    return result.Release();
}

bool TProposeCarTagHookAction::Perform(const IChatUserContext::TPtr userContext, TChatContext& context, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatTx, const TChatRobotScriptItem& /*item*/) const {
    TString operatedObjectId;

    if (Object == "!current_car") {
        TVector<TString> cars;
        if (!server->GetDriveAPI()->GetCurrentSessionsObjects(userContext->GetUserId(), cars, Now()) || cars.size() != 1) {
            chatTx.SetErrorInfo("ProposeCarTagHookAction::Perform", "cannot get current_car");
            return false;
        }
        operatedObjectId = cars.front();
    } else {
        operatedObjectId = userContext->Unescape(Object, context);
    }

    if (!operatedObjectId) {
        chatTx.SetErrorInfo("ProposeCarTagHookAction::Perform", "cannot get operated object_id");
        return false;
    }

    auto tagName = userContext->Unescape(Tag, context);
    auto tagImpl = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(tagName, userContext->GetMessage());
    if (!tagImpl) {
        chatTx.SetErrorInfo("ProposeCarTagHookAction::Perform", "cannot create tag " + tagName);
        return false;
    }

    TDBTag dbTag;
    dbTag.SetData(tagImpl).SetObjectId(operatedObjectId);
    return !!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().ProposeTag(dbTag, userContext->GetUserId(), session);
}

bool TAbstractFeedbackHookAction::Parse(const NJson::TJsonValue& raw, TMessagesCollector& /*errors*/) {
    JREAD_STRING(raw, "value", Value);
    return true;
}

bool TAbstractFeedbackHookAction::Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatTx, const TChatRobotScriptItem& /*item*/) const {
    TString topicLink = userContext->GetChatId();
    if (userContext->GetChatTopic()) {
        topicLink += "." + userContext->GetChatTopic();
    }
    TVector<TDBTag> userTags;
    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({userContext->GetUserId()}, {}, userTags, session)) {
        return false;
    }
    for (auto&& tag : userTags) {
        auto impl = tag.GetTagAs<TSupportChatTag>();
        if (impl && impl->GetTopicLink() == topicLink) {
            if (!PatchTag(tag)) {
                chatTx.SetErrorInfo("AbstractFeedbackHookAction::Perform", "cannot PatchTag");
                return false;
            }
            return server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(tag, userContext->GetUserId(), session);
        }
    }
    return true;
}

IHookAction::TPtr TFeedbackOverallHookAction::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TFeedbackOverallHookAction> result(new TFeedbackOverallHookAction());
    result->SetValue(NChatScript::GetMaybeParametrizedField(GetValue(), params));
    return result.Release();
}

IHookAction::TPtr TFeedbackDetailsHookAction::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TFeedbackDetailsHookAction> result(new TFeedbackDetailsHookAction());
    result->SetValue(NChatScript::GetMaybeParametrizedField(GetValue(), params));
    return result.Release();
}

bool TFeedbackOverallHookAction::PatchTag(TDBTag& tag) const {
    auto tagImpl = tag.MutableTagAs<TSupportChatTag>();
    if (!tagImpl) {
        return false;
    }
    tagImpl->MutableFeedback().SetOverallQuality(GetValue());
    return true;
}

bool TFeedbackDetailsHookAction::PatchTag(TDBTag& tag) const {
    auto tagImpl = tag.MutableTagAs<TSupportChatTag>();
    if (!tagImpl) {
        return false;
    }
    tagImpl->MutableFeedback().MutableIssues().push_back(GetValue());
    return true;
}

const TVector<NUserDocument::EType> TSetDocumentStatusesHookAction::PhotosOrder = TVector<NUserDocument::EType>(
    {
        NUserDocument::EType::LicenseFront,
        NUserDocument::EType::LicenseBack,
        NUserDocument::EType::PassportBiographical,
        NUserDocument::EType::PassportRegistration,
        NUserDocument::EType::PassportSelfie
    }
);

bool TSetDocumentStatusesHookAction::Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw.Has("statuses")) {
        errors.AddMessage("set_photo_statuses", "don't have statuses");
        return false;
    }
    if (!raw["statuses"].IsString()) {
        errors.AddMessage("set_photo_statuses", "'statuses' is not string");
        return false;
    }
    Statuses = raw["statuses"].GetString();
    if (Statuses.size() != 5) {
        errors.AddMessage("set_photo_statuses", "statuses should be 5 chars long");
        return false;
    }
    VStatus.resize(5);
    for (size_t i = 0; i < 5; ++i) {
        if (Statuses[i] == '.') {
            VStatus[i] = NUserDocument::EVerificationStatus::NotYetProcessed;
            continue;
        }
        TString statusStr((size_t)1, Statuses[i]);
        if (!TryFromString(statusStr, VStatus[i])) {
            errors.AddMessage("set_photo_statuses", "'statusStr' does not represent valid status");
            return false;
        }
    }
    return true;
}

IHookAction::TPtr TSetDocumentStatusesHookAction::GetTemplateImpl(const TVector<TString>& /*params*/) const {
    THolder<TSetDocumentStatusesHookAction> result(new TSetDocumentStatusesHookAction());
    result->SetStatuses(Statuses);
    return result.Release();
}

bool TSetDocumentStatusesHookAction::Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatTx, const TChatRobotScriptItem& /*item*/) const {
    if (!server->GetDriveAPI()->HasDocumentPhotosManager()) {
        chatTx.SetErrorInfo("SetDocumentStatusesHookAction::Perform", "DocumentPhotosManager is missing");
        return false;
    }
    auto mapping = server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().GetTypeToRecentPhotoMapping(userContext->GetUserId(), PhotosOrder);
    for (size_t i = 0; i < PhotosOrder.size(); ++i) {
        if (VStatus[i] == NUserDocument::EVerificationStatus::NotYetProcessed) {
            continue;
        }
        auto it = mapping.find(PhotosOrder[i]);
        if (it == mapping.end()) {
            chatTx.SetErrorInfo("SetDocumentStatusesHookAction::Perform", "cannot find " + ToString(PhotosOrder[i]));
            return false;
        }
        it->second.SetVerificationStatus(VStatus[i]);
        if (!server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().Upsert(it->second, session)) {
            return false;
        }
    }
    return true;
}

bool TStartDeviceVerifyHookAction::Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        NJson::ParseField(raw, "verification_method", NJson::Stringify(VerificationMethod), false, errors);
}

IHookAction::TPtr TStartDeviceVerifyHookAction::GetTemplateImpl(const TVector<TString>& /*params*/) const {
    THolder<TStartDeviceVerifyHookAction> result(new TStartDeviceVerifyHookAction());
    result->SetVerificationMethod(VerificationMethod);
    return result.Release();
}

bool TStartDeviceVerifyHookAction::Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatTx, const TChatRobotScriptItem& /*item*/) const {
    if (!server->GetUserDevicesManager()) {
        chatTx.SetErrorInfo("StartDeviceVerifyHookAction::Perform", "UserDevicesManager is missing");
        return false;
    }
    TSet<TString> errorCodes;
    TDeviceIdVerificationContext divc;
    divc.ApplicationId = userContext->GetApplicationId();
    divc.ClientIp = userContext->GetClientIp();
    divc.EnableGpsPackageName = IRequestProcessor::IsAndroid(userContext->GetPlatform());
    return server->GetUserDevicesManager()->StartDeviceIdVerification(userContext->GetUserId(), userContext->GetDeviceId(), userContext->GetPhone(), divc, userContext->GetReplyContext(), session, VerificationMethod, errorCodes).Defined();
}

bool TStartPhoneVerifyHookAction::Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        NJson::ParseField(raw, "verification_method", NJson::Stringify(VerificationMethod), false, errors)
        && NJson::ParseField(raw, "track_tag", TrackTag, false, errors)
        && NJson::ParseField(raw, "phone", Phone, false, errors);
}

IHookAction::TPtr TStartPhoneVerifyHookAction::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TStartPhoneVerifyHookAction> result(new TStartPhoneVerifyHookAction());
    result->SetVerificationMethod(VerificationMethod);
    result->SetTrackTag(TrackTag);
    result->SetPhone(NChatScript::GetMaybeParametrizedField(Phone, params));
    return result.Release();
}

bool TStartPhoneVerifyHookAction::Perform(const IChatUserContext::TPtr userContext, TChatContext& context, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatTx, const TChatRobotScriptItem& item) const {
    if (!server->GetUserDevicesManager()) {
        return false;
    }
    TSet<TString> errorCodes;
    TDeviceIdVerificationContext divc;
    divc.ApplicationId = userContext->GetApplicationId();
    divc.ClientIp = userContext->GetClientIp();
    divc.EnableGpsPackageName = IRequestProcessor::IsAndroid(userContext->GetPlatform());

    auto userId = userContext->GetUserId();

    auto userFetchResult = server->GetDriveAPI()->GetUsersData()->FetchInfo(userId, session);
    auto userPtr = userFetchResult.GetResultPtr(userId);
    if (!userPtr) {
        chatTx.SetErrorInfo("StartDeviceVerifyHookAction::Perform", "Incorrect userId " + userId);
        return false;
    }

    TPhoneNormalizer normalizer;
    auto phone = normalizer.TryNormalize(userContext->Unescape(Phone, context));

    auto user = *userPtr;
    user.SetPhone(phone);
    user.SetPhoneVerified(false);

    if (!server->GetDriveAPI()->GetUsersData()->UpdateUser(user, userId, session)) {
        return false;
    }

    TString token;
    if (!server->GetUserDevicesManager()->StartPhoneVerification(token, phone, divc, userContext->GetReplyContext(), VerificationMethod, errorCodes)) {
        if (!errorCodes.empty()) {
            auto primaryError = *errorCodes.begin();
            item.AddSpecialValueToContext(context, EContextDataType::PhoneVerificationError, primaryError);
        }
        return true;
    }

    auto tag = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(TrackTag, "Для регистрации " + phone);
    if (!tag) {
        chatTx.SetErrorInfo("StartPhoneVerifyHookAction::Perform", "cannot create tag " + TrackTag);
        return false;
    }
    auto passportTag = dynamic_cast<TPassportTrackTag*>(tag.Get());
    if (!passportTag) {
        chatTx.SetErrorInfo("StartPhoneVerifyHookAction::Perform", "cannot cast tag " + tag->GetName() + " to PassportTrackTag");
        return false;
    }
    passportTag->SetTrackId(token);

    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, userId, userId, server, session)) {
        return false;
    }
    return true;
}

bool TAddTagHookAction::Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    JREAD_STRING(raw, "object", Object);
    if (Object != "!current_car" && Object != "!current_user") {
        errors.AddMessage("add_tag_hook_action", "object is not in {'!current_car', '!current_user'}");
        return false;
    }
    JREAD_STRING(raw, "tag", Tag);
    if (raw.Has("data")) {
        TagData = raw["data"];
    }
    TagData["tag_name"] = Tag;
    return true;
}

IHookAction::TPtr TAddTagHookAction::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TAddTagHookAction> result(new TAddTagHookAction());
    result->SetObject(Object);
    result->SetTag(NChatScript::GetMaybeParametrizedField(Tag, params));
    result->SetTagData(TagData);
    return result.Release();
}

bool TAddTagHookAction::Perform(const IChatUserContext::TPtr userContext, TChatContext& context, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatTx, const TChatRobotScriptItem& /*item*/) const {
    if (!server || !server->GetDriveAPI()) {
        return false;
    }
    NJson::TJsonValue unescapedTagData = userContext->UnescapeJson(TagData, context);
    auto tag = IJsonSerializableTag::BuildFromJson(server->GetDriveAPI()->GetTagsManager(), unescapedTagData);
    if (!tag) {
        chatTx.SetErrorInfo("AddTagHookAction::Perform", TStringBuilder() << "cannot construct tag from " << unescapedTagData);
        return false;
    }

    TString objectId;
    if (Object == "!current_car") {
        TVector<TString> cars;
        if (!server->GetDriveAPI()->GetCurrentSessionsObjects(userContext->GetUserId(), cars, Now()) || cars.size() != 1) {
            return false;
        }
        objectId = cars.front();

        if (!objectId) {
            chatTx.SetErrorInfo("AddTagHookAction::Perform", "cannot get current_car");
            return false;
        }
        return server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(tag, userContext->GetUserId(), objectId, server, session).Defined();
    } else if (Object == "!current_user") {
        objectId = userContext->GetUserId();
        return server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, userContext->GetUserId(), objectId, server, session).Defined();
    }
    return false;
}

bool TModifyTagsAction::Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw["modifications"].IsArray()) {
        errors.AddMessage("ParseAction", "incorrect modifications");
        return false;
    }
    for (const auto& modificationJson : raw["modifications"].GetArray()) {
        TModification modification;
        if (!modification.Parse(modificationJson, errors)) {
            return false;
        }
        Modifications.emplace_back(std::move(modification));
    }
    return NJson::ParseField(raw, "object", Object, true, errors) && NJson::ParseField(raw, "entity", NJson::Stringify(Entity), true, errors);
}

IHookAction::TPtr TModifyTagsAction::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TModifyTagsAction> result(new TModifyTagsAction());
    result->SetObject(NChatScript::GetMaybeParametrizedField(Object, params));
    result->SetEntity(Entity);
    for (const auto& modificationImpl : Modifications) {
        TModification modification;
        modification.SetAction(modificationImpl.GetAction());
        modification.SetTagName(NChatScript::GetMaybeParametrizedField(modificationImpl.GetTagName(), params));
        modification.SetTagData(NChatScript::GetMaybeParametrizedField(modificationImpl.GetTagData(), params));
        result->MutableModifications().emplace_back(std::move(modification));
    }
    return result.Release();
}

ITag::TPtr PatchTagData(ITag::TConstPtr baseTag, const NJson::TJsonValue& patchJson, const NDrive::IServer& server) {
    if (!baseTag) {
        return nullptr;
    }
    NJson::TJsonValue baseJson = baseTag->SerializeToJson();
    return IJsonSerializableTag::BuildFromJson(server.GetDriveAPI()->GetTagsManager(), NJson::MergeJson(patchJson, baseJson));
}

bool TModifyTagsAction::Perform(const IChatUserContext::TPtr userContext, TChatContext& context, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& /*chatsSession*/, const TChatRobotScriptItem& /*item*/) const {
    if (!server || !server->GetDriveAPI()) {
        return false;
    }

    const auto object = userContext->Unescape(Object, context);
    const IEntityTagsManager& entityTagsManager = server->GetDriveAPI()->GetEntityTagsManager(Entity);

    TModificationContext modificationContext(userContext, server);
    for (const auto& modification : Modifications) {
        NJson::TJsonValue newTagJson;
        ITag::TPtr newTag;
        if (modification.GetTagData()) {
            TString unescapedTagData = userContext->Unescape(modification.GetTagData(), context);
            if (!NJson::ReadJsonFastTree(unescapedTagData, &newTagJson)) {
                session.SetErrorInfo("ModifyTagsAction::Perform", "cannot read json from " + unescapedTagData);
                return false;
            }

            if (modification.GetAction() == TAbstractTagsModification::EModificationAction::Add || modification.GetAction() == TAbstractTagsModification::EModificationAction::Evolve || modification.GetAction() == TAbstractTagsModification::EModificationAction::Propose) {
                newTag = IJsonSerializableTag::BuildFromJson(server->GetDriveAPI()->GetTagsManager(), newTagJson);
                if (!newTag) {
                    session.SetErrorInfo("ModifyTagsAction::Perform", "cannot build tag from " + newTagJson.GetStringRobust());
                    return false;
                }
            }
        }

        TVector<TDBTag> tags;
        if (!entityTagsManager.RestoreTags({ object }, { modification.GetTagName() }, tags, session)) {
            ERROR_LOG << "Errors reading tags " << session.GetStringReport() << Endl;
            return false;
        }

        switch (modification.GetAction()) {
        case TAbstractTagsModification::EModificationAction::Add:
            if (!newTag) {
                newTag = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(modification.GetTagName());
                if (!newTag) {
                    session.SetErrorInfo("ModifyTagsAction::Perform", "cannot create tag " + modification.GetTagName());
                    return false;
                }
            }
            modificationContext.AddTag(object, newTag);
            break;
        case TAbstractTagsModification::EModificationAction::Remove:
            for (const auto& dbTag : tags) {
                modificationContext.RemoveTag(object, dbTag);
            }
            break;
        case TAbstractTagsModification::EModificationAction::Evolve:
            if (!newTag) {
                session.SetErrorInfo("ModifyTagsAction::Perform", "new tag is missing");
                return false;
            }
            for (const auto& dbTag : tags) {
                if (!newTag->CopyOnEvolve(*dbTag.GetData(), nullptr, *server)) {
                    session.SetErrorInfo("ModifyTagsAction::Perform", "cannot CopyOnEvolve");
                    return false;
                }
                modificationContext.EvolveTag(object, dbTag, newTag);
            }
            break;
        case TAbstractTagsModification::EModificationAction::Update:
            for (const auto& dbTag : tags) {
                auto patchTag = PatchTagData(dbTag.GetData(), newTagJson, *server);
                if (!patchTag) {
                    session.SetErrorInfo("ModifyTagsAction::Perform", "cannot PatchTagData");
                    return false;
                }
                modificationContext.UpdateTag(object, dbTag, patchTag);
            }
            break;
        case TAbstractTagsModification::EModificationAction::DirectUpdate:
            for (const auto& dbTag : tags) {
                TDBTag newDBTag = dbTag.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
                auto patchTag = PatchTagData(newDBTag.GetData(), newTagJson, *server);
                if (!patchTag) {
                    session.SetErrorInfo("ModifyTagsAction::Perform", "cannot PatchTagData");
                    return false;
                }
                newDBTag.SetData(patchTag);
                modificationContext.UpdateTag(object, newDBTag);
            }
            break;
        case TAbstractTagsModification::EModificationAction::Propose: {
            if (!newTag) {
                newTag = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(modification.GetTagName());
                if (!newTag) {
                    session.SetErrorInfo("ModifyTagsAction::Perform", "cannot create tag " + modification.GetTagName());
                    return false;
                }
            }

            TDBTag dbTag;
            dbTag.SetData(newTag).SetObjectId(object);

            modificationContext.ProposeTag(object, dbTag);
            break;
        }
        }
    }

    return modificationContext.ApplyModification(modificationContext.GetTagModifications().begin(), modificationContext.GetTagModifications().end(), entityTagsManager, session);
}

bool TAlterFlagsHookAction::Parse(const NJson::TJsonValue& raw, TMessagesCollector& /*errors*/) {
    JREAD_FROM_STRING(raw, "flag", Flag);
    JREAD_BOOL(raw, "enabled", Enabled);
    return true;
}

IHookAction::TPtr TAlterFlagsHookAction::GetTemplateImpl(const TVector<TString>& /*params*/) const {
    THolder<TAlterFlagsHookAction> result(new TAlterFlagsHookAction());
    return result.Release();
}

bool TAlterFlagsHookAction::Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */,  const NDrive::IServer* server, NDrive::TEntitySession& /*session*/, NDrive::TEntitySession& chatsSession, const TChatRobotScriptItem& /*item*/) const {
    auto chatRobot = server->GetChatRobot(userContext->GetChatId());
    if (!chatRobot) {
        chatsSession.SetErrorInfo("AlterFlagsHookAction::Perform", "cannot GetChatRobot " + userContext->GetChatId());
        return false;
    }
    NDrive::NChat::TChat chat;
    if (!chatRobot->GetChat(userContext->GetUserId(), userContext->GetChatTopic(), chat, true, chatsSession.GetTransaction())) {
        return false;
    }
    if (Enabled) {
        chat.SetFlag(Flag);
    } else {
        chat.DropFlag(Flag);
    }
    return server->GetChatEngine()->GetChats().Upsert(chat, userContext->GetUserId(), chatsSession);
}


bool TClassifyHookAction::Parse(const NJson::TJsonValue& raw, TMessagesCollector& /*errors*/) {
    JREAD_STRING(raw, "categorizer_node_ids", NodeIdsRaw);
    NodeIds = SplitString(NodeIdsRaw, ",");
    return true;
}

IHookAction::TPtr TClassifyHookAction::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TClassifyHookAction> result(new TClassifyHookAction());
    result->SetNodeIdsRaw(NChatScript::GetMaybeParametrizedField(NodeIdsRaw, params));
    result->SetNodeIds(SplitString(result->GetNodeIdsRaw(), ","));
    return result.Release();
}

bool TClassifyHookAction::Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */,  const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatTx, const TChatRobotScriptItem& /*item*/) const {
    TVector<TDBTag> dbTags;
    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({userContext->GetUserId()}, {}, dbTags, session)) {
        return false;
    }

    auto soughtTopicLink = userContext->GetChatId() + (!!userContext->GetChatTopic() ? ("." + userContext->GetChatTopic()) : "");
    TString tagId;
    for (auto&& tag : dbTags) {
        auto impl = tag.GetTagAs<TSupportChatTag>();
        if (!impl) {
            continue;
        }
        if (impl->GetTopicLink() == soughtTopicLink) {
            tagId = tag.GetTagId();
        }
    }

    auto execute = [&](NDrive::TEntitySession& tx) {
        for (auto&& nodeId : NodeIds) {
            if (!server->GetSupportCenterManager() || !server->GetSupportCenterManager()->GetSupportRequestCategorizer()) {
                return false;
            }
            auto cat = server->GetSupportCenterManager()->GetSupportRequestCategorizer();
            if (!cat->AddCategorization(TSupportRequestCategorization(tagId, nodeId, "automatically from chat script", {}), userContext->GetUserId(), tx)) {
                return false;
            }
        }
        return true;
    };

    if (!execute(chatTx)) {
        return false;
    }

    return true;
}

bool TEndpointCallHookAction::Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw.Has("env") || !raw["env"].IsString() || !TryFromString(raw["env"].GetString(), Env)) {
        errors.AddMessage("endpoint_call_hook_action", "could not parse env");
        return false;
    }

    if (!raw.Has("route") || !raw["route"].IsString()) {
        errors.AddMessage("endpoint_call_hook_action", "could not parse route");
        return false;
    }
    Route = raw["route"].GetString();

    if (raw.Has("cgi") && !raw["cgi"].IsString()) {
        errors.AddMessage("endpoint_call_hook_action", "could not parse cgi");
        return false;
    } else if (raw.Has("cgi")) {
        Cgi = raw["cgi"].GetString();
    }

    if (raw.Has("post") && !raw["post"].IsString()) {
        errors.AddMessage("endpoint_call_hook_action", "could not parse post");
        return false;
    } else if (raw.Has("post")) {
        Post = raw["post"].GetString();
    }

    if (raw.Has("actor_user_id") && !raw["actor_user_id"].IsString()) {
        errors.AddMessage("endpoint_call_hook_action", "could not parse actor_user_id");
        return false;
    } else if (raw.Has("actor_user_id")) {
        ActorUserId = raw["actor_user_id"].GetString();
    }

    if (raw.Has("json_mapper") && !raw["json_mapper"].IsArray()) {
        errors.AddMessage("endpoint_call_hook_action", "could not parse json_mapper");
        return false;
    } else if (raw.Has("json_mapper") && !JsonMapper.DeserializeFromJson(raw["json_mapper"], errors)) {
        return false;
    }

    return true;
}

IHookAction::TPtr TEndpointCallHookAction::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TEndpointCallHookAction> result(new TEndpointCallHookAction());
    result->SetEnv(Env);
    result->SetRoute(NChatScript::GetMaybeParametrizedField(Route, params));
    result->SetCgi(NChatScript::GetMaybeParametrizedField(Cgi, params));
    result->SetPost(NChatScript::GetMaybeParametrizedField(Post, params));
    result->SetActorUserId(NChatScript::GetMaybeParametrizedField(ActorUserId, params));
    return result.Release();
}

bool TEndpointCallHookAction::Perform(const IChatUserContext::TPtr userContext, TChatContext& context, const NDrive::IServer* server, NDrive::TEntitySession& /*session*/, NDrive::TEntitySession& chatTx, const TChatRobotScriptItem& /*item*/) const {
    auto selfRequester = server->GetSelfHttpRequester();
    if (!selfRequester) {
        chatTx.SetErrorInfo("EndpointCallHookAction::Perform", "SelfRequester is missing");
        return false;
    }
    auto requester = selfRequester->GetForEnv(Env);
    if (!requester) {
        chatTx.SetErrorInfo("EndpointCallHookAction::Perform", TStringBuilder() << "SelfRequester is not configured for " << Env);
        return false;
    }

    auto cgi = userContext->Unescape(Cgi, context);
    auto extraCgi = server->GetSettings().GetValueDef<TString>("self_requester.extra_cgi", "");
    if (extraCgi) {
        if (!cgi) {
            cgi = extraCgi;
        } else {
            cgi += "&" + extraCgi;
        }
    }

    NNeh::THttpRequest request;
    request
        .SetUri(userContext->Unescape(Route, context))
        .SetCgiData(cgi)
        .SetPostData(userContext->Unescape(Post, context));

    if (ActorUserId == "[[[!current_user]]]") {
        const auto& whitelist = selfRequester->GetHeaderWhitelist();
        auto headers = userContext->GetReplyContext()->GetRequestData().HeadersIn();
        for (auto&& header : headers) {
            if (!whitelist.empty() && !whitelist.contains(header.first)) {
                continue;
            }
            request.AddHeader(header.first, header.second);
        }
    } else {
        request.AddHeader("Authorization", selfRequester->GetDelegationAuthHeader());
        if (ActorUserId != "[[[!root_user]]]") {
            request.AddHeader("UserIdDelegation", ActorUserId);
        }
    }

    auto result = requester->SendMessageSync(request, Now() + selfRequester->GetRequestTimeout());
    if (result.Code() / 100 != 2) {
        chatTx.AddErrorMessage("EndpointCallHookAction::Perform", TStringBuilder()
            << "SelfRequester failed: " << request.GetDebugRequest() << " " << result.GetDebugReply()
        );
    }

    context.SetLastRequestCode(result.Code());
    context.SetLastRequestData(NJson::TJsonValue::UNDEFINED);
    if (!NJson::ReadJsonFastTree(result.Content(), &context.MutableLastRequestData())) {
        chatTx.SetErrorInfo("EndpointCallHookAction::Perform", "cannot read json from " + result.Content());
        return false;
    }
    JsonMapper.SaveFieldsToContext(context, context.GetLastRequestData());

    return result.Code() / 100 != 5;
}

bool TEditDictionaryHookAction::Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    if (!raw.Has("tag_name") || !raw["tag_name"].IsString()) {
        errors.AddMessage("edit_dictionary_hook_action", "could not parse tag_name");
        return false;
    }
    JREAD_STRING(raw, "tag_name", TagName);

    if (!raw.Has("field_name") || !raw["field_name"].IsString()) {
        errors.AddMessage("edit_dictionary_hook_action", "could not parse field_name");
        return false;
    }
    JREAD_STRING(raw, "field_name", FieldName);

    if (!raw.Has("value") || !raw["value"].IsString()) {
        errors.AddMessage("edit_dictionary_hook_action", "could not parse value");
        return false;
    }
    JREAD_STRING(raw, "value", Value);

    JREAD_BOOL_OPT(raw, "create_if_missing", CreateIfMissing);

    return true;
}

IHookAction::TPtr TEditDictionaryHookAction::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TEditDictionaryHookAction> result(new TEditDictionaryHookAction());
    result->SetTagName(NChatScript::GetMaybeParametrizedField(TagName, params));
    result->SetFieldName(NChatScript::GetMaybeParametrizedField(FieldName, params));
    result->SetValue(NChatScript::GetMaybeParametrizedField(Value, params));
    result->SetCreateIfMissing(CreateIfMissing);
    return result.Release();
}

bool TEditDictionaryHookAction::Perform(const IChatUserContext::TPtr userContext, TChatContext& context, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatTx, const TChatRobotScriptItem& /*item*/) const {
    if (!server) {
        return false;
    }

    auto tagName = userContext->Unescape(TagName, context);
    auto targetField = userContext->Unescape(FieldName, context);
    auto targetValue = userContext->Unescape(Value, context);

    TVector<TDBTag> dbTags;
    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({userContext->GetUserId()}, {tagName}, dbTags, session)) {
        return false;
    }
    if (dbTags.size() == 0 && !CreateIfMissing) {
        chatTx.SetErrorInfo("EditDictionaryHookAction::Perform", "cannot restore tag " + tagName);
        return false;
    }
    if (dbTags.size() == 0) {
        auto tag = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(tagName, "created from chat");
        if (!tag) {
            chatTx.SetErrorInfo("EditDictionaryHookAction::Perform", "cannot create tag " + tagName);
            return false;
        }
        auto impl = dynamic_cast<TUserDictionaryTag*>(tag.Get());
        if (!impl) {
            chatTx.SetErrorInfo("EditDictionaryHookAction::Perform", "cannot cast tag " + tag->GetName() + " to UserDictionaryTag");
            return false;
        }
        impl->SetField(targetField, targetValue);
        return server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, userContext->GetUserId(), userContext->GetUserId(), server, session).Defined();
    }

    auto tag = dbTags.front();
    auto impl = tag.MutableTagAs<TUserDictionaryTag>();
    if (!impl) {
        chatTx.SetErrorInfo("EditDictionaryHookAction::Perform", "cannot cast tag " + tag->GetName() + " to UserDictionaryTag");
        return false;
    }
    impl->SetField(targetField, targetValue);
    return server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(tag, userContext->GetUserId(), session);
}

bool TSendComplaintPhotos::Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        NJson::ParseField(raw, "car_tag_id", CarTagId, true, errors) &&
        NJson::ParseField(raw, "source_name", SourceName, errors) &&
        NJson::ParseField(raw, "check_confirmed", CheckConfirmed, false, errors);
}

IHookAction::TPtr TSendComplaintPhotos::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TSendComplaintPhotos> result(new TSendComplaintPhotos());
    result->SetCarTagId(NChatScript::GetMaybeParametrizedField(CarTagId, params));
    result->SetSourceName(NChatScript::GetMaybeParametrizedField(SourceName, params));
    result->SetCheckConfirmed(CheckConfirmed);
    return result.Release();
}

bool TSendComplaintPhotos::Perform(const IChatUserContext::TPtr userContext, TChatContext& context, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatsSession, const TChatRobotScriptItem& /*item*/) const {
    if (!server || !server->GetDriveAPI()->HasMDSClient()) {
        return false;
    }
    auto chatRobot = server->GetChatRobot(userContext->GetChatId());
    if (!chatRobot) {
        chatsSession.SetErrorInfo("SendComplaintPhotos::Perform", "cannot GetChatRobot " + userContext->GetChatId());
        return false;
    }
    auto tagId = userContext->Unescape(CarTagId, context);
    auto sourceName = userContext->Unescape(SourceName, context);
    auto optionalTag = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTag(tagId, session);
    if (!optionalTag) {
        return false;
    }
    auto impl = optionalTag->MutableTagAs<TCarComplaintTag>();
    if (!impl) {
        chatsSession.SetErrorInfo("SendComplaintPhotos::Perform", "cannot cast tag " + optionalTag.GetRef()->GetName() + " to CarComplaintTag");
        return false;
    }
    TVector<NDrive::NChat::TMessage> messages;
    for (auto&& source : impl->GetComplaintSources()) {
        if (CheckConfirmed && !source->IsConfirmed() || !sourceName.empty() && source->GetSourceName() != sourceName) {
            continue;
        }
        for (auto&& imagePath : source->GetImagePaths()) {
            messages.emplace_back(imagePath, 0, NDrive::NChat::IMessage::EMessageType::Image);
        }
    }
    if (!messages.empty()) {
        return chatRobot->AcceptMessages(userContext->GetUserId(), userContext->GetChatTopic(), messages, chatRobot->GetChatConfig().GetRobotUserId(), chatsSession);
    }
    return true;
}

bool TSetDocumentCheckStatus::Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
    return
        NJson::ParseField(raw, "check_types", CheckTypes, true, errors) &&
        NJson::ParseField(raw, "status", Status, true, errors);
}

IHookAction::TPtr TSetDocumentCheckStatus::GetTemplateImpl(const TVector<TString>& params) const {
    THolder<TSetDocumentCheckStatus> result(new TSetDocumentCheckStatus());
    result->SetCheckTypes(CheckTypes);
    result->SetStatus(NChatScript::GetMaybeParametrizedField(Status, params));
    return result.Release();
}

bool TSetDocumentCheckStatus::Perform(const IChatUserContext::TPtr userContext, TChatContext& context, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatsSession, const TChatRobotScriptItem& /*item*/) const {
    Y_UNUSED(session);
    TString actualStatus = userContext->Unescape(Status, context);
    if (!server || !server->GetUserDocumentsChecksManager()) {
        return false;
    }
    for (auto&& type : CheckTypes) {
        TString actualType = userContext->Unescape(type, context);
        if (!server->GetUserDocumentsChecksManager()->SetStatus(userContext->GetUserId(), actualType, actualStatus, userContext->GetUserId(), *server, chatsSession)) {
            return false;
        }
    }
    return true;
}
