#pragma once

#include "ifaces.h"

#include <drive/backend/chat_robots/configuration/chat_script.h>

#include <drive/backend/chat/engine.h>
#include <drive/backend/tags/tag_modification.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/user_devices/manager.h>
#include <drive/backend/users/user_documents.h>

#include <drive/library/cpp/self_requester/client.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

class TProposeCarTagHookAction : public IHookAction {
    R_FIELD(TString, Object);
    R_FIELD(TString, Tag);

    static TFactory::TRegistrator<TProposeCarTagHookAction> Registrator;

public:
    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

    virtual bool Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& /*chatsSession*/, const TChatRobotScriptItem& /*item*/) const override;
};

class TAbstractFeedbackHookAction : public IHookAction {
    R_FIELD(TString, Value);

protected:
    virtual bool PatchTag(TDBTag& /*tag*/) const {
        return true;
    }

public:
    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override = 0;

    virtual bool Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& /*chatsSession*/, const TChatRobotScriptItem& /*item*/) const override;
};

class TFeedbackOverallHookAction : public TAbstractFeedbackHookAction {
    static TFactory::TRegistrator<TFeedbackOverallHookAction> Registrator;

protected:
    virtual bool PatchTag(TDBTag& tag) const override;

public:
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TFeedbackDetailsHookAction : public TAbstractFeedbackHookAction {
    static TFactory::TRegistrator<TFeedbackDetailsHookAction> Registrator;

protected:
    virtual bool PatchTag(TDBTag& tag) const override;

public:
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override;
};

class TSetDocumentStatusesHookAction : public IHookAction {
    R_FIELD(TString, Statuses);

    static TFactory::TRegistrator<TSetDocumentStatusesHookAction> Registrator;

    static const TVector<NUserDocument::EType> PhotosOrder;
    TVector<NUserDocument::EVerificationStatus> VStatus;

public:
    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

    virtual bool Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& /*chatsSession*/, const TChatRobotScriptItem& /*item*/) const override;
};

class TStartDeviceVerifyHookAction : public IHookAction {
    static TFactory::TRegistrator<TStartDeviceVerifyHookAction> Registrator;
    R_FIELD(IUserDevicesManager::EVerificationMethod, VerificationMethod, IUserDevicesManager::EVerificationMethod::Sms);

public:
    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

    virtual bool Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& /*chatsSession*/, const TChatRobotScriptItem& /*item*/) const override;
};

class TStartPhoneVerifyHookAction : public IHookAction {
    static TFactory::TRegistrator<TStartPhoneVerifyHookAction> Registrator;
    R_FIELD(IUserDevicesManager::EVerificationMethod, VerificationMethod, IUserDevicesManager::EVerificationMethod::Sms);
    R_FIELD(TString, TrackTag, TPassportTrackTag::TypeName);
    R_FIELD(TString, Phone);

public:
    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

    virtual bool Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& /*chatsSession*/, const TChatRobotScriptItem& item) const override;
};

class TAddTagHookAction : public IHookAction {
    static TFactory::TRegistrator<TAddTagHookAction> Registrator;

    R_FIELD(TString, Object);
    R_FIELD(TString, Tag);
    R_FIELD(NJson::TJsonValue, TagData);

public:
    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

    virtual bool Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& /*chatsSession*/, const TChatRobotScriptItem& /*item*/) const override;
};

class TModifyTagsAction : public IHookAction {
public:
    class TModification {
        R_FIELD(TAbstractTagsModification::EModificationAction, Action);
        R_FIELD(TString, TagName);
        R_FIELD(TString, TagData);

    public:
        bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) {
            return NJson::ParseField(raw, "action", NJson::Stringify(Action), true, errors)
                && NJson::ParseField(raw, "tag_name", TagName, true, errors)
                && NJson::ParseField(raw, "tag_data", TagData, Action == TAbstractTagsModification::EModificationAction::Evolve || Action == TAbstractTagsModification::EModificationAction::Update, errors);
        }
    };

private:
    class TModificationContext : public ITagsModificationContextImplBase {
        using TBase = ITagsModificationContextImplBase;
        const IChatUserContext::TPtr Context;
    public:
        TModificationContext(const IChatUserContext::TPtr userContext, const NDrive::IServer* server)
            : TBase(*server)
            , Context(userContext)
        {}

        virtual const TString& GetProcessName() const override {
            return Context ? Context->GetChatTopic() : Default<TString>();
        }

        virtual const TString& GetRobotUserId() const override {
            return Context ? Context->GetUserId() : Default<TString>();
        }
    };

private:
    R_FIELD(TVector<TModification>, Modifications);
    R_FIELD(TString, Object);
    R_FIELD(NEntityTagsManager::EEntityType, Entity);

    static TFactory::TRegistrator<TModifyTagsAction> Registrator;

public:
    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

    virtual bool Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& /*chatsSession*/, const TChatRobotScriptItem& /*item*/) const override;
};

class TAlterFlagsHookAction : public IHookAction {
    static TFactory::TRegistrator<TAlterFlagsHookAction> Registrator;

    R_FIELD(NDrive::NChat::TChat::EChatFlags, Flag);
    R_FIELD(bool, Enabled, false);

public:
    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

    virtual bool Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& /*chatsSession*/, const TChatRobotScriptItem& /*item*/) const override;
};

class TClassifyHookAction : public IHookAction {
    static TFactory::TRegistrator<TClassifyHookAction> Registrator;

    R_FIELD(TString, NodeIdsRaw);
    R_FIELD(TVector<TString>, NodeIds);

public:
    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

    virtual bool Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& /*chatsSession*/, const TChatRobotScriptItem& /*item*/) const override;
};

class TEndpointCallHookAction : public IHookAction {
private:
    static TFactory::TRegistrator<TEndpointCallHookAction> Registrator;

    R_FIELD(TSelfHttpRequesterConfig::EEnv, Env, TSelfHttpRequesterConfig::EEnv::TestLocal);
    R_FIELD(TString, Route);
    R_FIELD(TString, Cgi);
    R_FIELD(TString, Post);

    R_FIELD(TString, ActorUserId, "[[[!current_user]]]");
    R_FIELD(TJsonMapper, JsonMapper);

public:
    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

    virtual bool Perform(const IChatUserContext::TPtr userContext, TChatContext& context, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& /*chatsSession*/, const TChatRobotScriptItem& /*item*/) const override;
};

class TEditDictionaryHookAction : public IHookAction {
private:
    static TFactory::TRegistrator<TEditDictionaryHookAction> Registrator;

    R_FIELD(TString, TagName);
    R_FIELD(TString, FieldName);
    R_FIELD(TString, Value);
    R_FIELD(bool, CreateIfMissing, false);

public:
    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

    virtual bool Perform(const IChatUserContext::TPtr userContext, TChatContext& /* context */, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& /*chatsSession*/, const TChatRobotScriptItem& /*item*/) const override;
};

class TSendComplaintPhotos: public IHookAction {
private:
    static TFactory::TRegistrator<TSendComplaintPhotos> Registrator;

    R_FIELD(TString, SourceName);
    R_FIELD(TString, CarTagId);
    R_FIELD(bool, CheckConfirmed, true);

public:
    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

    virtual bool Perform(const IChatUserContext::TPtr userContext, TChatContext& context, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatsSession, const TChatRobotScriptItem& /*item*/) const override;
};

class TSetDocumentCheckStatus: public IHookAction {
private:
    static inline TFactory::TRegistrator<TSetDocumentCheckStatus> Registrator{IHookAction::EType::SetDocumentsCheckStatus};

    R_FIELD(TSet<TString>, CheckTypes);
    R_FIELD(TString, Status);

public:
    virtual bool Parse(const NJson::TJsonValue& raw, TMessagesCollector& errors) override;
    virtual IHookAction::TPtr GetTemplateImpl(const TVector<TString>& params) const override;

    virtual bool Perform(const IChatUserContext::TPtr userContext, TChatContext& context, const NDrive::IServer* server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatsSession, const TChatRobotScriptItem& /*item*/) const override;
};
