#pragma once

#include "media_resource.h"

#include <drive/backend/chat_robots/configuration/context.h>

#include <drive/backend/chat/engine.h>
#include <drive/backend/processor/processor.h>

#include <drive/library/cpp/searchserver/context/replier.h>

#include <library/cpp/http/misc/httpcodes.h>

#include <util/generic/string.h>

class IChatRobot;
class TBlob;
class TChatRobotConfig;
class TChatRobotScriptItem;
class TConstDBTag;
class TGeoCoord;
class TInstant;
class TUserPermissions;

class TChatContext;

class IChatUserContext {
private:
    using TDictMapping = TMap<TString, TString>;
    using TDictMappings = TVector<TDictMapping>;
    using TDictMappingsByName = TMap<TString, TDictMappings>;
    using TTagNameCountMap = TMap<TString, ui32>;

private:
    R_FIELD(TString, ApplicationId);
    R_FIELD(TString, ChatId);
    R_FIELD(TString, ChatTopic);
    R_FIELD(TString, Message);
    R_FIELD(TString, DeviceId);
    R_FIELD(TString, Phone);
    R_FIELD(TString, ClientIp);
    R_FIELD(TString, Origin);
    R_FIELD(IReplyContext::TPtr, ReplyContext);
    R_FIELD(TString, UserId);
    R_FIELD(IRequestProcessor::EApp, Platform, IRequestProcessor::EApp::Unknown);
    R_FIELD(ui32, AppBuild, 0);
    R_FIELD(TString, RegistrationStep);
    R_FIELD(ELocalization, Locale, DefaultLocale);
    R_READONLY(TSet<TString>, AreaIds);
    R_READONLY(TSet<TString>, PerformedTags);
    R_READONLY(TTagNameCountMap, UserTags);
    R_READONLY(TDictMappingsByName, DictionaryTagsData);
    R_FIELD(bool, Debtor, false);
    R_FIELD(bool, FirstRiding, true);
    R_FIELD(TSet<TString>, DefaultEmails);

public:
    using TPtr = TAtomicSharedPtr<IChatUserContext>;

public:
    virtual void SpecifyGeo(const TGeoCoord& c) = 0;
    virtual const TString GetStatus() const = 0;
    virtual bool HasAction(const TString& action) const = 0;

    virtual TAtomicSharedPtr<IChatRobot> GetChatRobot() const = 0;

    virtual TString Unescape(TString pattern, const TChatContext& context) const = 0;
    virtual NJson::TJsonValue UnescapeJson(const NJson::TJsonValue& json, const TChatContext& context) const = 0;
    virtual TIntrusiveConstPtr<TUserPermissions> GetUserPermissions() const = 0;

    virtual ~IChatUserContext() = default;

    virtual const NDrive::IServer& GetServer() const = 0;
};

class IChatResponsePostProcessingCallback {
public:
    virtual void OnSuccess() = 0;
    virtual void OnFailure(const TString& errorMessage, const HttpCodes code = HttpCodes::HTTP_INTERNAL_SERVER_ERROR) = 0;
    virtual ~IChatResponsePostProcessingCallback() = default;
};

class IChatMediaResourcePostUploadCallback {
public:
    virtual void OnSuccess(const TString& resourceId) = 0;
    virtual void OnFailure(const TString& errorMessage, const HttpCodes code = HttpCodes::HTTP_INTERNAL_SERVER_ERROR) = 0;
    virtual ~IChatMediaResourcePostUploadCallback() = default;
};

class IChatMediaResourceAcquisitionCallback {
public:
    virtual void OnSuccess(const TBlob& content, const TString& contentType) = 0;
    virtual void OnFailure(const TString& errorMessage, const HttpCodes code = HttpCodes::HTTP_INTERNAL_SERVER_ERROR) = 0;

    virtual void OnRedirect(const TString& /*location*/) {
    }

    virtual ~IChatMediaResourceAcquisitionCallback() = default;
};

struct TMessageAttachment {
    TString Data;
    TString ContentType;

    TMessageAttachment() = default;
    TMessageAttachment(const TString& data, const TString& contentType = "")
        : Data(data)
        , ContentType(contentType)
    {
    }
};

class TNextActionInfo {
    R_FIELD(TString, NextNodeId);
    R_FIELD(TString, DocumentRevision);
    R_FIELD(bool, Failed, false);
    R_FIELD(NDrive::NChat::TMessage, Message);
    R_FIELD(TVector<TMessageAttachment>, Attachments);

public:
    TNextActionInfo() = default;

    TNextActionInfo(const NDrive::NChat::TMessage& message, TVector<TMessageAttachment> attachments = {})
        : Message(message)
        , Attachments(attachments)
    {
    }

    TNextActionInfo(const TString& nextNodeId)
        : NextNodeId(nextNodeId)
    {
    }

    void AddAttachment(const TString& attachment) {
        Attachments.push_back({attachment});
    }
};

class IChatRobot {
    using TMessage = NDrive::NChat::TMessage;
    using TChat = NDrive::NChat::TChat;

public:
    using TPtr = TAtomicSharedPtr<IChatRobot>;

public:
    virtual TMaybe<TVector<NDrive::NChat::TChat>> GetChats(const TSet<TString> searchIds, NDrive::TEntitySession& session) const = 0;
    virtual TMaybe<TVector<NDrive::NChat::TChat>> GetChats(const TString& chatId, const TString& topic, NDrive::TEntitySession& session) const = 0;
    virtual bool HasChat(const TString& userId, const TString& topic, NDrive::TEntitySession* sessionExt = nullptr) const = 0;
    virtual bool EnsureChat(const TString& userId, const TString& topic, NDrive::TEntitySession& session, const bool isRead = false, TVector<TContextMapInfo> contextMap = {}) const = 0;
    virtual bool ResetChat(const TString& userId, const TString& topic) const = 0;
    virtual bool CreateCleanChat(const TString& userId, const TString& topic) const = 0;
    virtual bool RemoveChat(const TString& userId, const TString& topic, const TString& operatorId) const = 0;
    virtual bool MoveToStep(const TString& userId, const TString& topic, const TString& stepName, NDrive::TEntitySession* sessionExternal = nullptr, const bool sendMessages = false, const TString& newChatTitle = "", const TString& operatorId = "robot-frontend", const TMaybe<ui64> lastMessageId = {}, TVector<TContextMapInfo> contextMap = {}) const = 0;
    virtual bool GetIsChatCompleted(const TString& userId, const TString& topic, const TInstant actuality) const = 0;
    virtual bool GetIsChatInClosedNode(const TString& userId, const TString& topic, const TInstant actuality) const = 0;
    virtual bool GetCurrentScriptItem(const TString& userId, const TString& topic, TChatRobotScriptItem& result, const TInstant actuality) const = 0;
    virtual TMaybe<TNextActionInfo> GetFirstResubmitStep(const IChatUserContext::TPtr context, TChatContext& stateContext, NDrive::TEntitySession& chatSession, ui32 maskOverride) const = 0;
    virtual TMaybe<TNextActionInfo> GetNextResubmitStep(const IChatUserContext::TPtr context, const TChatRobotScriptItem& currentScriptItem, TChatContext& stateContext, NDrive::TEntitySession& chatSession) const = 0;
    virtual ui64 GetLastViewedMessageIdExcept(const TString& userId, const TString& topic, const TString& exceptUserId) const = 0;
    virtual ui64 GetLastViewedMessageId(const TString& userId, const TString& topic, const TString& viewerId) const = 0;
    virtual bool GetOrCreateChat(const TString& userId, const TString& topic, ui32& resultState, TChatContext& resultContext, NDrive::TEntitySession& session) const = 0;
    virtual bool UpdateChat(const TChatContext& chatContext, const TString& userId, const TString& topic, const TString& currentStep, NDrive::TEntitySession& session) const = 0;

    virtual TString UnescapeMacroses(ELocalization locale, const TString& rawText, const TString& userId, const TString& topic, const TString& prefix = "", const TString& suffix = "", const TChatContext* context = nullptr) const = 0;

    virtual TVector<std::pair<TString, TString>> GetUsersInNodes(const TString& topic, const TSet<TString>& nodeNames) const = 0;
    virtual bool GetHistoryStatsByNodes(const TDuration& windowSize, TMap<TString, size_t>& result) const = 0;

    virtual NJson::TJsonValue GetCurrentActionMessagesReport(ELocalization locale, const TString& userId, const TString& topic, bool withMetainfo = false, const TInstant timestampOverride = TInstant::Zero()) const = 0;
    virtual bool GetActualStateId(const TString& userId, const TString& topic, TString& result) const = 0;
    virtual bool GetChatContextByChat(const TString& /*userId*/, const TString& /*topic*/, TChatContext& /*result*/, const TInstant /*actuality*/) const {
        return true;
    }
    virtual bool RefreshStates(const TInstant actiality) const = 0;

    virtual TSet<TString> GetNonRobotParticipants(const NDrive::NChat::TMessageEvents& messages) const = 0;
    virtual TMaybe<size_t> GetMessagesCount(const TString& userId, const TString& topic, const NDrive::NChat::TMessage::EMessageType type, NDrive::TEntitySession& session) const = 0;

    virtual bool DoPreEntryActions(const IChatUserContext::TPtr userContext, const TChatRobotScriptItem& item, TChatContext& context, NDrive::TEntitySession& tagsSession, NDrive::TEntitySession& chatsSession) const = 0;
    virtual bool SendPreActionMessages(const ui32 chatRoomId, const TString& chatItemId, NDrive::TEntitySession& session, const TString& operatorId, const TMaybe<TChatContext>& chatContext = {}, const TMaybe<ui64> lastMessageId = {}) const = 0;

    // The only methods to be implemented in actual chat
    virtual NThreading::TFuture<void> AcceptUserResponse(const IChatUserContext::TPtr context, const TMessage& message, const TVector<TMessageAttachment>& attachments, const TString& operatorId) const = 0;
    virtual bool MaybeContinueChat(const IChatUserContext::TPtr context, const TString& operatorId, TChatContext& stateContext) const = 0;
    virtual void AcceptMediaResource(const IChatUserContext::TPtr context, const TString& resourceId, const TString& contentType, const TString& content, TAtomicSharedPtr<IChatMediaResourcePostUploadCallback> callback) const = 0;
    virtual NThreading::TFuture<TChatResourceAcquisitionResult> GetMediaResource(const IChatUserContext::TPtr context, const TString& resourceId, const bool needsFurtherProcessing) const = 0;
    virtual bool RegisterMediaResource(const TString& userId, const TString& resourceId, const TString& contentType, const bool shared, NDrive::TEntitySession& session) const = 0;
    virtual bool UpdateMediaResourceDescription(const TMediaResourceDescription description, const TString& actorUserId, NDrive::TEntitySession& session) const = 0;
    virtual NThreading::TFuture<void> UploadResourcePreview(const TString& content, const TMediaResourceDescription& description, const TString& contentType) const = 0;

    virtual bool AcceptMessages(const TString& userId, const TString& topic, TVector<TMessage>& messages, const TString& operatorId, NDrive::TEntitySession& session) const = 0;

    // Override for actual previews
    virtual NThreading::TFuture<TChatResourceAcquisitionResult> GetMediaResourcePreview(const IChatUserContext::TPtr context, const TString& resourceId, const bool needsFurtherProcessing, const TMap<TString, TString>& typeResourceOverrides) const = 0;

    virtual bool ArchiveChat(const TString& userId, const TString& topic) const = 0;

    virtual NDrive::TEntitySession BuildChatEngineSession(const bool readOnly = false) const = 0;

    virtual TString GetChatTitle(const TString& userId, const TString& topic) const = 0;
    virtual TString GetChatTitle(const NDrive::NChat::TChat& chat) const = 0;
    virtual bool OverrideTitle(const TString& userId, const TString& topic, const TString& newTitle, NDrive::TEntitySession& session) const = 0;
    virtual TString GetChatIcon(const NDrive::NChat::TChat& chat) const = 0;

    virtual bool GetChat(const TString& userId, const TString& topic, NDrive::NChat::TChat& chat, const bool fallback = true, NStorage::ITransaction::TPtr transactionExt = nullptr) const = 0;
    virtual TString GetFaqUrl() const = 0;
    virtual TString GetSupportUrl() const = 0;

    virtual const TChatRobotConfig& GetChatConfig() const = 0;

    virtual bool IsExistsForUser(const TString& userId, const TString& topic) const = 0;
    virtual bool DeduceChatContinuation(const TString& /*userId*/, const TString& /*topic*/, const ui32 /*chatRoomId*/, NDrive::TEntitySession& /*session*/) const = 0;

    virtual bool SendArbitraryMessage(const TString& userId, const TString& topic, const TString& operatorId, TMessage& message, NDrive::TEntitySession& session, const TChatContext* externalContext = nullptr) const = 0;
    virtual bool EditMessage(const TString& operatorId, const ui64 messageId, const NDrive::NChat::TMessageEdit messageEdit, NDrive::TEntitySession& session) const = 0;
    virtual bool DeleteMessage(const TString& operatorId, const ui64 message, NDrive::TEntitySession& session) const = 0;

    virtual bool SetChatOperatorName(const TString& userId, const TString& topic, const TString& operatorName, NDrive::TEntitySession& session) const = 0;
    virtual bool AddToContext(const TVector<TContextMapInfo>& values, const TString& userId, const TString& topic, NDrive::TEntitySession& session) const = 0;

    virtual NDrive::NChat::TOptionalMessageEvents GetChatMessages(const TString& userId, const TString& topic, NDrive::TEntitySession& session, const ui32 viewerTraits = NDrive::NChat::TMessage::AllKnownTraits, const ui64 startId = 0) const = 0;
    virtual NDrive::NChat::TOptionalMessageEvents GetChatMessagesSinceTimestamp(const TString& userId, const TString& topic, const TInstant startTs, NDrive::TEntitySession& session, const ui32 viewerTraits = NDrive::NChat::TMessage::AllKnownTraits) const = 0;
    virtual NDrive::NChat::TOptionalMessageEvents GetChatMessagesRange(const TString& userId, const TString& topic, const ui32 viewerTraits, const ui64 startId, const ui64 finishId, NDrive::TEntitySession& session) const = 0;
    virtual bool MarkMessagesRead(const NDrive::NChat::TMessageEvent& lastMessage, const TString& userId, NDrive::TEntitySession& session) const = 0;

    virtual NDrive::NChat::TExpectedMessageEvents GetCachedMessages(const TString& searchId) const = 0;
    virtual bool UpdateCachedMessages(const TSet<TString>& searchIds, NDrive::TEntitySession& session, const TInstant& requestTime) const = 0;

    virtual TMaybe<NDrive::NChat::TChatMessages> GetUserChatsMessages(const TVector<TChat>& chats, NDrive::TEntitySession& session, const TRange<ui64>& idRange = {}, const TRange<TInstant>& timestampRange = {}, const ui32 viewerTraits = NDrive::NChat::TMessage::AllKnownTraits) const = 0;
    virtual NDrive::NChat::TOptionalMessageEvents GetUnreadMessages(const ui32 chatId, const TString& userId, const ui32 viewerTraits, NDrive::TEntitySession& session) const = 0;

    virtual bool GetLastMessage(const ui32& chatRoomId, NDrive::TEntitySession& session, const ui32 viewerTraits, TMaybe<NDrive::NChat::TMessageEvent>& result, const TString& messageFromUserId = "") const = 0;
    virtual bool GetLastMessage(const TString& chatUserId, const TString& topic, NDrive::TEntitySession& session, const ui32 messageTraits, TMaybe<NDrive::NChat::TMessageEvent>& result, const TString& messageFromUserId = "") const = 0;

    virtual NJson::TJsonValue GetMessageReport(ELocalization locale, const NDrive::NChat::TMessageEvent& message, const TString& userId, const TString& topic) const = 0;

    virtual TMaybe<ui64> GetFirstEventIdByTime(const TString& userId, const TString& topic, const TInstant since, const bool isGoingBack, NDrive::TEntitySession& session) const = 0;

    virtual TMaybe<size_t> GetMessagesCount(const ui32 chatId, const ui32 viewerTraits, NDrive::TEntitySession& session) const = 0;
    virtual TMaybe<size_t> GetUnreadMessagesCount(const ui32 chatId, const TString& userId, const ui32 viewerTraits, NDrive::TEntitySession& session) const = 0;
    virtual bool UpdateLastViewedMessageId(const TString& userId, const TString& topic, const TString& viewerId, const ui64 sentMessageId, NDrive::TEntitySession& session) const = 0;

    virtual bool RefreshEngine(const TInstant actiality) const = 0;
    virtual bool Refresh(const TInstant actuality) const = 0;
    virtual TVector<TChat> GetTopics(IChatUserContext::TPtr ctx, bool onlyActive = false) const = 0;
    virtual TString GetChatSearchId(const TString& userId, const TString& topic) const = 0;

    virtual void CheckIn(const TString& /*userId*/, const TString& /*operatorUserId*/, const TString& /*topic*/, const TString& /*origin*/, const TString& /*externalUserId*/, const TGeoCoord* /*coord*/, const TInstant /*actuality*/) const = 0;

    static void ParseTopicLink(const TString& topicLink, TString& chatId, TString& topic) {
        if (topicLink.find(".") != TString::npos) {
            size_t separatorPos = topicLink.find(".");
            topic = topicLink.substr(separatorPos + 1, topicLink.size() - separatorPos - 1);
            chatId = topicLink.substr(0, separatorPos);
        } else {
            chatId = topicLink;
            topic = "";
        }
    }

    virtual ~IChatRobot() = default;
};

namespace NDrive::NChat{
    struct TBadRequestException: public virtual yexception {
    };

    struct TAccessForbiddenException: public virtual yexception {
    };
}
