#include "media_storage.h"

#include "abstract.h"

#include <drive/library/cpp/image_transformation/resize.h>
#include <drive/library/cpp/image_transformation/video_first_frame.h>
#include <drive/library/cpp/mds/client.h>

class TUploadCallbackWrapper : public TS3Client::IBaseCallback {
private:
    TAtomicSharedPtr<IChatMediaResourcePostUploadCallback> HighLevelCallback;
    const TSimpleUnencryptedMediaStorage* Storage;
    const TSimpleUnencryptedMediaStorageTable* Table;
    const TChatRobotMediaHistoryManager* HistoryWriter;
    TMediaResourceDescription Descr;

public:
    TUploadCallbackWrapper(TAtomicSharedPtr<IChatMediaResourcePostUploadCallback> highLevelCallback, const TSimpleUnencryptedMediaStorage* storage, const TSimpleUnencryptedMediaStorageTable* table, const TChatRobotMediaHistoryManager* historyWriter, const TMediaResourceDescription descr)
        : TS3Client::IBaseCallback(TS3Client::ERequestType::PutKey)
        , HighLevelCallback(highLevelCallback)
        , Storage(storage)
        , Table(table)
        , HistoryWriter(historyWriter)
        , Descr(descr)
    {
        Y_UNUSED(Storage);
    }

private:
    void DoOnSuccess(const TString& /*path*/, const THttpReplyData<TString>& /*reply*/) override {
        Descr.SetUploadedAt(Now());
        auto session = Table->template BuildTx<NSQL::Writable>();
        if (!Table->Upsert(Descr, session) || !HistoryWriter->AddHistory(Descr, Descr.GetUserId(), EObjectHistoryAction::UpdateData, session) || !session.Commit()) {
            HighLevelCallback->OnFailure("cannot update db entry: " + session.GetStringReport());
            return;
        }
        HighLevelCallback->OnSuccess(Descr.GetId());
    }

    void DoOnFailure(const TString& /*path*/, const THttpReplyData<TString>& reply) override {
        HighLevelCallback->OnFailure(TStringBuilder() << "mds update failed: " << reply.GetHttpCode() << " " << reply.GetReportSafe());
    }
};

bool TSimpleUnencryptedMediaStorage::DoRebuildCacheUnsafe() const {
    UserToResources.clear();

    NStorage::TObjectRecordsSet<TRecordType> records;
    {
        auto table = Database->GetTable(Table->GetTableName());
        auto transaction = Database->CreateTransaction(true);
        auto result = table->GetRows("", records, transaction);

        if (!result->IsSucceed()) {
            ERROR_LOG << "Cannot refresh data for " << Table->GetTableName() << Endl;
            return false;
        }
    }

    for (auto&& record : records) {
        if (record.GetShared()) {
            SharedResources[record.GetId()] = record;
        } else {
            UserToResources[record.GetUserId()].emplace(record.GetId(), record);
        }
    }

    return true;
}

bool TSimpleUnencryptedMediaStorage::DoAcceptHistoryEventUnsafe(const TAtomicSharedPtr<TObjectEvent<TRecordType>>& ev, const bool /*isNew*/) {
    if (ev->GetShared()) {
        SharedResources[ev->GetId()] = *ev;
    } else {
        UserToResources[ev->GetUserId()][ev->GetId()] = *ev;
    }
    return true;
}

TStringBuf TSimpleUnencryptedMediaStorage::GetEventObjectId(const TObjectEvent<TRecordType>& object) const {
    return object.GetId();
}

bool TSimpleUnencryptedMediaStorage::GetResourceMeta(TMediaResourceDescription& descr, const TString& userId, const TString& resourceId, const TInstant freshness) const {
    if (!RefreshCache(freshness)) {
        return false;
    }
    auto rg = MakeObjectReadGuard();
    {
        auto sharedResourceIt = SharedResources.find(resourceId);
        if (sharedResourceIt != SharedResources.end()) {
            descr = sharedResourceIt->second;
            return true;
        }
    }
    auto userIt = UserToResources.find(userId);
    if (userIt == UserToResources.end()) {
        return false;
    }
    auto it = userIt->second.find(resourceId);
    if (it == userIt->second.end() || it->second.GetUserId() != userId) {
        return false;
    }
    descr = it->second;
    return true;
}

TMaybe<TMediaResourceDescription> TSimpleUnencryptedMediaStorage::RegisterResource(const TString& userId, const TString& resourceId, const TString& contentType, const bool shared, NDrive::TEntitySession& session) const {
    {
        auto rg = MakeObjectReadGuard();
        auto userIt = UserToResources.find(userId);
        if (userIt != UserToResources.end()) {
            auto resourceIt = userIt->second.find(resourceId);
            if (resourceIt != userIt->second.end()) {
                return resourceIt->second;
            }
        }
    }
    TMediaResourceDescription descr(resourceId, userId);
    descr.SetShared(shared);
    descr.SetContentType(contentType);
    if (Table->Insert(descr, session) && HistoryWriter->AddHistory(descr, userId, EObjectHistoryAction::Add, session)) {
        return descr;
    }
    return Nothing();
}

void TSimpleUnencryptedMediaStorage::UploadResource(const TString& userId, const TString& resourceId, const TString& contentType, const TString& content, TAtomicSharedPtr<IChatMediaResourcePostUploadCallback> callback) const {
    if (!content.size()) {
        callback->OnFailure("content is empty");
        return;
    }
    TMediaResourceDescription descr;
    if (!GetResourceMeta(descr, userId, resourceId, Now())) {
        callback->OnFailure("can't get description");
        return;
    }

    auto bucket = MdsClient.GetBucket(Config.GetBucketName());
    if (!bucket) {
        callback->OnFailure("bucket not defined");
        return;
    }

    descr.SetContentType(contentType);
    TAtomicSharedPtr<TS3Client::IBaseCallback> callbackWrap(new TUploadCallbackWrapper(callback, this, Table.Get(), HistoryWriter.Get(), descr));
    bucket->PutKey(descr.BuildPath(), content, callbackWrap, descr.GetContentType());
}

NThreading::TFuture<void> TSimpleUnencryptedMediaStorage::UploadResourcePreview(const TString& content, const TMediaResourceDescription& description, const TString& contentType) const {
    if (!content.size()) {
        return NThreading::TExceptionFuture() << "TSimpleUnencryptedMediaStorage::UploadResourcePreview: content is empty";
    }
    auto bucket = MdsClient.GetBucket(Config.GetBucketName());
    if (!bucket) {
        return NThreading::TExceptionFuture() << "TSimpleUnencryptedMediaStorage::UploadResourcePreview: bucket not defined";
    }
    return bucket->CheckReply(bucket->PutKey(description.BuildPreviewPath(), content, contentType));
}

NThreading::TFuture<TChatResourceAcquisitionResult> TSimpleUnencryptedMediaStorage::AcquireResource(const TString& userId, const TString& resourceId, bool needsFurtherProcessing) const {
    TMediaResourceDescription descr;
    if (!GetResourceMeta(descr, userId, resourceId)) {
        return NThreading::TExceptionFuture() << "TSimpleUnencryptedMediaStorage::AcquireResource: cannot get description";
    }
    if (!needsFurtherProcessing) {
        return NThreading::MakeFuture(TChatResourceAcquisitionResult(TBlob{}, descr, MdsClient.GetTmpFilePath(Config.GetBucketName(), descr.BuildPath())));
    }
    auto bucket = MdsClient.GetBucket(Config.GetBucketName());
    if (!bucket) {
        return NThreading::TExceptionFuture() << "TSimpleUnencryptedMediaStorage::AcquireResource: cannot get bucket '" << Config.GetBucketName() << "'";
    }
    return bucket->GetFile(descr.BuildPath(), descr.GetContentType(), true).Apply([
            descr,
            filePath = MdsClient.GetTmpFilePath(Config.GetBucketName(), descr.BuildPath())
        ] (const auto& r) {
        return TChatResourceAcquisitionResult(TBlob::FromString(r.GetValue().GetContent()), descr);
    });
}

NThreading::TFuture<TChatResourceAcquisitionResult> TSimpleUnencryptedMediaStorage::AcquireResourcePreview(const TString& userId, const TString& resourceId, bool needsFurtherProcessing, const TMap<TString, TString>& typeResourceOverrides) const {
    TMediaResourceDescription descr;
    if (!GetResourceMeta(descr, userId, resourceId)) {
        return NThreading::TExceptionFuture() << "TSimpleUnencryptedMediaStorage::AcquireResourcePreview: cannot get description";
    }
    TString previewPath = descr.BuildPreviewPath();
    bool hasPreview = descr.GetHasPreview();
    auto previewOverrideIt = typeResourceOverrides.find(descr.GetContentType());
    if (previewOverrideIt != typeResourceOverrides.end()) {
        previewPath = previewOverrideIt->second;
        hasPreview = true;
    }
    if (!needsFurtherProcessing && hasPreview) {
        return NThreading::MakeFuture(TChatResourceAcquisitionResult(TBlob{}, descr, MdsClient.GetTmpFilePath(Config.GetBucketName(), previewPath)));
    }
    auto bucket = MdsClient.GetBucket(Config.GetBucketName());
    if (!bucket) {
        return NThreading::TExceptionFuture() << "TSimpleUnencryptedMediaStorage::AcquireResource: cannot get bucket '" << Config.GetBucketName() << "'";
    }
    if (needsFurtherProcessing && hasPreview) {
        return bucket->GetFile(previewPath).Apply([descr, previewPath] (const auto& r) {
            return TChatResourceAcquisitionResult(TBlob::FromString(r.GetValue().GetContent()), descr);
        });
    }
    return bucket->GetFile(descr.BuildPath()).Apply([
        descr,
        previewPath,
        ffMpegPath = Config.GetFFMpegPath(),
        resizeMaxSide = Config.GetPreviewResizeMaxSide()
    ] (const auto& r) mutable {
        auto contentType = descr.GetContentType();
        TString previewType = contentType;
        TBlob preview;
        TBlob sourceContentBlob = TBlob::FromString(r.GetValue().GetContent());
        if (contentType.EndsWith("jpeg") || contentType.EndsWith("png")) {
            auto resizer = NImageTransformation::TResize(resizeMaxSide);
            TMessagesCollector errors;
            TString format = "jpg";
            if (contentType.EndsWith("png")) {
                format = "png";
            }
            resizer.Transform(sourceContentBlob, preview, errors, format);
        } else if (contentType.EndsWith("mp4") || contentType.StartsWith("video")) {
            TBlob transformResult;
            NImageTransformation::TFirstFrameExtractor extractor(ffMpegPath);
            if (!extractor.Extract(sourceContentBlob, preview)) {
                preview = std::move(sourceContentBlob);
            }
            previewType = "image/jpeg";
        } else {
            preview = std::move(sourceContentBlob);
        }
        TChatResourceAcquisitionResult result { preview, descr };
        result.SetPreviewType(previewType);
        if (preview.Size() != 0) {
            result.SetGenerated(true);
        }
        return result;
    });
}

bool TSimpleUnencryptedMediaStorage::ForceRefresh(const TInstant actuality) const {
    return RefreshCache(actuality);
}

TExpected<TMediaResourceDescription, TString> TSimpleUnencryptedMediaStorage::UploadResourceToMdsByLink(const TString& resourceName, const TString& resourceLink, const TString& contentType, const TString& userId, const bool shared) const {
    TMediaResourceDescription existingDescription;
    if (GetResourceMeta(existingDescription, userId, resourceName)) {
        return existingDescription;
    }
    TString bucket, path, file;
    TMessagesCollector errors;
    if (!MdsClient.ParseMdsLink(resourceLink, errors, bucket, path)) {
        return MakeUnexpected<TString>("MediaStorage: Can't parse resource link " + resourceLink + " " + errors.GetStringReport());
    }

    auto inputFileBucket = MdsClient.GetBucket(bucket);
    if (!inputFileBucket) {
        return MakeUnexpected<TString>("MediaStorage: Bucket " + bucket + " not defined");
    }

    INFO_LOG << "SaveResourceByLink: parsed link " << resourceLink << " as bucket: " << bucket << "; path: " << path << Endl;
    if (inputFileBucket->GetFile(path, file, errors) / 100 != 2) {
        return MakeUnexpected<TString>("MediaStorage: " + errors.GetStringReport());
    }
    if (file.empty()) {
        return MakeUnexpected<TString>("MediaStorage: Downloaded file " + resourceName + " is empty (link " + resourceLink + ")");
    }

    TMediaResourceDescription description(resourceName, userId);
    description.SetShared(shared);
    description.SetContentType(contentType);
    auto clientBucket = MdsClient.GetBucket(Config.GetBucketName());
    if (!clientBucket) {
        return MakeUnexpected<TString>("MediaStorage: Bucket " + Config.GetBucketName() + " not defined");
    }
    auto resultCode = clientBucket->PutKey(description.BuildPath(), file, errors, contentType);
    if (resultCode < 200 || resultCode >= 300) {
        return MakeUnexpected<TString>("MediaStorage: could add " + description.BuildPath() + " to bucket " + Config.GetBucketName() + " errors: " + errors.GetStringReport());
    }
    return description;
}

bool TSimpleUnencryptedMediaStorage::AddResourceDescription(TMediaResourceDescription& description, NDrive::TEntitySession& session) const {
    description.SetUploadedAt(Now());
    return Table->Upsert(description, session) && HistoryWriter->AddHistory(description, description.GetUserId(), EObjectHistoryAction::Add, session);
}

bool TSimpleUnencryptedMediaStorage::UpdateResourceMeta(const TMediaResourceDescription description, const TString& actorUserId, NDrive::TEntitySession& session) const {
    return Table->Upsert(description, session) && HistoryWriter->AddHistory(description, actorUserId, EObjectHistoryAction::UpdateData, session);
}
