#pragma once

#include "media_resource.h"

#include <drive/backend/chat_robots/configuration/config.h>

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/cache.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

class IChatMediaResourcePostUploadCallback;
class IChatMediaResourceAcquisitionCallback;
class TMediaResourceDescription;
class TChatResourceAcquisitionResult;

class IChatMediaStorage {
public:
    virtual TMaybe<TMediaResourceDescription> RegisterResource(const TString& userId, const TString& resourceId, const TString& contentType, const bool shared, NDrive::TEntitySession& session) const = 0;
    virtual void UploadResource(const TString& userId, const TString& resourceId, const TString& contentType, const TString& content, TAtomicSharedPtr<IChatMediaResourcePostUploadCallback> callback) const = 0;
    virtual NThreading::TFuture<TChatResourceAcquisitionResult> AcquireResource(const TString& userId, const TString& resourceId, bool needsFurtherProcessing) const = 0;
    virtual NThreading::TFuture<TChatResourceAcquisitionResult> AcquireResourcePreview(const TString& userId, const TString& resourceId, bool needsFurtherProcessing, const TMap<TString, TString>& typeResourceOverrides) const = 0;

    virtual bool GetResourceMeta(TMediaResourceDescription& descr, const TString& userId, const TString& resourceId, const TInstant freshness = TInstant::Zero()) const = 0;

    virtual TExpected<TMediaResourceDescription, TString> UploadResourceToMdsByLink(const TString& resourceName, const TString& resourceLink, const TString& contentType, const TString& userId, const bool shared) const = 0;
    virtual bool AddResourceDescription(TMediaResourceDescription& description, NDrive::TEntitySession& session) const = 0;
    virtual bool UpdateResourceMeta(const TMediaResourceDescription description, const TString& actorUserId, NDrive::TEntitySession& session) const = 0;
    virtual NThreading::TFuture<void> UploadResourcePreview(const TString& content, const TMediaResourceDescription& description, const TString& contentType) const = 0;

    virtual ~IChatMediaStorage() = default;
};

class TSimpleUnencryptedMediaStorageTable : public TDBEntities<TMediaResourceDescription> {
private:
    using TBase = TDBEntities<TMediaResourceDescription>;
    using TEntity = TMediaResourceDescription;

public:
    using TBase::TBase;

    virtual TString GetTableName() const override {
        return "chat_robot_media";
    }

    virtual TString GetColumnName() const override {
        return "id";
    }

    virtual TString GetMainId(const TMediaResourceDescription& e) const override {
        return e.GetId();
    }

    virtual void UpdateUniqueCondition(NStorage::TTableRecord& unique, const TEntity& info) const override {
        unique.Set("user_id", info.GetUserId());
    }
};

class TChatRobotMediaHistoryManager : public TDatabaseHistoryManager<TMediaResourceDescription> {
private:
    using TBase = TDatabaseHistoryManager<TMediaResourceDescription>;

public:
    TChatRobotMediaHistoryManager(const IHistoryContext& context)
        : TBase(context, "chat_robot_media_history")
    {
    }
};

class TSimpleUnencryptedMediaStorage : public TAutoActualizingSnapshot<TMediaResourceDescription>, public IChatMediaStorage {
    using TCacheBase = TAutoActualizingSnapshot<TMediaResourceDescription>;
    using TDBTable = TSimpleUnencryptedMediaStorageTable;
    using TRecordType = TMediaResourceDescription;
    using THistoryManager = TChatRobotMediaHistoryManager;
    using THistoryReader = TAtomicSharedPtr<TCallbackSequentialTableImpl<TObjectEvent<TMediaResourceDescription>, TString>>;

private:
    NStorage::IDatabase::TPtr Database;
    const TS3Client& MdsClient;
    const TMediaStorageConfig Config;

    THolder<TDBTable> Table;
    THolder<THistoryManager> HistoryWriter;

    mutable TMap<TString, TMap<TString, TMediaResourceDescription>> UserToResources;
    mutable TMap<TString, TMediaResourceDescription> SharedResources;

    virtual bool DoRebuildCacheUnsafe() const override;

    virtual bool DoAcceptHistoryEventUnsafe(const TAtomicSharedPtr<TObjectEvent<TRecordType>>& ev, const bool isNew) override;

    virtual TStringBuf GetEventObjectId(const TObjectEvent<TRecordType>& object) const override;

public:
    TSimpleUnencryptedMediaStorage(const THistoryReader& reader, NStorage::IDatabase::TPtr db, const TS3Client& mdsClient, const TMediaStorageConfig& config)
        : TCacheBase("chat_robot_media", reader)
        , Database(db)
        , MdsClient(mdsClient)
        , Config(config)
        , HistoryWriter(new THistoryManager(THistoryContext(db)))
    {
        Table.Reset(new TDBTable(db));
    }

    virtual bool GetResourceMeta(TMediaResourceDescription& descr, const TString& userId, const TString& resourceId, const TInstant freshness = TInstant::Zero()) const override;

    virtual TMaybe<TMediaResourceDescription> RegisterResource(const TString& userId, const TString& resourceId, const TString& contentType, const bool shared, NDrive::TEntitySession& session) const override;
    virtual void UploadResource(const TString& userId, const TString& resourceId, const TString& contentType, const TString& content, TAtomicSharedPtr<IChatMediaResourcePostUploadCallback> callback) const override;
    virtual NThreading::TFuture<TChatResourceAcquisitionResult> AcquireResource(const TString& userId, const TString& resourceId, bool needsFurtherProcessing) const override;
    virtual NThreading::TFuture<TChatResourceAcquisitionResult> AcquireResourcePreview(const TString& userId, const TString& resourceId, bool needsFurtherProcessing, const TMap<TString, TString>& typeResourceOverrides) const override;
    virtual NThreading::TFuture<void> UploadResourcePreview(const TString& content, const TMediaResourceDescription& description, const TString& contentType) const override;
    virtual TExpected<TMediaResourceDescription, TString> UploadResourceToMdsByLink(const TString& resourceName, const TString& resourceLink, const TString& contentType, const TString& userId, const bool shared) const override;
    virtual bool AddResourceDescription(TMediaResourceDescription& description, NDrive::TEntitySession& session) const override;
    virtual bool UpdateResourceMeta(const TMediaResourceDescription description, const TString& actorUserId, NDrive::TEntitySession& session) const override;
    virtual bool ForceRefresh(const TInstant actuality) const;
};
