#include "bot.h"

#include <drive/backend/areas/areas.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/user_origin.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/tags/tags_manager.h>

bool TRegistrationChatBot::DeduceChatContinuation(const TString& userId, const TString& /*topic*/, const ui32 chatRoomId, NDrive::TEntitySession& session) const {
    auto userFR = DriveAPI.GetUsersData()->FetchInfo(userId);
    auto userPtr = userFR.GetResultPtr(userId);
    if (!userPtr) {
        return false;
    }
    auto status = userPtr->GetStatus();
    if (status == NDrive::UserStatusRejected) {
        return SetInitialAction(userId, chatRoomId, "rejected_general", session);
    }
    if (status == NDrive::UserStatusBlocked) {
        return DeduceBlockedChatContinuation(userId, chatRoomId, session);
    }
    if (status == NDrive::UserStatusActive) {
        return SetInitialAction(userId, chatRoomId, "finish_ok", session);
    }
    return false;
}

bool TRegistrationChatBot::DeduceBlockedChatContinuation(const TString& userId, const ui32 chatRoomId, NDrive::TEntitySession& session) const {
    auto chatMessagesGroup = DeduceBanMessagesGroup(userId);
    if (chatMessagesGroup) {
        return SetInitialAction(userId, chatRoomId, chatMessagesGroup, session);
    } else {
        return false;
    }
}

bool TRegistrationChatBot::SetInitialAction(const TString& userId, const ui32 chatRoomId, const TString& actionId, NDrive::TEntitySession& session, const ui32 queuedResubmit) const {
    TChatRobotState currentState;
    currentState.SetCurrentStep(actionId);
    currentState.SetResubmitMask(0);
    currentState.SetQueuedResubmitMask(queuedResubmit);
    currentState.SetOriginalResubmitMask(0);

    TChatRobotScriptItem scriptItem;
    if (!ChatConfig.GetChatScript().GetScriptItemById(actionId, scriptItem)) {
        ERROR_LOG << "can't get script item" << Endl;
        return false;
    }

    auto operatorUserId = TString{"robot-frontend"};
    if (!SendPreActionMessages(chatRoomId, scriptItem.GetId(), session, operatorUserId)) {
        ERROR_LOG << "can't send pre action messages" << Endl;
        return false;
    }

    return UpdateChatRobotState(userId, "", currentState, session);
}

void TRegistrationChatBot::CheckIn(const TString& userId, const TString& operatorUserId, const TString& /*topic*/, const TString& origin, const TString& externalUserId, const TGeoCoord* location, TInstant locationActuality) const {
    auto eg = NDrive::BuildEventGuard("registration_check_in");
    auto session = Server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
    if (externalUserId || origin) {
        if (!TUserOriginTag::Set(userId, origin, externalUserId, *Server, session)) {
            ERROR_LOG << "RegistrationChatBot::CheckIn: cannot set user origin: " << session.GetStringReport() << Endl;
        }
    }
    if (location) {
        if (Server->GetDriveAPI()->GetUsersData()->ActualizeRegistrationGeo(userId, userId, *location, session)) {
            ERROR_LOG << "unable to actualize registration geo for " << userId << ": " << session.GetStringReport() << Endl;
        }
    }

    const TAreasDB* areas = Server->GetDriveAPI() ? Server->GetDriveAPI()->GetAreasDB() : nullptr;
    TVector<TStandRegistrationPromoTag> tags;
    if (areas && location) {
        areas->ProcessHardTagsInPoint<TStandRegistrationPromoTag>(*location, [&tags](const TStandRegistrationPromoTag* tag) {
            if (tag) {
                tags.push_back(*tag);
            }
            return true;
        }, TInstant::Zero());
    }

    TVector<TDBTag> userTags;
    bool restoreTagsOK = false;
    if (tags) {
        restoreTagsOK = Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(userId, {}, userTags, session);
    }

    if (restoreTagsOK) {
        TSet<TString> tagNames;
        for (auto&& tag : userTags) {
            tagNames.insert(tag->GetName());
        }

        for (auto&& tag : tags) {
            if (tagNames.contains(tag.GetMarkupTagName(Server->GetDriveAPI()->GetTagsManager().GetTagsMeta()))) {
                continue;
            }

            if (!(tag.GetStartInstant() <= locationActuality && locationActuality <= tag.GetFinishInstant())) {
                continue;
            }

            auto bonusTagName = tag.GetBonusTagName(Server->GetDriveAPI()->GetTagsManager().GetTagsMeta());

            auto bonusTag = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(bonusTagName, "Был отмечен тегом на начисление бонуса после регистрации");
            {
                auto operationTag = dynamic_cast<TOperationTag*>(bonusTag.Get());
                if (!!operationTag) {
                    operationTag->SetAmount(tag.GetBonusAmount(Server->GetDriveAPI()->GetTagsManager().GetTagsMeta()));
                }
            }

            auto markupTag = IJsonSerializableTag::BuildWithComment<TUniqueUserTag>(tag.GetMarkupTagName(Server->GetDriveAPI()->GetTagsManager().GetTagsMeta()), "Поучаствовал в промоакции");
            if (!markupTag) {
                ERROR_LOG << "unable to construct markup tag" << Endl;
                continue;
            }

            if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(markupTag, operatorUserId, userId, Server, session)) {
                ERROR_LOG << "unable to markup participation" << Endl;
                continue;
            }
            if (bonusTagName && !Server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(bonusTag, operatorUserId, userId, Server, session)) {
                ERROR_LOG << "unable to accrue bonus" << Endl;
                continue;
            }
        }
    }
    if (!session.Commit()) {
        ERROR_LOG << "RegistrationChatBot::CheckIn: cannot commit transaction: " << session.GetStringReport() << Endl;
    }
}
