#include "robot_manager.h"

#include <drive/backend/chat_robots/channel/bot.h>
#include <drive/backend/chat_robots/registration/bot.h>
#include <drive/backend/chat_robots/registration/light.h>
#include <drive/backend/chat_robots/support/bot.h>

#include <drive/backend/data/chargable.h>
#include <drive/backend/notifications/native_chat/chat.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/roles/manager.h>

TChatRobotsManager::TChatRobotsManager(const NDrive::IServer* server, const TChatRobotsConfig& robotsConfig)
    : Server(server)
{
    const TString storageDBName = robotsConfig.GetStateStorageDatabase();
    auto db = server->GetDatabase(storageDBName);
    AssertCorrectConfig(!!db, "no such database: " + storageDBName);

    ChatRobotStateStorage.Reset(new TChatRobotStatePostgresStorage(THistoryContext(db), robotsConfig.GetStatesHistoryConfig()));
    Y_ENSURE_BT(ChatRobotStateStorage->Start());

    ChatViewTrackerHistoryReader.Reset(new TCallbackSequentialTableImpl<TObjectEvent<TViewTrackerState>, ui32>(THistoryContext(db), "chat_views_history", robotsConfig.GetViewsHistoryConfig()));
    Y_ENSURE_BT(ChatViewTrackerHistoryReader->Start());
    ChatViewTracker.Reset(new TChatViewTracker(ChatViewTrackerHistoryReader, db));
    Y_ENSURE_BT(ChatViewTracker->RebuildCache());

    AssertCorrectConfig(!!server->GetMDSClient(), "no mds client");
    ChatMediaHistoryReader.Reset(new TCallbackSequentialTableImpl<TObjectEvent<TMediaResourceDescription>, TString>(THistoryContext(db), "chat_robot_media_history", robotsConfig.GetMediaHistoryConfig()));
    Y_ENSURE_BT(ChatMediaHistoryReader->Start());
    ChatRobotMediaStorage.Reset(new TSimpleUnencryptedMediaStorage(ChatMediaHistoryReader, db, *server->GetMDSClient(), robotsConfig.GetMediaStorageConfig()));
    Y_ENSURE_BT(ChatRobotMediaStorage->RebuildCache());

    for (auto&& chatRobotConfig : robotsConfig.GetBotConfigs()) {
        CHECK_WITH_LOG(RegisterChatRobot(chatRobotConfig));
    }

    for (auto&& notifierName : server->GetNotifierNames()) {
        auto notifier = server->GetNotifier(notifierName);
        if (!notifier) {
            continue;
        }
        auto nativeChat = dynamic_cast<TNativeChatNotifier*>(notifier.Get());
        if (nativeChat) {
            RegisterChatRobot(nativeChat->GetConfig());
        }
    }

    RegisterGlobalMessageProcessor(this);
}

IChatRobot::TPtr TChatRobotsManager::GetChatRobot(const TString& name) const {
    TReadGuard rg(MutexChatRobots);
    auto chatRobotIt = ChatRobots.find(name);
    if (chatRobotIt != ChatRobots.end()) {
        return chatRobotIt->second;
    }
    return nullptr;
}

const TSimpleUnencryptedMediaStorage* TChatRobotsManager::GetChatRobotsMediaStorage() const {
    return ChatRobotMediaStorage.Get();
}

const TAtomicSharedPtr<TChatViewTracker> TChatRobotsManager::GetChatViewTracker() const {
    return ChatViewTracker;
}

const TAtomicSharedPtr<TChatRobotStatePostgresStorage> TChatRobotsManager::GetChatRobotStateStorage() const {
    return ChatRobotStateStorage;
}

const TMap<TString, IChatRobot::TPtr> TChatRobotsManager::GetChatRobots() const {
    TReadGuard rg(MutexChatRobots);
    return ChatRobots;
}

bool TChatRobotsManager::RegisterChatRobot(const TChatRobotConfig& config) const {
    TWriteGuard wg(MutexChatRobots);
    IChatRobot::TPtr bot;

    switch (config.GetBotClassName()) {
    case NChatRobot::ELogic::LightRegistration:
        if (!GetChatViewTracker() || !GetChatRobotStateStorage()) {
            return false;
        }
        bot.Reset(new TLightRegistrationChatBot(Server, config, GetChatViewTracker(), GetChatRobotStateStorage(), *GetChatRobotsMediaStorage()));
        break;
    case NChatRobot::ELogic::Registration:
        if (!GetChatViewTracker() || !GetChatRobotStateStorage()) {
            return false;
        }
        bot.Reset(new TRegistrationChatBot(Server, config, GetChatViewTracker(), GetChatRobotStateStorage(), *GetChatRobotsMediaStorage()));
        break;
    case NChatRobot::ELogic::Support:
        if (!GetChatRobotsMediaStorage() || !GetChatRobotStateStorage()) {
            return false;
        }
        bot.Reset(new TSupportChatBot(Server, config, GetChatViewTracker(), GetChatRobotStateStorage(), *GetChatRobotsMediaStorage()));
        break;
    case NChatRobot::ELogic::Channel:
        if (!GetChatViewTracker() || !GetChatRobotStateStorage()) {
            return false;
        }
        bot.Reset(new TChannelChatBot(Server, config, GetChatViewTracker(), GetChatRobotStateStorage()));
        break;
    }

    ChatRobots[config.GetChatId()] = std::move(bot);

    return true;
}

bool TChatRobotsManager::UnregisterChatRobot(const TString& robotId) const {
    TWriteGuard wg(MutexChatRobots);
    auto robotIt = ChatRobots.find(robotId);
    if (robotIt == ChatRobots.end()) {
        return false;
    }
    ChatRobots.erase(robotIt);
    return true;
}

TString TChatRobotsManager::Name() const {
    return "mp_chat_robots_manager";
}

bool TChatRobotsManager::Process(IMessage* message) {
    {
        TOfferBookingCompleted* impl = dynamic_cast<TOfferBookingCompleted*>(message);
        if (impl) {
            HandleOfferBookingEvent(impl->GetOffer());
            return true;
        }
    }
    return false;
}

void TChatRobotsManager::HandleOfferBookingEvent(const ICommonOffer::TPtr offer) const {
    auto action = Server->GetDriveAPI()->GetRolesManager()->GetAction(offer->GetBehaviourConstructorId());
    auto userId = offer->GetUserId();
    auto offerBuilder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
    if (offerBuilder) {
        auto introscreensChatId = Server->GetSettings().GetValueDef<TString>("chats.introscreen_chat_id", "");
        NDrive::TEntitySession session;
        const auto& tags = offerBuilder->GetGrouppingTags();
        auto introscreensChat = Server->GetChatRobot(introscreensChatId);
        if (introscreensChat) {
            TSet<TString> sentLandings;
            for (auto&& tag : tags) {
                auto landingId = Server->GetSettings().GetValueDef<TString>("offers." + tag + ".after_book_landing_id", "");
                if (landingId && sentLandings.emplace(landingId).second) {
                    NDrive::NChat::TMessage message;
                    message.SetText(landingId);
                    message.SetType(NDrive::NChat::TMessage::EMessageType::Introscreen);
                    if (!session) {
                        session = Server->GetChatEngine()->BuildSession();
                    }
                    introscreensChat->SendArbitraryMessage(userId, "", "robot-frontend", message, session);
                }
            }
        }

        for (auto&& tag : tags) {
            auto chatId = Server->GetSettings().GetValueDef<TString>("offers." + tag + ".after_book_chat_id", "");
            auto chatRobot = GetChatRobot(chatId);
            if (chatRobot) {
                if (!session) {
                    session = Server->GetChatEngine()->BuildSession();
                }
                chatRobot->EnsureChat(userId, offer->GetOfferId(), session, true);
            }
        }

        if (session && !session.Commit()) {
            ERROR_LOG << "Could not commit chat session on processing TOfferBookingCompleted event: " << session.GetStringReport() << Endl;
        }
    }
}

TChatRobotsManager::~TChatRobotsManager() {
    if (ChatRobotStateStorage && !ChatRobotStateStorage->Stop()) {
        ERROR_LOG << "cannot stop ChatRobotStateStorage manager" << Endl;
    }
    if (ChatViewTrackerHistoryReader && !ChatViewTrackerHistoryReader->Stop()) {
        ERROR_LOG << "cannot stop ChatViewTrackerHistoryReader manager" << Endl;
    }
    if (ChatMediaHistoryReader && !ChatMediaHistoryReader->Stop()) {
        ERROR_LOG << "cannot stop ChatMediaHistoryReader manager" << Endl;
    }
    UnregisterGlobalMessageProcessor(this);
}
