#include "documents.h"

#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/user_document_photos/manager.cpp>
#include <drive/backend/database/drive_api.h>

IChatScriptAction::TFactory::TRegistrator<TSingleDocumentScriptAction> LicenseBackActionRegistrator(NChatRobot::EUserAction::LicenseBack);
IChatScriptAction::TFactory::TRegistrator<TSingleDocumentScriptAction> LicenseFrontActionRegistrator(NChatRobot::EUserAction::LicenseFront);
IChatScriptAction::TFactory::TRegistrator<TSingleDocumentScriptAction> LicenseSelfieActionRegistrator(NChatRobot::EUserAction::LicenseSelfie);
IChatScriptAction::TFactory::TRegistrator<TSingleDocumentScriptAction> PassportBiographicalActionRegistrator(NChatRobot::EUserAction::PassportBiographical);
IChatScriptAction::TFactory::TRegistrator<TSingleDocumentScriptAction> PassportRegistrationActionRegistrator(NChatRobot::EUserAction::PassportRegistration);
IChatScriptAction::TFactory::TRegistrator<TSingleDocumentScriptAction> PassportSelfieActionRegistrator(NChatRobot::EUserAction::PassportSelfie);
IChatScriptAction::TFactory::TRegistrator<TDocumentVideoAction> LicenseVideoActionRegistrator(NChatRobot::EUserAction::LicenseVideo);
IChatScriptAction::TFactory::TRegistrator<TDocumentVideoAction> PassportVideoActionRegistrator(NChatRobot::EUserAction::PassportVideo);
IChatScriptAction::TFactory::TRegistrator<TDocumentVideoAction> VideoSelfieActionRegistrator(NChatRobot::EUserAction::VideoSelfie);

TMaybe<NUserDocument::EType> TCompletePassportScriptAction::GetDocumentType(NChatRobot::EUserAction actionType, size_t attachmentNumber) const {
    Y_UNUSED(actionType);
    if (attachmentNumber == 0) {
        return NUserDocument::EType::PassportBiographical;
    } else {
        return NUserDocument::EType::PassportRegistration;
    }
}

TMaybe<NUserDocument::EType> TCompleteLicenseScriptAction::GetDocumentType(NChatRobot::EUserAction actionType, size_t attachmentNumber) const {
    Y_UNUSED(actionType);
    if (attachmentNumber == 0) {
        return NUserDocument::EType::LicenseFront;
    } else {
        return NUserDocument::EType::LicenseBack;
    }
}

TMaybe<NUserDocument::EType> TSingleDocumentScriptAction::GetDocumentType(NChatRobot::EUserAction actionType, size_t attachmentNumber) const {
    Y_UNUSED(attachmentNumber);
    auto actionTypeStr = ToString(actionType);
    NUserDocument::EType documentType;
    if (!TryFromString(actionTypeStr, documentType)) {
        return Nothing();
    }
    return documentType;
}

TMaybe<NUserDocument::EType> TDocumentVideoAction::GetDocumentType(NChatRobot::EUserAction actionType, size_t attachmentNumber) const {
    Y_UNUSED(attachmentNumber);
    auto actionTypeStr = ToString(actionType);
    NUserDocument::EType documentType;
    if (!TryFromString(actionTypeStr, documentType)) {
        return Nothing();
    }
    return documentType;
}

NDrive::NChat::TMessage::EMessageType ICompleteDocumentScriptAction::GetMessageType() const {
    return NDrive::NChat::TMessage::EMessageType::UserDocumentPhotos;
}

NDrive::NChat::TMessage::EMessageType TSingleDocumentScriptAction::GetMessageType() const {
    return NDrive::NChat::TMessage::EMessageType::UserDocumentPhoto;
}

NDrive::NChat::TMessage::EMessageType TDocumentVideoAction::GetMessageType() const {
    return NDrive::NChat::TMessage::EMessageType::UserDocumentVideo;
}

TString JoinAttachmentsToString(const TVector<TMessageAttachment>& attachments) {
    TStringStream ss;
    for (auto&& attach : attachments) {
        if (!ss.empty()) {
            ss << ", ";
        }
        ss << attach.Data << " " << attach.ContentType;
    }
    return ss.Str();
}

void LogFailedRecognition(const TString& userId, const TVector<TMessageAttachment>& attachments, const NJson::TJsonValue& errorsReport) {
    NDrive::TEventLog::TUserIdGuard userIdGuard(userId);
    NDrive::TEventLog::Log("DocumentsRecognitionFail", NJson::TMapBuilder
        ("user_id", userId)
        ("photo_ids", JoinAttachmentsToString(attachments))
        ("errors", errorsReport)
    );
}

bool UpdateDocumentRevision(const TString& userId, const TString& updatedRevision, const TString& operatorId, NUserDocument::EType documentType, const NDrive::IServer& server, NDrive::TEntitySession& session) {
    if (updatedRevision.empty()) {
        return true;
    }
    if (documentType != NUserDocument::EType::PassportBiographical && documentType != NUserDocument::EType::LicenseFront && documentType != NUserDocument::EType::LicenseBack) {
        return true;
    }
    if (auto userData = server.GetDriveAPI()->GetUsersData()->RestoreUser(userId, session)) {
        if (documentType == NUserDocument::EType::PassportBiographical) {
            userData->SetPassportDatasyncRevision(updatedRevision);
        } else {
            userData->SetDrivingLicenseDatasyncRevision(updatedRevision);
        }
        if (!server.GetDriveAPI()->GetUsersData()->UpdateUser(*userData, operatorId, session)) {
            return false;
        }
    } else {
        return false;
    }
    return true;
}

NThreading::TFuture<TNextActionInfo> GetFailedFuture(const TString& userId, const TString& errorPrefix, const TVector<TMessageAttachment>& attachments, bool shouldFail, const NThreading::TFuture<void>& waiter, TNextActionInfo result) {
    if (shouldFail) {
        return NThreading::TExceptionFuture() << errorPrefix << JoinAttachmentsToString(attachments) << " error: " << NThreading::GetExceptionMessage(waiter);
    } else {
        result.SetFailed(true);
        LogFailedRecognition(userId, attachments, NThreading::GetExceptionInfo(waiter));
        return NThreading::MakeFuture(result);
    }
}

NThreading::TFuture<TNextActionInfo> GetFailedFuture(const TString& userId, const TString& errorPrefix, const TVector<TMessageAttachment>& attachments, bool shouldFail, const NDrive::TEntitySession& session, TNextActionInfo result) {
    if (shouldFail) {
        return NThreading::TExceptionFuture() << errorPrefix << JoinAttachmentsToString(attachments) << " error: " << session.GetStringReport();
    } else {
        result.SetFailed(true);
        LogFailedRecognition(userId, attachments, session.GetReport());
        return NThreading::MakeFuture(result);
    }
}

NThreading::TFuture<TNextActionInfo> RecognizeAndUpdateDocuments(const TVector<NS3::TFile>& documents, IChatUserContext::TPtr context, bool shouldFail, const TDriveUserData& userData, TNextActionInfo result, TMap<TString, TUserDocumentPhoto> photoById) {
    TVector<NThreading::TFuture<TRecognizedDocument>> recognizerFutures;
    for (auto&& document : documents) {
        auto photoData = photoById.find(document.GetKey());
        if (photoData != photoById.end()) {
            auto recognizer = IPhotoRecognizerClient::TFactory::Construct(photoData->second.GetType());
            if (recognizer) {
                recognizerFutures.push_back(recognizer->GetPatchedRecognizedDocument(&context->GetServer(), userData, document.GetContent(), document.GetKey()));
            }
        }
    }

    auto eventLogState = NDrive::TEventLog::CaptureState();
    return NThreading::WaitExceptionOrAll(recognizerFutures).Apply([
        eventLogState = std::move(eventLogState),
        recognizerFutures = std::move(recognizerFutures),
        result,
        context,
        userData,
        shouldFail,
        photoById
    ] (const auto& waiter) mutable {
        auto elsg = NDrive::TEventLog::Guard(eventLogState);
        if (waiter.HasException() || !waiter.HasValue()) {
            return GetFailedFuture(context->GetUserId(), "IDocumentsScriptAction::ProcessAsyncOperations cannot recognize documents ", result.GetAttachments(), shouldFail, waiter, result);
        }
        TRecognizedDocument finalDocumentData;
        for (size_t i = 0; i < recognizerFutures.size(); ++i) {
            if (i == 0) {
                finalDocumentData = recognizerFutures[i].GetValue();
            } else {
                finalDocumentData.MergeData(recognizerFutures[i].GetValue().OptionalPassportData(), recognizerFutures[i].GetValue().OptionalLicenseData());
            }
            if (!recognizerFutures[i].GetValue().GetPhotoId().empty()) {
                auto photoIt = photoById.find(recognizerFutures[i].GetValue().GetPhotoId());
                if (photoIt != photoById.end()) {
                    photoIt->second.SetRecognizerMeta(recognizerFutures[i].GetValue().GetConfidences());
                }
            }
        }

        NThreading::TFuture<TUserPassportData> passportDataFuture = NThreading::MakeFuture(TUserPassportData());
        NThreading::TFuture<TUserDrivingLicenseData> licenseDataFuture = NThreading::MakeFuture(TUserDrivingLicenseData());
        if (finalDocumentData.HasPassportData() && !userData.GetPassportDatasyncRevision().empty()) {
            passportDataFuture = context->GetServer().GetDriveAPI()->GetPrivateDataClient().GetPassport(userData, userData.GetPassportDatasyncRevision());
        }
        if (finalDocumentData.HasLicenseData() && !userData.GetDrivingLicenseDatasyncRevision().empty()) {
            licenseDataFuture = context->GetServer().GetDriveAPI()->GetPrivateDataClient().GetDrivingLicense(userData, userData.GetDrivingLicenseDatasyncRevision());
        }

        TVector<NThreading::TFuture<void>> futures = { passportDataFuture.IgnoreResult(), licenseDataFuture.IgnoreResult() };
        return NThreading::WaitExceptionOrAll(futures).Apply([
            finalDocumentData, shouldFail, result, context, photoById, userData,
            eventLogState = std::move(eventLogState),
            passportDataFuture = std::move(passportDataFuture),
            licenseDataFuture = std::move(licenseDataFuture)
        ] (const auto& waiter) mutable -> NThreading::TFuture<TNextActionInfo> {
            auto elsg = NDrive::TEventLog::Guard(eventLogState);
            if (waiter.HasException() || !waiter.HasValue()) {
                return GetFailedFuture(context->GetUserId(), "IDocumentsScriptAction::ProcessAsyncOperations cannot get from datasync ", result.GetAttachments(), shouldFail, waiter, result);
            }
            TVector<NThreading::TFuture<void>> privateDataUpdateResults;
            if (finalDocumentData.HasPassportData()) {
                TUserPassportData patchedPassport = passportDataFuture.GetValue();
                patchedPassport.Patch(finalDocumentData.GetPassportDataRef());
                privateDataUpdateResults.push_back(context->GetServer().GetDriveAPI()->GetPrivateDataClient().UpdatePassport(userData, finalDocumentData.GetDocumentRevision(), patchedPassport));
            }
            if (finalDocumentData.HasLicenseData()) {
                TUserDrivingLicenseData patchedLicense = licenseDataFuture.GetValue();
                patchedLicense.Patch(finalDocumentData.GetLicenseDataRef());
                privateDataUpdateResults.push_back(context->GetServer().GetDriveAPI()->GetPrivateDataClient().UpdateDrivingLicense(userData, finalDocumentData.GetDocumentRevision(), patchedLicense));
            }
            if (!privateDataUpdateResults.empty()) {
                result.SetDocumentRevision(finalDocumentData.GetDocumentRevision());
            }
            return NThreading::WaitExceptionOrAll(privateDataUpdateResults).Apply([
                eventLogState = std::move(eventLogState),
                shouldFail,
                result,
                context,
                photoById
            ] (const auto& waiter) -> NThreading::TFuture<TNextActionInfo> {
                auto elsg = NDrive::TEventLog::Guard(eventLogState);
                if (waiter.HasException() || !waiter.HasValue()) {
                    return GetFailedFuture(context->GetUserId(), "IDocumentsScriptAction::RecognizeAndUpdateDocuments failed passport datasync update ", result.GetAttachments(), shouldFail, waiter, result);
                }
                auto session = context->GetServer().GetDriveAPI()->template BuildTx<NSQL::Writable>();
                for (auto&& [id, photo] : photoById) {
                    if (!context->GetServer().GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().Upsert(photo, session)) {
                        return GetFailedFuture(context->GetUserId(), "IDocumentsScriptAction::RecognizeAndUpdateDocuments photo update failed, id ", {id}, shouldFail, session, result);
                    }
                }
                if (!session.Commit()) {
                    return GetFailedFuture(context->GetUserId(), "IDocumentsScriptAction::RecognizeAndUpdateDocuments photo update session commit failed ", result.GetAttachments(), shouldFail, session, result);
                }
                return NThreading::MakeFuture(result);
            });
        });
    });
}

NThreading::TFuture<TNextActionInfo> IDocumentScriptAction::ProcessAsyncOperations(const NDrive::NChat::TMessage& message, const TVector<TMessageAttachment>& attachments) const {
    Y_UNUSED(message);

    if (!GetContext()->GetServer().GetDriveAPI()->HasDocumentPhotosManager()) {
        return NThreading::TExceptionFuture() << "IDocumentsScriptAction::ProcessAsyncOperations document photo manager not configured";
    }

    if (attachments.size() == 0) {
        return NThreading::TExceptionFuture() << "IDocumentsScriptAction::ProcessAsyncOperations No documents are attached";
    }

    if (attachments[0].Data.StartsWith("!r")) {
        TNextActionInfo result;
        TVector<NThreading::TFuture<NS3::TFile>> documentFutures;
        TMap<TString, TUserDocumentPhoto> photoById;
        TMap<TString, TUserDocumentVideo> videoById;
        auto session = GetContext()->GetServer().GetDriveAPI()->BuildTx<NSQL::Writable>();
        for (size_t i = 0; i < attachments.size(); ++i) {
            auto resourceId = attachments[i].Data.substr(3, attachments[0].Data.size() - 3);
            if (auto resourceType = GetDocumentType(GetCurrentScriptItem().GetActionType(), i)) {
                if (*resourceType & NUserDocument::Video) {
                    auto video = GetContext()->GetServer().GetDriveAPI()->GetDocumentPhotosManager().GetUserBackgroundVideosDB().Fetch(resourceId, session);
                    if (!video) {
                        return NThreading::TExceptionFuture() << "IDocumentsScriptAction::ProcessAsyncOperations cannot FetchInfo: " << session.GetStringReport();
                    }
                    if (video->GetUserId() != GetContext()->GetUserId()) {
                        return NThreading::TExceptionFuture() << "IDocumentsScriptAction::ProcessAsyncOperations user id mismatch";
                    }
                    video->SetType(*resourceType);
                    videoById[resourceId] = std::move(*video);
                    if (GetCurrentScriptItem().GetUseRecognition()) {
                        WARNING_LOG << "Recognition is not supported for video" << Endl;
                    }
                } else {
                    auto photo = GetContext()->GetServer().GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().Fetch(resourceId, session);
                    if (!photo) {
                        return NThreading::TExceptionFuture() << "IDocumentsScriptAction::ProcessAsyncOperations cannot FetchInfo: " << session.GetStringReport();
                    }
                    if (photo->GetUserId() != GetContext()->GetUserId()) {
                        return NThreading::TExceptionFuture() << "IDocumentsScriptAction::ProcessAsyncOperations user id mismatch";
                    }
                    photo->SetType(*resourceType);
                    photoById[resourceId] = std::move(*photo);
                    if (GetCurrentScriptItem().GetUseRecognition()) {
                        documentFutures.push_back(GetContext()->GetServer().GetDriveAPI()->GetDocumentPhotosManager().GetDocumentPhoto(resourceId, GetContext()->GetServer()));
                    }
                }
            } else {
                return NThreading::TExceptionFuture() << "IDocumentsScriptAction::ProcessAsyncOperations unknown document type " + ToString(GetCurrentScriptItem().GetActionType());
            }
            result.AddAttachment(resourceId);
        }
        NDrive::NChat::TMessage documentMessage;
        documentMessage.SetType(GetMessageType());
        result.SetMessage(documentMessage);

        if (GetCurrentScriptItem().GetUseRecognition() && !videoById) {
            TDriveUserData userData;
            if (auto optionalUserData = GetContext()->GetServer().GetDriveAPI()->GetUsersData()->GetCachedObject(GetContext()->GetUserId())) {
                userData = std::move(*optionalUserData);
            }
            auto eventLogState = NDrive::TEventLog::CaptureState();
            return NThreading::WaitExceptionOrAll(documentFutures).Apply([
                    eventLogState = std::move(eventLogState),
                    context = GetContext(),
                    documentFutures = std::move(documentFutures),
                    shouldFail = !GetCurrentScriptItem().GetNextStepsError().HasFurtherSteps(),
                    userData, result, photoById
                ] (const auto& waiter) mutable -> NThreading::TFuture<TNextActionInfo>
            {
                auto elsg = NDrive::TEventLog::Guard(eventLogState);
                if (waiter.HasException() || !waiter.HasValue()) {
                    return GetFailedFuture(context->GetUserId(), "IDocumentsScriptAction::ProcessAsyncOperations cannot get documents ", result.GetAttachments(), shouldFail, waiter, result);
                }
                TVector<NS3::TFile> documents;
                for (auto documentFuture : documentFutures) {
                    documents.push_back(documentFuture.ExtractValue());
                }
                return RecognizeAndUpdateDocuments(documents, context, shouldFail, userData, result, photoById);
            });
        } else {
            const auto& photosDB = GetContext()->GetServer().GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB();
            for (auto&& [id, photo] : photoById) {
                if (!photosDB.Upsert(photo, session)) {
                    return NThreading::TExceptionFuture() << "IDocumentsScriptAction::ProcessAsyncOperations upsert of photo " << id << " failed: " + session.GetStringReport();
                }
            }
            const auto& videoDB = GetContext()->GetServer().GetDriveAPI()->GetDocumentPhotosManager().GetUserBackgroundVideosDB();
            for (auto&& [id, video]: videoById) {
                if (!videoDB.Upsert(video, session)) {
                    return NThreading::TExceptionFuture() << "IDocumentsScriptAction::ProcessAsyncOperations upsert of video " << id << " failed: " + session.GetStringReport();
                }
            }
            if (!session.Commit()) {
                return NThreading::TExceptionFuture() << "IDocumentsScriptAction::ProcessAsyncOperations session failed: " + session.GetStringReport();
            }
            return NThreading::MakeFuture(result);
        }
    }

    TVector<NThreading::TFuture<TDocumentPhotosManager::TAddDocumentResult>> futures;
    TMap<TString, TString> photoContentById;
    auto useRecognition = GetCurrentScriptItem().GetUseRecognition();
    for (size_t i = 0; i < attachments.size(); ++i) {
        auto documentType = GetDocumentType(GetCurrentScriptItem().GetActionType(), i);
        if (!documentType) {
            return NThreading::TExceptionFuture() << "IDocumentsScriptAction::ProcessAsyncOperations unknown document type " + ToString(GetCurrentScriptItem().GetActionType());
        }
        TString media;
        NThreading::TFuture<TDocumentPhotosManager::TAddDocumentResult> addDocumentFuture;
        try {
            media = Base64Decode(attachments[i].Data);
        } catch (const std::exception& e) {
            return NThreading::TExceptionFuture() << "IDocumentsScriptAction::ProcessAsyncOperations: " << "malformed base64: " << FormatExc(e);
        }
        if (*documentType & NUserDocument::Video) {
            useRecognition = false;
            addDocumentFuture = GetContext()->GetServer().GetDriveAPI()->GetDocumentPhotosManager().AddDocumentVideo(GetContext()->GetUserId(), GetTopicLink(), *documentType, media, GetContext()->GetServer(), !useRecognition, true);
        } else {
            addDocumentFuture = GetContext()->GetServer().GetDriveAPI()->GetDocumentPhotosManager().AddDocumentPhoto(GetContext()->GetUserId(), GetTopicLink(), *documentType, media, "", GetContext()->GetServer(), !useRecognition, true);
        }
        futures.push_back(std::move(addDocumentFuture));
    }
    auto messageType = GetMessageType();
    auto eventLogState = NDrive::TEventLog::CaptureState();
    return NThreading::WaitExceptionOrAll(futures).Apply([
        messageType, futures, useRecognition,
        context = GetContext(),
        eventLogState = std::move(eventLogState),
        shouldFail = !GetCurrentScriptItem().GetNextStepsError().HasFurtherSteps()
    ] (const auto& waiter) -> NThreading::TFuture<TNextActionInfo> {
        auto elsg = NDrive::TEventLog::Guard(eventLogState);
        if (waiter.HasValue()) {
            TDriveUserData userData;
            if (auto optionalUserData = context->GetServer().GetDriveAPI()->GetUsersData()->GetCachedObject(context->GetUserId())) {
                userData = std::move(*optionalUserData);
            }
            TNextActionInfo result;
            TVector<NS3::TFile> documents;
            TMap<TString, TUserDocumentPhoto> photoById;
            for (auto&& future : futures) {
                if (future.GetValue().GetVideo().GetId()) {
                    result.AddAttachment(future.GetValue().GetVideo().GetId());
                    documents.emplace_back(future.GetValue().GetVideo().GetId(), future.GetValue().GetVideoContent(), "video/mp4");
                }
                if (future.GetValue().GetPhoto().GetId()) {
                    result.AddAttachment(future.GetValue().GetPhoto().GetId());
                    photoById[future.GetValue().GetPhoto().GetId()] = future.GetValue().GetPhoto();
                    documents.emplace_back(future.GetValue().GetPhoto().GetId(), future.GetValue().GetPhotoContent(), "image/jpeg");
                }
            }
            NDrive::NChat::TMessage documentMessage;
            documentMessage.SetType(messageType);
            result.SetMessage(documentMessage);
            if (useRecognition) {
                return RecognizeAndUpdateDocuments(documents, context, shouldFail, userData, result, photoById);
            } else {
                return NThreading::MakeFuture(result);
            }
        } else {
            return NThreading::TExceptionFuture() << "IDocumentsScriptAction::ProcessAsyncOperations error adding photos and videos " << NThreading::GetExceptionMessage(waiter);
        }
    });
}

TMaybe<TString> IDocumentScriptAction::AcceptMessage(const TNextActionInfo& actionInfo, TChatContext& stateContext, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagsSession) const {
    Y_UNUSED(tagsSession);
    if (actionInfo.IsFailed()) {
        return GetCurrentScriptItem().GetNextStepsError().GetNextNode(GetContext(), stateContext);
    }
    auto documentType = GetDocumentType(GetCurrentScriptItem().GetActionType(), 0);
    if (!documentType) {
        chatSession.SetErrorInfo("IDocumentsScriptAction::AcceptMessage",  "unknown document type " + ToString(GetCurrentScriptItem().GetActionType()));
        return Nothing();
    }
    if (!UpdateDocumentRevision(GetContext()->GetUserId(), actionInfo.GetDocumentRevision(), GetContext()->GetChatRobot()->GetChatConfig().GetRobotUserId(), *documentType, GetContext()->GetServer(), tagsSession)) {
        return Nothing();
    }
    if (!AcceptRegularMessage(actionInfo.GetMessage(), actionInfo.GetAttachments(), stateContext, chatSession, false, false)) {
        return Nothing();
    }
    auto nextStepWithResubmit = GetContext()->GetChatRobot()->GetNextResubmitStep(GetContext(), GetCurrentScriptItem(), stateContext, chatSession);
    if (!nextStepWithResubmit) {
        return Nothing();
    }
    if (!nextStepWithResubmit->GetMessage().GetText().empty() && !AcceptRegularMessage(nextStepWithResubmit->GetMessage(), nextStepWithResubmit->GetAttachments(), stateContext, chatSession, false, true)) {
        return Nothing();
    }
    return nextStepWithResubmit->GetNextNodeId().empty() ? actionInfo.GetNextNodeId() : nextStepWithResubmit->GetNextNodeId();
}
