#include "feedback.h"

#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/feedback.h>
#include <drive/backend/sessions/manager/billing.h>

IChatScriptAction::TFactory::TRegistrator<TCurrentFeedbackScriptAction> TCurrentFeedbackScriptAction::Registrator(TCurrentFeedbackScriptAction::GetType());
IChatScriptAction::TFactory::TRegistrator<TPastFeedbackScriptAction> TPastFeedbackScriptAction::Registrator(TPastFeedbackScriptAction::GetType());

TMaybe<TString> IFeedbackScriptAction::AcceptMessage(const TNextActionInfo& actionInfo, TChatContext& stateContext, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagsSession) const {
    TString sessionId, deviceId, nextNodeId;
    if (!FindSessionAndDevice(actionInfo.GetMessage().GetText(), sessionId, deviceId, tagsSession)) {
        return Nothing();
    }
    if (!sessionId) {
        nextNodeId = GetCurrentScriptItem().GetNextStepsIncorrect().GetNextNode(GetContext(), stateContext);
    }
    AddTag = (GetOperatorId() == GetContext()->GetUserId());
    if (!AcceptRegularMessage(actionInfo.GetMessage(), actionInfo.GetAttachments(), stateContext, chatSession)) {
        return Nothing();
    }
    if (sessionId && deviceId) {
        ITag::TPtr tag = IJsonSerializableTag::BuildWithComment<TFeedbackTraceTag>(GetCurrentScriptItem().GetFeedbackTag(), actionInfo.GetMessage().GetText());
        auto feedbackTag = tag ? std::dynamic_pointer_cast<TFeedbackTraceTag>(tag) : nullptr;
        if (!feedbackTag) {
            chatSession.SetErrorInfo("IFeedbackScriptAction", "bad feedback trace tag");
            return Nothing();
        }
        if (!feedbackTag->Set(deviceId, GetContext()->GetUserId(), Now(), &GetContext()->GetServer(), tagsSession)
            || !GetContext()->GetServer().GetDriveAPI()->GetTagsManager().GetTraceTags().AddTags({feedbackTag}, GetContext()->GetUserId(), sessionId, &GetContext()->GetServer(), tagsSession))
        {
            return Nothing();
        }
        if (!feedbackTag->GetEffectiveBonusAmount()) {
            nextNodeId = GetCurrentScriptItem().GetNextStepsNoBonus().GetNextNode(GetContext(), stateContext);
        } else {
            nextNodeId = GetCurrentScriptItem().GetNextSteps().GetNextNode(GetContext(), stateContext);
        }
    }
    return nextNodeId;
}

bool TCurrentFeedbackScriptAction::FindSessionAndDevice(const TString& messageText, TString& sessionId, TString& deviceId, NDrive::TEntitySession& tagsSession) const {
    Y_UNUSED(messageText);
    TVector<TDBTag> perfTags;
    if (!GetContext()->GetServer().GetDriveAPI()->GetTagsManager().GetDeviceTags().RestorePerformerTags({GetContext()->GetUserId()}, perfTags, tagsSession)) {
        return false;
    }
    if (perfTags.size() == 1) {
        const TDBTag& perfTag = perfTags.front();
        deviceId = perfTag.GetObjectId();
        auto optionalSession = GetContext()->GetServer().GetDriveDatabase().GetSessionManager().GetTagSession(perfTag.GetTagId(), tagsSession);
        if (!optionalSession) {
            return false;
        }
        auto sessionInfo = *optionalSession;
        if (sessionInfo && sessionInfo->GetUserId() == GetContext()->GetUserId()) {
            sessionId = sessionInfo->GetSessionId();
        }
    }
    return true;
}

bool TPastFeedbackScriptAction::FindSessionAndDevice(const TString& messageText, TString& sessionId, TString& deviceId, NDrive::TEntitySession& tagsSession) const {
    auto ydbTx = GetContext()->GetServer().GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("past_feedback_script_action", &GetContext()->GetServer());
    auto optionalSessions = GetContext()->GetServer().GetDriveAPI()->GetMinimalCompiledRides().Get<TMinimalCompiledRiding>({ messageText }, tagsSession, ydbTx);
    if (!optionalSessions) {
        return false;
    }
    if (!optionalSessions->empty()) {
        sessionId = optionalSessions->front().GetSessionId();
        deviceId = optionalSessions->front().TMinimalCompiledRiding::GetObjectId();
    }
    return true;
}
