#include "options.h"

#include <drive/backend/chat/message.h>
#include <drive/library/cpp/trust/entity.h>

IChatScriptAction::TFactory::TRegistrator<TTreeScriptAction> TTreeScriptAction::Registrator(TTreeScriptAction::GetType());
IChatScriptAction::TFactory::TRegistrator<TContextButtonScriptAction> TContextButtonScriptAction::Registrator(TContextButtonScriptAction::GetType());

TMaybe<TString> IOptionsScriptAction::AcceptMessage(const TNextActionInfo& actionInfo, TChatContext& stateContext, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagsSession) const {
    Y_UNUSED(tagsSession);
    auto evlog = NDrive::GetThreadEventLogger();
    TVector<TTreeRouteText> supportMessagesRoute;
    TString nextStepId;
    if (!GetCurrentScriptItem().GetSchema().FillTreeMessagesRoute(GetContext(), actionInfo.GetMessage(), stateContext, supportMessagesRoute, nextStepId)) {
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "IOptionsScriptAction::AcceptMessage")
                ("error", "Failed to fill tree message route")
                ("next_node", actionInfo.GetNextNodeId())
                ("fallback_node", GetCurrentScriptItem().GetFallbackNode())
            );
        }
        if (!GetCurrentScriptItem().GetFallbackNode()) {
            chatSession.SetErrorInfo("IOptionsScriptAction::AcceptMessage", "unexpected message and no fallback node");
            return Nothing();
        }
        AddTag = true;
        if (!AcceptRegularMessage(actionInfo.GetMessage(), actionInfo.GetAttachments(), stateContext, chatSession)) {
            if (evlog) {
                evlog->AddEvent(NJson::TMapBuilder
                    ("event", "IOptionsScriptAction::AcceptMessage")
                    ("error", "Failed to accept regular message")
                    ("next_node", actionInfo.GetNextNodeId())
                    ("session_info", chatSession.GetStringReport())
                );
            }
            return Nothing();
        }
        nextStepId = GetCurrentScriptItem().GetFallbackNode();
    } else {
        if (supportMessagesRoute.size() > 0) {
            auto finalRoute = supportMessagesRoute.back();
            NDrive::NChat::TMessage actualMessage = actionInfo.GetMessage();
            if (finalRoute.GetType() == NChatRobot::ETreeSchema::PhoneNumber) {
                TString phoneDescriptionStr = actionInfo.GetMessage().GetText();
                NJson::TJsonValue json;
                if (!ReadJsonFastTree(phoneDescriptionStr, &json)) {
                    chatSession.SetErrorInfo("IOptionsScriptAction::AcceptMessage", "cannot parse phone data");
                    return Nothing();
                }
                GetCurrentScriptItem().GetJsonMapper().SaveFieldsToContext(stateContext, json);
                auto actualMessageText = GetContext()->Unescape(finalRoute.GetMessageText(), stateContext);
                actualMessage.SetText(actualMessageText);
                actualMessage.SetType(NDrive::NChat::TMessage::EMessageType::PhoneNumber);
            } else if (finalRoute.GetType() == NChatRobot::ETreeSchema::CreditCardSchema) {
                actualMessage.SetText("");
                actualMessage.SetType(NDrive::NChat::TMessage::EMessageType::CreditCardBinding);
                auto card = FindCreditCardById(GetContext(), actionInfo.GetMessage().GetText());
                if (!card) {
                    chatSession.SetErrorInfo("IOptionsScriptAction::AcceptMessage", "cannot get user cards");
                    return Nothing();
                }
                if (card->GetId().empty()) {
                    if (!GetCurrentScriptItem().GetFallbackNode()) {
                        chatSession.SetErrorInfo("IOptionsScriptAction::AcceptMessage", "unexpected message and no fallback node");
                        return Nothing();
                    }
                    nextStepId = GetCurrentScriptItem().GetFallbackNode();
                } else {
                    actualMessage.SetText(card->GetAccount());
                    GetCurrentScriptItem().AddSpecialValueToContext(stateContext, EContextDataType::CreditCardId, card->GetId());
                    GetCurrentScriptItem().AddSpecialValueToContext(stateContext, EContextDataType::CreditCardMask, card->GetAccount());
                }
            } else {
                actualMessage.SetType(NDrive::NChat::TMessage::EMessageType::Plaintext);
                actualMessage.SetText(supportMessagesRoute.back().GetHistoryText());
                if (GetCurrentScriptItem().GetActionType() == NChatRobot::EUserAction::Tree && !GetContext()->GetChatRobot()->OverrideTitle(GetContext()->GetUserId(), GetContext()->GetChatTopic(), supportMessagesRoute.front().GetMenuText(), chatSession)) {
                    return Nothing();
                }
            }
            if (!GetContext()->GetChatRobot()->SendArbitraryMessage(GetContext()->GetUserId(), GetContext()->GetChatTopic(), GetOperatorId(), actualMessage, chatSession, &stateContext)) {
                return Nothing();
            }
        }
    }
    return nextStepId;
}
