#include "phone_verification.h"

#include <drive/backend/user_devices/manager.h>
#include <drive/backend/database/drive_api.h>

TMaybe<TString> IVerificationScriptAction::AcceptMessage(const TNextActionInfo& actionInfo, TChatContext& stateContext, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagsSession) const {
    const TString& messageText = actionInfo.GetMessage().GetText();
    TString nextNodeId;
    if (actionInfo.GetMessage().GetType() == NDrive::NChat::TMessage::EMessageType::PhoneVerificationCode) {
        TSet<TString> errorCodes;
        auto isFinishDone = AcceptCode(messageText, errorCodes, tagsSession);
        if (!isFinishDone) {
            chatSession.SetErrorInfo("IVerificationScriptAction::AcceptCode", tagsSession.GetStringReport());
            return Nothing();
        }
        if (!*isFinishDone) {
            if (!errorCodes.empty()) {
                auto primaryError = *errorCodes.begin();
                GetCurrentScriptItem().AddSpecialValueToContext(stateContext, EContextDataType::PhoneVerificationError, primaryError);
            }
            nextNodeId = GetCurrentScriptItem().GetFallbackNode();
        }
        NDrive::NChat::TMessage actualMessage = actionInfo.GetMessage();
        actualMessage.SetType(NDrive::NChat::TMessage::EMessageType::Plaintext);
        if (!AcceptRegularMessage(actualMessage, actionInfo.GetAttachments(), stateContext, chatSession)) {
            return Nothing();
        }
    } else if (actionInfo.GetMessage().GetType() == NDrive::NChat::TMessage::EMessageType::NodeId) {
        TMaybe<TChatRobotScriptItem> nextScriptItem = GetChatScript().GetScriptItemById(messageText);
        if (!nextScriptItem) {
            chatSession.SetErrorInfo("IVerificationScriptAction::AcceptMessage", "node " + messageText + " does not exist");
            return Nothing();
        }
        nextNodeId = messageText;
    } else {
        chatSession.SetErrorInfo("TDeviceVerificationScriptAction::AcceptMessage", "message type is not supported");
        return Nothing();
    }
    return nextNodeId;
}

IChatScriptAction::TFactory::TRegistrator<TDeviceVerificationScriptAction> TDeviceVerificationScriptAction::Registrator(TDeviceVerificationScriptAction::GetType());

TMaybe<bool> TDeviceVerificationScriptAction::AcceptCode(const TString& messageText, TSet<TString>& errorCodes, NDrive::TEntitySession& tagsSession) const {
    if (!GetContext()->GetServer().GetUserDevicesManager()) {
        tagsSession.SetErrorInfo("TDeviceVerificationScriptAction::AcceptMessage", "devices manager not configured");
        return {};
    }
    auto deadline = TInstant::Zero();
    if (!GetContext()->GetServer().GetUserDevicesManager()->FinishDeviceIdVerification(GetContext()->GetUserId(), GetContext()->GetDeviceId(), messageText, GetContext()->GetClientIp(), deadline, errorCodes, tagsSession)) {
        return {};
    }
    return true;
}

IChatScriptAction::TFactory::TRegistrator<TPhoneVerificationScriptAction> TPhoneVerificationScriptAction::Registrator(TPhoneVerificationScriptAction::GetType());

TMaybe<bool> TPhoneVerificationScriptAction::AcceptCode(const TString& messageText, TSet<TString>& errorCodes, NDrive::TEntitySession& tagsSession) const {
    if (!GetContext()->GetServer().GetUserDevicesManager()) {
        tagsSession.SetErrorInfo("TPhoneVerificationScriptAction::AcceptMessage", "devices manager not configured");
        return {};
    }

    TVector<TDBTag> userTags;
    if (!GetContext()->GetServer().GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(GetContext()->GetUserId(), { TPassportTrackTag::TypeName }, userTags, tagsSession)) {
        return {};
    }

    const TPassportTrackTag* trackTag;
    if (userTags.empty() || !(trackTag = userTags.begin()->GetTagAs<TPassportTrackTag>())) {
        tagsSession.SetErrorInfo("TPhoneVerificationScriptAction::AcceptMessage", "incorrect track", EDriveSessionResult::InternalError);
        return {};
    }

    auto deadline = TInstant::Zero();
    if (!GetContext()->GetServer().GetUserDevicesManager()->FinishPhoneVerification(trackTag->GetTrackId(), messageText, GetContext()->GetClientIp(), deadline, errorCodes)) {
        return false;
    }
    auto userId = GetContext()->GetUserId();
    auto userFetchResult = GetContext()->GetServer().GetDriveAPI()->GetUsersData()->FetchInfo(userId, tagsSession);
    auto userPtr = userFetchResult.GetResultPtr(userId);
    if (!userPtr) {
        tagsSession.SetErrorInfo("TPhoneVerificationScriptAction::AcceptMessage", "Incorrect userId " + userId);
        return {};
    }

    auto user = *userPtr;
    user.SetPhoneVerified(true);

    if (!GetContext()->GetServer().GetDriveAPI()->GetUsersData()->UpdateUser(user, userId, tagsSession)) {
        return {};
    }
    return true;
}
