#include "script_action.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/database/drive_api.h>

#include <drive/backend/data/container_tag.h>
#include <drive/backend/data/support_tags.h>
#include <drive/library/cpp/trust/entity.h>

bool IChatScriptAction::ShouldUseSuggest(const NDrive::NChat::TMessage& message) const {
    Y_UNUSED(message);
    return GetCurrentScriptItem().GetUseClassifier() && GetCurrentScriptItem().GetNodeResolver();
}

TString IChatScriptAction::GetTopicLink() const {
    auto topicLink = Context->GetChatId();
    if (Context->GetChatTopic()) {
        topicLink += "." + Context->GetChatTopic();
    }
    return topicLink;
}

bool UndeferSupportTag(const TDBTag& tag, const TString& defaultTagName, bool moveCall, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    auto dbTag = tag;
    auto implPtr = dbTag.GetTagAs<TSupportChatTag>();
    if (!implPtr) {
        session.SetErrorInfo("UndeferDialogue", "tag is not support chat tag", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    if (implPtr->GetName() == TSupportChatTag::DeferredName || (implPtr->GetName() != defaultTagName && moveCall)) {
        TString supportLine = defaultTagName;
        if (implPtr->GetOriginalSupportLine() && !moveCall) {
            supportLine = implPtr->GetOriginalSupportLine();
        }
        auto newTag = dbTag.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
        if (!newTag) {
            session.SetErrorInfo("UndeferDialogue", "can't copy tag", EDriveSessionResult::InternalError);
            return false;
        }
        newTag->SetName(supportLine);
        auto evolved = server->GetDriveDatabase().GetTagsManager().GetUserTags().DirectEvolveTag("robot-frontend", dbTag, newTag.GetData(), session);
        if (!evolved) {
            return false;
        }
        implPtr = evolved->GetTagAs<TSupportChatTag>();
        if (!implPtr) {
            session.SetErrorInfo("UndeferDialogue", "evolved tag is not support chat tag", EDriveSessionResult::IncorrectRequest);
            return false;
        }
        dbTag = std::move(*evolved);
    }

    if (implPtr->IsMuted()) {
        return TSupportChatTag::ChangeMutedStatus(dbTag, false, dbTag.GetObjectId(), server, session);
    }
    return true;
}

bool UndeferContainer(const TDBTag& tag, const TString& defaultTagName, bool moveCall, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    auto permissions = server->GetDriveAPI()->GetUserPermissions(tag.GetObjectId(), TUserPermissionsFeatures());
    if (!permissions) {
        session.SetErrorInfo("UndeferContainer", "can't get permissions", EDriveSessionResult::InternalError);
        return false;
    }
    auto optionalTag = IContainerTag::UndeferContainer(tag, permissions, server, server->GetDriveAPI()->GetTagsManager().GetUserTags(), session);
    if (!optionalTag) {
        session.AddErrorMessage("UndeferContainer", "Container undefer failed");
        return false;
    }
    auto implPtr = optionalTag->GetTagAs<TSupportChatTag>();
    if (!implPtr) {
        return true;
    }
    return UndeferSupportTag(*optionalTag, defaultTagName, moveCall, server, session);
}

bool UndeferDialogue(const TDBTag& tag, const TString& defaultTagName, bool moveCall, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    auto containerPtr = tag.GetTagAs<IContainerTag>();
    if (containerPtr) {
        return UndeferContainer(tag, defaultTagName, moveCall, server, session);
    }
    return UndeferSupportTag(tag, defaultTagName, moveCall, server, session);
}

bool IChatScriptAction::MatchingUndefer(const TDBTag& tag, const TString& topicLink, const TString& supportLineTag, bool moveCall, const NDrive::IServer* server, NDrive::TEntitySession& session, const bool setSessionError) {
    ITag::TPtr containerImpl;
    auto containerTag = tag.GetTagAs<IContainerTag>();
    auto topicLinkTag = tag.GetTagAs<ITopicLinkOwner>();
    if (containerTag) {
        containerImpl = containerTag->RestoreTag();
        topicLinkTag = dynamic_cast<ITopicLinkOwner*>(containerImpl.Get());
    }
    if (!topicLinkTag || topicLinkTag->GetTopicLink() == topicLink) {
        return UndeferDialogue(tag, supportLineTag, moveCall, server, session);
    } else if (setSessionError) {
        session.SetErrorInfo("MatchingUndefer", "Tag " + tag.GetTagId() + " is not ITopicLinkOwner or topic link " + topicLink + " is not matching");
    }
    return false;
}

bool IChatScriptAction::MaybeAddSupportRequestTags(const TString& supportLineTag, bool moveCall, NDrive::TEntitySession& tagsSession) const {
    auto topicLink = Context->GetChatId();
    if (Context->GetChatTopic()) {
        topicLink += "." + Context->GetChatTopic();
    }
    TSet<TString> tagNames = Context->GetServer().GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TSupportChatTag::TypeName});
    tagNames.emplace(TSupportChatTag::DeferredContainerName);
    auto userTags = Context->GetServer().GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(Context->GetUserId(), MakeVector(tagNames), tagsSession);
    if (!userTags) {
        return false;
    }
    for (auto&& tag : *userTags) {
        if (MatchingUndefer(tag, topicLink, supportLineTag, moveCall, &Context->GetServer(), tagsSession)) {
            return true;
        }
    }
    auto tag = Context->GetServer().GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(supportLineTag, "Обращение из чата");
    if (!tag) {
        tagsSession.SetErrorInfo("IChatScriptAction::MaybeAddSupportRequestTags", TStringBuilder() << "cannot create tag " << supportLineTag);
        return false;
    }
    {
        auto tagSupport = dynamic_cast<TSupportChatTag*>(tag.Get());
        if (!tagSupport) {
            tagsSession.SetErrorInfo("IChatScriptAction::MaybeAddSupportRequestTags", TStringBuilder() << "cannot cast tag " << tag->GetName() << " to SupportChatTag");
            return false;
        }
        tagSupport->SetTopicLink(topicLink);
    }
    auto optionalAddedTags = Context->GetServer().GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, Context->GetChatRobot()->GetChatConfig().GetRobotUserId(), Context->GetUserId(), &Context->GetServer(), tagsSession);
    return optionalAddedTags.Defined();
}

TMaybe<NDrive::NTrustClient::TPaymentMethod> IChatScriptAction::FindCreditCardById(const IChatUserContext::TPtr context, const TString& cardId) const {
    auto permissions = context->GetUserPermissions();
    if (!permissions) {
        return Nothing();
    }
    auto userCards = context->GetServer().GetDriveAPI()->GetUserPaymentMethodsSync(*permissions, context->GetServer(), false);
    if (!userCards) {
        return Nothing();
    }
    for (auto&& card : *userCards) {
        if (card.Check(cardId)) {
            return card;
        }
    }
    TVector<TString> parts = SplitString(cardId, "*");
    ui32 bin;
    if (parts.size() == 2 && TryFromString<ui32>(parts[0], bin)) {
        TString suffix = parts[1];
        for (auto&& card : *userCards) {
            if (card.GetBIN() == bin && card.GetSuffix() == suffix) {
                return card;
            }
        }
    }
    return NDrive::NTrustClient::TPaymentMethod();
}

bool IChatScriptAction::AcceptRegularMessage(const NDrive::NChat::TMessage& chatMessage, const TVector<TMessageAttachment>& attachments, TChatContext& stateContext, NDrive::TEntitySession& chatSession, bool registerResource, bool fromRobot) const {
    auto actualMessage = chatMessage;
    if (actualMessage.GetText() && !attachments.empty()) {
        chatSession.SetErrorInfo("IChatScriptAction::AcceptRegularMessage", "both message and are attachments defined");
        return false;
    }
    if (actualMessage.GetText().empty() && attachments.empty()) {
        chatSession.SetErrorInfo("IChatScriptAction::AcceptRegularMessage", "text and attachments are empty");
        return false;
    }
    auto operatorId = fromRobot ? Context->GetChatRobot()->GetChatConfig().GetRobotUserId() : OperatorId;
    if (actualMessage.GetText()) {
        if (!Context->GetChatRobot()->SendArbitraryMessage(Context->GetUserId(), Context->GetChatTopic(), operatorId, actualMessage, chatSession, &stateContext)) {
            return false;
        }
    } else {
        if (actualMessage.GetType() == NDrive::NChat::TMessage::EMessageType::Unknown || actualMessage.GetType() == NDrive::NChat::TMessage::EMessageType::Plaintext) {
            actualMessage.SetType(NDrive::NChat::TMessage::EMessageType::MediaResources);
        }
        TString messageText = "";
        for (auto&& attachment : attachments) {
            if (messageText != "") {
                messageText += ",";
            }
            messageText += ToLowerUTF8(attachment.Data);
            if (registerResource && !Context->GetChatRobot()->RegisterMediaResource(Context->GetUserId(), attachment.Data, attachment.ContentType, false, chatSession)) {
                return false;
            }
        }
        actualMessage.SetText(messageText);
        if (!Context->GetChatRobot()->SendArbitraryMessage(Context->GetUserId(), Context->GetChatTopic(), operatorId, actualMessage, chatSession, &stateContext)) {
            return false;
        }
    }
    return true;
}

NThreading::TFuture<TNextActionInfo> IChatScriptAction::ProcessAsyncOperations(const NDrive::NChat::TMessage& message, const TVector<TMessageAttachment>& attachments) const {
    TNextActionInfo result{message, attachments};
    if (ShouldUseSuggest(message)) {
        TMaybe<NDrive::NChat::TMessageEvents> optionalMessages;
        {
            auto readOnlySession = Context->GetServer().GetChatEngine()->BuildSession(true);
            optionalMessages = Context->GetChatRobot()->GetChatMessages(Context->GetUserId(), Context->GetChatTopic(), readOnlySession, NDrive::NChat::TMessage::AllKnownTraits, 0);
            if (!optionalMessages) {
                return NThreading::TExceptionFuture() << "IChatScriptAction::ProcessAsyncOperations: " << readOnlySession.GetStringReport();
            }
        }
        optionalMessages->emplace_back(TObjectEvent<NDrive::NChat::TMessage>(message, EObjectHistoryAction::Add, Now(), GetOperatorId(), GetOperatorId(), ""));
        if (!GetCurrentScriptItem().GetNodeResolver()) {
            return NThreading::TExceptionFuture() << "IChatScriptAction::ProcessAsyncOperations: " << "Node resolver not configured";
        }
        return GetCurrentScriptItem().GetNodeResolver()->GetNextNodeFuture(Context, *optionalMessages).Apply([result](const auto& f) mutable {
            result.SetNextNodeId(f.GetValue());
            return result;
        });
    }
    return NThreading::MakeFuture(result);
}

NThreading::TFuture<void> IChatScriptAction::ProcessUserResponse(const NDrive::NChat::TMessage& message, const TVector<TMessageAttachment>& attachments) const {
    TString nextNodeId;
    auto eventLogState = NDrive::TEventLog::CaptureState();
    auto nextNodeFuture = ProcessAsyncOperations(message, attachments);
    return nextNodeFuture.Apply([
        context = Context,
        eventLogState = std::move(eventLogState),
        scriptItem = CurrentScriptItem,
        operatorId = OperatorId,
        message,
        attachments
    ] (const NThreading::TFuture<TNextActionInfo>& f) {
        auto elsg = NDrive::TEventLog::Guard(eventLogState);
        auto preliminaryNextActionInfo = f.GetValue();
        if (auto currentScriptAction = IChatScriptAction::Construct(context, scriptItem, operatorId)) {
            return currentScriptAction->OnUserAction(preliminaryNextActionInfo);
        } else {
            return NThreading::MakeErrorFuture<void>(std::make_exception_ptr(yexception() << "IChatScriptAction::ProcessUserResponse: cannot construct script action"));
        }
    });
}

NThreading::TFuture<void> IChatScriptAction::OnUserAction(const TNextActionInfo& actionInfo) const {
    const bool sameDB = Context->GetServer().GetChatEngine()->GetDatabaseName() == Context->GetServer().GetDriveAPI()->GetDatabaseName();
    auto chatSession = Context->GetServer().GetChatEngine()->BuildSession(false);

    NDrive::TEntitySession tagsSession;
    if (!sameDB) {
        tagsSession = Context->GetServer().GetDriveAPI()->template BuildTx<NSQL::Writable>();
    }
    TChatContext stateContext(sameDB ? &chatSession : &tagsSession, &chatSession);

    NDrive::NChat::TMessage message = actionInfo.GetMessage();
    ui32 chatRoomId;
    if (!Context->GetChatRobot()->GetOrCreateChat(Context->GetUserId(), Context->GetChatTopic(), chatRoomId, stateContext, chatSession)) {
        return NThreading::TExceptionFuture() << "cannot GetOrCreateChat: " << chatSession.GetStringReport();
    }
    if (!CurrentScriptItem.SaveFieldsToContext(stateContext, Context)) {
        return NThreading::TExceptionFuture() << "cannot save fields to context";
    }
    if (Context->GetUserId() != OperatorId) {
        message.SetIsRateable(Context->GetChatRobot()->GetChatConfig().GetChatScript().HasRatings());
        if (AcceptRegularMessage(message, actionInfo.GetAttachments(), stateContext, chatSession) && chatSession.Commit()) {
            return NThreading::MakeFuture();
        } else {
            return NThreading::TExceptionFuture() << "cannot AcceptRegularMessage: " << chatSession.GetStringReport();
        }
    }
    if (!sameDB) {
        if (!MoveToNextNode(actionInfo, stateContext, chatRoomId, chatSession, tagsSession, actionInfo.GetNextNodeId())) {
            return NThreading::TExceptionFuture() << "Tags session: " << tagsSession.GetStringReport() << "; Chat session: " << chatSession.GetStringReport();
        }
    } else if (!MoveToNextNode(actionInfo, stateContext, chatRoomId, chatSession, chatSession, actionInfo.GetNextNodeId())) {
        return NThreading::TExceptionFuture() << "cannot MoveToNextNode: " << chatSession.GetStringReport();
    }
    if (!tagsSession.Commit()) {
        return NThreading::TExceptionFuture() << "cannot Commit tags session: " << tagsSession.GetStringReport();
    }
    if (!chatSession.Commit()) {
        return NThreading::TExceptionFuture() << "cannot Commit chats session: " << chatSession.GetStringReport();
    }
    return NThreading::MakeFuture();
}

bool IChatScriptAction::MoveToNextNode(const TNextActionInfo& actionInfo, TChatContext stateContext, ui32 chatRoomId, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagsSession, TString nextNodeId) const {
    auto optionalNextNodeId = AcceptMessage(actionInfo, stateContext, chatSession, tagsSession);
    if (!optionalNextNodeId) {
        chatSession.AddErrorMessage("IChatScriptAction::MoveToNextNode", "failed to accept message");
        return false;
    }
    nextNodeId = optionalNextNodeId->empty() ? nextNodeId : *optionalNextNodeId;

    const TString& userId = Context->GetUserId();
    const TString& topic = Context->GetChatTopic();
    TMaybe<TChatRobotScriptItem> nextScriptItem;
    if (nextNodeId.empty()) {
        nextScriptItem = Context->GetChatRobot()->GetChatConfig().GetChatScript().GetNextScriptItem(CurrentScriptItem, Context, stateContext);
    } else {
        nextScriptItem = Context->GetChatRobot()->GetChatConfig().GetChatScript().GetScriptItemById(nextNodeId);
    }

    bool shouldAddTag = AddTag;
    if (nextScriptItem && nextScriptItem->GetId() != CurrentScriptItem.GetId()) {
        shouldAddTag &= !nextScriptItem->GetPutTagOnEntry();
    }
    if (!OnExit(shouldAddTag, stateContext, chatSession, tagsSession)) {
        chatSession.AddErrorMessage("IChatScriptAction::MoveToNextNode", "on exit trigger failed");
        return false;
    }

    TString nextStep;
    if (nextScriptItem && nextScriptItem->GetId() && nextScriptItem->GetId() != CurrentScriptItem.GetId()) {
        nextStep = nextScriptItem->GetId();
        if (auto nextScriptActions = IChatScriptAction::Construct(Context, *nextScriptItem, OperatorId)) {
            if (!nextScriptActions->OnEnter(chatRoomId, stateContext, chatSession, tagsSession, nextStep)) {
                chatSession.AddErrorMessage("IChatScriptAction::MoveToNextNode", "on enter trigger of " + nextScriptItem->GetId() + " failed");
                return false;
            }
        } else {
            chatSession.SetErrorInfo("IChatScriptAction::MoveToNextNode", "cannot construct action " + ToString(nextScriptItem->GetActionType()));
            return false;
        }
    }
    if (!Context->GetChatRobot()->UpdateChat(stateContext, userId, topic, nextStep, chatSession)) {
        chatSession.AddErrorMessage("IChatScriptAction::MoveToNextNode", "failed to update chat");
        return false;
    }
    return true;
}

bool IChatScriptAction::OnEnter(const ui32 chatRoomId, TChatContext& stateContext, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagsSession, TString& nextNodeId, const bool sendMessages, const TMaybe<ui64> lastMessageId) const {
    if (auto eventLogger = NDrive::GetThreadEventLogger()) {
        eventLogger->AddEvent(NJson::TMapBuilder
            ("event", "OnEnter")
            ("step", nextNodeId)
        );
    }
    if (!CurrentScriptItem.SaveFieldsToContext(stateContext, Context)) {
        chatSession.SetErrorInfo("IChatScriptAction::OnEnter", "cannot SaveFieldsToContext");
        return false;
    }
    auto chatRobotImpl = Context->GetChatRobot();
    if (chatRobotImpl && chatRobotImpl->GetChatConfig().IsPersistVisitedStates()) {
        stateContext.MutableVisitedStates().push_back(nextNodeId);
    }
    if (sendMessages && chatRobotImpl && !chatRobotImpl->SendPreActionMessages(chatRoomId, CurrentScriptItem.GetId(), chatSession, Context->GetChatRobot()->GetChatConfig().GetRobotUserId(), stateContext, lastMessageId)) {
        chatSession.AddErrorMessage("IChatScriptAction::OnEnter", "cannot SendPreActionMessages");
        return false;
    }
    TString supportLineTag = CurrentScriptItem.GetSupportLineTag();
    if (supportLineTag && CurrentScriptItem.GetPutTagOnEntry() && !MaybeAddSupportRequestTags(supportLineTag, CurrentScriptItem.GetMoveSupportCall(), tagsSession)) {
        chatSession.AddErrorMessage("IChatScriptAction::OnEnter", "cannot MaybeAddSupportRequestTags: " + tagsSession.GetStringReport());
        return false;
    }
    if (CurrentScriptItem.HasPreEntryActions() && !Context->GetChatRobot()->DoPreEntryActions(Context, CurrentScriptItem, stateContext, tagsSession, chatSession)) {
        chatSession.AddErrorMessage("IChatScriptAction::OnEnter", "cannot DoPreEntryActions: " + tagsSession.GetStringReport());
        return false;
    }
    return true;
}

bool IChatScriptAction::OnExit(const bool shouldAddTag, TChatContext& stateContext, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagsSession) const {
    Y_UNUSED(stateContext);
    Y_UNUSED(chatSession);
    if (auto eventLogger = NDrive::GetThreadEventLogger()) {
        eventLogger->AddEvent(NJson::TMapBuilder
            ("event", "OnExit")
            ("step", CurrentScriptItem.GetId())
        );
    }
    TString supportLineTag;
    if (shouldAddTag && !CurrentScriptItem.GetSuppressSupportCall()) {
        supportLineTag = CurrentScriptItem.GetSupportLineTag();
    }
    if (supportLineTag && !MaybeAddSupportRequestTags(supportLineTag, CurrentScriptItem.GetMoveSupportCall(), tagsSession)) {
        return false;
    }
    return true;
}

bool IChatScriptAction::OnExternalEvent(TChatContext& stateContext) const {
    auto* chatSessionPtr = stateContext.GetChatSession();
    auto* tagSessionPtr = stateContext.GetTagsSession();
    if (!chatSessionPtr || !tagSessionPtr) {
        ERROR_LOG << "IChatScriptAction::OnExternalEvent: missing db session" << Endl;
        return false;
    }

    ui32 chatRoomId;
    if (!Context->GetChatRobot()->GetOrCreateChat(Context->GetUserId(), Context->GetChatTopic(), chatRoomId, stateContext, *chatSessionPtr)) {
        return false;
    }
    if (!OnExternalEvent(chatRoomId, stateContext, *chatSessionPtr, *tagSessionPtr)) {
        return false;
    }
    return true;
}
