#pragma once

#include <drive/backend/data/support_tags.h>
#include <drive/backend/chat/message.h>
#include <drive/backend/chat_robots/configuration/chat_script_item.h>
#include <drive/backend/chat_robots/configuration/chat_script.h>
#include <drive/backend/chat_robots/configuration/config.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/util/types/messages_collector.h>

namespace NDrive {
    namespace NTrustClient {
        class TPaymentMethod;
    }
}

class IChatScriptAction {
    R_READONLY(IChatUserContext::TPtr, Context);
    R_READONLY(TChatRobotScriptItem, CurrentScriptItem);
    R_READONLY(TString, OperatorId);

public:
    using TFactory = NObjectFactory::TParametrizedObjectFactory<IChatScriptAction, NChatRobot::EUserAction, const IChatUserContext::TPtr, const TChatRobotScriptItem&, const TString&>;
    using TPtr = TAtomicSharedPtr<IChatScriptAction>;

    IChatScriptAction(const IChatUserContext::TPtr ctx, const TChatRobotScriptItem& scriptItem, const TString& operatorId)
        : Context(ctx)
        , CurrentScriptItem(scriptItem)
        , OperatorId(operatorId)
    {
    }

    virtual ~IChatScriptAction() = default;

    NThreading::TFuture<void> ProcessUserResponse(const NDrive::NChat::TMessage& message, const TVector<TMessageAttachment>& attachments) const;
    virtual bool OnEnter(const ui32 chatRoomId, TChatContext& stateContext, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagsSession, TString& nextNodeId, const bool sendMessages = true, const TMaybe<ui64> lastMessageId = {}) const;
    virtual bool OnExit(const bool shouldAddTag, TChatContext& stateContext, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagsSession) const;
    virtual bool OnExternalEvent(TChatContext& stateContext) const;
    static IChatScriptAction::TPtr Construct(const IChatUserContext::TPtr ctx, const TChatRobotScriptItem& scriptItem, const TString& operatorId) {
        return IChatScriptAction::TFactory::Construct(scriptItem.GetActionType(), ctx, scriptItem, operatorId);
    }
    static bool MatchingUndefer(const TDBTag& tag, const TString& topicLink, const TString& supportLineTag, bool moveCall, const NDrive::IServer* server, NDrive::TEntitySession& session, const bool setSessionError = false);

protected:
    virtual bool ShouldUseSuggest(const NDrive::NChat::TMessage& message) const;
    TString GetTopicLink() const;
    bool AcceptRegularMessage(const NDrive::NChat::TMessage& chatMessage, const TVector<TMessageAttachment>& attachments, TChatContext& context, NDrive::TEntitySession& chatSession, bool registerResource = true, bool fromRobot = false) const;
    bool MaybeAddSupportRequestTags(const TString& supportLineTag, bool moveCall, NDrive::TEntitySession& tagsSession) const;
    TMaybe<NDrive::NTrustClient::TPaymentMethod> FindCreditCardById(const IChatUserContext::TPtr context, const TString& cardId) const;

    virtual NThreading::TFuture<TNextActionInfo> ProcessAsyncOperations(const NDrive::NChat::TMessage& message, const TVector<TMessageAttachment>& attachments) const;

    NThreading::TFuture<void> OnUserAction(const TNextActionInfo& nextActionInfo) const;
    virtual TMaybe<TString> AcceptMessage(const TNextActionInfo& actionInfo,TChatContext& stateContext, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagsSession) const = 0;

    virtual bool OnExternalEvent(const ui32 /*chatRoomId*/, TChatContext& /*stateContext*/, NDrive::TEntitySession& /*chatSession*/, NDrive::TEntitySession& /*tagsSession*/) const {
        return true;
    }

    const TChatRobotScript& GetChatScript() const {
        return Context->GetChatRobot()->GetChatConfig().GetChatScript();
    }

private:
    bool MoveToNextNode(const TNextActionInfo& nextNodeIdByAction, TChatContext stateContext, ui32 chatRoomId, NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagsSession, TString nextNodeId) const;

protected:
    mutable bool AddTag = false;
};
