#include "simple.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/billing/accounts/trust.h>
#include <drive/backend/billing/trust/charge_logic.h>
#include <drive/backend/chat_robots/script_actions/user_message.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/feedback.h>
#include <drive/backend/data/support_tags.h>
#include <drive/backend/game/logic.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/promo_codes/manager.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/user_devices/manager.h>
#include <drive/backend/user_document_photos/manager.h>

#include <drive/library/cpp/covid_pass/client.h>

#include <rtline/library/json/proto/adapter.h>
#include <rtline/util/algorithm/ptr.h>

namespace {
    TString GetRegistrationChatPhotoActionDefault(NUserDocument::EType type) {
        switch (type) {
        case NUserDocument::EType::LicenseFront:
            return "сторону прав с фотографией";
        case NUserDocument::EType::LicenseBack:
            return "сторону прав с таблицей";
        case NUserDocument::EType::PassportBiographical:
            return "основной разворот паспорта";
        case NUserDocument::EType::PassportRegistration:
            return "разворот паспорта с пропиской";
        case NUserDocument::EType::PassportSelfie:
            return "себя с паспортом";
        case NUserDocument::EType::LicenseSelfie:
            return "себя с правами";
        default:
            return "unknown code: " + ToString(type);
        }
    }

    TString CreateResubmitPhotosMessage(const TVector<NUserDocument::EType>& photoTypes, const ILocalization * localization, ELocalization locale, const TVector<TDocumentResubmitOverride>& resubmitOverrides) {
        TString result = localization ? localization->GetLocalString(locale, "registration.resubmit_message.start", "Нужно еще раз сфотографировать ") : "Нужно еще раз сфотографировать ";
        for (size_t i = 0; i < photoTypes.size(); ++i) {
            if (i > 0) {
                result += ", ";
            }
            auto defaultMessage = GetRegistrationChatPhotoActionDefault(photoTypes[i]);
            auto overridenActionIt = FindIf(resubmitOverrides, [requestedType = photoTypes[i]](const auto& resubmitOverride){ return requestedType == resubmitOverride.ResubmitItem; });
            auto resourceKey = overridenActionIt != resubmitOverrides.end() ? overridenActionIt->Localization : ToString(photoTypes[i]);
            result += localization ? localization->GetLocalString(locale, "registration.resubmit_message." + resourceKey, defaultMessage) : defaultMessage;
        }
        return result;
    }

    TVector<TString> ChatTitles = TVector<TString>({
        "\U0001F3B1",
        "\U0001F697",
        "\U0001F699",
        "\U0001F3D9",
        "\U00002601",
        "\U0001F30E",
        "\U0001F3C0",
        "\U000026BD",
        "\U0001F5BC",
        "\U0001F916"
    });

    const TVector<TString> ChatIconUrls = TVector<TString>({
        "https://carsharing.s3.yandex.net/drive/static/chat/chat-icon.png",
        "https://carsharing.s3.yandex.net/drive/static/chat/chat-icon-1.png",
        "https://carsharing.s3.yandex.net/drive/static/chat/chat-icon-2.png",
        "https://carsharing.s3.yandex.net/drive/static/chat/chat-icon-3.png",
        "https://carsharing.s3.yandex.net/drive/static/chat/chat-icon-4.png",
        "https://carsharing.s3.yandex.net/drive/static/chat/chat-icon-5.png",
        "https://carsharing.s3.yandex.net/drive/static/chat/chat-icon-6.png",
        "https://carsharing.s3.yandex.net/drive/static/chat/chat-icon-7.png",
        "https://carsharing.s3.yandex.net/drive/static/chat/chat-icon-8.png",
        "https://carsharing.s3.yandex.net/drive/static/chat/chat-icon-9.png"
    });
};

const TSet<TString> TSimpleChatBot::InitialRegistrationSteps = TSet<TString>(
    {
        "intro",
        "support_email",
        "agreements",
        "driver_license",
        "passport",
        "passport_selfie"
    }
);

class TDocumentMediaUpdateCallbackWrapper : public IDocumentMediaUpdateCallback {
private:
    const TAtomicSharedPtr<IChatMediaResourcePostUploadCallback> HighLevelCallback;

public:
    TDocumentMediaUpdateCallbackWrapper(TAtomicSharedPtr<IChatMediaResourcePostUploadCallback> highLevelCallback)
        : HighLevelCallback(highLevelCallback)
    {
    }

    void OnSuccess(const TString& photoId, const TString& /*updateMeta*/) override {
        if (HighLevelCallback) {
            HighLevelCallback->OnSuccess(photoId);
        }
    }

    void OnFailure(const TString& updateMeta) override {
        if (HighLevelCallback) {
            HighLevelCallback->OnFailure("MDS upload error: " + updateMeta);
        }
    }
};

bool TSimpleChatBot::UpdateChatRobotStateContext(const TChatContext& chatContext, const TString& userId, const TString& topic, const TChatRobotState& state, NDrive::TEntitySession& session) const {
    TChatRobotState currentState = state;
    {
        currentState.MutableExecutionContext()->SetLastRequestCode(chatContext.GetLastRequestCode());
        currentState.MutableExecutionContext()->SetLastRequestData(chatContext.GetLastRequestData().GetStringRobust());
        currentState.MutableExecutionContext()->SetOperatorName(chatContext.GetOperatorName());
    }

    for (auto&& [key, value] : chatContext.GetContextMap()) {
        NChatRobotState::TChatExecutionContext::TContextField* protoEvent = currentState.MutableExecutionContext()->AddContextMap();
        protoEvent->SetKey(key);
        protoEvent->SetValue(value);
    }
    chatContext.WriteResubmitDataToState(currentState);

    if (!chatContext.GetVisitedStates().empty()) {
        const auto& visitedStates = chatContext.GetVisitedStates();
        TChatRobotState historyState = currentState;
        for (size_t i = 0; i < visitedStates.size() - 1; ++i) {
            historyState.SetCurrentStep(visitedStates[i]);
            AddChatRobotStateHistoryEvent(userId, topic, historyState, session);
        }
    }

    return UpdateChatRobotState(userId, topic, currentState, session);
}

bool TSimpleChatBot::RegisterMediaResource(const TString& userId, const TString& resourceId, const TString& contentType, const bool shared, NDrive::TEntitySession& session) const {
    return MediaStorage.RegisterResource(userId, resourceId, contentType, shared, session).Defined();
}

IChatScriptAction::TPtr TSimpleChatBot::GetChatScriptAction(const IChatUserContext::TPtr context, const TString& operatorId, NDrive::TEntitySession& readOnlySession) const {
    TChatRobotScriptItem currentScriptItem;
    {
        TString currentId;
        auto currentStateExp = GetChatRobotState(context->GetUserId(), context->GetChatTopic(), TInstant::Zero(), &readOnlySession, true);
        if (currentStateExp) {
            currentId = currentStateExp->GetCurrentStep();
        } else {
            if (currentStateExp.GetError() != EChatError::NotFound) {
                return nullptr;
            }
            currentId = ChatConfig.GetInitialStepId();
        }
        if (!ChatConfig.GetChatScript().GetScriptItemById(currentId, currentScriptItem)) {
            readOnlySession.SetErrorInfo("TSimpleChatBot::GetChatScriptAction", "cannot find current script item");
            return nullptr;
        }
    }
    return IChatScriptAction::Construct(context, currentScriptItem, operatorId);
}

NThreading::TFuture<void> TSimpleChatBot::AcceptUserResponse(const IChatUserContext::TPtr context, const TMessage& message, const TVector<TMessageAttachment>& attachments, const TString& operatorId) const {
    IChatScriptAction::TPtr scriptAction;
    {
        auto readOnlySession = BuildChatEngineSession(true);
        scriptAction = GetChatScriptAction(context, operatorId, readOnlySession);
    }
    if (!scriptAction) {
        return NThreading::TExceptionFuture() << "TSimpleChatBot::AcceptUserResponse cannot construct script action";
    }
    return scriptAction->ProcessUserResponse(message, attachments);
}

bool TSimpleChatBot::MaybeContinueChat(const IChatUserContext::TPtr context, const TString& operatorId, TChatContext& stateContext) const {
    auto* chatSessionPtr = stateContext.GetChatSession();
    if (!chatSessionPtr) {
        return false;
    }
    auto scriptAction = GetChatScriptAction(context, operatorId, *chatSessionPtr);
    if (!scriptAction) {
        return false;
    }

    return scriptAction->OnExternalEvent(stateContext);
}

void TSimpleChatBot::AcceptMediaResource(const IChatUserContext::TPtr context, const TString& resourceId, const TString& contentType, const TString& content, TAtomicSharedPtr<IChatMediaResourcePostUploadCallback> callback) const {
    if (resourceId) {
        if (!contentType) {
            if (!!callback) {
                callback->OnFailure("empty content type", HTTP_BAD_REQUEST);
            }
            return;
        }
        MediaStorage.UploadResource(context->GetUserId(), resourceId, contentType, content, callback);
    } else {
        TAtomicSharedPtr<TDocumentMediaUpdateCallbackWrapper> wrapper = new TDocumentMediaUpdateCallbackWrapper(callback);
        DriveAPI.GetDocumentPhotosManager().AddDocumentPhoto(context->GetUserId(), GetFullChatId(context->GetChatTopic()), NUserDocument::EType::Unknown, content, "", wrapper, *Server, "primary");
    }
}

NThreading::TFuture<TChatResourceAcquisitionResult> TSimpleChatBot::GetMediaResource(const IChatUserContext::TPtr context, const TString& resourceId, const bool needsFurtherProcessing) const {
    if (!DriveAPI.HasDocumentPhotosManager()) {
        return NThreading::TExceptionFuture() << "TSimpleChatBot::GetMediaResource: document photo manager not configured";
    }
    if (resourceId.empty()) {
        return NThreading::TExceptionFuture() << "TSimpleChatBot::GetMediaResource: empty resource id";
    }

    auto photoFR = DriveAPI.GetDocumentPhotosManager().GetUserPhotosDB().FetchInfo(resourceId);
    if (!photoFR) {
        return NThreading::TExceptionFuture() << "TSimpleChatBot::GetMediaResource: document photo manager not configured" + photoFR.GetException().GetReport().GetStringRobust();
    }
    if (photoFR.size() != 1) {
        return MediaStorage.AcquireResource(context->GetUserId(), resourceId, needsFurtherProcessing);
    }
    auto photoPtr = photoFR.GetResultPtr(resourceId);
    if (photoPtr->GetUserId() != context->GetUserId()) {
        return NThreading::MakeErrorFuture<TChatResourceAcquisitionResult>(std::make_exception_ptr(NDrive::NChat::TAccessForbiddenException() << "no access to resource"));
    }
    TMediaResourceDescription description(resourceId, photoPtr->GetUserId());
    return DriveAPI.GetDocumentPhotosManager().GetDocumentPhoto(resourceId, *Server).Apply([description] (const auto& reply) mutable {
        description.SetContentType(reply.GetValue().GetContentType());
        return TChatResourceAcquisitionResult(TBlob::FromString(reply.GetValue().GetContent()), description);
    });
}

NThreading::TFuture<TChatResourceAcquisitionResult> TSimpleChatBot::GetMediaResourcePreview(const IChatUserContext::TPtr context, const TString& resourceId, const bool needsFurtherProcessing, const TMap<TString, TString>& typeResourceOverrides) const {
    if (!resourceId) {
        return NThreading::MakeErrorFuture<TChatResourceAcquisitionResult>(std::make_exception_ptr(NDrive::NChat::TBadRequestException() << "bad resource id"));
    }
    return MediaStorage.AcquireResourcePreview(context->GetUserId(), resourceId, needsFurtherProcessing, typeResourceOverrides);
}

NThreading::TFuture<void> TSimpleChatBot::UploadResourcePreview(const TString& content, const TMediaResourceDescription& description, const TString& contentType) const {
    return MediaStorage.UploadResourcePreview(content, description, contentType);
}

bool TSimpleChatBot::UpdateMediaResourceDescription(const TMediaResourceDescription description, const TString& actorUserId, NDrive::TEntitySession& session) const {
    return MediaStorage.UpdateResourceMeta(description, actorUserId, session);
}

NJson::TJsonValue TSimpleChatBot::GetMessageReport(ELocalization locale, const NDrive::NChat::TMessageEvent& message, const TString& userId, const TString& topic) const {
    auto messageReport = IChatRobotImpl::GetMessageReport(locale, message, userId, topic);
    if (message.GetType() == NDrive::NChat::TMessage::EMessageType::MediaResources) {
        NJson::TJsonValue contentTypes = NJson::JSON_ARRAY;
        NJson::TJsonValue cachedFullResources = NJson::JSON_ARRAY;
        NJson::TJsonValue cachedPreviews = NJson::JSON_ARRAY;

        auto resourceIds = SplitString(message.GetText(), ",");
        for (auto&& resourceId : resourceIds) {
            TMediaResourceDescription descr;
            if (!MediaStorage.GetResourceMeta(descr, userId, resourceId)) {
                contentTypes.AppendValue(NJson::JSON_NULL);
                cachedFullResources.AppendValue(NJson::JSON_NULL);
                cachedPreviews.AppendValue(NJson::JSON_NULL);
            } else {
                if (descr.GetContentType()) {
                    contentTypes.AppendValue(descr.GetContentType());
                } else {
                    contentTypes.AppendValue(NJson::JSON_NULL);
                }
                cachedFullResources.AppendValue(descr.BuildPath());
                if (descr.GetHasPreview()) {
                    cachedPreviews.AppendValue(descr.BuildPreviewPath());
                } else {
                    cachedPreviews.AppendValue(NJson::JSON_NULL);
                }
            }
        }
        messageReport["content_types"] = std::move(contentTypes);
        messageReport["cached_previews"] = std::move(cachedPreviews);
        messageReport["cached_images"] = std::move(cachedFullResources);
    }
    return messageReport;
}

TString TSimpleChatBot::GetChatIcon(const NDrive::NChat::TChat& chat) const {
    if (!ChatConfig.GetIcon()) {
        auto chatId = chat.GetId();
        return ChatIconUrls[chatId % ChatIconUrls.size()];
    } else {
        return ChatConfig.GetIcon();
    }
}

bool TSimpleChatBot::AskToResubmit(const TString& userId, const TString& topic, const ui32 resubmitMask, NDrive::TEntitySession& session, const TString& operatorId, const bool force, const TVector<TDocumentResubmitOverride>& resubmitOverrides) const {
    if (resubmitMask == 0) {
        session.SetErrorInfo("SimpleChatBot::AskToResubmit", "zero resubmit mask", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    TChatRobotState currentState;
    if (!GetChatRobotState(userId, topic, currentState)) {
        currentState.SetCurrentStep("photos_being_verified");
        currentState.SetQueuedResubmitMask(0);
        currentState.SetResubmitMask(0);
        currentState.SetOriginalResubmitMask(0);
    }
    TChatContext chatContext(nullptr, &session);
    if (!GetChatContext(currentState, chatContext)) {
        NDrive::TEventLog::Log("SimpleChatBot", NJson::TMapBuilder("error", "cannot get context for AskToResubmit"));
    }

    if (force) {
        chatContext.SetQueuedResubmitMask(0);
        chatContext.SetResubmitMask(0);
        chatContext.SetOriginalResubmitMask(0);
    }

    if (InitialRegistrationSteps.contains(currentState.GetCurrentStep())) {
        session.SetErrorInfo("SimpleChatBot::AskToResubmit", "Cannot resubmit on initial registration steps " + currentState.GetCurrentStep(), EDriveSessionResult::InternalError);
        return false;
    }

    TChatRobotScriptItem currentScriptItem;
    if (!ChatConfig.GetChatScript().GetScriptItemById(currentState.GetCurrentStep(), currentScriptItem)) {
        session.SetErrorInfo("SimpleChatBot::AskToResubmit", "GetScriptItemById step: " + currentState.GetCurrentStep(), EDriveSessionResult::InternalError);
        return false;
    }

    if (chatContext.GetResubmitMask()) {
        if (!currentScriptItem.GetActionType()) {
            chatContext.SetQueuedResubmitMask(0);
            chatContext.SetResubmitMask(0);
            chatContext.SetOriginalResubmitMask(0);
        } else {
            session.SetErrorInfo("SimpleChatBot::AskToResubmit", TStringBuilder() << "currentScriptItem.ActionType: " << currentScriptItem.GetActionType(), EDriveSessionResult::InternalError);
            return false;
        }
    }
    chatContext.SetResubmitOverrides(resubmitOverrides);
    if (!currentScriptItem.IsSkippable()) {
        // If the current chat item has continuation, wait until this storyline comes to an end and then request to resubmit.
        if (currentScriptItem.GetActionType() && currentScriptItem.GetActionType() != NChatRobot::EUserAction::EnterMap && !force) {
            chatContext.SetQueuedResubmitMask(chatContext.GetQueuedResubmitMask() | resubmitMask);
            return UpdateChatRobotStateContext(chatContext, userId, topic, currentState, session);
        }
    }

    chatContext.SetResubmitMask(resubmitMask);
    chatContext.SetOriginalResubmitMask(resubmitMask);

    ui32 chatRoomId;
    if (!GetOrCreateChatRoom(userId, topic, chatRoomId, &session)) {
        return false;
    }

    auto nextChatItem = GetResubmitChatItem(chatContext, currentState, true, userId, session);
    if (!nextChatItem) {
        return false;
    }
    if (resubmitMask & (resubmitMask - 1)) {
        TMessage resubmitIntroMessage;
        resubmitIntroMessage.SetChatId(chatRoomId);
        resubmitIntroMessage.SetText(CreateResubmitMessage(chatContext));
        if (!ChatEngine.AddMessage(resubmitIntroMessage, operatorId, session, GetChatSearchId(userId, topic))) {
            return false;
        }
    }
    if (!SendPreActionMessages(chatRoomId, nextChatItem->GetId(), session, operatorId, chatContext)) {
        return false;
    }
    currentState.SetCurrentStep(nextChatItem->GetId());

    return UpdateChatRobotStateContext(chatContext, userId, topic, currentState, session);
}

TString TSimpleChatBot::CreateResubmitMessage(const TChatContext& chatContext) const {
    auto resubmitMask = chatContext.GetResubmitMask();
    TVector<NUserDocument::EType> requestedTypes;
    for (auto&& type : NUserDocument::AllTypes) {
        if (resubmitMask & type) {
            resubmitMask -= type;
            requestedTypes.push_back(type);
        }
    }
    const auto& resubmitOverrides = chatContext.GetResubmitOverrides();
    return CreateResubmitPhotosMessage(requestedTypes, Server->GetLocalization(), ChatConfig.GetLocale(), resubmitOverrides);
}

TMaybe<TChatRobotScriptItem> TSimpleChatBot::GetResubmitChatItem(const TChatContext& chatContext, const TChatRobotState& currentState, bool isFirstRequest, const TString& userId, NDrive::TEntitySession& session) const {
    auto resubmitMask = chatContext.GetResubmitMask();
    TVector<NUserDocument::EType> requestedTypes;
    for (auto&& type : NUserDocument::AllTypes) {
        if (resubmitMask & type) {
            resubmitMask -= type;
            requestedTypes.push_back(type);
        }
    }
    if (requestedTypes.empty()) {
        session.SetErrorInfo("SimpleChatBot::GetResubmitChatItem", "request resubmit types are empty");
        return {};
    }

    TString actionName;
    if (requestedTypes.size() > 1 || !isFirstRequest) {
        actionName = isFirstRequest ? "start_" : "cont_";
    } else {
        actionName = "only_";
    }

    actionName += "resubmit_";
    const auto& resubmitOverrides = chatContext.GetResubmitOverrides();
    auto overridenActionIt = FindIf(resubmitOverrides, [requestedType = requestedTypes[0]](const auto& resubmitOverride){ return requestedType == resubmitOverride.ResubmitItem; });
    if (overridenActionIt != resubmitOverrides.end()) {
        actionName += overridenActionIt->Node;
    } else {
        if (requestedTypes[0] == NUserDocument::EType::LicenseBack) {
            actionName += "license_back";
        } else if (requestedTypes[0] == NUserDocument::EType::LicenseFront) {
            actionName += "license_front";
        } else if (requestedTypes[0] == NUserDocument::EType::LicenseSelfie) {
            actionName += "license_selfie";
        } else if (requestedTypes[0] == NUserDocument::EType::PassportBiographical) {
            actionName += "passport_biographical";
        } else if (requestedTypes[0] == NUserDocument::EType::PassportRegistration) {
            actionName += "passport_registration";
        } else if (requestedTypes[0] == NUserDocument::EType::PassportSelfie) {
            actionName += "passport_selfie";
        }
    }

    TChatRobotScriptItem result;
    if (!ChatConfig.GetChatScript().GetScriptItemById(actionName, result)) {
        session.SetErrorInfo("SimpleChatBot::GetResubmitChatItem", "cannot GetScriptItemById: " + actionName);
        return {};
    }
    TChatRobotState actualResubmitState = currentState;
    chatContext.WriteResubmitDataToState(actualResubmitState);
    session.Committed().Subscribe([
            isFirstRequest,
            result,
            userId,
            state = std::move(actualResubmitState)
        ](const NThreading::TFuture<void>& committed) {
        if (committed.HasValue()) {
            NDrive::TEventLog::Log("GetResubmitChatItem", NJson::TMapBuilder
                ("current_state", NJson::ToJson(NJson::Proto(state)))
                ("is_first_request", isFirstRequest)
                ("result", result.GetId())
                ("user_id", userId)
            );
        }
    });
    return result;
}

bool TSimpleChatBot::ApproveFinal(const TString& userId, const TString& topic, const TString& scriptItem) const {
    auto operatorUserId = "robot-frontend";
    TChatRobotState currentState;
    if (!GetChatRobotState(userId, topic, currentState)) {
        return false;
    }
    TChatContext chatContext;
    if (!GetChatContext(currentState, chatContext)) {
        return false;
    }
    ui32 chatRoomId;
    if (!GetOrCreateChatRoom(userId, topic, chatRoomId)) {
        return false;
    }
    TChatRobotScriptItem nextChatItem;
    auto session = BuildChatEngineSession();
    if (!ChatConfig.GetChatScript().GetScriptItemById(scriptItem, nextChatItem)) {
        return false;
    }
    if (currentState.GetCurrentStep() != scriptItem && !SendPreActionMessages(chatRoomId, nextChatItem.GetId(), session, operatorUserId, chatContext)) {
        return false;
    }

    currentState.SetCurrentStep(nextChatItem.GetId());
    chatContext.SetResubmitMask(0);
    chatContext.SetQueuedResubmitMask(0);
    chatContext.SetOriginalResubmitMask(0);

    return UpdateChatRobotStateContext(chatContext, userId, topic, currentState, session) && session.Commit();
}

bool TSimpleChatBot::SendRejectionMessages(const TString& userId, const TString& topic, NDrive::TEntitySession& session) const {
    ui32 chatRoomId;
    if (!GetOrCreateChatRoom(userId, topic, chatRoomId)) {
        return false;
    }

    TChatRobotState currentState;
    if (!GetChatRobotState(userId, topic, currentState)) {
        return false;
    }
    TChatContext chatContext(nullptr, &session);
    if (!GetChatContext(currentState, chatContext)) {
        return false;
    }

    if (currentState.GetCurrentStep() == "rejected_general") {
        return true;
    }

    currentState.SetCurrentStep("rejected_general");

    auto operatorUserId = "robot-frontend";
    TChatRobotScriptItem nextChatItem;
    return ChatConfig.GetChatScript().GetScriptItemById("rejected_general", nextChatItem) && SendPreActionMessages(chatRoomId, nextChatItem.GetId(), session, operatorUserId, chatContext) && UpdateChatRobotState(userId, topic, currentState, session);
}

bool TSimpleChatBot::MaybeSendFraudReason(const TString& userId, const TString& topic, const TString& fraudReason, NDrive::TEntitySession& session) const {
    ui32 chatRoomId;
    if (!GetOrCreateChatRoom(userId, topic, chatRoomId)) {
        return false;
    }

    TChatRobotScriptItem nextChatItem;
    if (!ChatConfig.GetChatScript().GetScriptItemById("fraud_reason_" + ToLowerUTF8(fraudReason), nextChatItem)) {
        // Not explainable fraud reason, like photoshop of selfie screencap.
        return true;
    }

    auto operatorUserId = "robot-frontend";
    return SendPreActionMessages(chatRoomId, nextChatItem.GetId(), session, operatorUserId);
}

TMaybe<TNextActionInfo> TSimpleChatBot::CreateActionInfoByResubmit(const TString& userId, const TString& topic, TChatContext& stateContext, TChatRobotState& currentState, bool delayed, NDrive::TEntitySession& chatSession) const {
    TNextActionInfo result;
    auto nextChatItem = GetResubmitChatItem(stateContext, currentState, true, userId, chatSession);
    if (!nextChatItem) {
        return Nothing();
    }
    if (stateContext.GetResubmitMask() & (stateContext.GetResubmitMask() - 1)) {
        TMessage resubmitIntroMessage;
        resubmitIntroMessage.SetText(CreateResubmitMessage(stateContext));
        result.SetMessage(resubmitIntroMessage);
    }
    currentState.SetCurrentStep(nextChatItem->GetId());

    if (!delayed && !UpdateChatRobotStateContext(stateContext, userId, topic, currentState, chatSession)) {
        return Nothing();
    }

    result.SetNextNodeId(nextChatItem->GetId());
    return result;
}

TMaybe<TNextActionInfo> TSimpleChatBot::GetFirstResubmitStep(const IChatUserContext::TPtr context, TChatContext& stateContext, NDrive::TEntitySession& chatSession, ui32 maskOverride) const {
    if (maskOverride == 0) {
        return TNextActionInfo();
    }
    auto userId = context->GetUserId();
    auto topic = context->GetChatTopic();
    TChatRobotState currentState;
    if (!GetChatRobotState(userId, topic, currentState, TInstant::Zero(), &chatSession) ||
        !stateContext.InitResubmitDataFromState(currentState))
    {
        return Nothing();
    }
    ui32 resubmitMask = stateContext.GetResubmitMask() | stateContext.GetQueuedResubmitMask() | maskOverride;
    stateContext.SetResubmitMask(resubmitMask);
    if (!stateContext.GetOriginalResubmitMask()) {
        stateContext.SetOriginalResubmitMask(maskOverride);
    }
    return CreateActionInfoByResubmit(userId, topic, stateContext, currentState, false, chatSession);
}

TMaybe<TNextActionInfo> TSimpleChatBot::GetNextResubmitStep(const IChatUserContext::TPtr context, const TChatRobotScriptItem& currentScriptItem, TChatContext& stateContext, NDrive::TEntitySession& chatSession) const {
    auto userId = context->GetUserId();
    auto topic = context->GetChatTopic();

    TChatRobotState currentState;
    if (!GetChatRobotState(userId, topic, currentState, TInstant::Zero(), &chatSession) ||
        !stateContext.InitResubmitDataFromState(currentState))
    {
        return Nothing();
    }

    TChatRobotScriptItem newScriptItem;
    if (!ChatConfig.GetChatScript().GetNextScriptItem(currentScriptItem, context, stateContext, newScriptItem)) {
        if (stateContext.GetResubmitMask() == 0) {
            return TNextActionInfo{};
        }
        stateContext.SetResubmitMask(stateContext.GetResubmitMask() & (stateContext.GetResubmitMask() - 1));
        TString nextChatItemId;
        if (stateContext.GetResubmitMask() == 0) {
            ui32 mask = stateContext.GetOriginalResubmitMask();
            stateContext.SetOriginalResubmitMask(0);
            stateContext.MutableResubmitOverrides().clear();

            if (mask & (mask - 1)) {
                nextChatItemId = "resubmit_many_complete";
            } else {
                nextChatItemId = "resubmit_one_complete";
            }
        } else {
            if (auto nextChatItem = GetResubmitChatItem(stateContext, currentState, false, userId, chatSession)) {
                nextChatItemId = nextChatItem->GetId();
            } else {
                return Nothing();
            }
        }

        return TNextActionInfo{nextChatItemId};
    }

    if (!newScriptItem.HasFurtherSteps() && stateContext.GetQueuedResubmitMask()) {
        TNextActionInfo result;
        auto resubmitMask = stateContext.GetQueuedResubmitMask();
        stateContext.SetQueuedResubmitMask(0);
        currentState.SetCurrentStep(newScriptItem.GetId());

        if (InitialRegistrationSteps.contains(currentState.GetCurrentStep())) {
            chatSession.SetErrorInfo("SimpleChatBot::GetNextResubmitStep", "Cannot resubmit on initial registration steps" + currentState.GetCurrentStep(), EDriveSessionResult::InternalError);
            return Nothing();
        }

        if (stateContext.GetResubmitMask()) {
            if (!newScriptItem.GetActionType()) {
                stateContext.SetQueuedResubmitMask(0);
                stateContext.SetResubmitMask(0);
                stateContext.SetOriginalResubmitMask(0);
            } else {
                chatSession.SetErrorInfo("SimpleChatBot::GetNextResubmitStep", TStringBuilder() << "newScriptItem.ActionType: " << newScriptItem.GetActionType(), EDriveSessionResult::InternalError);
                return Nothing();
            }
        }
        if (!newScriptItem.IsSkippable()) {
            // If the current chat item has continuation, wait until this storyline comes to an end and then request to resubmit.
            if (newScriptItem.GetActionType() && newScriptItem.GetActionType() != NChatRobot::EUserAction::EnterMap) {
                stateContext.SetQueuedResubmitMask(stateContext.GetQueuedResubmitMask() | resubmitMask);
                return result;
            }
        }
        stateContext.SetResubmitMask(resubmitMask);
        stateContext.SetOriginalResubmitMask(resubmitMask);

        return CreateActionInfoByResubmit(userId, topic, stateContext, currentState, true, chatSession);
    }
    return TNextActionInfo{newScriptItem.GetId()};
}

void TSimpleChatBot::ReportCallbackSuccess(TAtomicSharedPtr<IChatResponsePostProcessingCallback> callback) const {
    if (!!callback) {
        callback->OnSuccess();
    }
}

void TSimpleChatBot::ReportCallbackFailure(TAtomicSharedPtr<IChatResponsePostProcessingCallback> callback, const TString& message, const HttpCodes code) const {
    ERROR_LOG << "ChatCallbackFailure: " << message << Endl;
    if (!!callback) {
        callback->OnFailure(message, code);
    }
}

TString TSimpleChatBot::DeduceBanMessagesGroup(const TString& userId, NDrive::TEntitySession* sessionExternal) const {
    TVector<TDBTag> userTags;

    TVector<TString> tagNames;
    {
        auto tagDescrs = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(TUserProblemTag::TypeName);
        for (auto&& descr : tagDescrs) {
            tagNames.push_back(descr->GetName());
        }
    }

    if (!sessionExternal) {
        auto session = DriveAPI.BuildTx<NSQL::ReadOnly>();
        if (!DriveAPI.GetTagsManager().GetUserTags().RestoreEntityTags(userId, tagNames, userTags, session)) {
            return "";
        }
    } else {
        if (!DriveAPI.GetTagsManager().GetUserTags().RestoreEntityTags(userId, tagNames, userTags, *sessionExternal)) {
            return "";
        }
    }

    TString chatMessagesGroup = "ban_generic";
    ui32 chatMessagesGroupPriority = 0;
    for (auto&& tag : userTags) {
        auto tagPtr = tag.GetTagAs<TUserProblemTag>();
        if (!tagPtr) {
            continue;
        }
        if (tagPtr->GetChatMessagesGroupId(DriveAPI.GetTagsManager().GetTagsMeta()) && tagPtr->GetChatMessageGroupPriority(DriveAPI.GetTagsManager().GetTagsMeta()) >= chatMessagesGroupPriority) {
            chatMessagesGroupPriority = tagPtr->GetChatMessageGroupPriority(DriveAPI.GetTagsManager().GetTagsMeta());
            chatMessagesGroup = tagPtr->GetChatMessagesGroupId(DriveAPI.GetTagsManager().GetTagsMeta());
        }
    }

    return chatMessagesGroup;
}

bool TSimpleChatBot::SendBanMessages(const TString& userId, const TString& topic, NDrive::TEntitySession& tagsSession, const TString& operatorId) const {
    TString nextChatItemId = DeduceBanMessagesGroup(userId, &tagsSession);
    if (!nextChatItemId) {
        return true;
    }

    ui32 chatRoomId;
    if (!GetOrCreateChatRoom(userId, topic, chatRoomId)) {
        return false;
    }

    TChatRobotState currentState;
    if (!GetChatRobotState(userId, topic, currentState)) {
        return false;
    }

    auto execute = [&](NDrive::TEntitySession& tx) {

    if (nextChatItemId == "ban_old_license_intro") {
        // If the reason is just old license and there will be a resubmit, it can either be added to resubmit queue
        if (currentState.GetResubmitMask() || currentState.GetQueuedResubmitMask()) {
            auto qrm = currentState.GetQueuedResubmitMask();
            qrm |= NUserDocument::EType::LicenseBack | NUserDocument::EType::LicenseFront;
            currentState.SetQueuedResubmitMask(qrm);
            auto isUpdateOk = UpdateChatRobotState(userId, topic, currentState, tx);
            return isUpdateOk;
        }

        // If there is no expected resubmit and the registration is still underway, request one
        // Don't interrupt regular flow
        TChatRobotScriptItem currentScriptItem;
        if (!ChatConfig.GetChatScript().GetScriptItemById(currentState.GetCurrentStep(), currentScriptItem)) {
            return false;
        }
        if (currentScriptItem.GetActionType()) {
            auto isReaskOk = AskToResubmit(userId, topic, NUserDocument::EType::LicenseBack | NUserDocument::EType::LicenseFront, tx, operatorId);
            return isReaskOk;
        }
    }

    return MoveToStep(userId, chatRoomId, topic, nextChatItemId, currentState, tx, true);
    };

    if (ChatEngine.GetDatabaseName() == DriveAPI.GetDatabaseName()) {
        return execute(tagsSession);
    } else {
        auto chatSession = BuildChatEngineSession();
        if (!execute(chatSession)) {
            tagsSession.MergeErrorMessages(chatSession.GetMessages(), "ChatSession");
            return false;
        }
        if (!chatSession.Commit()) {
            tagsSession.SetErrorInfo("SimpleChatBot::SendBanMessages", "cannot commit ChatSession", EDriveSessionResult::TransactionProblem);
            tagsSession.MergeErrorMessages(chatSession.GetMessages(), "ChatSession");
            return false;
        }
        return true;
    }
}

bool TSimpleChatBot::DoGetIsChatCompleted(const TString& userId, const TString& topic, const TInstant actuality) const {
    TChatRobotState currentState;
    if (!GetChatRobotState(userId, topic, currentState, actuality)) {
        return false;
    }

    return !currentState.GetOriginalResubmitMask();
}

TString TSimpleChatBot::GetFullChatId(const TString& topic) const {
    return topic.size() ? ChatConfig.GetChatId() + "." + topic : ChatConfig.GetChatId();
}

bool TSimpleChatBot::GetChatContext(const TChatRobotState& state, TChatContext& chatContext) const {
    if (state.HasExecutionContext()) {
        chatContext.SetLastRequestCode(state.GetExecutionContext().GetLastRequestCode());
        if (!NJson::ReadJsonFastTree(state.GetExecutionContext().GetLastRequestData(), &chatContext.MutableLastRequestData())) {
            return false;
        }
        TMap<TString, TString> contextMap;
        for (auto&& pair : state.GetExecutionContext().GetContextMap()) {
            contextMap[pair.GetKey()] = pair.GetValue();
        }
        chatContext.SetContextMap(std::move(contextMap));
        chatContext.SetOperatorName(state.GetExecutionContext().GetOperatorName());
    }
    return chatContext.InitResubmitDataFromState(state);
}
