#pragma once

#include "abstract.h"
#include "condition.h"
#include "media_storage.h"

#include <drive/backend/billing/interfaces/entities.h>
#include <drive/backend/chat_robots/script_actions/script_action.h>
#include <drive/backend/database/drive_api.h>

class TSimpleChatBot : public IStatefulChatRobot<NChatRobotState::TChatRobotState> {
public:
    using TChatRobotState = NChatRobotState::TChatRobotState;

private:
    using TBase = IStatefulChatRobot<NChatRobotState::TChatRobotState>;
    using TMessage = NDrive::NChat::TMessage;
    static const TSet<TString> InitialRegistrationSteps;

    const IChatMediaStorage& MediaStorage;

    void ReportCallbackSuccess(TAtomicSharedPtr<IChatResponsePostProcessingCallback> callback) const;
    void ReportCallbackFailure(TAtomicSharedPtr<IChatResponsePostProcessingCallback> callback, const TString& message, const HttpCodes code = HttpCodes::HTTP_INTERNAL_SERVER_ERROR) const;
    TMaybe<TNextActionInfo> CreateActionInfoByResubmit(const TString& userId, const TString& topic, TChatContext& stateContext, TChatRobotState& currentState, bool delayed, NDrive::TEntitySession& chatSession) const;
    IChatScriptAction::TPtr GetChatScriptAction(const IChatUserContext::TPtr context, const TString& operatorId, NDrive::TEntitySession& readOnlySession) const;

protected:
    virtual TString CreateResubmitMessage(const TChatContext& chatContext) const;
    virtual TMaybe<TChatRobotScriptItem> GetResubmitChatItem(const TChatContext& chatContext, const TChatRobotState& currentState, bool isFirstRequest, const TString& userId, NDrive::TEntitySession& session) const;

    virtual TString DeduceBanMessagesGroup(const TString& userId, NDrive::TEntitySession* sessionExternal = nullptr) const;
    bool DoGetIsChatCompleted(const TString& userId, const TString& topic, const TInstant actuality) const override;

    virtual bool GetChatContext(const TChatRobotState& state, TChatContext& chatContext) const override;

    virtual bool GetChatContextByChat(const TString& userId, const TString& topic, TChatContext& result, const TInstant actuality) const override {
        TChatRobotState resultState;
        auto session = BuildChatEngineSession(true);
        if (!GetChatRobotState(userId, topic, resultState, actuality, &session)) {
            return true;
        }
        return GetChatContext(resultState, result);
    }

public:
    TSimpleChatBot(const NDrive::IServer* server, const TChatRobotConfig& chatConfig, const TAtomicSharedPtr<TChatViewTracker> messageViewTracker, const TAtomicSharedPtr<TChatRobotStatePostgresStorage> robotStateStorage, const IChatMediaStorage& mediaStorage)
        : TBase(server, chatConfig, messageViewTracker, robotStateStorage)
        , MediaStorage(mediaStorage)
    {
    }

    virtual NJson::TJsonValue GetMessageReport(ELocalization locale, const NDrive::NChat::TMessageEvent& message, const TString& userId, const TString& topic) const override;

    virtual bool UpdateChatRobotStateContext(const TChatContext& chatContext, const TString& userId, const TString& topic, const TChatRobotState& state, NDrive::TEntitySession& session) const override;
    virtual TMaybe<TNextActionInfo> GetNextResubmitStep(const IChatUserContext::TPtr context, const TChatRobotScriptItem& currentScriptItem, TChatContext& stateContext, NDrive::TEntitySession& chatSession) const override;
    virtual TMaybe<TNextActionInfo> GetFirstResubmitStep(const IChatUserContext::TPtr context, TChatContext& stateContext, NDrive::TEntitySession& chatSession, ui32 maskOverride) const override;

    virtual NThreading::TFuture<void> AcceptUserResponse(const IChatUserContext::TPtr context, const TMessage& message, const TVector<TMessageAttachment>& attachments, const TString& operatorId) const override;
    virtual bool MaybeContinueChat(const IChatUserContext::TPtr context, const TString& operatorId, TChatContext& stateContext) const override;
    virtual void AcceptMediaResource(const IChatUserContext::TPtr context, const TString& resourceId, const TString& contentType, const TString& content, TAtomicSharedPtr<IChatMediaResourcePostUploadCallback> callback) const override;
    virtual NThreading::TFuture<TChatResourceAcquisitionResult> GetMediaResource(const IChatUserContext::TPtr context, const TString& resourceId, const bool needsFurtherProcessing) const override;
    virtual NThreading::TFuture<TChatResourceAcquisitionResult> GetMediaResourcePreview(const IChatUserContext::TPtr context, const TString& resourceId, const bool needsFurtherProcessing, const TMap<TString, TString>& typeResourceOverrides) const override;
    virtual bool RegisterMediaResource(const TString& userId, const TString& resourceId, const TString& contentType, const bool shared, NDrive::TEntitySession& session) const override;
    virtual NThreading::TFuture<void> UploadResourcePreview(const TString& content, const TMediaResourceDescription& description, const TString& contentType) const override;
    virtual bool UpdateMediaResourceDescription(const TMediaResourceDescription description, const TString& actorUserId, NDrive::TEntitySession& session) const override;

    virtual bool ApproveFinal(const TString& userId, const TString& topic, const TString& scriptItem) const;
    virtual bool SendRejectionMessages(const TString& userId, const TString& topic, NDrive::TEntitySession& session) const;
    virtual bool MaybeSendFraudReason(const TString& userId, const TString& topic, const TString& fraudReason, NDrive::TEntitySession& session) const;
    virtual bool SendBanMessages(const TString& userId, const TString& topic, NDrive::TEntitySession& session, const TString& operatorId) const;

    virtual bool AskToResubmit(const TString& userId, const TString& topic, const ui32 resubmitMask, NDrive::TEntitySession& session, const TString& operatorId, const bool force = false, const TVector<TDocumentResubmitOverride>& resubmitOverrides = {}) const;

    virtual TString GetChatIcon(const NDrive::NChat::TChat& chat) const override;
    virtual TString GetFullChatId(const TString& topic) const;
};
