#include "state_storage.h"

#include <drive/backend/logging/evlog.h>

bool TChatRobotSerializedState::Parse(const NStorage::TTableRecord& row) {
    return TBaseDecoder::DeserializeFromTableRecord(*this, row);
}

bool TChatRobotSerializedState::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, TalkId);
    READ_DECODER_VALUE(decoder, values, SerializedState);
    READ_DECODER_VALUE(decoder, values, CurrentStep);
    return true;
}

TString TChatRobotSerializedState::GetObjectId(const TChatRobotSerializedState& object) {
    return object.GetTalkId();
}

TChatRobotSerializedState TChatRobotSerializedState::FromHistoryEvent(const TObjectEvent<TChatRobotSerializedState>& historyEvent) {
    return historyEvent;
}

NStorage::TTableRecord TChatRobotSerializedState::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("talk_id", TalkId);
    result.Set("state_proto", SerializedState);
    result.Set("current_step", CurrentStep);
    return result;
}

TStringBuf TChatRobotStatePostgresStorage::GetEventObjectId(const TObjectEvent<TRecordType>& object) const {
    return object.GetTalkId();
}

bool TChatRobotStatePostgresStorage::DoRebuildCacheUnsafe() const {
    States.clear();

    NStorage::TObjectRecordsSet<TRecordType> records;
    {
        auto table = Database->GetTable(StatesTable->GetTableName());
        auto transaction = Database->CreateTransaction(true);
        auto result = table->GetRows("", records, transaction);

        if (!result->IsSucceed()) {
            ERROR_LOG << "Cannot refresh data for " << StatesTable->GetTableName() << Endl;
            return false;
        }
    }

    for (auto&& record : records) {
        States[record.GetTalkId()] = record.GetSerializedState();
    }

    return true;
}

void TChatRobotStatePostgresStorage::AcceptHistoryEventUnsafe(const TObjectEvent<TRecordType>& ev) const {
    if (ev.GetHistoryAction() != EObjectHistoryAction::Remove) {
        States[ev.GetTalkId()] = ev.GetSerializedState();
    } else {
        auto stateIt = States.find(ev.GetTalkId());
        if (stateIt != States.end()) {
            States.erase(stateIt);
        }
    }
}

TExpected<TString, EChatError> TChatRobotStatePostgresStorage::DirectGetSerializedState(const TString& talkId, NDrive::TEntitySession* sessionExt) const {
    TDBTable::TFetchResult stateFetchResult;
    if (sessionExt) {
        stateFetchResult = StatesTable->FetchInfo(talkId, *sessionExt);
        if (!stateFetchResult) {
            return MakeUnexpected(EChatError::FetchFailed);
        }
    } else {
        stateFetchResult = StatesTable->FetchInfo(talkId);
    }
    auto statePtr = stateFetchResult.GetResultPtr(talkId);
    if (!statePtr) {
        return MakeUnexpected(EChatError::NotFound);
    }
    return statePtr->GetSerializedState();
}

TExpected<TString, EChatError> TChatRobotStatePostgresStorage::GetSerializedState(const TString& talkId, const TInstant actuality, NDrive::TEntitySession* sessionExt) const {
    if (sessionExt) {
        // If there is an external session, we can't make use of history manager, since the changes may not be commited yet
        return DirectGetSerializedState(talkId, sessionExt);
    }

    if (!RefreshCache(actuality)) {
        return MakeUnexpected(EChatError::FetchFailed);
    }
    auto g = MakeObjectReadGuard();
    if (!States.contains(talkId)) {
        return MakeUnexpected(EChatError::NotFound);
    }
    return States[talkId];
}

bool TChatRobotStatePostgresStorage::SetSerializedState(const TString& talkId, const TString& serializedState, const TString& currentStep, NDrive::TEntitySession& session) const {
    auto eg = NDrive::BuildEventGuard("set_serialized_state");
    auto newState = TChatRobotSerializedState(talkId, serializedState, currentStep);
    {
        auto eg = NDrive::BuildEventGuard("upsert_state");
        if (!StatesTable->Upsert(newState, session)) {
            session.AddErrorMessage("chat_robot_state", "unable to upsert chat robot state in db table");
            return false;
        }
    }

    AddSerializedStateHistoryEvent(newState, session);

    {
        auto eg = NDrive::BuildEventGuard("update_state_mutex");
        auto g = MakeObjectWriteGuard();
        States[talkId] = serializedState;
    }

    return true;
}

bool TChatRobotStatePostgresStorage::AddSerializedStateHistoryEvent(const TChatRobotSerializedState& newState, NDrive::TEntitySession& session) const {
    auto eg = NDrive::BuildEventGuard("add_state_history");
    if (!HistoryManager->AddHistory(newState, "robot-frontend", EObjectHistoryAction::UpdateData, session)) {
        session.AddErrorMessage("chat_robot_state_history", "unable to write history");
        return false;
    }
    return true;
}

bool TChatRobotStatePostgresStorage::AddSerializedStateHistoryEvent(const TString& talkId, const TString& serializedState, const TString& currentStep, NDrive::TEntitySession& session) const {
    auto newState = TChatRobotSerializedState(talkId, serializedState, currentStep);
    AddSerializedStateHistoryEvent(newState, session);
    return true;
}

bool TChatRobotStatePostgresStorage::RemoveState(const TString& talkId, const TString& operatorId, NDrive::TEntitySession& session) const {
    NStorage::TObjectRecordsSet<TChatRobotSerializedState> removedState;
    if (!StatesTable->Remove(talkId, session, &removedState)) {
        session.AddErrorMessage("chat_robot_state", "unable to remove chat robot state in db table");
        return false;
    }
    const auto& step = removedState.front().GetCurrentStep();

    auto g = MakeObjectReadGuard();

    auto cachedStateIt = States.find(talkId);
    TString cachedState = cachedStateIt != States.end() ? cachedStateIt->second : "";

    if (!HistoryManager->AddHistory(TChatRobotSerializedState(talkId, cachedState, step), operatorId, EObjectHistoryAction::Remove, session)) {
        session.AddErrorMessage("chat_robot_state_history", "unable to write history");
        return false;
    }

    return true;
}

TMap<TString, TString> TChatRobotStatePostgresStorage::GetRawCachedStates() const {
    auto g = MakeObjectReadGuard();
    return States;
}

bool TChatRobotStatePostgresStorage::ForceRefresh(const TInstant actuality) const {
    return RefreshCache(actuality);
}
