#pragma once

#include "error_type.h"

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/manager.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>
#include <rtline/util/types/expected.h>

class TChatRobotSerializedState {
    R_READONLY(TString, TalkId);
    R_READONLY(TString, SerializedState);
    R_READONLY(TString, CurrentStep);

public:
    class TChatRobotSerializedStateDecoder: public TBaseDecoder {
        R_FIELD(i32, TalkId, -1);
        R_FIELD(i32, SerializedState, -1);
        R_FIELD(i32, CurrentStep, -1);

    public:
        TChatRobotSerializedStateDecoder() = default;

        TChatRobotSerializedStateDecoder(const TMap<TString, ui32>& decoderBase) {
            TalkId = GetFieldDecodeIndex("talk_id", decoderBase);
            SerializedState = GetFieldDecodeIndex("state_proto", decoderBase);
            CurrentStep = GetFieldDecodeIndex("current_step", decoderBase);
        }
    };

    using TDecoder = TChatRobotSerializedStateDecoder;

    TChatRobotSerializedState() = default;

    TChatRobotSerializedState(const TString& talkId, const TString& serializedState, const TString& currentStep)
        : TalkId(talkId)
        , SerializedState(serializedState)
        , CurrentStep(currentStep)
    {
    }

    bool DeserializeWithDecoder(const TChatRobotSerializedStateDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

    static TString GetObjectId(const TChatRobotSerializedState& object);

    static TChatRobotSerializedState FromHistoryEvent(const TObjectEvent<TChatRobotSerializedState>& historyEvent);

    NStorage::TTableRecord SerializeToTableRecord() const;

    bool Parse(const NStorage::TTableRecord& row);
};

class TChatRobotStateStorageTable : public TDBEntities<TChatRobotSerializedState> {
private:
    using TBase = TDBEntities<TChatRobotSerializedState>;

public:
    using TBase::TBase;

    virtual TString GetTableName() const override {
        return "chat_robot_state";
    }

    virtual TString GetColumnName() const override {
        return "talk_id";
    }

    virtual TString GetMainId(const TChatRobotSerializedState& e) const override {
        return e.GetTalkId();
    }
};

class TChatRobotStateHistoryManager : public TIndexedAbstractHistoryManager<TChatRobotSerializedState> {
private:
    using TBase = TIndexedAbstractHistoryManager<TChatRobotSerializedState>;

public:
    TChatRobotStateHistoryManager(const IHistoryContext& context, const THistoryConfig& hConfig)
        : TBase(context, "chat_robot_state_history", hConfig)
    {
    }
};

class TChatRobotStatePostgresStorage : public TDBCacheWithHistoryOwner<TChatRobotStateHistoryManager, TChatRobotSerializedState> {
private:
    using TCacheBase = TDBCacheWithHistoryOwner<TChatRobotStateHistoryManager, TChatRobotSerializedState>;
    using TDBTable = TChatRobotStateStorageTable;
    using TRecordType = TChatRobotSerializedState;
    using THistoryReader = TAtomicSharedPtr<TCallbackSequentialTableImpl<TObjectEvent<TChatRobotSerializedState>, TString>>;

private:
    THolder<TDBTable> StatesTable;
    NStorage::IDatabase::TPtr Database;
    mutable TMap<TString, TString> States;

private:
    virtual bool DoRebuildCacheUnsafe() const override;
    virtual void AcceptHistoryEventUnsafe(const TObjectEvent<TRecordType>& ev) const override;
    virtual TStringBuf GetEventObjectId(const TObjectEvent<TRecordType>& object) const override;

public:
    TChatRobotStatePostgresStorage(const IHistoryContext& historyContext, const THistoryConfig& historyConfig)
        : TCacheBase("chat_robot_state", historyContext, historyConfig)
        , Database(historyContext.GetDatabase())
    {
        StatesTable.Reset(new TDBTable(Database));
    }

    virtual TExpected<TString, EChatError> DirectGetSerializedState(const TString& talkId, NDrive::TEntitySession* sessionExt = nullptr) const;
    virtual TExpected<TString, EChatError> GetSerializedState(const TString& talkId, const TInstant actuality = TInstant::Zero(), NDrive::TEntitySession* sessionExt = nullptr) const;
    virtual bool SetSerializedState(const TString& talkId, const TString& serializedState, const TString& currentStep, NDrive::TEntitySession& session) const;

    virtual bool AddSerializedStateHistoryEvent(const TChatRobotSerializedState& newState, NDrive::TEntitySession& session) const;
    virtual bool AddSerializedStateHistoryEvent(const TString& talkId, const TString& serializedState, const TString& currentStep, NDrive::TEntitySession& session) const;

    virtual bool RemoveState(const TString& talkId, const TString& operatorId, NDrive::TEntitySession& session) const;

    virtual TMap<TString, TString> GetRawCachedStates() const;

    virtual bool ForceRefresh(const TInstant actuality) const;
};
