#pragma once

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/chat_robots/ifaces.h>

#include <drive/library/cpp/taxi/support_chat_suggest/client.h>
#include <rtline/library/json/cast.h>
#include <rtline/util/types/accessor.h>
#include <rtline/util/types/expected.h>
#include <rtline/util/types/messages_collector.h>

#include <library/cpp/object_factory/object_factory.h>

#include <util/generic/set.h>

class TSuggestSchema {
    R_FIELD(TString, Node);
    R_FIELD(TString, Text);

public:
    NJson::TJsonValue GetChatReport() const;
};

class TNodeResolveParameters {
    R_FIELD(TString, ClassificationResult);
    R_FIELD(TString, NodeId);
    R_FIELD(i32, MinConfidence, 0);
    R_FIELD(i32, MaxConfidence, 100);
    R_OPTIONAL(TSuggestSchema, Schema);
    R_OPTIONAL(TVector<TSuggestSchema>, SureSchema);
    R_FIELD(bool, AppendSuggest, false);

public:
    bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector& errors);
    bool IsMatching(const TString& classificationResult, const i32 confidence) const;
    void AddNodeTextMappings(TMap<TString, TString>& map) const;
};

class TClassificationFeatures {
    R_FIELD(TSet<TString>, UserTags);

public:
    NJson::TJsonValue GenerateFeaturesJson(const IChatUserContext::TPtr context, bool userOptionsSuggest = false) const;
    bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector& errors);
};

class INodeResolver {
public:
    using TPtr = TAtomicSharedPtr<INodeResolver>;
    using TFactory = NObjectFactory::TObjectFactory<INodeResolver, TString>;

private:
    R_FIELD(TString, DefaultNode);
    R_FIELD(TString, FallbackNode);
    R_FIELD(TVector<TNodeResolveParameters>, ResolveParameters);
    R_FIELD(TDuration, RequestTimeout, TDuration::Seconds(5));
    R_FIELD(TClassificationFeatures, ClassificationFeatures);
    R_FIELD(ui32, SuggestOptionsCount, 3);
    R_FIELD(size_t, MessageLimit, 15);

public:
    virtual TString GetNextNode(const IChatUserContext::TPtr context, const NDrive::NChat::TMessageEvents& messages) const = 0;
    virtual NThreading::TFuture<TString> GetNextNodeFuture(const IChatUserContext::TPtr context, const NDrive::NChat::TMessageEvents& messages) const = 0;
    virtual NThreading::TFuture<TTaxiSupportChatSuggestClient::TSuggestResponse> GetSuggest(const IChatUserContext::TPtr context, const NDrive::NChat::TMessageEvents& messages, bool userOptionsSuggest = false) const = 0;
    virtual NJson::TJsonValue GetSuggestedChatOptions(const TTaxiSupportChatSuggestClient::TSuggestResponse& suggestResponse) const = 0;
    virtual bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector& errors);
    virtual NJson::TJsonValue SerializeToJson() const;
    virtual TString GetType() const = 0;
    virtual void MergeFields(INodeResolver::TPtr toCopy);
    virtual TString GetSuggestText(const TString& node) const;

    static TExpected<INodeResolver::TPtr, TString> Construct(const NJson::TJsonValue& json, const INodeResolver::TPtr toCopy = nullptr);

    virtual ~INodeResolver() = default;

protected:
    virtual TMaybe<TNodeResolveParameters> TryFindResolveParameters(const TString& classificationResult, const i32 confidence, const bool withSchema) const;
    virtual TMaybe<TNodeResolveParameters> TryFindResolveParameters(const TString& classificationResult, const bool withSchema) const;

private:
    TMap<TString, TString> SuggestNodeToText;
};

class TSupportPredictionNodeResolver: public INodeResolver {
public:
    virtual TString GetNextNode(const IChatUserContext::TPtr context, const NDrive::NChat::TMessageEvents& messages) const = 0;
    virtual NThreading::TFuture<TString> GetNextNodeFuture(const IChatUserContext::TPtr context, const NDrive::NChat::TMessageEvents& messages) const = 0;

protected:
    TString MatchPredictions(const TVector<NDrive::TSupportPrediction::TElement>& predictions, const TString& logEvent) const;
    TVector<TNodeResolveParameters> GetTopPredictions(const TVector<NDrive::TSupportPrediction::TElement>& predictions, const TSet<TString>& skipTopics, const ui32 count, const bool withSchema) const;
};

class TTaxiSupportChatNodeResolver: public TSupportPredictionNodeResolver {
public:
    virtual NThreading::TFuture<TTaxiSupportChatSuggestClient::TSuggestResponse> GetSuggest(const IChatUserContext::TPtr context, const NDrive::NChat::TMessageEvents& messages, bool userOptionsSuggest = false) const override;
    virtual TString GetNextNode(const IChatUserContext::TPtr context, const NDrive::NChat::TMessageEvents& messages) const override;
    virtual NThreading::TFuture<TString> GetNextNodeFuture(const IChatUserContext::TPtr context, const NDrive::NChat::TMessageEvents& messages) const override;
    virtual NJson::TJsonValue GetSuggestedChatOptions(const TTaxiSupportChatSuggestClient::TSuggestResponse& suggestResponse) const override;

    TString GetType() const override {
        return "taxi_support_chat_classification";
    }

private:
    static TFactory::TRegistrator<TTaxiSupportChatNodeResolver> Registrator;
};
