#include "view_tracker.h"

bool TViewTrackerState::DeserializeWithDecoder(const TViewTrackerStateDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, ChatId);
    READ_DECODER_VALUE(decoder, values, UserId);
    READ_DECODER_VALUE(decoder, values, LastViewedMessageId);
    return true;
}

TString TViewTrackerState::GetObjectId(const TViewTrackerState& object) {
    return ToString(object.GetChatId());
}

TViewTrackerState TViewTrackerState::FromHistoryEvent(const TObjectEvent<TViewTrackerState>& historyEvent) {
    return historyEvent;
}

NStorage::TTableRecord TViewTrackerState::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("user_id", UserId);
    result.Set("chat_id", ChatId);
    result.Set("last_viewed_message_id", LastViewedMessageId);
    return result;
}

bool TViewTrackerState::Parse(const NStorage::TTableRecord& row) {
    return TBaseDecoder::DeserializeFromTableRecord(*this, row);
}

bool TChatViewTracker::DoRebuildCacheUnsafe() const {
    States.clear();

    NStorage::TObjectRecordsSet<TRecordType> records;
    {
        auto table = GetDatabase().GetTable(StatesTable->GetTableName());
        auto transaction = GetDatabase().CreateTransaction(true);
        auto result = table->GetRows("", records, transaction);

        if (!result->IsSucceed()) {
            ERROR_LOG << "Cannot refresh data for " << StatesTable->GetTableName() << Endl;
            return false;
        }
    }

    for (auto&& record : records) {
        States.emplace(std::make_pair(record.GetChatId(), record.GetUserId()), record.GetLastViewedMessageId());
    }

    return true;
}

ui32 TChatViewTracker::GetEventObjectId(const TObjectEvent<TRecordType>& ev) const {
    return ev.GetChatId();
}

bool TChatViewTracker::DoAcceptHistoryEventUnsafe(const TAtomicSharedPtr<TObjectEvent<TRecordType>>& ev, const bool isNewEvent) {
    if (!isNewEvent) {
        return true;
    }
    auto key = std::make_pair(ev->GetChatId(), ev->GetUserId());
    if (ev->GetHistoryAction() != EObjectHistoryAction::Remove) {
        States[key] = Max(States[key], ev->GetLastViewedMessageId());
    } else {
        auto stateIt = States.find(key);
        if (stateIt != States.end()) {
            States.erase(stateIt);
        }
    }
    return true;
}

ui64 TChatViewTracker::GetLastViewedMessageId(const TString& userId, const ui32 chatId) const {
    auto g = MakeObjectReadGuard();
    auto key = std::make_pair(chatId, userId);
    auto stateIt = States.find(key);
    return stateIt != States.end() ? stateIt->second : 0;
}

ui64 TChatViewTracker::GetLastViewedMessageIdExcept(const TString& userId, const ui32 chatId) const {
    auto g = MakeObjectReadGuard();
    ui64 result = 0;
    auto it = States.lower_bound(std::make_pair(chatId, ""));
    while (it != States.end() && it->first.first == chatId) {
        if (it->first.second != userId) {
            result = Max(result, it->second);
        }
        ++it;
    }
    return result;
}

bool TChatViewTracker::UpdateLastViewedMessageId(const TString& userId, const ui32 chatId, const ui64 messageId, NDrive::TEntitySession& session) const {
    if (GetLastViewedMessageId(userId, chatId) >= messageId) {
        return true;
    }

    auto newState = TViewTrackerState(chatId, userId, messageId);

    if (!StatesTable->Upsert(newState, session)) {
        session.AddErrorMessage("view_tracker_state", "unable to upsert chat robot state in db table");
        return false;
    }

    if (!HistoryWriter.AddHistory(newState, "robot-frontend", EObjectHistoryAction::UpdateData, session)) {
        session.AddErrorMessage("view_tracker_state_history", "unable to write history");
        return false;
    }

    auto g = MakeObjectWriteGuard();
    auto presentValue = States[std::make_pair(chatId, userId)];
    States[std::make_pair(chatId, userId)] = Max(presentValue, messageId);

    return true;
}

bool TChatViewTracker::ForceRefresh(const TInstant actuality) const {
    return RefreshCache(actuality);
}
