#pragma once

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/manager.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

class TViewTrackerState {
    R_READONLY(ui32, ChatId, 0);
    R_READONLY(TString, UserId);
    R_READONLY(ui64, LastViewedMessageId, 0);

public:
    class TViewTrackerStateDecoder: public TBaseDecoder {
        R_FIELD(i32, ChatId, -1);
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, LastViewedMessageId, -1);

    public:
        TViewTrackerStateDecoder() = default;

        TViewTrackerStateDecoder(const TMap<TString, ui32>& decoderBase) {
            ChatId = GetFieldDecodeIndex("chat_id", decoderBase);
            UserId = GetFieldDecodeIndex("user_id", decoderBase);
            LastViewedMessageId = GetFieldDecodeIndex("last_viewed_message_id", decoderBase);
        }
    };

    using TDecoder = TViewTrackerStateDecoder;

public:
    TViewTrackerState() = default;
    TViewTrackerState(const ui32 chatId, const TString& userId, const ui32 lastViewedMessageId)
        : ChatId(chatId)
        , UserId(userId)
        , LastViewedMessageId(lastViewedMessageId)
    {
    }

    bool DeserializeWithDecoder(const TViewTrackerStateDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

    static TString GetObjectId(const TViewTrackerState& object);
    static TViewTrackerState FromHistoryEvent(const TObjectEvent<TViewTrackerState>& historyEvent);

    NStorage::TTableRecord SerializeToTableRecord() const;
    bool Parse(const NStorage::TTableRecord& row);
};

class TViewTrackerStateTable : public TDBEntities<TViewTrackerState> {
private:
    using TBase = TDBEntities<TViewTrackerState>;
    using TEntity = TViewTrackerState;

public:
    using TBase::TBase;

    virtual TString GetTableName() const override {
        return "chat_views";
    }

    virtual TString GetColumnName() const override {
        return "chat_id";
    }

    virtual TString GetMainId(const TViewTrackerState& e) const override {
        return ToString(e.GetChatId());
    }

    virtual void UpdateUniqueCondition(NStorage::TTableRecord& unique, const TEntity& info) const override {
        unique.Set("user_id", info.GetUserId());
    }
};

class TChatViewTrackerHistoryManager : public TDatabaseHistoryManager<TViewTrackerState> {
private:
    using TBase = TDatabaseHistoryManager<TViewTrackerState>;

public:
    TChatViewTrackerHistoryManager(const IHistoryContext& context)
        : TBase(context, "chat_views_history")
    {
    }
};

class TChatViewTracker
    : public TAutoActualizingSnapshot<TViewTrackerState, ui32>
    , public TDatabaseSessionConstructor
{
    using TCacheBase = TAutoActualizingSnapshot<TViewTrackerState, ui32>;
    using TRecordType = TViewTrackerState;
    using TDBTable = TViewTrackerStateTable;
    using THistoryReader = TAtomicSharedPtr<TCallbackSequentialTableImpl<TObjectEvent<TViewTrackerState>, ui32>>;

private:
    TChatViewTrackerHistoryManager HistoryWriter;
    THolder<TDBTable> StatesTable;
    mutable TMap<std::pair<ui32, TString>, ui64> States;

private:
    virtual bool DoRebuildCacheUnsafe() const override;
    virtual ui32 GetEventObjectId(const TObjectEvent<TRecordType>& ev) const override;
    virtual bool DoAcceptHistoryEventUnsafe(const TAtomicSharedPtr<TObjectEvent<TRecordType>>& ev, const bool isNewEvent) override;

public:
    TChatViewTracker(const THistoryReader& reader, NStorage::IDatabase::TPtr db)
        : TCacheBase("chat_views", reader)
        , TDatabaseSessionConstructor(db)
        , HistoryWriter(THistoryContext(db))
    {
        StatesTable.Reset(new TDBTable(db));
    }

    virtual ui64 GetLastViewedMessageId(const TString& userId, const ui32 chatId) const;
    virtual ui64 GetLastViewedMessageIdExcept(const TString& userId, const ui32 chatId) const;
    virtual bool UpdateLastViewedMessageId(const TString& userId, const ui32 chatId, const ui64 messageId, NDrive::TEntitySession& session) const;

    virtual bool ForceRefresh(const TInstant actuality) const;
};
