#include "client.h"

#include <drive/library/cpp/clickhouse/client.h>
#include <drive/telematics/server/data/clickhouse/sensor.h>

#include <rtline/library/storage/sql/query.h>
#include <library/cpp/logger/global/global.h>

NDrive::TClickHouseBackendClient::TClickHouseBackendClient(const NClickHouse::TAsyncClientOptions& options)
    : Client(MakeHolder<NClickHouse::TAsyncClient>(options))
{
}

NDrive::TClickHouseBackendClient::~TClickHouseBackendClient() {
}

template <class T>
NSQL::TQueryOptions CreateQueryOptions(const T& query) {
    NSQL::TQueryOptions queryOptions;
    if (const auto& objectIds = query.GetObjectIds(); !objectIds.empty()) {
        queryOptions.SetGenericCondition("object_id", objectIds);
    }
    if (const auto& timestamp = query.GetTimestamp()) {
        TRange<ui64> seconds;
        seconds.From = timestamp.From ? MakeMaybe<ui32>(timestamp.From->Seconds()) : Nothing();
        seconds.To = timestamp.To ? MakeMaybe<ui32>(timestamp.To->Seconds()) : Nothing();
        queryOptions.SetGenericCondition("timestamp", seconds);
    }
    if (query.HasLimit()) {
        queryOptions.SetLimit(query.GetLimitRef());
    }
    return queryOptions;
}

NThreading::TFuture<NDrive::TClickHouseBackendClient::TAllOrdersSummaryResult> NDrive::TClickHouseBackendClient::Get(const TOrdersSummaryQuery& query, TInstant deadline) const {
    auto queryOptions = CreateQueryOptions(query);
    TString timestamp = TStringBuilder() << TimestampFieldBuilder(query.GetAggregationType()) << "(FROM_UNIXTIME(start_timestamp)))) AS timestamp";
    TVector<TString> fields {
        "object_id",
        timestamp,
        "sum(mileage) AS mileage_sum",
        "count(session_id) AS count",
        "offer_type",
        "sum(finish_timestamp - start_timestamp) AS duration",
        "insurance_type",
        "sum(entry_to_eco_zones_in_germany) AS entry_to_eco_zones_in_germany_sum",
        "sum(snow_chains) AS snow_chains_sum",
        "sum(gps) AS gps_sum",
        "sum(roof_rack) AS roof_rack_sum",
        "sum(child_seat) AS child_seat_sum",
        "sum(total_payment) AS total_payment_sum",
        "sum(riding_duration) AS riding_duration",
        "min(currency) AS currency",
        "max(start_timestamp) AS start_time",
        "max(finish_timestamp) AS finish_time"
    };
    TVector<TString> groupBy {
        "object_id",
        "timestamp",
        "offer_type",
        "insurance_type"
    };
    auto customCondition = TStringBuilder() << "finished = " << ToString(query.GetFinished())
        << " AND mileage < "
        << ToString(query.GetMileage());
    const auto tariff = query.GetTariff();
    if (!tariff.empty()) {
        customCondition << " AND offer_type = '" << tariff << "'";
    }
    queryOptions.AddCustomCondition(customCondition);

    queryOptions.SetGroupBy(groupBy);
    auto customQuery = queryOptions.PrintQuery(NClickHouse::TSimpleQuoter::Instance(), NDrive::GetCompiledRidesClickHouseTable(), fields);
    auto block = Yensured(Client)->Execute(customQuery, deadline);
    auto carsRental = block.Apply([](const NThreading::TFuture<NClickHouse::TBlock>& b) {
        const auto& block = b.GetValue();
        NDrive::TOrderSummaryClickHouseRecords records;
        Extract(block, records);
        TAllOrdersSummaryResult result;
        for (auto&& record : records) {
            if (record.ObjectId) {
                result[record.ObjectId].push_back(record);
            }
        }
        return result;
    });
    return carsRental;
}

NThreading::TFuture<NDrive::TClickHouseBackendClient::TAllMileageSummaryResult> NDrive::TClickHouseBackendClient::Get(const TMileageSummaryQuery& query, TInstant deadline) const {
    auto queryOptions = CreateQueryOptions(query);
    TString timestamp = TStringBuilder() << TimestampFieldBuilder(query.GetAggregationType()) << "(timestamp))) AS timestamp";
    TVector<TString> fields {
        "object_id",
        timestamp,
        "max(value_float) - min(value_float) AS mileage",
    };
    TVector<TString> groupBy {
        "object_id",
        "timestamp",
    };

    queryOptions.SetGroupBy(groupBy);
    auto customQuery = queryOptions.PrintQuery(NClickHouse::TSimpleQuoter::Instance(), NDrive::GetSensorHistoryClickHouseTable(), fields);
    auto block = Yensured(Client)->Execute(customQuery, deadline);
    auto carsRental = block.Apply([](const NThreading::TFuture<NClickHouse::TBlock>& b) {
        const auto& block = b.GetValue();
        NDrive::TMileageSummaryClickHouseRecords records;
        Extract(block, records);
        TAllMileageSummaryResult result;
        for (auto&& record : records) {
            if (record.ObjectId) {
                result[record.ObjectId].push_back(record);
            }
        }
        return result;
    });
    return carsRental;
}

TString NDrive::TClickHouseBackendClient::TimestampFieldBuilder(const TString& startOf) const {
    TString agregationCHString;
    if (startOf == "month") {
        agregationCHString = "toDateTime(toStartOfMonth";
    } else if (startOf == "week") {
        agregationCHString = "toDateTime(toStartOfWeek";
    } else {
        agregationCHString = "toStartOfDay(";
    }
    TString result = TStringBuilder() << "toUnixTimestamp(" << agregationCHString;
    return result;
}
