#include "host_filter.h"

#include <util/system/hostname.h>


THostFilter& THostFilter::SetHostPattern(const TString& hostPattern) {
    HostPattern = hostPattern;
    if (!!HostPattern) {
        HostPatternRegex = TRegExMatch(HostPattern);
    }
    return *this;
}

THostFilter& THostFilter::SetFQDNHostPattern(const TString& hostPattern) {
    FQDNHostPattern = hostPattern;
    if (!!FQDNHostPattern) {
        FQDNHostPatternRegex = TRegExMatch(FQDNHostPattern);
    }
    return *this;
}

bool THostFilter::CheckHost(const IServerBase& server) const {
    if (!!CType && CType != server.GetCType()) {
        return false;
    }
    try {
        if (!!HostPattern) {
            const TString hostName = GetHostName();
            if (!HostPatternRegex.Match(hostName.c_str())) {
                return false;
            }
        }
        if (!!FQDNHostPattern) {
            const TString fqdnHostName = GetFQDNHostName();
            if (!FQDNHostPatternRegex.Match(fqdnHostName.c_str())) {
                return false;
            }
        }
    } catch (const std::exception& e) {
        ERROR_LOG << FormatExc(e) << Endl;
        return false;
    }
    return true;
}


NJson::TJsonValue THostFilter::SerializeToJson() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    NJson::InsertNonNull(result, "host_pattern", HostPattern);
    NJson::InsertNonNull(result, "fqdn_host_pattern", FQDNHostPattern);
    NJson::InsertNonNull(result, "ctype", CType);
    return result;
}


bool THostFilter::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_STRING_OPT(jsonInfo, "host_pattern", HostPattern);
    JREAD_STRING_OPT(jsonInfo, "fqdn_host_pattern", FQDNHostPattern);
    JREAD_STRING_OPT(jsonInfo, "ctype", CType);
    SetHostPattern(HostPattern);
    SetFQDNHostPattern(FQDNHostPattern);
    return true;
}


NDrive::TScheme THostFilter::GetScheme(const IServerBase& /*server*/, const TString& defaults) const {
    NDrive::TScheme result;
    result.Add<TFSString>("host_pattern", "Шаблон для имени хоста");
    result.Add<TFSString>("fqdn_host_pattern", "Шаблон для fqdn имени хоста");
    result.Add<TFSString>("ctype", "CType для запуска").SetDefault(defaults);
    return result;
}
