#include "scheme_adapters.h"

#include <drive/backend/abstract/base.h>

#include <util/string/join.h>
#include <util/string/subst.h>

namespace NDrive {
    namespace NPrivate {
        NJson::TJsonValue GetSettingJsonValue(const TString& settingKey) {
            if (!settingKey || !NDrive::HasServer() || !NDrive::GetServer().HasSettings()) {
                return {};
            }
            auto& settings = NDrive::GetServer().GetSettings();
            return settings.GetJsonValue(settingKey);
        }
    }

    TLocalizedVariantAdapter::TLocalizedVariantAdapter(const TString& value, const TString& description, const ELocalization locale, const TString& resourcePrefix)
        : Value(value)
        , Description(description)
        , Locale(locale)
        , ResourcePrefix(resourcePrefix)
    {
        PrefixHasSpaces = (ResourcePrefix.find(' ') != TString::npos);
        ValueHasSpaces = (Value.find(' ') != TString::npos);
    }

    TString TLocalizedVariantAdapter::GetResourceName() const {
        TString safePrefix = GetSafeValue(ResourcePrefix, PrefixHasSpaces);
        TString safeValue = GetSafeValue(Value, ValueHasSpaces);
        return (safePrefix) ? JoinSeq(".", { safePrefix, safeValue }) : safeValue;
    }

    TString TLocalizedVariantAdapter::GetLocalizedValue() const {
        return (NDrive::HasServer()) ? NDrive::GetServer().GetLocalization()->GetLocalString(Locale, GetResourceName(), Value) : Value;
    }

    TLocalizedVariantAdapter::operator TCompoundVariant() const {
        return TCompoundVariant(Value, GetLocalizedValue(), Description);
    }

    TString TLocalizedVariantAdapter::GetSafeValue(const TString& value, const bool hasSpaces) const {
        return (hasSpaces) ? SubstGlobalCopy(value, ' ', '_') : value;
    }

    TLocalizedSettingVariantsAdapter::TLocalizedSettingVariantsAdapter(const TString& settingKey, const TString& resourcePrefix, const ELocalization locale)
        : SettingKey(settingKey)
        , ResourcePrefix(resourcePrefix)
        , Locale(locale)
    {
    }

    TLocalizedSettingVariantsAdapter::TCompoundVariants TLocalizedSettingVariantsAdapter::Cast(const TString& settingKey, const TString& resourcePrefix, const ELocalization locale) {
        return static_cast<TCompoundVariants>(TLocalizedSettingVariantsAdapter(settingKey, resourcePrefix, locale));
    }

    TLocalizedSettingVariantsAdapter::operator TCompoundVariants() const {
        TCompoundVariants variants;

        NJson::TJsonValue data = NPrivate::GetSettingJsonValue(SettingKey);
        TVector<TString> items;
        if (NJson::TryFromJson(data, items)) {
            for (const auto& item: items) {
                variants.emplace(TLocalizedVariantAdapter(item, {}, Locale, ResourcePrefix));
            }
        }

        return variants;
    }
}
