#pragma once

#include <drive/backend/abstract/localization.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/library/json/parse.h>

#include <util/system/type_name.h>
#include <util/string/cast.h>

namespace NDrive {
    namespace NPrivate {
        NJson::TJsonValue GetSettingJsonValue(const TString& settingKey);
    }

    class TLocalizedVariantAdapter {
    public:
        using TCompoundVariant = TFSVariants::TCompoundVariant;
        using TCompoundVariants = TFSVariants::TCompoundVariants;

        explicit TLocalizedVariantAdapter(const TString& value, const TString& description = {}, const ELocalization locale = ELocalization::Rus, const TString& resourcePrefix = {});

        template <typename T>
        explicit TLocalizedVariantAdapter(const T& value, const TString& description = {}, const ELocalization locale = ELocalization::Rus, const TString& resourcePrefix = {})
            : TLocalizedVariantAdapter(::ToString(value), description, locale, resourcePrefix)
        {
        }

        template <typename T>
        static TLocalizedVariantAdapter FromEnum(const T& value, const TString& description = {}, const ELocalization locale = ELocalization::Rus, const TMaybe<TString>& resourcePrefix = {}) {
            return TLocalizedVariantAdapter(value, description, locale, (resourcePrefix) ? *resourcePrefix : TypeName<T>());
        }

        TString GetResourceName() const;
        TString GetLocalizedValue() const;

        operator TCompoundVariant() const;

    private:
        TString GetSafeValue(const TString& value, const bool hasSpaces) const;

    private:
        R_FIELD(TString, Value);
        R_FIELD(TString, Description);
        R_FIELD(ELocalization, Locale);
        R_FIELD(TString, ResourcePrefix);

        bool PrefixHasSpaces = false;
        bool ValueHasSpaces = false;
    };

    template <typename E>
    class TLocalizedEnumVariantsAdapter {
    public:
        using TCompoundVariant = TLocalizedVariantAdapter::TCompoundVariant;
        using TCompoundVariants = TLocalizedVariantAdapter::TCompoundVariants;

        explicit TLocalizedEnumVariantsAdapter(const ELocalization locale = ELocalization::Rus, const TMaybe<TString>& resourcePrefix = {})
            : Locale(locale)
            , ResourcePrefix(resourcePrefix)
        {
        }

        static TCompoundVariants Cast(const ELocalization locale = ELocalization::Rus, const TMaybe<TString>& resourcePrefix = {}) {
            return static_cast<TCompoundVariants>(TLocalizedEnumVariantsAdapter<E>(locale, resourcePrefix));
        }

        operator TCompoundVariants() const {
            TCompoundVariants variants;
            for (auto&& [name, value]: GetEnumNames<E>()) {
                variants.emplace(TLocalizedVariantAdapter::FromEnum<E>(name, {}, Locale, ResourcePrefix));
            }
            return variants;
        }

    private:
        const ELocalization Locale;
        TMaybe<TString> ResourcePrefix;
    };

    class TLocalizedSettingVariantsAdapter {
    public:
        using TCompoundVariant = TLocalizedVariantAdapter::TCompoundVariant;
        using TCompoundVariants = TLocalizedVariantAdapter::TCompoundVariants;

        explicit TLocalizedSettingVariantsAdapter(const TString& settingKey, const TString& resourcePrefix, const ELocalization locale = ELocalization::Rus);

        static TCompoundVariants Cast(const TString& settingKey, const TString& resourcePrefix, const ELocalization locale = ELocalization::Rus);

        operator TCompoundVariants() const;

    private:
        TString SettingKey;
        TString ResourcePrefix;
        const ELocalization Locale;
    };

    template <typename T>
    class TSettingValueAdapter {
        explicit TSettingValueAdapter(const TString& settingKey)
            : SettingKey(settingKey)
        {
        }

        static TMaybe<T> Cast(const TString& settingKey) {
            return static_cast<T>(TSettingValueAdapter<T>(settingKey));
        }

        static T CastDef(const TString& settingKey, T fallback = Default<T>()) {
            TMaybe<T> value = Cast(settingKey);
            return (value) ? *value : fallback;
        }

        operator TMaybe<T>() const {
            NJson::TJsonValue data = NPrivate::GetSettingJsonValue(SettingKey);
            return NJson::TryFromJson<TMaybe<T>>(data);
        }

    private:
        const TString SettingKey;
    };
}

using TLocalizedVariantAdapter = NDrive::TLocalizedVariantAdapter;

template <typename E>
using TLocalizedEnumVariantsAdapter = NDrive::TLocalizedEnumVariantsAdapter<E>;

using TLocalizedSettingVariantsAdapter = NDrive::TLocalizedSettingVariantsAdapter;

template <typename T>
using TSettingValueAdapter = NDrive::TSettingValueAdapter<T>;
