#include "compiled_riding.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/actions/correctors.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/offers/standart.h>
#include <drive/backend/proto/snapshot.pb.h>
#include <drive/backend/roles/manager.h>

#include <drive/library/cpp/compression/simple.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/algorithm/type_traits.h>
#include <rtline/util/types/string_pool.h>

static Y_THREAD(TStringPool) BillRecordIconPool;
static Y_THREAD(TStringPool) BillRecordIdPool;
static Y_THREAD(TStringPool) BillRecordTitlePool;
static Y_THREAD(TStringPool) BillRecordTypePool;
static Y_THREAD(TStringPool) CompiledLocalEventTagNamesPool;
static Y_THREAD(TStringPool) CompiledRidingObjectIdPool;
static Y_THREAD(TStringPool) CompiledRidingOfferNamePool;

TMinimalCompiledRiding::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    SumPrice = GetFieldDecodeIndex("price", decoderBase);
    SessionId = GetFieldDecodeIndex("session_id", decoderBase);
    ObjectId = GetFieldDecodeIndex("object_id", decoderBase);
    StartInstant = GetFieldDecodeIndex("start", decoderBase);
    FinishInstant = GetFieldDecodeIndex("finish", decoderBase);
}

bool TMinimalCompiledRiding::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, SessionId);
    READ_DECODER_VALUE(decoder, values, SumPrice);
    READ_DECODER_VALUE_INSTANT(decoder, values, StartInstant);
    READ_DECODER_VALUE_INSTANT(decoder, values, FinishInstant);
    TString objectId;
    READ_DECODER_VALUE_TEMP(decoder, values, objectId, ObjectId);
    ObjectId = TlsRef(CompiledRidingObjectIdPool).Get(objectId);
    return true;
}

NStorage::TTableRecord TMinimalCompiledRiding::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("session_id", SessionId);
    result.Set("start", StartInstant.Seconds());
    result.Set("finish", FinishInstant.Seconds());
    result.Set("price", SumPrice);
    if (ObjectId) {
        result.Set("object_id", ObjectId);
    }
    return result;
}

NJson::TJsonValue TMinimalCompiledRiding::GetReport() const {
    NJson::TJsonValue result;
    result.InsertValue("session_id", SessionId);
    result.InsertValue("start", StartInstant.Seconds());
    result.InsertValue("finish", FinishInstant.Seconds());
    result.InsertValue("total_price", SumPrice);
    return result;
}

TCompiledRiding::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase)
    : TDecoderBase(decoderBase)
{
    MetaProto = GetFieldDecodeIndex("meta_proto", decoderBase);
}

TBillRecord TCompiledRiding::BuildTariffRecord() const {
    TBillRecord result;
    result.SetType("tariff");
    result.SetTitle(OfferName);
    return result;
}

void TCompiledRiding::RecalcBillInfo(const TBill& bill) {
    UselessDuration = bill.GetFreeDuration() - bill.GetPricedDuration();
}

void TCompiledRiding::RecalcEventsInfo(const TVector<TCompiledLocalEvent>& events) {
    AcceptanceStarted.Clear();
    AcceptanceFinished.Clear();
    AcceptanceDuration.Clear();
    ParkingDuration.Clear();
    RidingDuration.Clear();
    auto& acceptanceDuration = AcceptanceDuration;
    auto& parkingDuration = ParkingDuration;
    auto& ridingDuration = RidingDuration;
    auto& ridingInstant = AcceptanceFinished;
    TMaybe<TCompiledLocalEvent> currentEvent;
    Y_ASSERT(std::is_sorted(events.begin(), events.end()));
    for (auto&& ev : events) {
        if (ev.GetTagName() == "old_state_acceptance" && !AcceptanceStarted) {
            AcceptanceStarted = ev.GetInstant();
        }
        if (ev.GetTagName() == "old_state_riding" && !ridingInstant) {
            ridingInstant = ev.GetInstant();
        }
        if (currentEvent) {
            auto delta = ev.GetInstant() - currentEvent->GetInstant();
            if (currentEvent->GetTagName() == "old_state_acceptance") {
                if (!acceptanceDuration) {
                    acceptanceDuration = delta;
                } else {
                    acceptanceDuration = *acceptanceDuration + delta;
                }
            }
            if (currentEvent->GetTagName() == "old_state_riding") {
                if (!ridingDuration) {
                    ridingDuration = delta;
                } else {
                    ridingDuration = *ridingDuration + delta;
                }
            }
            if (currentEvent->GetTagName() == "old_state_parking") {
                if (!parkingDuration) {
                    parkingDuration = delta;
                } else {
                    parkingDuration = *parkingDuration + delta;
                }
            }
        }
        currentEvent = ev;
    }
}

bool TCompiledRiding::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext) {
    if (!TBase::DeserializeWithDecoder(decoder, values, hContext)) {
        return false;
    }
    if (decoder.GetMetaProto() != -1) {
        TString metaProtoString;
        READ_DECODER_VALUE_TEMP(decoder, values, metaProtoString, MetaProto);
        if (!!metaProtoString) {
            auto decoded = NDrive::Decompress(metaProtoString);
            if (!decoded) {
                ERROR_LOG << "cannot Decompress meta_proto for " << GetSessionId() << ": " << decoded.GetError().AsStrBuf() << Endl;
                return false;
            }
            NDrive::NProto::TCompiledRidingMeta proto;
            if (!proto.ParseFromString(*decoded)) {
                return false;
            }
            if (proto.HasUselessDuration()) {
                UselessDuration = TDuration::Seconds(proto.GetUselessDuration());
            } else if (proto.HasBill()) {
                TBill bill;
                if (!bill.DeserializeFromProto(proto.GetBill())) {
                    return false;
                }
                RecalcBillInfo(bill);
            }
            if (proto.HasAcceptanceFinished() || proto.HasAcceptanceDuration() || proto.HasRidingDuration() || proto.HasParkingDuration()) {
                AcceptanceFinished = TInstant::Seconds(proto.GetAcceptanceFinished());
                AcceptanceDuration = TDuration::Seconds(proto.GetAcceptanceDuration());
                RidingDuration = TDuration::Seconds(proto.GetRidingDuration());
                ParkingDuration = TDuration::Seconds(proto.GetParkingDuration());
            } else {
                TVector<TCompiledLocalEvent> events;
                for (auto&& i : proto.GetLocalEvents()) {
                    TCompiledLocalEvent ev;
                    if (!ev.DeserializeFromProto(i)) {
                        return false;
                    }
                    events.emplace_back(std::move(ev));
                }
                RecalcEventsInfo(events);
            }
            OfferName = TlsRef(CompiledRidingOfferNamePool).Get(proto.GetOfferName());
            if (proto.HasSnapshotsDiff()) {
                TSnapshotsDiff diff;
                if (!diff.DeserializeFromProto(proto.GetSnapshotsDiff())) {
                    return false;
                }
                SnapshotsDiff = std::move(diff);
            }
            if (proto.HasDelegationType()) {
                DelegationType = static_cast<ECarDelegationType>(proto.GetDelegationType());
            }
            if (!DeserializeFromProto(proto)) {
                return false;
            }
        }
    }
    return true;
}

NStorage::TTableRecord TCompiledRiding::SerializeToTableRecord() const {
    NStorage::TTableRecord result = TBase::SerializeToTableRecord();
    result.Set("duration", GetDuration().Seconds());
    NDrive::NProto::TCompiledRidingMeta ridingProto;
    ridingProto.SetOfferName(OfferName);
    SerializeToProto(ridingProto);
    ridingProto.SetUselessDuration(UselessDuration.Seconds());
    if (AcceptanceFinished) {
        ridingProto.SetAcceptanceFinished(AcceptanceFinished->Seconds());
    }
    if (AcceptanceDuration) {
        ridingProto.SetAcceptanceDuration(AcceptanceDuration->Seconds());
    }
    if (RidingDuration) {
        ridingProto.SetRidingDuration(RidingDuration->Seconds());
    }
    if (ParkingDuration) {
        ridingProto.SetParkingDuration(ParkingDuration->Seconds());
    }
    if (!!SnapshotsDiff) {
        *ridingProto.MutableSnapshotsDiff() = SnapshotsDiff->SerializeToProto();
    }
    if (DelegationType) {
        ridingProto.SetDelegationType(static_cast<ui32>(*DelegationType));
    }
    auto metaProtoString = ridingProto.SerializeAsString();
    auto metaProtoCompressed = NDrive::Compress(metaProtoString);
    result.Set("meta_proto", *metaProtoCompressed);
    return result;
}

NJson::TJsonValue TCompiledRiding::GetReport(ELocalization locale, NDriveSession::TReportTraits traits, const NDrive::IServer& server) const {
    NJson::TJsonValue result = TBase::GetReport();
    result.InsertValue("total_duration", GetDuration().Seconds());
    result.InsertValue("offer_name", OfferName);
    if (DelegationType) {
        result.InsertValue("delegation_type", ToString(DelegationType));
    }
    if (!!SnapshotsDiff) {
        result.InsertValue("diff", SnapshotsDiff->GetReport(locale, server));
    }
    PatchReport(result, locale, traits, server);
    return result;
}

TMap<TString, NYdb::EPrimitiveType> TCompiledRiding::GetYDBSchema() {
    using EPrimitiveType = NYdb::EPrimitiveType;
    TMap<TString, EPrimitiveType> schema = {
        {"session_id",             EPrimitiveType::String},
        {"object_id",              EPrimitiveType::String},
        {"price",                  EPrimitiveType::Int32},
        {"duration",               EPrimitiveType::Int32},
        {"start",                  EPrimitiveType::Int32},
        {"finish",                 EPrimitiveType::Int32},
        {"meta",                   EPrimitiveType::JsonDocument},
        {"meta_proto",             EPrimitiveType::String},
        {"hard_proto",             EPrimitiveType::String},
    };
    return schema;
}

TMaybe<NDrive::TSensor> TSnapshotsDiff::GetStartFuelLevel() const {
    if (!StartFuelLevel) {
        return {};
    }
    NDrive::TSensor result(CAN_FUEL_LEVEL_P);
    result.Value = *StartFuelLevel;
    if (StartFuelLevelTimestamp) {
        result.Timestamp = *StartFuelLevelTimestamp;
    }
    return result;
}

TMaybe<NDrive::TSensor> TSnapshotsDiff::GetStartFuelVolume() const {
    if (!StartFuelVolume) {
        return {};
    }
    NDrive::TSensor result(CAN_CUSTOM_FUEL_VOLUME);
    result.Value = *StartFuelVolume;
    if (StartFuelVolumeTimestamp) {
        result.Timestamp = *StartFuelVolumeTimestamp;
    }
    return result;
}

NJson::TJsonValue TSnapshotsDiff::GetReport(ELocalization locale, const IServerBase& server) const {
    NJson::TJsonValue result = SerializeToJson();
    if (!!Mileage) {
        result.InsertValue("hr_mileage", server.GetLocalization()->DistanceFormatKm(locale, *Mileage));
    } else {
        result.InsertValue("hr_mileage", NJson::JSON_NULL);
    }
    return result;
}

NJson::TJsonValue TSnapshotsDiff::SerializeToJson() const {
    NJson::TJsonValue report(NJson::JSON_MAP);
    report.InsertValue("mileage", NJson::ToJson(Mileage));
    if (Start) {
        report.InsertValue("start", NDrive::TLocation(*Start).ToJson());
    }
    if (Last) {
        if (Finished) {
            report.InsertValue("finish", NDrive::TLocation(*Last).ToJson());
        } else {
            report.InsertValue("current", NDrive::TLocation(*Last).ToJson());
        }
    }
    if (StartFuelLevel) {
        report.InsertValue("start_fuel_level", *StartFuelLevel);
    }
    if (LastFuelLevel) {
        if (Finished) {
            report.InsertValue("finish_fuel_level", *LastFuelLevel);
        } else {
            report.InsertValue("current_fuel_level", *LastFuelLevel);
        }
    }
    if (StartFuelLevelTimestamp) {
        report.InsertValue("start_fuel_level_timestamp", NJson::ToJson(StartFuelLevelTimestamp));
    }
    if (LastFuelLevelTimestamp) {
        if (Finished) {
            report.InsertValue("finish_fuel_level_timestamp", NJson::ToJson(LastFuelLevelTimestamp));
        } else {
            report.InsertValue("current_fuel_level_timestamp", NJson::ToJson(LastFuelLevelTimestamp));
        }
    }
    if (StartFuelVolume) {
        report.InsertValue("start_fuel_volume", *StartFuelVolume);
    }
    if (LastFuelVolume) {
        if (Finished) {
            report.InsertValue("finish_fuel_volume", *LastFuelVolume);
        } else {
            report.InsertValue("current_fuel_volume", *LastFuelVolume);
        }
    }
    if (StartFuelVolumeTimestamp) {
        report.InsertValue("start_fuel_volume_timestamp", NJson::ToJson(StartFuelVolumeTimestamp));
    }
    if (LastFuelVolumeTimestamp) {
        if (Finished) {
            report.InsertValue("finish_fuel_volume_timestamp", NJson::ToJson(LastFuelVolumeTimestamp));
        } else {
            report.InsertValue("current_fuel_volume_timestamp", NJson::ToJson(LastFuelVolumeTimestamp));
        }
    }
    if (StartMileage) {
        report.InsertValue("start_mileage", *StartMileage);
    }
    if (LastMileage) {
        if (Finished) {
            report.InsertValue("finish_mileage", *LastMileage);
        } else {
            report.InsertValue("current_mileage", *LastMileage);
        }
    }
    return report;
}

bool TSnapshotsDiff::DeserializeFromJson(const NJson::TJsonValue& info) {
    double mileage;
    if (info.Has("mileage")) {
        JREAD_DOUBLE(info, "mileage", mileage);
        Mileage = mileage;
    }

    const NJson::TJsonValue* startJson;
    if (info.GetValuePointer("start", &startJson)) {
        NDrive::TLocation start;
        if (!start.TryFromJson(*startJson)) {
            return false;
        }
        Start = start;
    }

    NDrive::TLocation last;
    const NJson::TJsonValue* lastJson;
    if (info.GetValuePointer("finish", &lastJson)) {
        Finished = true;
        if (!last.TryFromJson(*lastJson)) {
            return false;
        }
        Last = last;
    } else if (info.GetValuePointer("current", &lastJson)) {
        Finished = false;
        if (!last.TryFromJson(*lastJson)) {
            return false;
        }
        Last = last;
    }
    return
        NJson::ParseField(info["start_mileage"], StartMileage) &&
        NJson::ParseField(info["current_mileage"], LastMileage) &&
        NJson::ParseField(info["finish_mileage"], LastMileage) &&

        NJson::ParseField(info["current_fuel_volume_timestamp"], LastFuelVolumeTimestamp) &&
        NJson::ParseField(info["finish_fuel_volume_timestamp"], LastFuelVolumeTimestamp) &&
        NJson::ParseField(info["start_fuel_volume_timestamp"], StartFuelVolumeTimestamp) &&
        NJson::ParseField(info["current_fuel_volume"], LastFuelVolume) &&
        NJson::ParseField(info["finish_fuel_volume"], LastFuelVolume) &&
        NJson::ParseField(info["start_fuel_volume"], StartFuelVolume) &&

        NJson::ParseField(info["current_fuel_level_timestamp"], LastFuelLevelTimestamp) &&
        NJson::ParseField(info["finish_fuel_level_timestamp"], LastFuelLevelTimestamp) &&
        NJson::ParseField(info["start_fuel_level_timestamp"], StartFuelLevelTimestamp) &&
        NJson::ParseField(info["current_fuel_level"], LastFuelLevel) &&
        NJson::ParseField(info["finish_fuel_level"], LastFuelLevel) &&
        NJson::ParseField(info["start_fuel_level"], StartFuelLevel);
}

NDrive::NProto::TSnapshotsDiff TSnapshotsDiff::SerializeToProto() const {
    NDrive::NProto::TSnapshotsDiff result;
    if (Mileage) {
        result.SetMileage(*Mileage);
    }
    if (StartMileage) {
        result.SetStartMileage(*StartMileage);
    }
    if (LastMileage) {
        result.SetLastMileage(*LastMileage);
    }
    if (Start) {
        *result.MutableStart() = NDrive::TLocation(*Start).ToProto();
    }
    if (Last) {
        *result.MutableLast() = NDrive::TLocation(*Last).ToProto();
    }
    if (StartFuelLevel) {
        result.SetStartFuelLevel(*StartFuelLevel);
    }
    if (LastFuelLevel) {
        result.SetLastFuelLevel(*LastFuelLevel);
    }
    if (StartFuelLevelTimestamp) {
        result.SetStartFuelLevelTimestamp(StartFuelLevelTimestamp->Seconds());
    }
    if (LastFuelLevelTimestamp) {
        result.SetLastFuelLevelTimestamp(LastFuelLevelTimestamp->Seconds());
    }
    if (StartFuelVolume) {
        result.SetStartFuelVolume(*StartFuelVolume);
    }
    if (LastFuelVolume) {
        result.SetLastFuelVolume(*LastFuelVolume);
    }
    if (StartFuelVolumeTimestamp) {
        result.SetStartFuelVolumeTimestamp(StartFuelVolumeTimestamp->Seconds());
    }
    if (LastFuelVolumeTimestamp) {
        result.SetLastFuelVolumeTimestamp(LastFuelVolumeTimestamp->Seconds());
    }
    result.SetFinished(Finished);
    return result;
}

bool TSnapshotsDiff::DeserializeFromProto(const NDrive::NProto::TSnapshotsDiff& info) {
    if (info.HasMileage()) {
        Mileage = info.GetMileage();
    }
    if (info.HasStartMileage()) {
        StartMileage = info.GetStartMileage();
    }
    if (info.HasLastMileage()) {
        LastMileage = info.GetLastMileage();
    }
    if (info.HasStart()) {
        NDrive::TLocation start;
        if (!start.FromProto(info.GetStart())) {
            return false;
        }
        Start = std::move(start);
    }
    if (info.HasLast()) {
        NDrive::TLocation last;
        if (!last.FromProto(info.GetLast())) {
            return false;
        }
        Last = std::move(last);
    }
    if (info.HasStartFuelLevel()) {
        StartFuelLevel = info.GetStartFuelLevel();
    }
    if (info.HasLastFuelLevel()) {
        LastFuelLevel = info.GetLastFuelLevel();
    }
    if (info.HasStartFuelLevelTimestamp()) {
        StartFuelLevelTimestamp = TInstant::Seconds(info.GetStartFuelLevelTimestamp());
    }
    if (info.HasLastFuelLevelTimestamp()) {
        LastFuelLevelTimestamp = TInstant::Seconds(info.GetLastFuelLevelTimestamp());
    }
    if (info.HasStartFuelVolume()) {
        StartFuelVolume = info.GetStartFuelVolume();
    }
    if (info.HasLastFuelVolume()) {
        LastFuelVolume = info.GetLastFuelVolume();
    }
    if (info.HasStartFuelVolumeTimestamp()) {
        StartFuelVolumeTimestamp = TInstant::Seconds(info.GetStartFuelVolumeTimestamp());
    }
    if (info.HasLastFuelVolumeTimestamp()) {
        LastFuelVolumeTimestamp = TInstant::Seconds(info.GetLastFuelVolumeTimestamp());
    }
    Finished = info.GetFinished();
    return true;
}

const TString TBillRecord::BillingTypePrefix = "billing_";
const TString TBillRecord::DiscountType = "discount";
const TString TBillRecord::SectionType = "section";
const TString TBillRecord::TotalType = "total";
const TString TBillRecord::CashbackType = "cashback";

TBillRecord TBillRecord::Section(const TString& name) {
    TBillRecord result;
    result.SetTitle(name);
    result.SetType(SectionType);
    return result;
}

bool TBillRecord::IsAreaFees(const TString& type) {
    return type == "fee_drop_zone_fix" || type == "fee_drop_zone_max";
}

NJson::TJsonValue TBillRecord::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    auto localization = server.GetLocalization();
    NJson::TJsonValue result;
    result["cost"] = Cost;
    result["type"] = Type;
    if (Id) {
        result["id"] = Id;
    }
    if (Details) {
        result["details"] = Details;
    }
    if (Distance > 0) {
        result["distance"] = Distance;
    }
    if (Duration != TDuration::Max()) {
        result["duration"] = Duration.Seconds();
    }
    TStringBuf icon = Icon;
    TStringBuf title = Title;
    if (!icon && !title && Id) {
        auto optionalAction = server.GetDriveAPI()->GetRolesManager()->GetActionsDB().GetObject(Id);
        auto corrector = optionalAction ? optionalAction->GetAs<TDiscountOfferCorrector>() : nullptr;
        if (corrector) {
            icon = corrector->GetSmallIcon() ? corrector->GetSmallIcon() : corrector->GetIcon();
            title = localization ? localization->ApplyResources(corrector->GetDescription(), locale) : corrector->GetDescription();
        }
    }
    if (icon) {
        result["icon"] = icon;
    }
    result["title"] = title;
    return result;
}

NJson::TJsonValue TBillRecord::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    JWRITE(result, "cost", Cost);
    JWRITE(result, "type", Type);
    JWRITE(result, "title", Title);

    JWRITE_DEF(result, "icon", Icon, "");
    JWRITE_DEF(result, "id", Id, "");
    if (Duration != TDuration::Max()) {
        JWRITE_DURATION(result, "duration", Duration);
    }
    JWRITE_DEF(result, "distance", Distance, 0);
    JWRITE_DEF(result, "details", Details, "");
    return result;
}

bool TBillRecord::DeserializeFromJson(const NJson::TJsonValue& info) {
    JREAD_INT(info, "cost", Cost);
    JREAD_STRING(info, "type", Type);
    JREAD_STRING_OPT(info, "title", Title);

    JREAD_DURATION_OPT(info, "duration", Duration);
    JREAD_DOUBLE_OPT(info, "distance", Distance);
    JREAD_STRING_OPT(info, "details", Details);
    JREAD_STRING_OPT(info, "icon", Icon);
    JREAD_STRING_OPT(info, "id", Id);
    return true;
}

NDrive::NProto::TBillRecord TBillRecord::SerializeToProto() const {
    NDrive::NProto::TBillRecord result;
    result.SetCost(Cost);
    if (Icon) {
        result.SetIcon(Icon);
    }
    if (Title) {
        result.SetTitle(Title);
    }
    if (Type) {
        result.SetType(Type);
    }
    if (Id) {
        result.SetId(Id);
    }
    if (Details) {
        result.SetDetails(Details);
    }
    if (Duration != TDuration::Max()) {
        result.SetDuration(Duration.Seconds());
    }
    result.SetDistance(static_cast<float>(Distance));
    return result;
}

bool TBillRecord::DeserializeFromProto(const NDrive::NProto::TBillRecord& info) {
    Cost = info.GetCost();
    if (info.HasDuration()) {
        Duration = TDuration::Seconds(info.GetDuration());
    }
    Distance = info.GetDistance();
    Icon = TlsRef(BillRecordIconPool).Get(info.GetIcon());
    Id = TlsRef(BillRecordIdPool).Get(info.GetId());
    Title = TlsRef(BillRecordTitlePool).Get(info.GetTitle());
    Type = TlsRef(BillRecordTypePool).Get(info.GetType());
    Details = info.GetDetails();
    return true;
}

TBill TBill::GetBillForNonPlusUser(TBill&& bill) {
    for (auto& record : bill.MutableRecords()) {
        if (record.GetType() == TBillRecord::SectionType && record.GetTitle() == NDrive::TLocalization::CashbackBillSection()) {
            record.SetTitle(NDrive::TLocalization::CashbackBillNonPlusSection());
        }
        if (record.GetType() == TBillRecord::CashbackType) {
            record.SetDetails(NDrive::TLocalization::CashbackBillingNonPlusDetails());
        }
    }
    return bill;
}

NJson::TJsonValue TBill::GetReport(ELocalization locale, NDriveSession::TReportTraits traits, const NDrive::IServer& server) const {
    bool insertDiscountSection = traits & NDriveSession::ReportBillSections;
    bool insertBillingSection = traits & NDriveSession::ReportBillSections;
    bool insertCashbackSection = traits & NDriveSession::ReportBillSections;
    ui32 discounts = 0;
    NJson::TJsonValue result;
    if (insertDiscountSection) {
        for (auto&& i : Records) {
            if (i.GetType() == TBillRecord::DiscountType) {
                ++discounts;
            }
        }
    }
    for (auto&& i : Records) {
        if (TBillRecord::IsAreaFees(i.GetType())) {
            result.AppendValue(TBillRecord::Section("").GetReport(locale, server));
        }
        if (insertDiscountSection && i.GetType() == TBillRecord::DiscountType) {
            insertDiscountSection = false;
            auto discountSectionName = discounts > 1 ? NDrive::TLocalization::DiscountsBillSection() : NDrive::TLocalization::DiscountBillSection();
            result.AppendValue(TBillRecord::Section(discountSectionName).GetReport(locale, server));
        }
        if (insertBillingSection && i.GetType().StartsWith(TBillRecord::BillingTypePrefix)) {
            insertBillingSection = false;
            result.AppendValue(TBillRecord::Section(NDrive::TLocalization::BillingBillSection()).GetReport(locale, server));
        }
        if (insertCashbackSection && i.GetType() == TBillRecord::CashbackType) {
            insertCashbackSection = false;
            result.AppendValue(TBillRecord::Section(NDrive::TLocalization::CashbackBillSection()).GetReport(locale, server));
        }
        result.AppendValue(i.GetReport(locale, server));
    }
    return result;
}

NJson::TJsonValue TBill::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    NJson::TJsonValue& recordsJson = result.InsertValue("records", NJson::JSON_ARRAY);
    for (auto&& i : Records) {
        recordsJson.AppendValue(i.SerializeToJson());
    }
    JWRITE_DURATION(result, "free_duration", FreeDuration);
    JWRITE_DURATION(result, "priced_duration", PricedDuration);
    JWRITE(result, "cashback_percent", CashbackPercent);
    return result;
};

bool TBill::DeserializeFromJson(const NJson::TJsonValue& info) {
    NJson::TJsonValue::TArray arr;
    if (!info.GetArray(&arr)) {
        if (info.IsMap()) {
            const NJson::TJsonValue* recordsJson;
            if (!info.GetValuePointer("records", &recordsJson) || !recordsJson->GetArray(&arr)) {
                return false;
            }
        } else {
            return false;
        }
    }
    for (auto&& i : arr) {
        TBillRecord record;
        if (!record.DeserializeFromJson(i)) {
            return false;
        }
        Records.emplace_back(std::move(record));
    }
    JREAD_DURATION_OPT(info, "free_duration", FreeDuration);
    JREAD_DURATION_OPT(info, "priced_duration", PricedDuration);
    JREAD_BOOL_OPT(info, "cashback_percent", CashbackPercent);
    return true;
}

NDrive::NProto::TBill TBill::SerializeToProto() const {
    NDrive::NProto::TBill result;
    result.SetFreeDuration(FreeDuration.Seconds());
    result.SetPricedDuration(PricedDuration.Seconds());
    result.SetCashbackPercent(CashbackPercent);
    for (auto&& i : Records) {
        *result.AddRecord() = i.SerializeToProto();
    }
    return result;
}

bool TBill::DeserializeFromProto(const NDrive::NProto::TBill& info) {
    FreeDuration = TDuration::Seconds(info.GetFreeDuration());
    PricedDuration = TDuration::Seconds(info.GetPricedDuration());
    CashbackPercent = info.GetCashbackPercent();
    for (auto&& i : info.GetRecord()) {
        TBillRecord record;
        if (!record.DeserializeFromProto(i)) {
            return false;
        }
        Records.emplace_back(std::move(record));
    }
    return true;
}

TMultiBill& TMultiBill::AddBill(TBill&& bill) {
    Bills.push_back(std::move(bill));
    return *this;
}

NJson::TJsonValue TMultiBill::GetReport(ELocalization locale, NDriveSession::TReportTraits traits, const NDrive::IServer& server) const {
    NJson::TJsonValue result = NJson::TJsonValue(NJson::JSON_MAP);
    NJson::TJsonValue& billsJson = result.InsertValue("bills", NJson::JSON_ARRAY);
    ui32 sumPrice = 0;
    for (auto&& i : Bills) {
        sumPrice += i.GetSumPrice();
        billsJson.AppendValue(i.GetReport(locale, traits, server));
    }
    NJson::TJsonValue& recordsJson = result.InsertValue("records", NJson::JSON_ARRAY);
    {
        TBillRecord billRecord;
        billRecord
            .SetCost(sumPrice)
            .SetType(TBillRecord::TotalType)
            .SetTitle(NDrive::TLocalization::TotalBillHeader());
        recordsJson.AppendValue(billRecord.GetReport(locale, server));
    }
    return result;
}

TCompiledLocalEvent::TCompiledLocalEvent(const TCarTagHistoryEvent& ev)
    : Instant(ev.GetHistoryInstant())
    , HistoryEventId(ev.GetHistoryEventId())
    , TagName(ev->GetName())
    , HistoryAction(ev.GetHistoryAction())
{
}

TCompiledLocalEvent::TCompiledLocalEvent(const TCarTagHistoryEvent* ev)
    : TCompiledLocalEvent(*Yasserted(ev))
{
}

TCompiledLocalEvent::TCompiledLocalEvent(const TAtomicSharedPtr<TCarTagHistoryEvent>& ev)
    : TCompiledLocalEvent(*Yasserted(ev))
{
}

bool TCompiledLocalEvent::DeserializeFromProto(const NDrive::NProto::TCompiledLocalEvent& proto) {
    Instant = TInstant::Seconds(proto.GetInstant());
    HistoryEventId = proto.GetHistoryEventId();
    TagName = TlsRef(CompiledLocalEventTagNamesPool).Get(proto.GetTagName());
    HistoryAction = (EObjectHistoryAction)proto.GetHistoryAction();
    TransformationSkippedByExternalCommand = proto.GetTransformationSkippedByExternalCommand();
    return true;
}

NDrive::NProto::TCompiledLocalEvent TCompiledLocalEvent::SerializeToProto() const {
    NDrive::NProto::TCompiledLocalEvent result;
    result.SetInstant(Instant.Seconds());
    result.SetHistoryEventId(HistoryEventId);
    result.SetTagName(TagName);
    result.SetHistoryAction((ui32)HistoryAction);
    if (TransformationSkippedByExternalCommand) {
        result.SetTransformationSkippedByExternalCommand(true);
    }
    return result;
}

NJson::TJsonValue TCompiledLocalEvent::SerializeToJson() const {
    NJson::TJsonValue result;
    result.InsertValue("timestamp", Instant.Seconds());
    result.InsertValue("event_id", HistoryEventId);
    result.InsertValue("tag_name", TagName);
    result.InsertValue("action", ::ToString(HistoryAction));
    if (TransformationSkippedByExternalCommand) {
        result.InsertValue("transformation_skipped_by_external_command", TransformationSkippedByExternalCommand);
    }
    return result;
}

TFullCompiledRiding::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase)
    : TBase(decoderBase)
{
    HardProto = GetFieldDecodeIndex("hard_proto", decoderBase);
}

void TFullCompiledRiding::PatchReport(NJson::TJsonValue& result, ELocalization locale, NDriveSession::TReportTraits traits, const NDrive::IServer& server) const {
    if (!!Offer && (traits & NDriveSession::EReportTraits::ReportOffer)) {
        ICommonOffer::TReportOptions reportOptions(locale, traits);
        result.InsertValue("offer", Offer->BuildJsonReport(reportOptions, server));
    }
    if (!!Bill && (traits & NDriveSession::EReportTraits::ReportBillSections)) {
        result.InsertValue("bill", Bill->GetReport(locale, traits, server));
        auto cashback = Bill->GetCashbackSum();
        if (cashback > 0) {
            result.InsertValue("total_cashback", cashback);
        } else if (!!Offer && Offer->GetCashbackPercent() > 0) {
            result.InsertValue("no_cashback", true);
        }
    }
    if (IMEI && (traits & NDriveSession::EReportTraits::ReportIMEI)) {
        result.InsertValue("imei", IMEI);
    }
    if (LocalEvents.size()) {
        NJson::TJsonValue& events = result.InsertValue("events", NJson::JSON_ARRAY);
        for (auto&& i : LocalEvents) {
            events.AppendValue(i.SerializeToJson());
        }
    }
    if (ServicingInfos.size()) {
        result.InsertValue("servicing_info", NJson::ToJson(ServicingInfos));
    }
    if (Offer) {
        IOffer::TOfferSessionPatchData patchData {
              GetDuration()
            , GetStartInstant().Get()
            , LocalEvents
            , OptionalSnapshotsDiff() ? &GetSnapshotsDiffRef() : nullptr
        };
        Offer->PatchSessionReport(result, traits, locale, server, patchData);
    }
}

void TFullCompiledRiding::SerializeToProto(NDrive::NProto::TCompiledRidingMeta& ridingProto) const {
    {
        for (auto&& i : LocalEvents) {
            *ridingProto.AddLocalEvents() = i.SerializeToProto();
        }
    }
    if (Bill) {
        *ridingProto.MutableBill() = Bill->SerializeToProto();
    }
    for (auto&& servicingInfo : ServicingInfos) {
        auto servicingInfoProto = ridingProto.AddServicingInfo();
        servicingInfoProto->SetStart(servicingInfo.Start.Seconds());
        servicingInfoProto->SetFinish(servicingInfo.Finish.Seconds());
        servicingInfoProto->SetMileage(servicingInfo.Mileage);
    }
}

bool TFullCompiledRiding::DeserializeFromProto(const NDrive::NProto::TCompiledRidingMeta& proto) {
    TVector<TCompiledLocalEvent> localEvents;
    for (auto&& i : proto.GetLocalEvents()) {
        TCompiledLocalEvent ev;
        if (!ev.DeserializeFromProto(i)) {
            return false;
        }
        localEvents.emplace_back(std::move(ev));
    }
    SetLocalEvents(std::move(localEvents));
    if (proto.HasOffer()) {
        Offer = ICommonOffer::ConstructFromProto<IOffer>(proto.GetOffer());
    }
    if (proto.HasBill()) {
        TBill bill;
        if (!bill.DeserializeFromProto(proto.GetBill())) {
            return false;
        }
        SetBill(std::move(bill));
    }
    for (auto&& servicingInfoProto : proto.GetServicingInfo()) {
        TServicingInfo servicingInfo;
        servicingInfo.Start = TInstant::Seconds(servicingInfoProto.GetStart());
        servicingInfo.Finish = TInstant::Seconds(servicingInfoProto.GetFinish());
        servicingInfo.Mileage = servicingInfoProto.GetMileage();
        ServicingInfos.push_back(std::move(servicingInfo));
    }
    return true;
}

NStorage::TTableRecord TFullCompiledRiding::SerializeToTableRecord() const {
    NStorage::TTableRecord result = TBase::SerializeToTableRecord();
    NDrive::NProto::TCompiledRidingHard hardProto;
    *hardProto.MutableOffer() = Offer->SerializeToProto();
    if (Bill) {
        *hardProto.MutableBill() = Bill->SerializeToProto();
    }
    if (IMEI) {
        hardProto.SetIMEI(IMEI);
    }
    for (auto&& i : LocalEvents) {
        *hardProto.MutableLocalEvents()->AddEvents() = i.SerializeToProto();
    }
    auto hardProtoString = hardProto.SerializeAsString();
    auto hardProtoCompressed = NDrive::Compress(hardProtoString);
    result.Set("hard_proto", *Yensured(hardProtoCompressed));
    return result;
}

const TCompiledLocalEvent* TFullCompiledRiding::GetLastEventAt(TInstant timestamp) const {
    auto lk = LowerKey(LocalEvents, timestamp);
    if (lk == LocalEvents.end()) {
        return nullptr;
    }
    return &*lk;
}

const TCompiledLocalEvent* TFullCompiledRiding::GetNextEventAt(TInstant timestamp) const {
    auto ub = UpperBound(LocalEvents, timestamp);
    if (ub == LocalEvents.end()) {
        return nullptr;
    }
    return &*ub;
}

TBill TFullCompiledRiding::GetBillWithTariffRecord() const {
    auto bill = GetBillUnsafe();
    bill.MutableRecords().insert(bill.MutableRecords().begin(), BuildTariffRecord());
    return bill;
}

bool TFullCompiledRiding::GetPhaseDuration(const TString& tagName, TDuration& result) const {
    if (LocalEvents.empty()) {
        return false;
    } else {
        TCompiledLocalEvent currentEvent = LocalEvents.front();
        result = TDuration::Zero();
        for (auto&& i : LocalEvents) {
            if (currentEvent.GetTagName() == tagName) {
                result = result + (i.GetInstant() - currentEvent.GetInstant());
            }
            currentEvent = i;
        }
        return true;
    }
}

bool TFullCompiledRiding::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext) {
    if (!TBase::DeserializeWithDecoder(decoder, values, hContext)) {
        return false;
    }
    if (decoder.GetHardProto() != -1) {
        TString hardProtoString;
        READ_DECODER_VALUE_TEMP(decoder, values, hardProtoString, HardProto);
        if (!!hardProtoString) {
            auto decoded = NDrive::Decompress(hardProtoString);
            if (!decoded) {
                ERROR_LOG << "cannot Decompress hard_proto for " << GetSessionId() << ": " << decoded.GetError().AsStrBuf() << Endl;
                return false;
            }
            NDrive::NProto::TCompiledRidingHard proto;
            if (!proto.ParseFromString(*decoded)) {
                return false;
            }
            if (proto.HasOffer()) {
                Offer = ICommonOffer::ConstructFromProto<IOffer>(proto.GetOffer());
            }
            if (proto.HasBill()) {
                TBill bill;
                if (!bill.DeserializeFromProto(proto.GetBill())) {
                    return false;
                }
                SetBill(std::move(bill));
            }
            if (proto.HasIMEI()) {
                IMEI = proto.GetIMEI();
            }
            if (proto.HasLocalEvents()) {
                TVector<TCompiledLocalEvent> localEvents;
                for (auto&& i : proto.GetLocalEvents().GetEvents()) {
                    TCompiledLocalEvent ev;
                    if (!ev.DeserializeFromProto(i)) {
                        return false;
                    }
                    localEvents.emplace_back(std::move(ev));
                }
                SetLocalEvents(std::move(localEvents));
            }
        }
    }
    return true;
}

template struct TExpectedSizeOf<TMinimalCompiledRiding, 40>;
