#pragma once

#include "servicing.h"

#include <drive/backend/offers/abstract.h>
#include <drive/backend/sessions/common/report.h>

#include <drive/telematics/server/location/location.h>

#include <ydb/public/sdk/cpp/client/ydb_value/value.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>

class IHistoryContext;
class IServerBase;

namespace NDrive::NProto {
    class TBill;
    class TBillRecord;
    class TCompiledLocalEvent;
    class TCompiledRidingMeta;
    class TSnapshotsDiff;
}

class TSnapshotsDiff {
private:
    R_OPTIONAL(double, Mileage);
    R_OPTIONAL(double, StartMileage);
    R_OPTIONAL(double, LastMileage);
    R_OPTIONAL(NDrive::TSimpleLocation, Start);
    R_OPTIONAL(NDrive::TSimpleLocation, Last);
    R_OPTIONAL(ui8, StartFuelLevel);
    R_OPTIONAL(ui8, LastFuelLevel);
    R_OPTIONAL(TTimestamp, StartFuelLevelTimestamp);
    R_OPTIONAL(TTimestamp, LastFuelLevelTimestamp);
    R_OPTIONAL(float, StartFuelVolume);
    R_OPTIONAL(float, LastFuelVolume);
    R_OPTIONAL(TTimestamp, StartFuelVolumeTimestamp);
    R_OPTIONAL(TTimestamp, LastFuelVolumeTimestamp);
    R_FIELD(bool, Finished, true);

public:
    TMaybe<NDrive::TSensor> GetStartFuelLevel() const;
    TMaybe<NDrive::TSensor> GetStartFuelVolume() const;

    NJson::TJsonValue GetReport(ELocalization locale, const IServerBase& server) const;

    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& info);

    NDrive::NProto::TSnapshotsDiff SerializeToProto() const;
    bool DeserializeFromProto(const NDrive::NProto::TSnapshotsDiff& info);
};

class TBillRecord {
private:
    R_FIELD(i32, Cost, 0);
    R_FIELD(TString, Icon);
    R_FIELD(TString, Title);
    R_FIELD(TString, Type);
    R_FIELD(TString, Id);
    R_FIELD(TString, Details);
    R_FIELD(TDuration, Duration, TDuration::Max());
    R_FIELD(double, Distance, 0);

public:
    static const TString BillingTypePrefix;
    static const TString DiscountType;
    static const TString SectionType;
    static const TString TotalType;
    static const TString CashbackType;

public:
    static TBillRecord Section(const TString& name);
    static bool IsAreaFees(const TString& type);

public:
    bool HasDuration() const {
        return Duration != TDuration::Max();
    }

    NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const;

    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& info);

    NDrive::NProto::TBillRecord SerializeToProto() const;
    bool DeserializeFromProto(const NDrive::NProto::TBillRecord& info);
};

class TBill {
private:
    R_FIELD(TVector<TBillRecord>, Records);
    R_FIELD(TDuration, FreeDuration, TDuration::Max());
    R_FIELD(TDuration, PricedDuration, TDuration::Max());
    R_FIELD(ui64, CashbackPercent, 0);

public:
    static TBill GetBillForNonPlusUser(TBill&& bill);

public:
    i32 GetSumPrice() const {
        for (auto&& i : Records) {
            if (i.GetType() == TBillRecord::TotalType) {
                return i.GetCost();
            }
        }
        return 0;
    }

    i32 GetCashbackSum() const {
        for (auto&& i : Records) {
            if (i.GetType() == TBillRecord::CashbackType) {
                return i.GetCost();
            }
        }
        return 0;
    }

    NJson::TJsonValue GetReport(ELocalization locale, NDriveSession::TReportTraits traits, const NDrive::IServer& server) const;

    NJson::TJsonValue SerializeToJson() const;
    bool DeserializeFromJson(const NJson::TJsonValue& info);

    NDrive::NProto::TBill SerializeToProto() const;
    bool DeserializeFromProto(const NDrive::NProto::TBill& info);
};

class TMultiBill {
private:
    TVector<TBill> Bills;

public:
    TMultiBill& AddBill(TBill&& bill);

    NJson::TJsonValue GetReport(ELocalization locale, NDriveSession::TReportTraits traits, const NDrive::IServer& server) const;
};

class TCompiledLocalEvent {
private:
    R_FIELD(TInstant, Instant, TInstant::Zero());
    R_FIELD(ui64, HistoryEventId, 0);
    R_FIELD(TString, TagName);
    R_FIELD(EObjectHistoryAction, HistoryAction, EObjectHistoryAction::Unknown);
    R_FIELD(bool, TransformationSkippedByExternalCommand, false);

public:
    TCompiledLocalEvent() = default;
    TCompiledLocalEvent(const TCarTagHistoryEvent& ev);
    TCompiledLocalEvent(const TCarTagHistoryEvent* ev);
    TCompiledLocalEvent(const TAtomicSharedPtr<TCarTagHistoryEvent>& ev);

    inline bool operator<(const TCompiledLocalEvent& other) const {
        return std::make_pair(Instant, HistoryEventId) < std::make_pair(other.Instant, other.HistoryEventId);
    }
    inline bool operator<(TInstant timestamp) const {
        return Instant < timestamp;
    }

    bool DeserializeFromProto(const NDrive::NProto::TCompiledLocalEvent& proto);
    NDrive::NProto::TCompiledLocalEvent SerializeToProto() const;

    NJson::TJsonValue SerializeToJson() const;
};

inline bool operator<(TInstant timestamp, const TCompiledLocalEvent& ev) {
    return timestamp < ev.GetInstant();
}

class TMinimalCompiledRiding {
public:
    R_FIELD(TString, SessionId);
    R_FIELD(TString, ObjectId);
    R_FIELD(TTimestamp, StartInstant);
    R_FIELD(TTimestamp, FinishInstant);
    R_FIELD(ui32, SumPrice, 0);

public:
    virtual ~TMinimalCompiledRiding() = default;

    inline bool operator<(const TMinimalCompiledRiding& other) const {
        return std::tie(FinishInstant, SessionId) < std::tie(other.FinishInstant, other.SessionId);
    }

    TDuration GetDuration() const {
        return FinishInstant - StartInstant;
    }

    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, SumPrice, -1);
        R_FIELD(i32, SessionId, -1);
        R_FIELD(i32, ObjectId, -1);
        R_FIELD(i32, StartInstant, -1);
        R_FIELD(i32, FinishInstant, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);

        static TString GetFieldsForRequest() {
            return "session_id, object_id, start, finish, price";
        }
    };

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

protected:
    NStorage::TTableRecord SerializeToTableRecord() const;
    NJson::TJsonValue GetReport() const;
};

class TCompiledRiding: public TMinimalCompiledRiding {
public:
    R_FIELD(TString, OfferName);
    R_OPTIONAL(TSnapshotsDiff, SnapshotsDiff);
    R_OPTIONAL(TInstant, AcceptanceStarted);
    R_OPTIONAL(TInstant, AcceptanceFinished);
    R_OPTIONAL(TDuration, AcceptanceDuration);
    R_OPTIONAL(TDuration, RidingDuration);
    R_OPTIONAL(TDuration, ParkingDuration);
    R_OPTIONAL(ECarDelegationType, DelegationType);
    R_FIELD(TDuration, UselessDuration, TDuration::Zero());

private:
    using TBase = TMinimalCompiledRiding;

protected:
    void RecalcBillInfo(const TBill& bill);
    void RecalcEventsInfo(const TVector<TCompiledLocalEvent>& events);

public:
    static TMap<TString, NYdb::EPrimitiveType> GetYDBSchema();

public:
    class TDecoder: public TBase::TDecoder {
    private:
        using TDecoderBase = TBase::TDecoder;

    private:
        R_FIELD(i32, MetaProto, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);

        static TString GetFieldsForRequest() {
            return "session_id, object_id, price, duration, start, finish, meta_proto";
        }
    };

protected:
    TBillRecord BuildTariffRecord() const;
    NStorage::TTableRecord SerializeToTableRecord() const;

    virtual bool DeserializeFromProto(const NDrive::NProto::TCompiledRidingMeta& /*proto*/) {
        return true;
    }
    virtual void SerializeToProto(NDrive::NProto::TCompiledRidingMeta& /*proto*/) const {
    }
    virtual void PatchReport(NJson::TJsonValue& /*result*/, ELocalization /*locale*/, NDriveSession::TReportTraits /*traits*/, const NDrive::IServer& /*server*/) const {
    }

public:
    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

    NJson::TJsonValue GetReport(ELocalization locale, NDriveSession::TReportTraits traits, const NDrive::IServer& server) const;
};

class TFullCompiledRiding: public TCompiledRiding {
private:
    using TBase = TCompiledRiding;

private:
    TMaybe<TBill> Bill;

public:
    R_FIELD(TString, IMEI);
    R_FIELD(IOffer::TPtr, Offer);
    R_FIELD(TVector<TServicingInfo>, ServicingInfos);
    R_READONLY(TVector<TCompiledLocalEvent>, LocalEvents);

public:
    const TBill& GetBillUnsafe() const {
        return *Bill;
    }

    bool HasBill() const {
        return Bill.Defined();
    }

    TFullCompiledRiding& SetLocalEvents(const TVector<TCompiledLocalEvent>& events) {
        LocalEvents = events;
        TBase::RecalcEventsInfo(LocalEvents);
        return *this;
    }

    TFullCompiledRiding& SetBill(const TBill& bill) {
        Bill = bill;
        TBase::RecalcBillInfo(bill);
        return *this;
    }

public:
    class TDecoder: public TBase::TDecoder {
    private:
        using TBase = TBase::TDecoder;

    private:
        R_FIELD(i32, HardProto, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);

        static TString GetFieldsForRequest() {
            return "*";
        }
    };

protected:
    virtual void PatchReport(NJson::TJsonValue& result, ELocalization locale, NDriveSession::TReportTraits traits, const NDrive::IServer& server) const override;

    virtual void SerializeToProto(NDrive::NProto::TCompiledRidingMeta& ridingProto) const override;
    virtual bool DeserializeFromProto(const NDrive::NProto::TCompiledRidingMeta& proto) override;

public:
    using TCompiledRiding::TCompiledRiding;

    const TCompiledLocalEvent* GetLastEventAt(TInstant timestamp) const;
    const TCompiledLocalEvent* GetNextEventAt(TInstant timestamp) const;
    TBill GetBillWithTariffRecord() const;

    NStorage::TTableRecord SerializeToTableRecord() const;
    bool GetPhaseDuration(const TString& tagName, TDuration& result) const;
    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext);
};
