#include "manager.h"
#include <rtline/library/json/proto/adapter.h>
#include <drive/backend/proto/offer.pb.h>

TMinimalRidingHistoryManager::TMinimalRidingHistoryManager(const ITagsHistoryContext& context)
    : TBase(context, "compiled_rides")
{
}

template <class T>
TOptionalObjectEvents<T> TMinimalRidingHistoryManager::GetEventsFiltered(TQueryOptions&& queryOptions, NDrive::TEntitySession& tx, const TFilter& filter) const {
    if (const auto& objectIds = filter.GetObjectIds()) {
        queryOptions.SetGenericCondition("object_id", objectIds.BuildCondition());
    }
    return GetEvents<T>({}, filter.GetTimestamps(), tx, queryOptions);
}

template <class T>
TOptionalObjectEvents<T> TMinimalRidingHistoryManager::Get(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& tx, const TFilter& filter) const {
    if (sessionIds.empty()) {
        return TObjectEvents<T>();
    }
    auto queryOptions = TQueryOptions()
        .SetGenericCondition("session_id", MakeSet<TString>(sessionIds))
        .SetOrderBy({ "history_timestamp" })
        .SetSecondaryIndex( "compiled_rides_session_id_index")
    ;
    return GetEventsFiltered<T>(std::move(queryOptions), tx, filter);
}

template <class T>
TOptionalObjectEvents<T> TMinimalRidingHistoryManager::GetObject(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange, TMaybe<ui32> last) const {
    if (objectIds.empty()) {
        return TObjectEvents<T>();
    }
    auto queryOptions = TQueryOptions(last ? *last : 0, !!last);
    queryOptions.SetGenericCondition("object_id", MakeSet<TString>(objectIds));
    if (objectIds.size() == 1) {
        // Help YDB to use index completely.
        queryOptions.SetOrderBy({ "object_id", "history_timestamp" });
    } else {
        queryOptions.SetOrderBy({ "history_timestamp" });
    }
    queryOptions.SetSecondaryIndex("compiled_rides_object_id_index");
    auto events = GetEvents<T>({}, std::move(timestampRange), tx, queryOptions);
    if (last && events) {
        std::reverse(events->begin(), events->end());
    }
    return std::move(events);
}

template <class T>
TOptionalObjectEvents<T> TMinimalRidingHistoryManager::GetObjects(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange, TRange<TEventId> eventIdRange, ui32 limit) const {
    if (objectIds.empty()) {
        return TObjectEvents<T>();
    }
    auto queryOptions = TQueryOptions(limit);
    queryOptions.SetGenericCondition("object_id", MakeSet<TString>(objectIds));
    queryOptions.SetOrderBy({ "history_event_id" });
    return GetEvents<T>(std::move(eventIdRange), std::move(timestampRange), tx, queryOptions);
}

template <class T>
TOptionalObjectEvents<T> TMinimalRidingHistoryManager::GetUser(TConstArrayRef<TString> userIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange, TMaybe<ui32> last) const {
    if (userIds.empty()) {
        return TObjectEvents<T>();
    }
    auto queryOptions = TQueryOptions(last ? *last : 0, !!last);
    queryOptions.SetGenericCondition("history_user_id", MakeSet<TString>(userIds));
    if (userIds.size() == 1) {
        // Help YDB to use index completely.
        queryOptions.SetOrderBy({ "history_user_id", "history_timestamp" });
    } else {
        queryOptions.SetOrderBy({ "history_timestamp" });
    }
    queryOptions.SetSecondaryIndex("compiled_rides_history_user_id_index");

    auto events = GetEvents<T>({}, std::move(timestampRange), tx, queryOptions);
    if (last && events) {
        std::reverse(events->begin(), events->end());
    }
    return std::move(events);
}

TMaybe<TObjectEvent<TFullCompiledRiding>> TMinimalRidingHistoryManager::CreateCompiledRide(const TFullCompiledRiding& riding, const NDrive::IServer& server, const TString& userId, NDrive::TEntitySession& tx) const {
    TObjectEvent<TFullCompiledRiding> event(riding, EObjectHistoryAction::Add, riding.GetFinishInstant(), userId, "", "");
    auto result = CreateCompiledRides({ event }, server, tx);
    if (!result) {
        return {};
    }
    return result->front();
}

TMaybe<TVector<TObjectEvent<TFullCompiledRiding>>> TMinimalRidingHistoryManager::CreateCompiledRides(const TVector<TObjectEvent<TFullCompiledRiding>>& ridings, const NDrive::IServer& server, NDrive::TEntitySession& tx) const {
    NStorage::TObjectRecordsSet<TObjectEvent<TFullCompiledRiding>> addedHistory;
    if (!TBase::AddHistory(ridings, tx, &addedHistory)) {
        return {};
    }
    if (addedHistory.size() != ridings.size()) {
        tx.SetErrorInfo("CreateCompiledRides", TStringBuilder() << "affected " << addedHistory.size() << "/" << ridings.size() << " rows", EDriveSessionResult::InternalError);
        return {};
    }
    tx.Committed().Subscribe([ridings, &server](const NThreading::TFuture<void>& commited) {
        if (commited.HasValue()) {
            for (const auto& riding : ridings) {
                auto locale = DefaultLocale;
                auto compiledRidingReport = riding.GetReport(locale, NDriveSession::ReportAll, server);
                auto offer = riding.GetOffer();
                if (offer) {
                    auto serializedOffer = offer->SerializeToProto();
                    compiledRidingReport["offer_proto"] = NJson::ToJson(NJson::Proto(serializedOffer));
                }
                NDrive::TEventLog::Log("CompiledRiding", compiledRidingReport);
            }
        }
    });
    return addedHistory.DetachObjects();
}

template <class T>
TOptionalObjectEvents<T> TMinimalRidingHistoryManager::Get(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange) const {
    auto filter = TMinimalRidingHistoryManager::TFilter()
        .SetTimestamps(timestampRange)
    ;
    auto optionalSessions = Get<T>(sessionIds, tx, filter);
    if (!optionalSessions) {
        return optionalSessions;
    }
    TObjectEvents<T> compiledRides = std::move(*optionalSessions);

    if (sessionIds.size() != compiledRides.size() && ydbTx) {
        THashSet<TString> compiledSessionsId;
        for (const auto& compiledSession : compiledRides) {
            compiledSessionsId.insert(compiledSession.GetSessionId());
        }

        auto size = compiledRides.size();
        auto optionalCompiledSessions = Get<T>(sessionIds, ydbTx, filter);
        if (!optionalCompiledSessions) {
            tx.SetErrorInfo("TMinimalRidingHistoryManager::Get", "Cannot get session from ydb: " + ydbTx.GetStringReport());
            return optionalCompiledSessions;
        }
        for (auto&& compiledSession : *optionalCompiledSessions) {
            if (!compiledSessionsId.contains(compiledSession.GetSessionId())) {
                compiledRides.push_back(std::move(compiledSession));
            }
        }
        FixCompiledRidesOrder<T>(size, compiledRides);
    }
    return compiledRides;
}

template <class T>
TOptionalObjectEvents<T> TMinimalRidingHistoryManager::GetObject(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange, TMaybe<ui32> numdoc) const {
    auto optionalCompiledSessions = GetObject<T>(objectIds, tx, timestampRange, numdoc);
    TObjectEvents<T> compiledRides;
    if (!optionalCompiledSessions) {
        return optionalCompiledSessions;
    }

    THashSet<TString> compiledSessionsId;
    compiledRides = std::move(*optionalCompiledSessions);
    for (auto&& compiledSession : compiledRides) {
        compiledSessionsId.insert(compiledSession.GetSessionId());
    }

    if (ydbTx && (!numdoc || compiledRides.size() < *numdoc)) {
        auto size = compiledRides.size();
        auto optionalCompiledSessions = GetObject<T>(objectIds, ydbTx, timestampRange, numdoc);
        if (!optionalCompiledSessions) {
            return optionalCompiledSessions;
        }
        for (auto&& compiledSession : *optionalCompiledSessions) {
            if (!compiledSessionsId.contains(compiledSession.GetSessionId())) {
                compiledRides.push_back(std::move(compiledSession));
            }
        }
        FixCompiledRidesOrder<T>(size, compiledRides);
    }
    return compiledRides;
}

template <class T>
TOptionalObjectEvents<T> TMinimalRidingHistoryManager::GetUser(const TString& userId, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange, TMaybe<ui32> numdoc) const {
    auto optionalCompiledSessions = GetUser<T>(userId, tx, timestampRange, numdoc);
    TObjectEvents<T> compiledRides;
    if (!optionalCompiledSessions) {
        return optionalCompiledSessions;
    }
    THashSet<TString> compiledSessionsId;
    compiledRides = std::move(*optionalCompiledSessions);
    for (const auto& compiledSession : compiledRides) {
        compiledSessionsId.insert(compiledSession.GetSessionId());
    }

    if (ydbTx && (!numdoc || compiledRides.size() < *numdoc)) {
        auto size = compiledRides.size();
        auto optionalCompiledSessions = GetUser<T>(userId, ydbTx, timestampRange, numdoc);
        if (!optionalCompiledSessions) {
            return optionalCompiledSessions;
        }
        for (auto&& compiledSession : *optionalCompiledSessions) {
            if (!compiledSessionsId.contains(compiledSession.GetSessionId())) {
                compiledRides.emplace_back(compiledSession);
            }
        }
        FixCompiledRidesOrder<T>(size, compiledRides);
    }
    return compiledRides;
}

template <class T>
TOptionalObjectEvents<T> TMinimalRidingHistoryManager::GetObjects(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange, TRange<TEventId> eventIdRange, ui32 limit) const {
   auto optionalCompiledSessions = GetObjects<T>(objectIds, tx, timestampRange, eventIdRange, limit);
   TObjectEvents<T> compiledRides;
    if (!optionalCompiledSessions) {
        return optionalCompiledSessions;
    }

    THashSet<TString> compiledSessionsId;
    compiledRides = std::move(*optionalCompiledSessions);
    for (const auto& compiledSession : compiledRides) {
        compiledSessionsId.insert(compiledSession.GetSessionId());
    }

    if (ydbTx && compiledRides.size() < limit) {
        auto size = compiledRides.size();
        auto optionalCompiledSessions = GetObjects<T>(objectIds, ydbTx, timestampRange, eventIdRange, limit);
        if (!optionalCompiledSessions) {
            return optionalCompiledSessions;
        }
        for (auto&& compiledSession : *optionalCompiledSessions) {
            if (!compiledSessionsId.contains(compiledSession.GetSessionId())) {
                compiledRides.emplace_back(compiledSession);
            }
        }
        FixCompiledRidesOrder<T>(size, compiledRides);
    }
    return compiledRides;
}

template <class T>
void TMinimalRidingHistoryManager::FixCompiledRidesOrder(size_t oldSize, TObjectEvents<T>& compiledRides) {
    if (oldSize > 0 && compiledRides.size() > oldSize) {
        TObjectEvents<T> mergedRides(compiledRides.size());

        std::merge(compiledRides.begin() + oldSize, compiledRides.end(), compiledRides.begin(), compiledRides.begin() + oldSize, mergedRides.begin(),
        [](TObjectEvent<T> first, TObjectEvent<T> second){
            return first.GetHistoryTimestamp() < second.GetHistoryTimestamp();
        });
        compiledRides.swap(mergedRides);
    }
};

template <class T>
TOptionalObjectEvents<T> TMinimalRidingHistoryManager::GetByTimestamp(NDrive::TEntitySession& tx, NSQL::TQueryOptions queryOptions, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange, TMaybe<ui32> numdoc) const {
    auto optionalCompiledSessions = GetTimestampEvents<T>(tx, queryOptions, timestampRange);
    TObjectEvents<T> compiledRides;
    if (!optionalCompiledSessions) {
        return optionalCompiledSessions;
    }

    THashSet<TString> compiledSessionsId;
    compiledRides = std::move(*optionalCompiledSessions);
    std::reverse(compiledRides.begin(), compiledRides.end());
    for (const auto& compiledSession : compiledRides) {
        compiledSessionsId.insert(compiledSession.GetSessionId());
    }


    if (ydbTx && (!numdoc || compiledRides.size() < *numdoc)) {
        auto optionalCompiledSessions = GetTimestampEvents<T>(ydbTx, queryOptions, timestampRange);
        if (!optionalCompiledSessions) {
            tx.SetErrorInfo("TMinimalRidingHistoryManager::GetByTimestamp", "cannot get session from ydb: " + ydbTx.GetStringReport());
            return optionalCompiledSessions;
        }
        for (auto&& compiledSession : Reversed(*optionalCompiledSessions)) {
            if (!compiledSessionsId.contains(compiledSession.GetSessionId())) {
                compiledRides.push_back(std::move(compiledSession));
            }
        }
    }
    return compiledRides;
}

// Instantiate.
template TOptionalObjectEvents<TCompiledRiding> TMinimalRidingHistoryManager::Get(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& tx, const TFilter& filter = {}) const;
template TOptionalObjectEvents<TMinimalCompiledRiding> TMinimalRidingHistoryManager::Get(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& tx, const TFilter& filter = {}) const;
template TOptionalObjectEvents<TFullCompiledRiding> TMinimalRidingHistoryManager::Get(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& tx, const TFilter& filter = {}) const;

template TOptionalObjectEvents<TCompiledRiding> TMinimalRidingHistoryManager::GetObject(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> last = {}) const;
template TOptionalObjectEvents<TMinimalCompiledRiding> TMinimalRidingHistoryManager::GetObject(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> last = {}) const;
template TOptionalObjectEvents<TFullCompiledRiding> TMinimalRidingHistoryManager::GetObject(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> last = {}) const;

template TOptionalObjectEvents<TCompiledRiding> TMinimalRidingHistoryManager::GetObjects(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange = {}, TRange<TEventId> eventIdRange = {}, ui32 limit = 0) const;
template TOptionalObjectEvents<TMinimalCompiledRiding> TMinimalRidingHistoryManager::GetObjects(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange = {}, TRange<TEventId> eventIdRange = {}, ui32 limit = 0) const;
template TOptionalObjectEvents<TFullCompiledRiding> TMinimalRidingHistoryManager::GetObjects(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange = {}, TRange<TEventId> eventIdRange = {}, ui32 limit = 0) const;

template TOptionalObjectEvents<TCompiledRiding> TMinimalRidingHistoryManager::GetUser(TConstArrayRef<TString> userIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> last = {}) const;
template TOptionalObjectEvents<TMinimalCompiledRiding> TMinimalRidingHistoryManager::GetUser(TConstArrayRef<TString> userIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> last = {}) const;
template TOptionalObjectEvents<TFullCompiledRiding> TMinimalRidingHistoryManager::GetUser(TConstArrayRef<TString> userIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> last = {}) const;

template TOptionalObjectEvents<TCompiledRiding> TMinimalRidingHistoryManager::Get(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}) const;
template TOptionalObjectEvents<TMinimalCompiledRiding> TMinimalRidingHistoryManager::Get(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}) const;
template TOptionalObjectEvents<TFullCompiledRiding> TMinimalRidingHistoryManager::Get(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}) const;

template TOptionalObjectEvents<TCompiledRiding> TMinimalRidingHistoryManager::GetObject(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> numdoc = {}) const;
template TOptionalObjectEvents<TMinimalCompiledRiding> TMinimalRidingHistoryManager::GetObject(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> numdoc = {}) const;
template TOptionalObjectEvents<TFullCompiledRiding> TMinimalRidingHistoryManager::GetObject(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> numdoc = {}) const;

template TOptionalObjectEvents<TCompiledRiding> TMinimalRidingHistoryManager::GetUser(const TString& userId, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> numdoc = {}) const;
template TOptionalObjectEvents<TMinimalCompiledRiding> TMinimalRidingHistoryManager::GetUser(const TString& userId, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> numdoc = {}) const;
template TOptionalObjectEvents<TFullCompiledRiding> TMinimalRidingHistoryManager::GetUser(const TString& userId, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> numdoc = {}) const;

template TOptionalObjectEvents<TCompiledRiding> TMinimalRidingHistoryManager::GetByTimestamp(NDrive::TEntitySession& tx, NSQL::TQueryOptions queryOptions, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> numdoc = {}) const;
template TOptionalObjectEvents<TMinimalCompiledRiding> TMinimalRidingHistoryManager::GetByTimestamp(NDrive::TEntitySession& tx, NSQL::TQueryOptions queryOptions, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> numdoc = {}) const;
template TOptionalObjectEvents<TFullCompiledRiding> TMinimalRidingHistoryManager::GetByTimestamp(NDrive::TEntitySession& tx, NSQL::TQueryOptions queryOptions, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> numdoc = {}) const;

template TOptionalObjectEvents<TCompiledRiding> TMinimalRidingHistoryManager::GetObjects(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TRange<TEventId> eventIdRange = {}, ui32 limit = 0) const;
template TOptionalObjectEvents<TMinimalCompiledRiding> TMinimalRidingHistoryManager::GetObjects(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TRange<TEventId> eventIdRange = {}, ui32 limit = 0) const;
template TOptionalObjectEvents<TFullCompiledRiding> TMinimalRidingHistoryManager::GetObjects(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TRange<TEventId> eventIdRange = {}, ui32 limit = 0) const;
