#pragma once

#include "compiled_riding.h"

#include <drive/backend/database/history/manager.h>

class TMinimalRidingHistoryManager: public TDatabaseHistoryManager<TFullCompiledRiding> {
private:
    using TBase = TDatabaseHistoryManager<TFullCompiledRiding>;
    using TBase::AddHistory;

public:
    struct TFilter {
        R_FIELD(TRange<TInstant>, Timestamps);
        R_FIELD(NSQL::TStringContainer, ObjectIds);
    };

    TMinimalRidingHistoryManager(const ITagsHistoryContext& context);

    TMaybe<TObjectEvent<TFullCompiledRiding>> CreateCompiledRide(const TFullCompiledRiding& riding, const NDrive::IServer& server, const TString& userId, NDrive::TEntitySession& tx) const;
    TMaybe<TVector<TObjectEvent<TFullCompiledRiding>>> CreateCompiledRides(const TVector<TObjectEvent<TFullCompiledRiding>>& ridings, const NDrive::IServer& server, NDrive::TEntitySession& tx) const;

    template <class T>
    TOptionalObjectEvents<T> GetObjects(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TRange<TEventId> eventIdRange = {}, ui32 limit = 0) const;
    template <class T>
    TOptionalObjectEvents<T> Get(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& tx,  NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}) const;
    template <class T>
    TOptionalObjectEvents<T> GetObject(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> numdoc = {}) const;
    template <class T>
    TOptionalObjectEvents<T> GetUser(const TString& userId, NDrive::TEntitySession& tx, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> numdoc = {}) const;

    template <class T>
    TOptionalObjectEvents<T> GetByTimestamp(NDrive::TEntitySession& tx, NSQL::TQueryOptions queryOptions, NDrive::TEntitySession& ydbTx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> numdoc = {}) const;

private:
    template <class T>
    TOptionalObjectEvents<T> Get(TConstArrayRef<TString> sessionIds, NDrive::TEntitySession& tx, const TFilter& filter = {}) const;

    template <class T>
    TOptionalObjectEvents<T> GetObject(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> last = {}) const;

    template <class T>
    TOptionalObjectEvents<T> GetObject(const TString& objectId, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> last = {}) const {
        return GetObject<T>(NContainer::Scalar(objectId), tx, std::move(timestampRange), last);
    }

    template <class T>
    TOptionalObjectEvents<T> GetObjects(TConstArrayRef<TString> objectIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange = {}, TRange<TEventId> eventIdRange = {}, ui32 limit = 0) const;

    template <class T>
    TOptionalObjectEvents<T> GetUser(TConstArrayRef<TString> userIds, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> last = {}) const;
    template <class T>
    TOptionalObjectEvents<T> GetUser(const TString& userId, NDrive::TEntitySession& tx, TRange<TInstant> timestampRange = {}, TMaybe<ui32> last = {}) const {
        return GetUser<T>(NContainer::Scalar(userId), tx, std::move(timestampRange), last);
    }

    template <class T>
    TOptionalObjectEvents<T> GetTimestampEvents(NDrive::TEntitySession& tx, NSQL::TQueryOptions options, TRange<TInstant> timestampRange = {}) const {
        return GetEvents<T>({}, std::move(timestampRange), tx, options.SetSecondaryIndex("compiled_rides_history_timestamp_index"));
    };

    template <class T>
    TOptionalObjectEvents<T> GetEventsFiltered(TQueryOptions&& queryOptions, NDrive::TEntitySession& tx, const TFilter& filter = {}) const;

    template <class T>
    static void FixCompiledRidesOrder(size_t oldSize, TObjectEvents<T>& compiledRides);
};
