#include "photo_recognizer_client.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/library/cpp/taxi/antifraud/client.h>

#include <rtline/util/types/uuid.cpp>
#include <drive/library/cpp/raw_text/datetime.h>

TRecognizedDocument ParsePassportData(const TTaxiAntifraudClient::TRecognizedDocument& recognizedData) {
    TRecognizedDocument recognition;
    TUserPassportData result;
    TRecognitionConfidenceData confidences = NPhotoRecognizerUtils::GetEmptyPassportConfidences();
    for (auto&& value : recognizedData.GetRecognizedTexts()) {
        if (value.Type == "name") {
            result.SetFirstName(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidences.AddField(TRecognitionConfidenceData::EDocumentField::FirstName, value.Confidence);
        } else if (value.Type == "middle_name") {
            result.SetMiddleName(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidences.AddField(TRecognitionConfidenceData::EDocumentField::MiddleName, value.Confidence);
        } else if (value.Type == "surname") {
            result.SetLastName(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidences.AddField(TRecognitionConfidenceData::EDocumentField::LastName, value.Confidence);
        } else if (value.Type == "birth_date") {
            result.SetBirthDate(NPhotoRecognizerUtils::ExtractDateStrIfValid(value.Text));
            confidences.AddField(TRecognitionConfidenceData::EDocumentField::BirthDate, value.Confidence);
        } else if (value.Type == "number") {
            result.SetNumber(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidences.AddField(TRecognitionConfidenceData::EDocumentField::PassportNumber, value.Confidence);
        } else if (value.Type == "issued_by") {
            result.SetIssuedBy(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidences.AddField(TRecognitionConfidenceData::EDocumentField::IssuedBy, value.Confidence);
        } else if (value.Type == "issue_date") {
            result.SetIssueDate(NPhotoRecognizerUtils::ExtractDateIfValid(value.Text));
            confidences.AddField(TRecognitionConfidenceData::EDocumentField::IssueDate, value.Confidence);
        } else if (value.Type == "subdivision") {
            result.SetSubdivisionCode(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidences.AddField(TRecognitionConfidenceData::EDocumentField::PassportSubdivisionCode, value.Confidence);
        } else if (value.Type == "gender") {
            result.SetGender(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidences.AddField(TRecognitionConfidenceData::EDocumentField::Gender, value.Confidence);
        } else if (value.Type == "citizenship") {
            result.SetCitizenship(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidences.AddField(TRecognitionConfidenceData::EDocumentField::Citizenship, value.Confidence);
        } else if (value.Type == "birth_place") {
            result.SetBirthPlace(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidences.AddField(TRecognitionConfidenceData::EDocumentField::BirthPlace, value.Confidence);
        } else if (value.Type == "expiration_date") {
            result.SetExpirationDate(NPhotoRecognizerUtils::ExtractDateIfValid(value.Text));
            confidences.AddField(TRecognitionConfidenceData::EDocumentField::ExpirationDate, value.Confidence);
        }
    }
    if (recognizedData.HasQuasiFmsScore()) {
        confidences.SetQuasiFmsScore(recognizedData.GetQuasiFmsScoreRef());
    }
    if (recognizedData.HasFromScreenScore()) {
        confidences.SetFromScreenScore(recognizedData.OptionalFromScreenScore());
    }
    recognition.SetPassportData(result).SetConfidences(confidences);
    return recognition;
}

TRecognizedDocument ParseLicenseFrontData(const TTaxiAntifraudClient::TRecognizedDocument& recognizedData) {
    TRecognizedDocument recognition;
    TUserDrivingLicenseData result;
    TRecognitionConfidenceData confidencesFront = NPhotoRecognizerUtils::GetEmptyLicenseConfidencesFront();
    for (auto&& value : recognizedData.GetRecognizedTexts()) {
        if (value.Type == "name") {
            result.SetFirstName(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::FirstName, value.Confidence);
        } else if (value.Type == "middle_name") {
            result.SetMiddleName(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::MiddleName, value.Confidence);
        } else if (value.Type == "surname") {
            result.SetLastName(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::LastName, value.Confidence);
        } else if (value.Type == "birth_date") {
            result.SetBirthDate(NPhotoRecognizerUtils::ExtractDateStrIfValid(value.Text));
            confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::BirthDate, value.Confidence);
        } else if (value.Type == "number") {
            result.SetNumberFront(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::LicenseNumberFront, value.Confidence);
        } else if (value.Type == "issued_by") {
            result.SetIssuedBy(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::IssuedBy, value.Confidence);
        } else if (value.Type == "issue_date") {
            result.SetIssueDate(NPhotoRecognizerUtils::ExtractDateIfValid(value.Text));
            confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::IssueDate, value.Confidence);
        } else if (value.Type == "expiration_date") {
            result.SetCategoriesBValidToDateFront(NPhotoRecognizerUtils::ExtractDateIfValid(value.Text));
            confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::LicenseCategoriesBValidToFront, value.Confidence);
        }
    }
    confidencesFront.SetQuasiGibddScore(recognizedData.OptionalQuasiGibddScore());
    confidencesFront.SetFromScreenScore(recognizedData.OptionalFromScreenScore());
    confidencesFront.SetBadFormatScore(recognizedData.OptionalBadFormatScore());
    recognition.SetLicenseData(result).SetConfidences(confidencesFront);
    return recognition;
}

TRecognizedDocument ParseLicenseBackData(const TTaxiAntifraudClient::TRecognizedDocument& recognizedData) {
    TRecognizedDocument recognition;
    TUserDrivingLicenseData result;
    TRecognitionConfidenceData confidencesBack = NPhotoRecognizerUtils::GetEmptyLicenseConfidencesBack();
    for (auto&& value : recognizedData.GetRecognizedTexts()) {
        if (value.Type == "prev_number") {
            result.SetPrevLicenseNumber(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidencesBack.AddField(TRecognitionConfidenceData::EDocumentField::LicensePrevNumber, value.Confidence);
        } else if (value.Type == "experience_from") {
            result.SetExperienceFromStr(NPhotoRecognizerUtils::ExtractYearStrIfValid(value.Text));
            confidencesBack.AddField(TRecognitionConfidenceData::EDocumentField::LicenseExperienceFrom, value.Confidence);
        } else if (value.Type == "number") {
            result.SetNumberBack(NPhotoRecognizerUtils::ExtractStr(value.Text));
            confidencesBack.AddField(TRecognitionConfidenceData::EDocumentField::LicenseNumberBack, value.Confidence);
        } else if (value.Type == "issue_date") {
            result.SetCategoriesBValidFromDateStr(NPhotoRecognizerUtils::ExtractDateStrIfValid(value.Text));
            confidencesBack.AddField(TRecognitionConfidenceData::EDocumentField::IssueDate, value.Confidence);
        } else if (value.Type == "expiration_date") {
            result.SetCategoriesBValidToDate(NPhotoRecognizerUtils::ExtractDateIfValid(value.Text));
            confidencesBack.AddField(TRecognitionConfidenceData::EDocumentField::ExpirationDate, value.Confidence);
        }
    }
    confidencesBack.SetQuasiGibddScore(recognizedData.OptionalQuasiGibddScore());
    confidencesBack.SetFromScreenScore(recognizedData.OptionalFromScreenScore());
    confidencesBack.SetBadFormatScore(recognizedData.OptionalBadFormatScore());
    recognition.SetLicenseData(result).SetConfidences(confidencesBack);
    return recognition;
}

TRecognizedDocument TPassportRecognizerClient::GetEmptyDocument() const {
    TRecognitionConfidenceData confidences = NPhotoRecognizerUtils::GetEmptyPassportConfidences();
    TRecognizedDocument recognition;
    recognition.SetPassportData(TUserPassportData{}).SetConfidences(confidences);
    return recognition;
}

NThreading::TFuture<TRecognizedDocument> TPassportRecognizerClient::GetRecognizedDocument(const NDrive::IServer& server, const TString& photoBlob) const {
    if (!server.GetTaxiAntifraudClient()) {
        return NThreading::TExceptionFuture() << "TPassportRecognizerClient::GetRecognizedDocument TaxiAntifraudClient not configured";
    }
    auto recognizeFuture = server.GetTaxiAntifraudClient()->RecognizePassport(photoBlob);
    return recognizeFuture.Apply([](const auto& fut) -> NThreading::TFuture<TRecognizedDocument> {
        if (fut.HasException() || !fut.HasValue()) {
            return NThreading::TExceptionFuture() << "TPassportRecognizerClient::GetRecognizedDocument Bad value in future: " << NThreading::GetExceptionMessage(fut);
        }
        return NThreading::MakeFuture(ParsePassportData(fut.GetValue()));
    });
}

TRecognizedDocument TLicenseFrontRecognizerClient::GetEmptyDocument() const {
    TRecognitionConfidenceData confidences = NPhotoRecognizerUtils::GetEmptyLicenseConfidencesFront();
    TRecognizedDocument recognition;
    recognition.SetLicenseData(TUserDrivingLicenseData{}).SetConfidences(confidences);
    return recognition;
}

NThreading::TFuture<TRecognizedDocument> TLicenseFrontRecognizerClient::GetRecognizedDocument(const NDrive::IServer& server, const TString& photoBlob) const {
    if (!server.GetTaxiAntifraudClient()) {
        return NThreading::TExceptionFuture() << "TLicenseFrontRecognizerClient::GetRecognizedDocument TaxiAntifraudClient not configured";
    }
    auto recognizeFrontFuture = server.GetTaxiAntifraudClient()->RecognizeLicenseFront(photoBlob);
    return recognizeFrontFuture.Apply([] (const auto& fut) -> NThreading::TFuture<TRecognizedDocument> {
        if (fut.HasException() || !fut.HasValue()) {
            return NThreading::TExceptionFuture() << "TLicenseFrontRecognizerClient::GetRecognizedDocument Bad value in future: " << NThreading::GetExceptionMessage(fut);
        }
        return NThreading::MakeFuture(ParseLicenseFrontData(fut.GetValue()));
    });
}

TRecognizedDocument TLicenseBackRecognizerClient::GetEmptyDocument() const {
    TRecognitionConfidenceData confidences = NPhotoRecognizerUtils::GetEmptyLicenseConfidencesBack();
    TRecognizedDocument recognition;
    recognition.SetLicenseData(TUserDrivingLicenseData{}).SetConfidences(confidences);
    return recognition;
}

NThreading::TFuture<TRecognizedDocument> TLicenseBackRecognizerClient::GetRecognizedDocument(const NDrive::IServer& server, const TString& photoBlob) const {
    if (!server.GetTaxiAntifraudClient()) {
        return NThreading::TExceptionFuture() << "TLicenseBackRecognizerClient::GetRecognizedDrivingLicenseData TaxiAntifraudClient not configured";
    }
    auto recognizeBackFuture = server.GetTaxiAntifraudClient()->RecognizeLicenseBack(photoBlob);
    return recognizeBackFuture.Apply([] (const auto& fut) -> NThreading::TFuture<TRecognizedDocument> {
        if (fut.HasException() || !fut.HasValue()) {
            return NThreading::TExceptionFuture() << "TLicenseBackRecognizerClient::GetRecognizedDocument Bad value in future: " <<  NThreading::GetExceptionMessage(fut);
        }
        return NThreading::MakeFuture(ParseLicenseBackData(fut.GetValue()));
    });
}

void NormalizeData(TUserDrivingLicenseData& licenseData) {
    tm parsedTm;
    if (licenseData.HasBirthDate() && !NUtil::ParseFomattedDatetime(licenseData.GetBirthDateRef(), "%Y-%m-%d", parsedTm)) {
        licenseData.SetBirthDate(Nothing());
    }
    if (licenseData.HasExperienceFromStr() && !NUtil::ParseFomattedDatetime(licenseData.GetExperienceFromStrRef(), "%Y-%m-%d", parsedTm)) {
        licenseData.SetExperienceFromStr(Nothing());
    }
    if (licenseData.HasPrevLicenseIssueDateStrFront() && !NUtil::ParseFomattedDatetime(licenseData.GetPrevLicenseIssueDateStrFrontRef(), "%Y-%m-%d", parsedTm)) {
        licenseData.SetPrevLicenseIssueDateStrFront(Nothing());
    }
    if (licenseData.HasPrevLicenseIssueDateStr() && !NUtil::ParseFomattedDatetime(licenseData.GetPrevLicenseIssueDateStrRef(), "%Y-%m-%d", parsedTm)) {
        licenseData.SetPrevLicenseIssueDateStr(Nothing());
    }
    if (licenseData.HasSpecialCategoryBDateStrFront() && !NUtil::ParseFomattedDatetime(licenseData.GetSpecialCategoryBDateStrFrontRef(), "%Y-%m-%d", parsedTm)) {
        licenseData.SetSpecialCategoryBDateStrFront(Nothing());
    }
    if (licenseData.HasSpecialCategoryBDateStrBack() && !NUtil::ParseFomattedDatetime(licenseData.GetSpecialCategoryBDateStrBackRef(), "%Y-%m-%d", parsedTm)) {
        licenseData.SetSpecialCategoryBDateStrBack(Nothing());
    }
}

void NormalizeData(TUserPassportData& passportData) {
    tm parsedTm;
    if (passportData.HasBirthDate() && !NUtil::ParseFomattedDatetime(passportData.GetBirthDateRef(), "%Y-%m-%d", parsedTm)) {
        passportData.SetBirthDate(Nothing());
    }
    if (!passportData.HasMiddleName()) {
        passportData.SetMiddleName("");
    }
}

NThreading::TFuture<TRecognizedDocument> IPhotoRecognizerClient::GetPatchedRecognizedDocument(const NDrive::IServer* server, const TDriveUserData& userData, const TString& photoBlob, TString photoId) {
    if (photoId.empty()) {
        photoId = NUtil::CreateUUID();
    }

    auto recognizeFuture = GetRecognizedDocument(*server, photoBlob);
    return recognizeFuture.Apply([
            uid = userData.GetPassportUid(),
            photoId
        ] (const auto& fut) mutable
    {
        auto recognized = fut.GetValue();
        recognized.SetPhotoId(photoId);
        auto updatedRevision = "recognized-" + (photoId.empty() ? NUtil::CreateUUID() : photoId);
        recognized.SetDocumentRevision(updatedRevision);
        if (recognized.HasPassportData()) {
            auto userDocumentsData = recognized.GetPassportDataRef();
            NormalizeData(userDocumentsData);
            recognized.SetPassportData(userDocumentsData);
        }
        if (recognized.HasLicenseData()){
            auto userDocumentsData = recognized.GetLicenseDataRef();
            if (!userDocumentsData.HasNumberBack() && userDocumentsData.HasNumberFront()) {
                userDocumentsData.SetNumberBack(userDocumentsData.GetNumberFrontRef());
            }
            NormalizeData(userDocumentsData);
            recognized.SetLicenseData(userDocumentsData);
        }
        return recognized;
    });
}
