#pragma once

#include "recognizer_utils.h"

#include <drive/backend/users/user.h>
#include <drive/backend/users/user_documents.h>

class TRecognizedDocument {
    R_FIELD(TRecognitionConfidenceData, Confidences);

    R_OPTIONAL(TUserPassportData, PassportData);
    R_OPTIONAL(TUserDrivingLicenseData, LicenseData);

    R_OPTIONAL(double, QuasiGibddScore);
    R_OPTIONAL(double, QuasiFmsScore);
    R_OPTIONAL(double, FromScreenScore);
    R_OPTIONAL(double, BadFormatScore);

    R_FIELD(TString, DocumentRevision);
    R_FIELD(TString, PhotoId);

public:
    TRecognizedDocument() = default;

    void MergeData(const TMaybe<TUserPassportData>& otherPassport, const TMaybe<TUserDrivingLicenseData>& otherLicense) {
        if (otherPassport) {
            if (PassportData) {
                PassportData->Patch(*otherPassport);
            } else {
                PassportData = *otherPassport;
            }
        }
        if (otherLicense) {
            if (LicenseData) {
                LicenseData->Patch(*otherLicense);
            } else {
                LicenseData = *otherLicense;
            }
        }
    }
};

class IPhotoRecognizerClient {
public:
    using TFactory = NObjectFactory::TObjectFactory<IPhotoRecognizerClient, NUserDocument::EType>;

public:
    virtual NThreading::TFuture<TRecognizedDocument> GetRecognizedDocument(const NDrive::IServer& server, const TString& photoBlob) const = 0;
    NThreading::TFuture<TRecognizedDocument> GetPatchedRecognizedDocument(const NDrive::IServer* server, const TDriveUserData& userData, const TString& photoBlob, TString photoId);

    virtual ~IPhotoRecognizerClient() {
    }

private:
    virtual TRecognizedDocument GetEmptyDocument() const = 0;
};

class TLicenseFrontRecognizerClient: public IPhotoRecognizerClient {
private:
    static inline TFactory::TRegistrator<TLicenseFrontRecognizerClient> Registrator{NUserDocument::EType::LicenseFront};

public:
    virtual NThreading::TFuture<TRecognizedDocument> GetRecognizedDocument(const NDrive::IServer& server, const TString& photoBlob) const override;

private:
    virtual TRecognizedDocument GetEmptyDocument() const override;
};

class TLicenseBackRecognizerClient: public IPhotoRecognizerClient {
private:
    static inline TFactory::TRegistrator<TLicenseBackRecognizerClient> Registrator{NUserDocument::EType::LicenseBack};

public:
    virtual NThreading::TFuture<TRecognizedDocument> GetRecognizedDocument(const NDrive::IServer& server, const TString& photoBlob) const override;

private:
    virtual TRecognizedDocument GetEmptyDocument() const override;
};

class TPassportRecognizerClient: public IPhotoRecognizerClient {
private:
    static inline TFactory::TRegistrator<TPassportRecognizerClient> Registrator{NUserDocument::EType::PassportBiographical};

public:
    virtual NThreading::TFuture<TRecognizedDocument> GetRecognizedDocument(const NDrive::IServer& server, const TString& photoBlob) const override;

private:
    virtual TRecognizedDocument GetEmptyDocument() const override;
};
