#include "recognizer_utils.h"

#include <util/string/cast.h>

namespace NPhotoRecognizerUtils {
    TMaybe<TString> ExtractStr(const TString& recognized) {
        if (recognized) {
            return recognized;
        } else {
            return {};
        }
    }

    TMaybe<TString> ExtractDateStrIfValid(const TString& recognized) {
        // ugly workaround to work with dates earlier than 1970-01-01
        if (recognized.size() != 10|| recognized[2] != '.' || recognized[5] != '.') {
            return {};
        }
        TString dayStr = recognized.substr(0, 2);
        TString monthStr = recognized.substr(3, 2);
        TString yearStr = recognized.substr(6, 4);

        ui32 year;
        ui32 month;
        ui32 day;
        if (!TryFromString(yearStr, year) || !TryFromString(monthStr, month) || !TryFromString(dayStr, day)) {
            return {};
        }
        if (year < 1900 || month < 1 || month > 12 || day < 1 || day > DaysInMonth(year, month)) {
            return {};
        }
        return yearStr + "-" + monthStr + "-" + dayStr + "T00:00:00.000Z";
    }

    ui32 DaysInMonth(const ui32 year, const ui32 month) {
        if (month < 1 || month > 12) {
            return 0;
        }
        bool isLeapYear = year % 400 == 0 || (year % 100 != 0 && year % 4 == 0);
        return TVector<ui32>({31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31})[month - 1] + (month == 2 && isLeapYear);
    }

    TMaybe<TInstant> ExtractDateIfValid(const TString& recognized) {
        auto stringValue = ExtractDateStrIfValid(recognized);
        if (!stringValue) {
            return {};
        }

        TInstant result;
        if (!TInstant::TryParseIso8601(*stringValue, result)) {
            return {};
        }
        return result;
    }

    TMaybe<TString> ExtractYearStrIfValid(const TString& recognized) {
        if (recognized.size() != 4) {
            return {};
        }
        ui32 year;
        if (!TryFromString(recognized, year) || year < 1900) {
            return {};
        }
        return recognized + "-00-00T00:00:00.000Z";
    }

    TRecognitionConfidenceData GetEmptyPassportConfidences() {
        TRecognitionConfidenceData confidences;
        confidences.AddField(TRecognitionConfidenceData::EDocumentField::FirstName, 0);
        confidences.AddField(TRecognitionConfidenceData::EDocumentField::LastName, 0);
        confidences.AddField(TRecognitionConfidenceData::EDocumentField::MiddleName, 0);
        confidences.AddField(TRecognitionConfidenceData::EDocumentField::BirthDate, 0);
        confidences.AddField(TRecognitionConfidenceData::EDocumentField::IssueDate, 0);
        confidences.AddField(TRecognitionConfidenceData::EDocumentField::PassportNumber, 0);
        confidences.AddField(TRecognitionConfidenceData::EDocumentField::PassportSubdivisionCode, 0);
        confidences.AddField(TRecognitionConfidenceData::EDocumentField::Gender, 0);
        confidences.AddField(TRecognitionConfidenceData::EDocumentField::Citizenship, 0);
        confidences.AddField(TRecognitionConfidenceData::EDocumentField::ExpirationDate, 0);
        confidences.AddField(TRecognitionConfidenceData::EDocumentField::Country, 0);
        return confidences;
    }

    TRecognitionConfidenceData GetEmptyLicenseConfidencesFront() {
        TRecognitionConfidenceData confidencesFront;
        confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::FirstName, 0);
        confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::LastName, 0);
        confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::MiddleName, 0);
        confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::LicenseNumberFront, 0);
        confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::BirthDate, 0);
        confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::IssueDate, 0);
        confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::LicensePrevNumber, 0);
        confidencesFront.AddField(TRecognitionConfidenceData::EDocumentField::CountryFront, 0);
        return confidencesFront;
    }

    TRecognitionConfidenceData GetEmptyLicenseConfidencesBack() {
        TRecognitionConfidenceData confidencesBack;
        confidencesBack.AddField(TRecognitionConfidenceData::EDocumentField::LicenseNumberBack, 0);
        confidencesBack.AddField(TRecognitionConfidenceData::EDocumentField::LicenseCategoriesBValidFrom, 0);
        confidencesBack.AddField(TRecognitionConfidenceData::EDocumentField::LicenseCategoriesBValidTo, 0);
        confidencesBack.AddField(TRecognitionConfidenceData::EDocumentField::LicenseExperienceFrom, 0);
        confidencesBack.AddField(TRecognitionConfidenceData::EDocumentField::CountryBack, 0);
        return confidencesBack;
    }
}
