#include "car_tags_history_event.h"

#include <drive/backend/cars/hardware.h>
#include <drive/backend/cars_filter/cars_filter.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/url.h>
#include <drive/backend/device_snapshot/snapshot.h>
#include <drive/backend/tags/tag_description.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/user.h>

#include <drive/library/cpp/raw_text/datetime.h>

#include <rtline/library/json/cast.h>
#include <rtline/util/algorithm/container.h>

#include <library/cpp/string_utils/relaxed_escaper/relaxed_escaper.h>

#include <util/generic/serialized_enum.h>
#include <util/string/cast.h>

TCarTagHistoryEventFetchContext::TCarTagHistoryEventFetchContext(const NDrive::IServer* server, const TCarTagHistoryEvent& event, TString placeholderStart, TString placeholderEnd)
    : TBase(server, event)
{
    SetOptions(TOptions{placeholderStart, placeholderEnd});
}

const TString IEventContextFetcher::TypeNamePrefix = "event.";

bool IEventContextFetcher::FetchPerformer(const TContextType& context, TDriveUserData& userData) const {
    const auto& event = context.GetEntry();
    if (!event.GetHistoryUserId()) {
        ERROR_LOG << "Event does not have history user id: history event id - " << event.GetHistoryEventId() << Endl;
        return false;
    }
    auto gUsers = context.GetServer().GetDriveAPI()->GetUsersData()->FetchInfo(event.GetHistoryUserId());
    auto userDataPtr = gUsers.GetResultPtr(event.GetHistoryUserId());
    if (userDataPtr) {
        userData = std::move(*userDataPtr);
        return true;
    }
    ERROR_LOG << "Cannot restore history user instance: history event id - " << event.GetHistoryEventId() << Endl;
    return false;
}

TString TEventInstantContextFetcher::GetTypeName() {
    return TypeNamePrefix + "instant";
}

TEventInstantContextFetcher::TRegistrator TEventInstantContextFetcher::Registrator;

bool TEventInstantContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    result = context.GetEntry().GetHistoryInstant().ToStringLocalUpToSeconds();
    return true;
}

TString TEventDateContextFetcher::GetTypeName() {
    return TypeNamePrefix + "date";
}

TEventDateContextFetcher::TRegistrator TEventDateContextFetcher::Registrator;

bool TEventDateContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    NUtil::TTimeZone destinationTimezone;
    if (!NUtil::GetTimeZone(context.GetTimezoneName(), destinationTimezone)) {
        return false;
    }
    auto convertedTime = NUtil::ConvertTimeZone(context.GetEntry().GetHistoryInstant(), NUtil::GetUTCTimeZone(), destinationTimezone);
    result = NUtil::FormatDatetime(convertedTime, "%Y-%m-%d");
    return true;
}

TString TEventUserLoginContextFetcher::GetTypeName() {
    return TypeNamePrefix + "user_login";
}

TEventUserLoginContextFetcher::TRegistrator TEventUserLoginContextFetcher::Registrator;

bool TEventUserLoginContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    TDriveUserData userData;
    if (!FetchPerformer(context, userData)) {
        return false;
    }
    result = userData.GetLogin();
    return true;
}

const TString ITagContextFetcher::TypeNamePrefix = "tag.";

const ITag* ITagContextFetcher::FetchTag(const TContextType& context) const {
    return context.GetEntry().GetTagAs<ITag>();
}

TString TTagIdContextFetcher::GetTypeName() {
    return TypeNamePrefix + "id";
}

TTagIdContextFetcher::TRegistrator TTagIdContextFetcher::Registrator;

bool TTagIdContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    result = context.GetEntry().GetTagId();
    return true;
}

TString TTagNameContextFetcher::GetTypeName() {
    return TypeNamePrefix + "name";
}

TTagNameContextFetcher::TRegistrator TTagNameContextFetcher::Registrator;

bool TTagNameContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    auto tag = FetchTag(context);
    if (!tag) {
        return false;
    }
    result = tag->GetName();
    return true;
}

TString TTagDisplayNameContextFetcher::GetTypeName() {
    return TypeNamePrefix + "display_name";
}

TTagDisplayNameContextFetcher::TRegistrator TTagDisplayNameContextFetcher::Registrator;

bool TTagDisplayNameContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    auto tag = FetchTag(context);
    if (!tag) {
        return false;
    }
    const ITagsMeta& tagsMeta = context.GetServer().GetDriveAPI()->GetTagsManager().GetTagsMeta();
    auto tagDesctiptionPtr = tagsMeta.GetDescriptionByName(tag->GetName());
    result = (tagDesctiptionPtr) ? tagDesctiptionPtr->GetDisplayName() : tag->GetName();
    return true;
}

TString TTagCommentContextFetcher::GetTypeName() {
    return TypeNamePrefix + "comment";
}

TTagCommentContextFetcher::TRegistrator TTagCommentContextFetcher::Registrator;

bool TTagCommentContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    auto tag = FetchTag(context);
    if (!tag) {
        return false;
    }
    auto escaped = NEscJ::EscapeJ<false>(tag->GetComment());
    result = escaped;
    return true;
}

const TString ISnapshotContextFetcher::TypeNamePrefix = "snapshot.";

const THistoryDeviceSnapshot* ISnapshotContextFetcher::FetchSnapshot(const TContextType& context) const {
    return context.GetEntry()->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
}

TString TSnapshotMileageContextFetcher::GetTypeName() {
    return TypeNamePrefix + "mileage";
}

TSnapshotMileageContextFetcher::TRegistrator TSnapshotMileageContextFetcher::Registrator;

bool TSnapshotMileageContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    double mileage;
    auto snapshot = FetchSnapshot(context);
    if (snapshot && snapshot->GetMileage(mileage)) {
        result = ::ToString(mileage);
        return true;
    }
    return false;
}

const TString ICarContextFetcher::TypeNamePrefix = "car.";

bool ICarContextFetcher::FetchCar(const TContextType& context, TDriveCarInfo& carData) const {
    const auto& event = context.GetEntry();
    auto gCars = context.GetServer().GetDriveAPI()->GetCarsData()->GetCachedOrFetch(event.GetObjectId(event));
    auto carDataPtr = gCars.GetResultPtr(event.GetObjectId(event));
    if (carDataPtr) {
        carData = std::move(*carDataPtr);
        return true;
    }
    ERROR_LOG << "Cannot restore history car data: history event id - " << event.GetHistoryEventId() << Endl;
    return false;
}

TString TCarModelContextFetcher::GetTypeName() {
    return TypeNamePrefix + "model";
}

TCarModelContextFetcher::TRegistrator TCarModelContextFetcher::Registrator;

bool TCarModelContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    TDriveCarInfo carData;
    if (!FetchCar(context, carData)) {
        return false;
    }
    result = carData.GetModel();
    return true;
}

TString TCarNumberContextFetcher::GetTypeName() {
    return TypeNamePrefix + "number";
}

TCarNumberContextFetcher::TRegistrator TCarNumberContextFetcher::Registrator;

bool TCarNumberContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    TDriveCarInfo carData;
    if (!FetchCar(context, carData)) {
        return false;
    }
    result = carData.GetNumber();
    return true;
}

TString TCarVinContextFetcher::GetTypeName() {
    return TypeNamePrefix + "vin";
}

TCarVinContextFetcher::TRegistrator TCarVinContextFetcher::Registrator;

bool TCarVinContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    TDriveCarInfo carData;
    if (!FetchCar(context, carData)) {
        return false;
    }
    result = carData.GetVin();
    return true;
}

TString TCarIdContextFetcher::GetTypeName() {
    return TypeNamePrefix + "id";
}

TCarIdContextFetcher::TRegistrator TCarIdContextFetcher::Registrator;

bool TCarIdContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    const auto& event = context.GetEntry();
    result = event.GetObjectId(event);
    return true;
}

TString TCarInsurerContextFetcher::GetTypeName() {
    return TypeNamePrefix + "insurer";
}

TCarInsurerContextFetcher::TRegistrator TCarInsurerContextFetcher::Registrator;

bool TCarInsurerContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    const auto& event = context.GetEntry();
    TString carId = event.GetObjectId(event);

    TCarInsurancePolicy policy;

    const auto& gAttachments = context.GetServer().GetDriveAPI()->GetCarAttachmentAssignments();
    if (!gAttachments.GetEffectiveInsurancePolicy(carId, ModelingNow(), policy)) {
        ERROR_LOG << "Cannot obtain insurance policy: history event id - " << event.GetHistoryEventId() << Endl;
        return false;
    }

    result = ::ToString(policy.GetProvider());
    return true;
}

TString TCarUrlContextFetcher::GetTypeName() {
    return TypeNamePrefix + "url";
}

TCarUrlContextFetcher::TRegistrator TCarUrlContextFetcher::Registrator;

bool TCarUrlContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    TDriveCarInfo carData;
    if (FetchCar(context, carData)) {
        result = TCarsharingUrl().CarInfo(carData.GetId());
    } else {
        result = "";
    }
    return true;
}

const TString ICarTagHistoryEventContextFetcher::SettingPrefix = "context_fetcher.car_tag_history_event_context";
const TString ICarTagHistoryEventContextFetcher::LocalizationPrefix = "context_fetcher.car_tag_history_event_context";

bool ICarParamByTagContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    const auto& event = context.GetEntry();
    TString carId = event.GetObjectId(event);

    TSet<TString> tagNames;
    if (!NJson::TryFromJson(context.GetServer().GetSettings().GetJsonValue(SettingPrefix + "." + GetParamKey()), tagNames)) {
        ERROR_LOG << "Cannot restore " << GetParamKey() << " tag names: history event id - " << event.GetHistoryEventId() << Endl;
        return false;
    }

    TVector<TDBTag> tags;
    {
        const auto& deviceTagManager = context.GetServer().GetDriveAPI()->GetTagsManager().GetDeviceTags();
        auto tx = deviceTagManager.BuildTx<NSQL::ReadOnly>();
        if (!deviceTagManager.RestoreEntityTags(carId, MakeVector(tagNames), tags, tx)) {
            ERROR_LOG << "Cannot restore car " << GetParamKey() << " tags: history event id - " << event.GetHistoryEventId() << ": " << tx.GetStringReport() << Endl;
            return false;
        }
    }

    const TString defaultCity = GetParamValue(context.GetServer(), {}, "undefined");  // even if default param name is missed in settings

    result = (tags.size() == 1) ? GetParamValue(context.GetServer(), tags.front()->GetName(), defaultCity) : defaultCity;
    return true;
}

TString ICarParamByTagContextFetcher::GetParamValue(const NDrive::IServer& server, const TMaybe<TString>& tagName, const TString& fallback) const {
    const TString resourceId = JoinSeq(".", { LocalizationPrefix, GetParamKey(), tagName.GetOrElse("default") });
    return Yensured(server.GetLocalization())->GetLocalString(ELocalization::Rus, resourceId, fallback);
}

TString TCarCityContextFetcher::GetParamKey() const {
    return "car_city_tags";
}

TString TCarCityContextFetcher::GetTypeName() {
    return TypeNamePrefix + "city";
}

TCarCityContextFetcher::TRegistrator TCarCityContextFetcher::Registrator;

TString TCarPartnerContextFetcher::GetParamKey() const {
    return "car_partner_tags";
}

TString TCarPartnerContextFetcher::GetTypeName() {
    return TypeNamePrefix + "partner";
}

TCarPartnerContextFetcher::TRegistrator TCarPartnerContextFetcher::Registrator;

bool TCarFilterContextFetcherBase::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    if (!Parameters) {
        result = NJson::ToJson(false).GetStringRobust();
        return true;
    }

    const auto& event = context.GetEntry();
    const auto& carId = event.GetObjectId(event);

    NJson::TJsonValue carsFilterParameters = GetCarsFilterParameters(carId);

    TCarsFilter carsFilter;
    if (!carsFilter.DeserializeFromJson(carsFilterParameters)) {
        ERROR_LOG << "Cannot construct cars filter for parameters " << carsFilterParameters.GetStringRobust() << " in placeholder " << OriginalPlaceholderName << Endl;
        return false;
    }

    TSet<TString> allowedCarIds;
    if (!carsFilter.GetAllowedCarIds(allowedCarIds, &context.GetServer())) {
        ERROR_LOG << "Cannot filter allowed car ids in placeholder " << OriginalPlaceholderName << Endl;
        return false;
    }

    result = NJson::ToJson(allowedCarIds.contains(carId)).GetStringRobust();
    return true;
}

TString TCarAreaTagsCheckContextFetcher::GetTypeName() {
    return TypeNamePrefix + "does_match_area_tags";
}

TCarAreaTagsCheckContextFetcher::TRegistrator TCarAreaTagsCheckContextFetcher::Registrator;

NJson::TJsonValue TCarAreaTagsCheckContextFetcher::GetCarsFilterParameters(const TString& carId) const {
    const auto& areaTags = Parameters;  // variants obtained from server.GetFullAreaTagsList()
    return NJson::TMapBuilder
        ("include_cars", carId)
        ("area_tags", NJson::ToJson(areaTags));
}

TString TCarAreaIdsCheckContextFetcher::GetTypeName() {
    return TypeNamePrefix + "does_match_area_ids";
}

TCarAreaIdsCheckContextFetcher::TRegistrator TCarAreaIdsCheckContextFetcher::Registrator;

NJson::TJsonValue TCarAreaIdsCheckContextFetcher::GetCarsFilterParameters(const TString& carId) const {
    const auto& areaIds = Parameters;  // variants obtained from server.GetDriveAPI()->GetAreasDB()->GetAreaIds()
    return NJson::TMapBuilder
        ("include_cars", carId)
        ("include_areas", NJson::ToJson(areaIds));
}
