#include "json.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/url.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/notifications/startrek/startrek.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/raw_text/datetime.h>
#include <drive/library/cpp/startrek/client.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/string_utils/relaxed_escaper/relaxed_escaper.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>

#include <util/string/cast.h>
#include <util/string/escape.h>
#include <util/string/join.h>
#include <util/string/strip.h>
#include <util/string/subst.h>

const TString TJsonFetchContext::DefaultDateTimeFormat = "%d.%m.%Y %H:%M";
const TString TJsonFetchContext::DefaultTimezoneName = "Europe/Moscow";
const bool TJsonFetchContext::DefaultFillNotMatchedPlaceholders = true;
const TString TJsonFetchContext::DefaultActiveOnlySessionEndInstantResult = "Еще в поездке";

TString TJsonFetchContext::GetDateTimeFormat() const {
    if (HasDateTimeFormat()) {
        return GetDateTimeFormatRef();
    }
    return GetServer().GetSettings().GetValueDef<TString>(GetFullSettingKey("datetime_format"), DefaultDateTimeFormat);
}

TString TJsonFetchContext::GetTimezoneName() const {
    if (HasTimezoneName()) {
        return GetTimezoneNameRef();
    }
    return GetServer().GetSettings().GetValueDef<TString>(GetFullSettingKey("timezone_name"), DefaultTimezoneName);
}

bool TJsonFetchContext::DoFillNotMatchedPlaceholders() const {
    if (HasFillNotMatchedPlaceholders()) {
        return GetFillNotMatchedPlaceholdersRef();
    }
    return GetServer().GetSettings().GetValueDef<bool>(GetFullSettingKey("fill_not_matched_placeholders"), DefaultFillNotMatchedPlaceholders);
}

TString TJsonFetchContext::GetActiveOnlySessionEndInstantResult() const {
    if (HasActiveOnlySessionEndInstantResult()) {
        return GetActiveOnlySessionEndInstantResultRef();
    }
    return GetServer().GetSettings().GetValueDef<TString>(GetFullSettingKey("active_only_session_end_instant_result"), DefaultActiveOnlySessionEndInstantResult);
}

TString TJsonFetchContext::GetFullSettingKey(TStringBuf key) const {
    return TString("context_fetcher.json_context.common.") + key;
}

void TJsonFetchContext::SetCachedUserData(THolder<TDriveUserData>&& userData) const {
    CachedUserData = std::move(userData);
}

const TDriveUserData* TJsonFetchContext::GetCachedUserData() const {
    return CachedUserData.Get();
}

TString TJsonDefaultContextFetcher::GetTypeName() {
    return "default";
}

TJsonDefaultContextFetcher::TRegistrator TJsonDefaultContextFetcher::Registrator;

bool TJsonDefaultContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    auto* propertyPtr = context.GetEntry().GetValueByPath(OriginalPlaceholderName);
    if (!propertyPtr || !propertyPtr->IsDefined()) {
        if (context.DoFillNotMatchedPlaceholders()) {
            result = "";
            return true;
        }
        return false;  // NB. Missed placeholder should not be treated as an error or fail reason should be detailed
    }

    if (propertyPtr->IsBoolean()) {
        if (Parameters.size() >= 1 && ::ToLowerUTF8(Parameters[0]) == "eng") {
            result = (propertyPtr->GetBoolean()) ? "Yes" : "No";
        } else {
            result = (propertyPtr->GetBoolean()) ? "Да" : "Нет";
        }
    } else {
        // default parser cannot detect type (besides boolean), so type should be specified explicitly
        if (Parameters.size() >= 1) {
            const TString& parameterType = Parameters[0];
            if (parameterType.StartsWith("instant")) {
                TInstant timestamp;
                if (!TryFromJson(*propertyPtr, timestamp)) {
                    errors.AddMessage(__LOCATION__, "Cannot interpret value " + propertyPtr->GetStringRobust() + " as a timestamp in placeholder " + OriginalPlaceholderName);
                    return false;
                }
                if (parameterType.Contains("_isoformat")) {
                    result = timestamp.ToString();
                } else {
                    if (parameterType.Contains("_local")) {
                        timestamp = NUtil::ConvertTimeZone(timestamp, NUtil::GetUTCTimeZone(), NUtil::GetTimeZone(context.GetTimezoneName()));
                    }
                    result = NUtil::FormatDatetime(timestamp, context.GetDateTimeFormat());
                }
            }
        } else {
            result = NEscJ::EscapeJ<false>(propertyPtr->GetStringRobust());  // EscapeC providers characters not correctly interpreted by json reader
        }
    }

    return true;
}

TString TJsonGlobalSettingsContextFetcher::GetTypeName() {
    return "global_setting";
}

TJsonGlobalSettingsContextFetcher::TRegistrator TJsonGlobalSettingsContextFetcher::Registrator;

bool TJsonGlobalSettingsContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TMaybe<TString> settingKey = (Parameters.size() > 0) ? Parameters[0] : TMaybe<TString>();
    TMaybe<TString> settingValueMappingKey = (Parameters.size() > 1) ? Parameters[1] : TMaybe<TString>();
    TMaybe<TString> defaultValue = (Parameters.size() > 2) ? Parameters[2] : TMaybe<TString>();

    if (!settingKey) {
        if (context.DoFillNotMatchedPlaceholders()) {
            result = "";
            return true;
        }
        return false;  // NB. Empty setting key should not be treated as an error or fail reason should be detailed
    }

    if (settingValueMappingKey) {
        NJson::TJsonValue settingValueMapping = context.GetServer().GetSettings().GetJsonValue(*settingKey);
        if (!settingValueMapping.IsMap() || !settingValueMapping.Has(*settingValueMappingKey)) {
            if (defaultValue) {
                result = *defaultValue;
                return true;
            }
            errors.AddMessage(__LOCATION__, "Cannot treat " + (*settingKey) + " as a mapping as required in placeholder " + OriginalPlaceholderName);
            return false;
        }

        result = settingValueMapping[*settingValueMappingKey].GetStringRobust();
        return true;
    }

    return context.GetServer().GetSettings().GetValueStr(*settingKey, result);
}

TString TJsonLocalizationContextFetcher::GetTypeName() {
    return "localization";
}

TJsonLocalizationContextFetcher::TRegistrator TJsonLocalizationContextFetcher::Registrator;

bool TJsonLocalizationContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TMaybe<TString> localizationKey = (Parameters.size() > 0) ? Strip(Parameters[0]) : TMaybe<TString>();
    TMaybe<TString> localizationKeyPrefix = (Parameters.size() > 1) ? Strip(Parameters[1]) : TMaybe<TString>();
    TMaybe<TString> locale = (Parameters.size() > 2) ? Parameters[2] : TMaybe<TString>();
    TMaybe<TString> defaultValue = (Parameters.size() > 3) ? Parameters[3] : TMaybe<TString>();

    if (!localizationKey) {
        if (context.DoFillNotMatchedPlaceholders()) {
            result = "";
            return true;
        }
        return false;  // NB. Empty localization key should not be treated as an error or fail reason should be detailed
    }

    ELocalization resourceLocale = ELocalization::Rus;
    if (locale && !TryFromString(*locale, resourceLocale)) {
        errors.AddMessage(__LOCATION__, "Invalid locale name " + (*locale) + " in placeholder " + OriginalPlaceholderName);
        return false;
    }

    TVector<TString> keys;
    NJson::TJsonValue localizationKeyJson;
    if (NJson::ReadJsonTree(UnescapeC(*localizationKey), &localizationKeyJson) && NJson::TryFromJson(localizationKeyJson, keys)) {
        TVector<TString> localizedKeys;
        for (auto&& key: keys) {
            localizedKeys.emplace_back(GetLocalization(context.GetServer(), key, resourceLocale, localizationKeyPrefix, defaultValue));
        }
        auto serializedLocalizedKeys = NJson::ToJson(localizedKeys);
        result = NEscJ::EscapeJ<false>(serializedLocalizedKeys.GetStringRobust());
    } else {
        result = GetLocalization(context.GetServer(), *localizationKey, resourceLocale, localizationKeyPrefix, defaultValue);
    }

    return true;
}

TString TJsonLocalizationContextFetcher::GetLocalization(const NDrive::IServer& server, const TString& key, const ELocalization resourceLocale, const TMaybe<TString>& prefix, const TMaybe<TString>& defaultValue) const {
    TString resourceId = (prefix) ? JoinSeq(".", {*prefix, key}) : key;
    TString resourceDefaultValue = (defaultValue) ? *defaultValue : key;
    return server.GetLocalization()->GetLocalString(resourceLocale, resourceId, resourceDefaultValue);
}

// car info fetchers

const TString IJsonCarContextFetcher::TypeNamePrefix = "car.";

bool IJsonCarContextFetcher::FetchCar(const TContextType& context, TDriveCarInfo& carData, TMessagesCollector& /* errors */) const {
    auto carId = NJson::TryFromJson<TString>(GetCombinedContextField(context, "car_id"));
    if (carId) {
        auto gCars = context.GetServer().GetDriveAPI()->GetCarsData()->FetchInfo(*carId, TInstant::Zero());
        auto carDataPtr = gCars.GetResultPtr(*carId);
        if (carDataPtr) {
            carData = std::move(*carDataPtr);
            return true;
        }
    }
    return false;  // NB. Empty car id should not be treated as an error or fail reason should be detailed
}

TString TJsonCarModelContextFetcher::GetTypeName() {
    return TypeNamePrefix + "model";
}

TJsonCarModelContextFetcher::TRegistrator TJsonCarModelContextFetcher::Registrator;

bool TJsonCarModelContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TDriveCarInfo carData;
    if (!FetchCar(context, carData, errors)) {
        if (context.DoFillNotMatchedPlaceholders()) {
            result = "";
            return true;
        }
        return false;
    }

    TString modelCode = carData.GetModel();

    auto modelFetchResult = context.GetServer().GetDriveAPI()->GetModelsData()->FetchInfo(modelCode, TInstant::Zero());
    auto modelPtr = modelFetchResult.GetResultPtr(modelCode);
    if (!modelPtr) {
        result = modelCode;
    } else {
        result = modelPtr->GetName();
    }

    return true;
}

TString TJsonCarNumberContextFetcher::GetTypeName() {
    return TypeNamePrefix + "number";
}

TJsonCarNumberContextFetcher::TRegistrator TJsonCarNumberContextFetcher::Registrator;

bool TJsonCarNumberContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TDriveCarInfo carData;
    if (!FetchCar(context, carData, errors)) {
        if (context.DoFillNotMatchedPlaceholders()) {
            result = "";
            return true;
        }
        return false;
    }
    result = carData.GetNumber();
    return true;
}

TString TJsonCarVinContextFetcher::GetTypeName() {
    return TypeNamePrefix + "vin";
}

TJsonCarVinContextFetcher::TRegistrator TJsonCarVinContextFetcher::Registrator;

bool TJsonCarVinContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TDriveCarInfo carData;
    if (!FetchCar(context, carData, errors)) {
        if (context.DoFillNotMatchedPlaceholders()) {
            result = "";
            return true;
        }
        return false;
    }
    result = carData.GetVin();
    return true;
}

TString TJsonCarStsContextFetcher::GetTypeName() {
    return TypeNamePrefix + "sts";
}

TJsonCarStsContextFetcher::TRegistrator TJsonCarStsContextFetcher::Registrator;

bool TJsonCarStsContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TDriveCarInfo carData;
    if (!FetchCar(context, carData, errors)) {
        if (context.DoFillNotMatchedPlaceholders()) {
            result = "";
            return true;
        }
        return false;
    }
    result = ::ToString(carData.GetRegistrationID());
    return true;
}

TString TJsonCarUrlContextFetcher::GetTypeName() {
    return TypeNamePrefix + "url";
}

TJsonCarUrlContextFetcher::TRegistrator TJsonCarUrlContextFetcher::Registrator;

bool TJsonCarUrlContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TDriveCarInfo carData;
    if (FetchCar(context, carData, errors)) {
        result = TCarsharingUrl().CarInfo(carData.GetId());
    } else {
        result = "";
    }
    return true;
}

TString TJsonCarInsurerContextFetcher::GetTypeName() {
    return TypeNamePrefix + "insurer";
}

TJsonCarInsurerContextFetcher::TRegistrator TJsonCarInsurerContextFetcher::Registrator;

bool TJsonCarInsurerContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TDriveCarInfo carData;
    if (FetchCar(context, carData, errors)) {
        TCarInsurancePolicy policy;
        const auto& gAttachments = context.GetServer().GetDriveAPI()->GetCarAttachmentAssignments();
        if (!gAttachments.GetEffectiveInsurancePolicy(carData.GetId(), ModelingNow(), policy)) {
            ERROR_LOG << "Cannot obtain insurance policy: car id - " << carData.GetId() << Endl;
            return false;
        }

        result = ::ToString(policy.GetProvider());
    } else {
        result = "";
    }
    return true;
}

// user info fetchers

const TString IJsonUserContextFetcher::TypeNamePrefix = "user.";

bool IJsonUserContextFetcher::FetchUser(const TContextType& context, TDriveUserData& userData, TMessagesCollector& /* errors */) const {
    if (!context.GetCachedUserData()) {
        auto userId = NJson::TryFromJson<TString>(GetCombinedContextField(context, "user_id"));
        if (userId) {
            auto userDataPtr = context.GetServer().GetDriveAPI()->GetUsersData()->GetCachedObject(*userId);
            if (!userDataPtr) {
                return false;
            }
            context.SetCachedUserData(MakeHolder<TDriveUserData>(std::move(*userDataPtr)));
        }
    }
    userData = *context.GetCachedUserData();
    return true;  // NB. Empty user id should not be treated as an error or fail reason should be detailed
}

TString TJsonFullUserNameContextFetcher::GetTypeName() {
    return TypeNamePrefix + "full_name";
}

TJsonFullUserNameContextFetcher::TRegistrator TJsonFullUserNameContextFetcher::Registrator;

bool TJsonFullUserNameContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TDriveUserData userData;
    if (!FetchUser(context, userData, errors)) {
        if (context.DoFillNotMatchedPlaceholders()) {
            result = "";
            return true;
        }
        return false;
    }
    result = userData.GetFullName();
    return true;
}

TString TJsonUserLoginContextFetcher::GetTypeName() {
    return TypeNamePrefix + "login";
}

TJsonUserLoginContextFetcher::TRegistrator TJsonUserLoginContextFetcher::Registrator;

bool TJsonUserLoginContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TDriveUserData userData;
    if (!FetchUser(context, userData, errors)) {
        if (context.DoFillNotMatchedPlaceholders()) {
            result = "";
            return true;
        }
        return false;
    }
    result = userData.GetLogin();
    return true;
}

TString TJsonDisplayUserNameContextFetcher::GetTypeName() {
    return TypeNamePrefix + "display_name";
}

TJsonDisplayUserNameContextFetcher::TRegistrator TJsonDisplayUserNameContextFetcher::Registrator;

bool TJsonDisplayUserNameContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TDriveUserData userData;
    if (!FetchUser(context, userData, errors)) {
        if (context.DoFillNotMatchedPlaceholders()) {
            result = "";
            return true;
        }
        return false;
    }
    result = userData.GetDisplayName();
    return true;
}

TString TJsonUserPhoneContextFetcher::GetTypeName() {
    return TypeNamePrefix + "phone";
}

TJsonUserPhoneContextFetcher::TRegistrator TJsonUserPhoneContextFetcher::Registrator;

bool TJsonUserPhoneContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TDriveUserData userData;
    if (!FetchUser(context, userData, errors)) {
        if (context.DoFillNotMatchedPlaceholders()) {
            result = "";
            return true;
        }
        return false;
    }
    result = userData.GetPhone();
    return true;
}

TString TJsonUserEmailContextFetcher::GetTypeName() {
    return TypeNamePrefix + "email";
}

TJsonUserEmailContextFetcher::TRegistrator TJsonUserEmailContextFetcher::Registrator;

bool TJsonUserEmailContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TDriveUserData userData;
    if (!FetchUser(context, userData, errors)) {
        if (context.DoFillNotMatchedPlaceholders()) {
            result = "";
            return true;
        }
        return false;
    }
    result = userData.GetEmail();
    return true;
}

TString TJsonUserUrlContextFetcher::GetTypeName() {
    return TypeNamePrefix + "url";
}

TJsonUserUrlContextFetcher::TRegistrator TJsonUserUrlContextFetcher::Registrator;

bool TJsonUserUrlContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TDriveUserData userData;
    if (FetchUser(context, userData, errors)) {
        result = TCarsharingUrl().ClientInfo(userData.GetUserId());
    } else {
        result = "";
    }
    return true;
}

// session info fetchers

TString TJsonSessionUrlContextFetcher::GetTypeName() {
    return "session.url";
}

TJsonSessionUrlContextFetcher::TRegistrator TJsonSessionUrlContextFetcher::Registrator;

bool TJsonSessionUrlContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
    auto sessionId = NJson::TryFromJson<TString>(GetCombinedContextField(context, "session_id"));
    if (sessionId && *sessionId) {
        result = TCarsharingUrl().SessionPage(*sessionId);
    } else {
        result = "";
    }
    return true;
}

TString TJsonSessionEndInstantContextFetcher::GetTypeName() {
    return "session.end_instant";
}

TJsonSessionEndInstantContextFetcher::TRegistrator TJsonSessionEndInstantContextFetcher::Registrator;

bool TJsonSessionEndInstantContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    auto sessionId = NJson::TryFromJson<TString>(GetCombinedContextField(context, "session_id"));
    if (!sessionId || !*sessionId) {
        if (context.DoFillNotMatchedPlaceholders()) {
            result = "";
            return true;
        }
        return false;  // NB. Empty session id should not be treated as an error or fail reason should be detailed
    }

    THistoryRidesContext sessionsContext(context.GetServer());

    auto tx = context.GetServer().GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
    auto ydbTx = context.GetServer().GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("session_context_fetcher", &context.GetServer());
    if (!sessionsContext.InitializeSession(*sessionId, tx, ydbTx)) {
        errors.AddMessage(__LOCATION__, "Cannot initialize history rides context in placeholder " + OriginalPlaceholderName);
        return false;
    }

    auto sessionsIterator = sessionsContext.GetIterator();
    bool matchSession = false;

    THistoryRideObject sessionInfo;
    while (sessionsIterator.GetAndNext(sessionInfo)) {
        matchSession = true;
        if (!sessionInfo.IsActive()) {
            auto timestamp = sessionInfo.GetLastTS();
            if (Parameters && Parameters.front().EndsWith("_local")) {
                timestamp = NUtil::ConvertTimeZone(timestamp, NUtil::GetUTCTimeZone(), NUtil::GetTimeZone(context.GetTimezoneName()));
            }
            result = NUtil::FormatDatetime(timestamp, context.GetDateTimeFormat());
            return true;
        }
    }

    if (matchSession) {
        result = context.GetActiveOnlySessionEndInstantResult();
        return true;
    }

    errors.AddMessage(__LOCATION__, "No sessions with id " + *sessionId + " found in placeholder " + OriginalPlaceholderName);
    return false;
}

// startrek integration fetchers

const TString TJsonStartrekTicketBaseContextFetcher::TypeNamePrefix = "startrek.";

TMaybe<TStartrekTicket> TJsonStartrekTicketBaseContextFetcher::FetchTicket(const TContextType& context, TMessagesCollector& /* errors */) const {
    TMaybe<TString> transitId = (Parameters.size() > 0) ? Parameters[0] : TMaybe<TString>();
    if (!transitId) {
        return {};
    }

    auto notifyResultProvider = context.GetNotifyResultProvider();
    if (notifyResultProvider) {
        auto notifyResultPtr = notifyResultProvider->GetHandlingResult(*transitId);
        auto startrekNotifyResultPtr = std::dynamic_pointer_cast<TStartrekNotifierResult>(notifyResultPtr);
        if (startrekNotifyResultPtr) {
            return startrekNotifyResultPtr->GetTicket();
        }
    }

    return {};
}

TString TJsonStartrekTicketKeyContextFetcher::GetTypeName() {
    return TypeNamePrefix + "related_ticket_key";
}

TJsonStartrekTicketKeyContextFetcher::TRegistrator TJsonStartrekTicketKeyContextFetcher::Registrator;

bool TJsonStartrekTicketKeyContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    auto ticket = FetchTicket(context, errors);
    if (ticket) {
        result = ticket->GetKey();
        return !!result;
    }
    if (auto valuePtr = context.GetDynamicContext().FindPtr(OriginalPlaceholderName)) {
        result = *valuePtr;
        return !!result;
    }
    if (context.DoFillNotMatchedPlaceholders()) {
        result = "";
        return true;
    }
    return false;  // NB. Not fetched ticket key should not be treated as an error or fail reason should be detailed
}

TString TJsonStartrekTicketUrlContextFetcher::GetTypeName() {
    return TypeNamePrefix + "related_ticket_url";
}

TJsonStartrekTicketUrlContextFetcher::TRegistrator TJsonStartrekTicketUrlContextFetcher::Registrator;

bool TJsonStartrekTicketUrlContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    auto ticket = FetchTicket(context, errors);
    if (ticket) {
        result = context.GetServer().GetDriveAPI()->GetStartrekClient().GetTicketUri(*ticket);
        return true;
    }
    if (auto valuePtr = context.GetDynamicContext().FindPtr(OriginalPlaceholderName)) {
        result = *valuePtr;
        return !!result;
    }
    if (context.DoFillNotMatchedPlaceholders()) {
        result = "";
        return true;
    }
    return false;  // NB. Not fetched ticket url should not be treated as an error or fail reason should be detailed
}

TString TJsonPerformerLoginContextFetcher::GetTypeName() {
    return "performer.login";
}

TJsonPerformerLoginContextFetcher::TRegistrator TJsonPerformerLoginContextFetcher::Registrator;
TJsonPerformerLoginContextFetcher::TRegistrator JsonPerformerLoginContextFetcher("performer.username");  // alias

bool TJsonPerformerLoginContextFetcher::Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const {
    TJsonUserLoginContextFetcher userLoginFetcher(OriginalPlaceholderName, Parameters);
    NJson::TJsonValue performerContextEntry = NJson::TMapBuilder("user_id", GetCombinedContextField(context, "performer_id"));  // NB. Context is passed and hold by reference
    TContextType userContext(context.GetServer(), performerContextEntry, context.GetDynamicContext(), context.GetNotifyResultProvider());
    return userLoginFetcher.Fetch(userContext, result, errors);
}
