#pragma once

#include "base.h"

#include <drive/backend/abstract/localization.h>
#include <drive/backend/users/user.h>

#include <drive/library/cpp/context_fetcher/fetcher.h>

#define FACTORY_JSON_CONTEXT_FETCHER(ClassName, FetcherTypeName)         \
public:                                                                  \
    using TRegistrator = TFactory::TRegistrator<ClassName>;              \
    static auto GetTypeName() {                                          \
        return FetcherTypeName;                                          \
    }                                                                    \
    virtual TString GetType() const override {                           \
        return FetcherTypeName;                                          \
    }                                                                    \
private:                                                                 \
    static TRegistrator Registrator;

class TDriveCarInfo;

class TStartrekTicket;

class TJsonFetchContext: public TFetchContext<NJson::TJsonValue, TEntryAtomicSharedPtrHolder<NJson::TJsonValue>> {
    using TBase = TFetchContext<NJson::TJsonValue, TEntryAtomicSharedPtrHolder<NJson::TJsonValue>>;

public:
    using TBase::TBase;

    TString GetDateTimeFormat() const;
    TString GetTimezoneName() const;
    bool DoFillNotMatchedPlaceholders() const;
    TString GetActiveOnlySessionEndInstantResult() const;

    void SetCachedUserData(THolder<TDriveUserData>&& userData) const;
    const TDriveUserData* GetCachedUserData() const;

private:
    TString GetFullSettingKey(TStringBuf key) const;

private:
    static const TString DefaultDateTimeFormat;
    static const TString DefaultTimezoneName;
    static const bool DefaultFillNotMatchedPlaceholders;
    static const TString DefaultActiveOnlySessionEndInstantResult;

private:
    R_OPTIONAL(TString, DateTimeFormat);
    R_OPTIONAL(TString, TimezoneName);
    R_OPTIONAL(bool, FillNotMatchedPlaceholders);
    R_OPTIONAL(TString, ActiveOnlySessionEndInstantResult);

private:
    mutable TAtomicSharedPtr<TDriveUserData> CachedUserData;
};

class IJsonContextFetcher : public IContextFetcher<IJsonContextFetcher, TJsonFetchContext> {
private:
    using TBase = IContextFetcher<IJsonContextFetcher, TJsonFetchContext>;

public:
    using TBase::TBase;
};

class TJsonDefaultContextFetcher: public IJsonContextFetcher {
    using TBase = IJsonContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonDefaultContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

class TJsonGlobalSettingsContextFetcher: public IJsonContextFetcher {
    using TBase = IJsonContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonGlobalSettingsContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

class TJsonLocalizationContextFetcher: public IJsonContextFetcher {
    using TBase = IJsonContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonLocalizationContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;
    TString GetLocalization(const NDrive::IServer& server, const TString& key, const ELocalization resourceLocale, const TMaybe<TString>& prefix = {}, const TMaybe<TString>& defaultValue = {}) const;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

// car info fetchers

class IJsonCarContextFetcher: public IJsonContextFetcher {
    using TBase = IJsonContextFetcher;

protected:
    using TBase::TBase;

    bool FetchCar(const TContextType& context, TDriveCarInfo& carData, TMessagesCollector& errors) const;

    static const TString TypeNamePrefix;
};

class TJsonCarModelContextFetcher: public IJsonCarContextFetcher {
    using TBase = IJsonCarContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonCarModelContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

class TJsonCarNumberContextFetcher: public IJsonCarContextFetcher {
    using TBase = IJsonCarContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonCarNumberContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

class TJsonCarVinContextFetcher: public IJsonCarContextFetcher {
    using TBase = IJsonCarContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonCarVinContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

class TJsonCarStsContextFetcher: public IJsonCarContextFetcher {
    using TBase = IJsonCarContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonCarStsContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

class TJsonCarUrlContextFetcher: public IJsonCarContextFetcher {
    using TBase = IJsonCarContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonCarUrlContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

class TJsonCarInsurerContextFetcher: public IJsonCarContextFetcher {
    using TBase = IJsonCarContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonCarInsurerContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

// user info fetchers

class IJsonUserContextFetcher: public IJsonContextFetcher {
    using TBase = IJsonContextFetcher;

protected:
    using TBase::TBase;

    bool FetchUser(const TContextType& context, TDriveUserData& userData, TMessagesCollector& errors) const;

    static const TString TypeNamePrefix;
};

class TJsonUserLoginContextFetcher: public IJsonUserContextFetcher {
    using TBase = IJsonUserContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonUserLoginContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

class TJsonFullUserNameContextFetcher: public IJsonUserContextFetcher {
    using TBase = IJsonUserContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonFullUserNameContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

class TJsonDisplayUserNameContextFetcher: public IJsonUserContextFetcher {
    using TBase = IJsonUserContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonDisplayUserNameContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

class TJsonUserPhoneContextFetcher: public IJsonUserContextFetcher {
    using TBase = IJsonUserContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonUserPhoneContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

class TJsonUserEmailContextFetcher: public IJsonUserContextFetcher {
    using TBase = IJsonUserContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonUserEmailContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

class TJsonUserUrlContextFetcher: public IJsonUserContextFetcher {
    using TBase = IJsonUserContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonUserUrlContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

// session info fetchers

class TJsonSessionUrlContextFetcher: public IJsonContextFetcher {
    using TBase = IJsonContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonSessionUrlContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

class TJsonSessionEndInstantContextFetcher: public IJsonContextFetcher {
    using TBase = IJsonContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonSessionEndInstantContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

// startrek integration fetchers

class TJsonStartrekTicketBaseContextFetcher: public IJsonContextFetcher {
    using TBase = IJsonContextFetcher;

protected:
    using TBase::TBase;

    TMaybe<TStartrekTicket> FetchTicket(const TContextType& context, TMessagesCollector& errors) const;

    static const TString TypeNamePrefix;
};

class TJsonStartrekTicketKeyContextFetcher: public TJsonStartrekTicketBaseContextFetcher {
    using TBase = TJsonStartrekTicketBaseContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonStartrekTicketKeyContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

class TJsonStartrekTicketUrlContextFetcher: public TJsonStartrekTicketBaseContextFetcher {
    using TBase = TJsonStartrekTicketBaseContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonStartrekTicketUrlContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};

// performer fetcher

class TJsonPerformerLoginContextFetcher: public IJsonUserContextFetcher {
    using TBase = IJsonUserContextFetcher;

public:
    using TRegistrator = TFactory::TRegistrator<TJsonPerformerLoginContextFetcher>;

    using TBase::TBase;

    virtual bool Fetch(const TContextType& context, TString& result, TMessagesCollector& errors) const override;

    static TString GetTypeName();

private:
    static TRegistrator Registrator;
};
