#include "client.h"

namespace {

    NSimpleMeta::TConfig GetDefaultConfig() {
        NSimpleMeta::TConfig config = NSimpleMeta::TConfig::ForRequester();
        config.SetConnectTimeout(TDuration::MilliSeconds(150));
        config.SetGlobalTimeout(TDuration::Seconds(15));
        return config;
    }

}

namespace NDrive {

    TCarsharingBackendClient::TCarsharingBackendClient(const TString& endpoint, IAuth::TPtr auth/* = nullptr*/)
        : Client(MakeHolder<NNeh::THttpClient>(endpoint, GetDefaultConfig()))
        , Auth(auth)
    {
    }

    NThreading::TFuture<TAreaInfosReport> TCarsharingBackendClient::GetAreaInfos(TDuration timeout/* = TDuration::Seconds(15)*/) const {
        NNeh::THttpRequest req;
        req.SetUri("/api/staff/areas/info");
        return DoRequest(req, timeout).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
            TAreaInfosReport report;
            Y_ENSURE(NJson::TryFromJson(r.GetValue(), report));
            return report;
        });
    }

    NThreading::TFuture<NSignalq::TApiSignalqNotifySupportResponse> TCarsharingBackendClient::NotifySupport(const NSignalq::TApiSignalqNotifySupportRequest& requestParams, TDuration timeout/* = TDuration::Seconds(10)*/) const {
        NNeh::THttpRequest req;
        req.SetUri("/api/signalq/notify_support");
        return DoRequest(req, timeout, NJson::ToJson(requestParams)).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
            NSignalq::TApiSignalqNotifySupportResponse response;
            Y_ENSURE(NJson::TryFromJson(r.GetValue(), response));
            return NThreading::MakeFuture(std::move(response));
        });
    }

    NThreading::TFuture<NJson::TJsonValue> TCarsharingBackendClient::DoRequest(NNeh::THttpRequest request, TDuration timeout, const TMaybe<NJson::TJsonValue>& requestBody/* = Nothing*/) const {
        request.AddHeader("UserPermissionsCache", "true");
        if (requestBody) {
            request.SetPostData(*requestBody);
        }
        NDrive::UpdateRequest(request, Auth);
        auto reply = Yensured(Client)->SendAsync(request, Now() + timeout);
        return reply.Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) {
            const NUtil::THttpReply& reply = r.GetValue();
            reply.EnsureSuccessfulReply();
            return NJson::ReadJsonFastTree(reply.Content());
        });
    }

}
