#include "account_tags.h"
#include <drive/backend/database/drive_api.h>
#include <drive/backend/roles/manager.h>
#include <rtline/library/json/field.h>

const TString TAccountTemporaryActionTag::TypeName = "account_temp_action";
ITag::TFactory::TRegistrator<TAccountTemporaryActionTag> TAccountTemporaryActionTag::Registrator(TAccountTemporaryActionTag::TypeName);
TAccountTemporaryActionTag::TDescription::TFactory::TRegistrator<TAccountTemporaryActionTag::TDescription> TAccountTemporaryActionTag::DescriptionRegistrator(TAccountTemporaryActionTag::TypeName);

const TString TSimpleAccountTag::TypeName = "simple_account_tag";
ITag::TFactory::TRegistrator<TSimpleAccountTag> TSimpleAccountTag::Registrator(TSimpleAccountTag::TypeName);

bool TAccountTemporaryActionTag::IsActive(bool isPotentiallyActive) const {
    return Active && TBase::IsActive(isPotentiallyActive);
}

NDrive::TScheme TAccountTemporaryActionTag::GetScheme(const NDrive::IServer* server) const {
    auto scheme = TBase::GetScheme(server);
    scheme.Add<TFSBoolean>("is_active", "активность").SetDefault(true);
    scheme.Add<TFSNumeric>("required_sum", "Сумма для активации").SetVisual(TFSNumeric::EVisualType::Money);
    return scheme;
}

void TAccountTemporaryActionTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::InsertField(json, "is_active", Active);
    NJson::InsertNonNull(json, "required_sum", RequiredSum);
}

bool TAccountTemporaryActionTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (errors) {
        return NJson::ParseField(json, "is_active", Active, false, *errors)
            && NJson::ParseField(json, "required_sum", RequiredSum, false, *errors)
            && TBase::DoSpecialDataFromJson(json, errors);
    } else {
        return NJson::ParseField(json, "is_active", Active, false)
            && NJson::ParseField(json, "required_sum", RequiredSum, false)
            && TBase::DoSpecialDataFromJson(json, errors);
    }
}

TAccountTemporaryActionTag::TProto TAccountTemporaryActionTag::DoSerializeSpecialDataToProto() const {
    auto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetActivity(Active);
    if (RequiredSum) {
        proto.SetRequiredSum(*RequiredSum);
    }
    return proto;
}

bool TAccountTemporaryActionTag::DoDeserializeSpecialDataFromProto(const TAccountTemporaryActionTag::TProto& proto) {
    if (proto.HasActivity()) {
        Active = proto.GetActivity();
    }
    if (proto.HasRequiredSum()) {
        RequiredSum = proto.GetRequiredSum();
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

const TString TAdditionalRolesAccountTag::TypeName = "additional_roles";
ITag::TFactory::TRegistrator<TAdditionalRolesAccountTag> TAdditionalRolesAccountTag::Registrator(TAdditionalRolesAccountTag::TypeName);
TAdditionalRolesAccountTag::TDescription::TFactory::TRegistrator<TAdditionalRolesAccountTag::TDescription> TAdditionalRolesAccountTag::TDescription::Registrator(TAdditionalRolesAccountTag::TypeName);


NDrive::TScheme TExpiredDebtAccountTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    auto scheme = TBase::GetScheme(server);
    scheme.Add<TFSNumeric>("warning_since_month_day", "День месяца начала отображения предупреждения о задолженности").SetDefault(WarningSinceMonthDay);
    scheme.Add<TFSNumeric>("deactivate_since_month_day", "День месяца начала деактивации аккаунта").SetDefault(DeactivateSinceMonthDay);
    scheme.Add<TFSNumeric>("gmt", "Настройка часового пояса").SetDefault(Gmt);
    scheme.Add<TFSBoolean>("deactivate_account", "Деактивировать аккаунты?").SetDefault(false);
    return scheme;
}

NJson::TJsonValue TExpiredDebtAccountTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeMetaToJson();
    NJson::FieldsToJson(result, GetFields());
    return result;
}

bool TExpiredDebtAccountTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    return TBase::DoDeserializeMetaFromJson(value)
        && NJson::TryFieldsFromJson(value, GetFields());
}

bool TExpiredDebtAccountTag::CalculateValues(const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (!HasFirstExpiredDT() || !HasDebtAmount()) {
        return true;
    }

    auto description = GetDescriptionAs<TDescription>(*server, session);
    if (!description) {
        session.SetErrorInfo("TExpiredDebtAccountTag::CalculateValues", TStringBuilder() << "cannot find description for tag: " << TypeName);
        return false;
    }

    auto now = TInstant::Now();
    if (now < FirstExpiredDT) {
        session.SetErrorInfo("TExpiredDebtAccountTag::CalculateValues", TStringBuilder() << "current time less than expired time : " << now << "<" << FirstExpiredDT << " " << session.GetStringReport());
        return false;
    }

    auto dayChecker = [](ui64 value, ui64 def) -> ui64 {
        if (value > 0 && value < 29) {
            return value;
        }

        WARNING_LOG << "incorrect input day: " << value << " value interval [1, 28] default will be used " << def << Endl;
        return def;
    };

    //find approximate payment date
    auto approximateDate = *FirstExpiredDT + TDuration::Days(TDescription::DefaultWarningSinceMonthDay);

    std::tm approximateWarningTm;
    approximateDate.GmTime(&approximateWarningTm);
    approximateWarningTm.tm_sec = 0;
    approximateWarningTm.tm_hour = 0;

    ui64 warningMonthDay = dayChecker(description->GetWarningSinceMonthDay(), TDescription::DefaultWarningSinceMonthDay);
    ui64 deactivateMonthDay = dayChecker(description->GetDeactivateSinceMonthDay(), TDescription::DefaultDeactivateSinceMonthDay);

    if (warningMonthDay > deactivateMonthDay) {
        TStringBuilder s;
        s << "incorrect input warningMonthDay > deactivateMonthDay " << warningMonthDay << " > " << deactivateMonthDay;
        warningMonthDay = TDescription::DefaultWarningSinceMonthDay;
        deactivateMonthDay = TDescription::DefaultDeactivateSinceMonthDay;
        s << " default will be used " << warningMonthDay << " < " << deactivateMonthDay;
        WARNING_LOG << s << Endl;
    }

    //create warning date since start of the month
    approximateWarningTm.tm_mday = warningMonthDay;
    WarningDate = TInstant::Seconds(TimeGM(&approximateWarningTm));
    WarningDate = *WarningDate - TDuration::Hours(description->GetGmt());

    //create deactivate date since start of the month
    approximateWarningTm.tm_mday = deactivateMonthDay;
    DeactivateDate = TInstant::Seconds(TimeGM(&approximateWarningTm));
    DeactivateDate = *DeactivateDate - TDuration::Hours(description->GetGmt());

    NeedWarning = now > WarningDate;
    NeedDeactivate = now > DeactivateDate;

    return true;
}

bool TExpiredDebtAccountTag::MustBeDeactivated(const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto description = GetDescriptionAs<TDescription>(*server, session);
    if (!description) {
        session.SetErrorInfo("TExpiredDebtAccountTag::CalculateValues", TStringBuilder() << "deactivate status undefined, cannot find description for tag: " << TypeName << " " << session.GetStringReport());
        return false;
    }

    return NeedDeactivate && description->GetDeactivateAccount();
}

TString TExpiredDebtAccountTag::SerializeToLog() const {
    TStringBuilder s;
    s << "NeedWarning: " << NeedWarning << " ";
    s << "NeedDeactivate: " << NeedDeactivate << " ";
    if (FirstExpiredDT) {
        s << "FirstExpiredDT: " << FirstExpiredDT << " ";

        if (WarningDate) {
             s << "Warning date: " << WarningDate << " ";
        }

        if (DeactivateDate) {
            s << "Deactivating date: " << FirstExpiredDT << " ";
        }
    }

    if (DebtAmount) {
        s << "DebtAmount: " << DebtAmount << " ";
    }

    return s;
}

bool TExpiredDebtAccountTag::IsEqual(const TExpiredDebtAccountTag& r) const {
    return GetFields() == r.GetFields();
}

NJson::TJsonValue TExpiredDebtAccountTag::GetReport() const {
    NJson::TJsonValue report;
    if (!HasDebtAmount()) {
        return report;
    }

    report["debt_amount"] = GetDebtAmountRef();
    report["debt_status"] = GetNeedDeactivate() ? ::ToString(EDebtStatus::Deactivate)
        : (GetNeedWarning() ? ::ToString(EDebtStatus::Warning) : ::ToString(EDebtStatus::None));

    return report;
}

bool TExpiredDebtAccountTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    return TBase::DoSpecialDataFromJson(value, errors) && NJson::TryFieldsFromJson(value, GetFields());
}

void TExpiredDebtAccountTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    NJson::FieldsToJson(value, GetFields());
}

NDrive::NProto::TExpiredDebtAccountTag TExpiredDebtAccountTag::DoSerializeSpecialDataToProto() const {
    auto result = TBase::DoSerializeSpecialDataToProto();

    result.SetNeedWarning(NeedWarning);
    result.SetNeedDeactivate(NeedDeactivate);

    if (HasFirstExpiredDT()) {
        result.SetFirstExpiredDT(FirstExpiredDT->MicroSeconds());
    }

    if (HasDebtAmount()) {
        result.SetDebtAmount(*DebtAmount);
    }

    return result;
}

bool TExpiredDebtAccountTag::DoDeserializeSpecialDataFromProto(const NDrive::NProto::TExpiredDebtAccountTag& proto) {
    SetNeedWarning(proto.GetNeedWarning());
    SetNeedDeactivate(proto.GetNeedDeactivate());

    if (proto.HasDebtAmount()) {
        SetDebtAmount(proto.GetDebtAmount());
    }

    if (proto.HasFirstExpiredDT()) {
        SetFirstExpiredDT(TInstant::MicroSeconds(proto.GetFirstExpiredDT()));
    }

    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

const TString TExpiredDebtAccountTag::TypeName = "expired_debt_account_tag";
ITag::TFactory::TRegistrator<TExpiredDebtAccountTag> TExpiredDebtAccountTag::Registrator(TExpiredDebtAccountTag::TypeName);
TExpiredDebtAccountTag::TDescription::TFactory::TRegistrator<TExpiredDebtAccountTag::TDescription> TExpiredDebtAccountTag::TDescription::Registrator(TExpiredDebtAccountTag::TypeName);
