#include "achievement.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>

const TString TAchievementUserTag::TypeName = "achievement_user_tag";

NDrive::TScheme TAchievementUserTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSString>("image_url", "URL картинки в достижении");
    result.Add<TFSString>("no_level_image_url", "URL картинки в достижении (без уровня)");
    result.Add<TFSString>("animation_url", "URL анимации в достижении");
    result.Add<TFSString>("no_level_animation_url", "URL анимации в достижении (без уровня)");
    result.Add<TFSString>("sharing_video_url", "URL видео в достижении");
    result.Add<TFSNumeric>("max_level_value", "Максимальное значение в достижении");
    result.Add<TFSString>("text_color", "Цвет текста");
    result.Add<TFSArray>("gradient_color", "Градиент заднего фона").SetElement<TFSString>();
    result.Add<TFSString>("progress_bg_color", "Цвет фона полоски прогресса");
    result.Add<TFSString>("progress_value_color", "Цвет значения полоски прогресса");
    result.Add<TFSString>("no_level_text_color", "Цвет текста (без уровня)");
    result.Add<TFSArray>("no_level_gradient_color", "Градиент заднего фона (без уровня)").SetElement<TFSString>();
    result.Add<TFSString>("no_level_progress_bg_color", "Цвет фона полоски прогресса (без уровня)");
    result.Add<TFSString>("no_level_progress_value_color", "Цвет значения полоски прогресса (без уровня)");
    result.Add<TFSBoolean>("enabled", "Показывать ли пользователю. Можно переопределить через achievement_user_tag.TAG.enabled");
    result.Add<TFSBoolean>("empty_enabled", "Показывать ли пользователю без тега. Можно переопределить через achievement_user_tag.TAG.empty_enabled");
    result.Add<TFSNumeric>("priority", "Приоритет достижения. Чем больше, тем ачивка выше");
    return result;
}

NJson::TJsonValue TAchievementUserTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue json = TBase::DoSerializeMetaToJson();
    NJson::InsertField(json, "image_url", ImageUrl);
    NJson::InsertField(json, "no_level_image_url", NoLevelImageUrl);
    NJson::InsertField(json, "animation_url", AnimationUrl);
    NJson::InsertField(json, "no_level_animation_url", NoLevelAnimationUrl);
    NJson::InsertField(json, "sharing_video_url", SharingVideoUrl);
    NJson::InsertField(json, "max_level_value", MaxLevelValue);
    NJson::InsertField(json, "text_color", TextColor);
    NJson::InsertField(json, "gradient_color", GradientColor);
    NJson::InsertField(json, "progress_bg_color", ProgressBgColor);
    NJson::InsertField(json, "progress_value_color", ProgressValueColor);
    NJson::InsertField(json, "no_level_text_color", NoLevelTextColor);
    NJson::InsertField(json, "no_level_gradient_color", NoLevelGradientColor);
    NJson::InsertField(json, "no_level_progress_bg_color", NoLevelProgressBgColor);
    NJson::InsertField(json, "no_level_progress_value_color", NoLevelProgressValueColor);
    NJson::InsertField(json, "enabled", Enabled);
    NJson::InsertField(json, "empty_enabled", EmptyEnabled);
    NJson::InsertField(json, "priority", Priority);
    return json;
}

bool TAchievementUserTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& json) {
    return NJson::ParseField(json, "image_url", ImageUrl, false) &&
        NJson::ParseField(json, "no_level_image_url", NoLevelImageUrl, false) &&
        NJson::ParseField(json, "animation_url", AnimationUrl, false) &&
        NJson::ParseField(json, "sharing_video_url", SharingVideoUrl, false) &&
        NJson::ParseField(json, "no_level_animation_url", NoLevelAnimationUrl, false) &&
        NJson::ParseField(json, "max_level_value", MaxLevelValue, false) &&
        NJson::ParseField(json, "text_color", TextColor, false) &&
        NJson::ParseField(json, "gradient_color", GradientColor, false) &&
        NJson::ParseField(json, "progress_bg_color", ProgressBgColor, false) &&
        NJson::ParseField(json, "progress_value_color", ProgressValueColor, false) &&
        NJson::ParseField(json, "no_level_text_color", NoLevelTextColor, false) &&
        NJson::ParseField(json, "no_level_gradient_color", NoLevelGradientColor, false) &&
        NJson::ParseField(json, "no_level_progress_bg_color", NoLevelProgressBgColor, false) &&
        NJson::ParseField(json, "no_level_progress_value_color", NoLevelProgressValueColor, false) &&
        NJson::ParseField(json, "enabled", Enabled, false) &&
        NJson::ParseField(json, "empty_enabled", EmptyEnabled, false) &&
        NJson::ParseField(json, "priority", Priority, false) &&
        TBase::DoDeserializeMetaFromJson(json);
}

TSet<NEntityTagsManager::EEntityType> TAchievementUserTag::GetObjectType() const {
    return { NEntityTagsManager::EEntityType::User };
}

EUniquePolicy TAchievementUserTag::GetUniquePolicy() const {
    return EUniquePolicy::Rewrite;
}

NDrive::TScheme TAchievementUserTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("level", "Уровень достижения");
    result.Add<TFSNumeric>("level_value", "Текущее значение в уровне достижения");
    result.Add<TFSNumeric>("level_timestamp", "Время получения последнего уровня достижения")
        .SetVisual(NDrive::TFSNumeric::EVisualType::DateTime);
    return result;
}

void TAchievementUserTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    if (Level) {
        json["level"] = *Level;
    }
    if (LevelValue) {
        json["level_value"] = *LevelValue;
    }
    if (LevelTimestamp) {
        json["level_timestamp"] = LevelTimestamp->Seconds();
    }
}

bool TAchievementUserTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    return NJson::ParseField(json, "level", Level, false, errors) &&
        NJson::ParseField(json, "level_value", LevelValue, false, errors) &&
        NJson::ParseField(json, "level_timestamp", LevelTimestamp, false, errors) &&
        TBase::DoSpecialDataFromJson(json, errors);
}

TAchievementUserTag::TProto TAchievementUserTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    if (Level) {
        proto.SetLevel(*Level);
    }
    if (LevelValue) {
        proto.SetLevelValue(*LevelValue);
    }
    if (LevelTimestamp) {
        proto.SetLevelTimestamp(LevelTimestamp->Seconds());
    }
    return proto;
}

bool TAchievementUserTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (proto.HasLevel()) {
        Level = proto.GetLevel();
    }
    if (proto.HasLevelValue()) {
        LevelValue = proto.GetLevelValue();
    }
    if (proto.HasLevelTimestamp()) {
        LevelTimestamp = TInstant::Seconds(proto.GetLevelTimestamp());
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

namespace {
    void UpdateAchivementTemplate(TString& text, TMaybe<i32> level, TMaybe<double> levelValue, TAtomicSharedPtr<const TAchievementUserTag::TDescription> description) {
        const int intLevelValue = levelValue.GetOrElse(0);
        const int maxLevelValue = description->GetMaxLevelValue();
        SubstGlobal(text, "_Level_", ToString(level.GetOrElse(0)));
        SubstGlobal(text, "_LevelValue_", ToString(intLevelValue));
        SubstGlobal(text, "_MaxLevelValue_", ToString(maxLevelValue));
        SubstGlobal(text, "_RemainedLevelValue_", ToString(std::max(maxLevelValue - intLevelValue, 0)));
    }
}

bool TAchievementUserTag::IsEnabled(const NDrive::IServer& server, TUserPermissions::TConstPtr permissions) const {
    auto description = Yensured(GetDescriptionAs<TAchievementUserTag::TDescription>(server));
    return Yensured(permissions)->GetSetting<bool>("achievement_user_tag." + GetName() + ".enabled").GetOrElse(description->GetEnabled());
}

bool TAchievementUserTag::IsEmptyEnabled(const NDrive::IServer& server, TUserPermissions::TConstPtr permissions) const {
    if (!IsEnabled(server, permissions)) {
        return false;
    }
    auto description = Yensured(GetDescriptionAs<TAchievementUserTag::TDescription>(server));
    return Yensured(permissions)->GetSetting<bool>("achievement_user_tag." + GetName() + ".empty_enabled").GetOrElse(description->GetEmptyEnabled());
}

i32 TAchievementUserTag::GetPriority(const NDrive::IServer& server, TUserPermissions::TConstPtr /*permissions*/) const {
    auto description = Yensured(GetDescriptionAs<TAchievementUserTag::TDescription>(server));
    return description->GetPriority();
}

NJson::TJsonValue TAchievementUserTag::GetPublicReport(const NDrive::IServer& server, ELocalization localizationId) const {
    const auto& localization = *Yensured(server.GetLocalization());
    auto description = Yensured(GetDescriptionAs<TAchievementUserTag::TDescription>(server));
    NJson::TJsonValue result = NJson::JSON_MAP;
    NJson::InsertField(result, "name", GetName());
    auto getLocalString = [
        this, &localization, localizationId, description,
        name = "achievement_user_tag." + GetName()
    ](const TString& key) -> TString {
        TString levelKey = ToString(Level.GetOrElse(0)) + "." + key;
        TString value = localization.GetLocalString(localizationId, name + "." + key, "");
        value = localization.GetLocalString(localizationId, name + "." + levelKey, value);
        UpdateAchivementTemplate(value, Level, LevelValue, description);
        return value;
    };
    NJson::InsertField(result, "title", getLocalString("title"));
    NJson::InsertField(result, "full_subtitle", getLocalString("full_subtitle"));
    {
        TString image = Level ? description->GetImageUrl() : description->GetNoLevelImageUrl();
        UpdateAchivementTemplate(image, Level, LevelValue, description);
        NJson::InsertField(result, "image", image);
    }
    {
        TString animation = Level ? description->GetAnimationUrl() : description->GetNoLevelAnimationUrl();
        UpdateAchivementTemplate(animation, Level, LevelValue, description);
        NJson::InsertField(result, "animation", animation);
    }
    {
        TString video = description->GetSharingVideoUrl();
        UpdateAchivementTemplate(video, Level, LevelValue, description);
        NJson::InsertField(result, "sharing_video", video);
    }
    if (LevelValue && description->GetMaxLevelValue() > 0) {
        NJson::InsertField(result, "sub_title", getLocalString("sub_title"));
        NJson::InsertField(result, "progress_title", getLocalString(Level ? "progress_title" : "no_level_progress_title"));
        NJson::InsertField(result, "progress", std::min(std::max(*LevelValue / description->GetMaxLevelValue(), 0.0), 1.0));
    } else {
        NJson::InsertField(result, "sub_title", getLocalString("no_progress_sub_title"));
    }
    if (Level) {
        NJson::InsertField(result, "level", *Level);
    }
    if (LevelTimestamp) {
        NJson::InsertField(result, "last_level_ts", LevelTimestamp->Seconds());
    }
    if (auto&& color = Level ? description->GetTextColor() : description->GetNoLevelTextColor()) {
        NJson::InsertField(result, "text_color", color);
    }
    if (auto&& color = Level ? description->GetGradientColor() : description->GetNoLevelGradientColor()) {
        NJson::InsertField(result, "gradient_color", color);
    }
    if (auto&& color = Level ? description->GetProgressBgColor() : description->GetNoLevelProgressBgColor()) {
        NJson::InsertField(result, "progress_bg_color", color);
    }
    if (auto&& color = Level ? description->GetProgressValueColor() : description->GetNoLevelProgressValueColor()) {
        NJson::InsertField(result, "progress_value_color", color);
    }
    return result;
}

void TAchievementUserTag::Sort(TVector<TAchievementUserTag>& tags, const NDrive::IServer& server, TUserPermissions::TConstPtr permissions) {
    auto cmp = [&server, permissions](const TAchievementUserTag& lhs, const TAchievementUserTag& rhs) {
        return lhs.GetPriority(server, permissions) > rhs.GetPriority(server, permissions);
    };
    std::sort(tags.begin(), tags.end(), cmp);
}

ITag::TFactory::TRegistrator<TAchievementUserTag> TAchievementUserTag::Registrator(TAchievementUserTag::TypeName);
TTagDescription::TFactory::TRegistrator<TAchievementUserTag::TDescription> TAchievementUserTag::TDescription::Registrator(TAchievementUserTag::TypeName);
