#include "additional_service.h"

#include "chargable.h"
#include "delivery.h"
#include "support_tags.h"
#include "user_tags.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/data/container_tag.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/logging/evlog.h>
#include <drive/backend/offers/actions/additional_service.h>
#include <drive/backend/offers/offers/additional_service.h>
#include <drive/backend/proto/offer.pb.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/sessions/manager/billing.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/json/field.h>
#include <rtline/library/json/proto/adapter.h>
#include <rtline/util/algorithm/type_traits.h>

#include <util/generic/ptr.h>

const TString TAdditionalServiceOfferHolderTag::Planned = "additional_service_planned";
const TString TAdditionalServiceOfferHolderTag::Starting = "additional_service_starting";
const TString TAdditionalServiceOfferHolderTag::Started = "additional_service_started";

NDrive::TScheme TAdditionalServiceOfferHolderTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    if (server) {
        result.Add<TFSBoolean>("check_deposit", "Check deposit on evolution").SetDefault(CheckDeposit);
        result.Add<TFSBoolean>("start_billing_task", "Start billing").SetDefault(StartBillingTask);
    }
    return result;
}

NJson::TJsonValue TAdditionalServiceOfferHolderTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeMetaToJson();
    NJson::FieldsToJson(result, GetFields());
    return result;
}

bool TAdditionalServiceOfferHolderTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    return
        TBase::DoDeserializeMetaFromJson(value) &&
        NJson::TryFieldsFromJson(value, GetFields());
}

TOptionalDBTags TAdditionalServiceOfferHolderTag::RestoreOfferHolderTags(const TString& behaviourConstructorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
    const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
    auto registeredTagNames = tagsMeta.GetRegisteredTagNames({ Type() });
    auto tagNames = MakeVector(registeredTagNames);
    return TBase::RestoreOfferHolderTags(behaviourConstructorId, tagNames, server, tx);
}

bool TAdditionalServiceOfferHolderTag::OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    Y_ENSURE(server);
    if (!TBase::OnAfterRemove(self, userId, server, tx)) {
        return false;
    }
    auto tag = self.GetTagAs<TAdditionalServiceOfferHolderTag>();
    if (!tag) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnAfterRemove", "cannot cast tag to TAdditionalServiceOfferHolderTag");
        return false;
    }
    if (!tag->HasParentSessionId()) {
        return true;
    }
    const auto& database = server->GetDriveDatabase();
    const auto& deviceTagManager = database.GetTagsManager().GetDeviceTags();
    auto optionalSession = database.GetSessionManager().GetSession(tag->GetParentSessionIdRef(), tx);
    if (!optionalSession) {
        return false;
    }
    auto session = *optionalSession;
    if (!session) {
        // Ignore case when session is already dropped.
        return true;
    }
    auto&& tagName = tag->GetName();
    if (tagName == TAdditionalServiceOfferHolderTag::Starting) {
        auto object = deviceTagManager.RestoreObject(session->GetObjectId(), tx);
        if (!object) {
            tx.AddErrorMessage("TAdditionalServiceOfferHolderTag::OnAfterRemove", TStringBuilder() << "cannot restore object: " << session->GetObjectId());
            return false;
        }
        for (auto&& tag : object->GetTags()) {
            auto servicingTag = tag.GetTagAs<TServicingTag>();
            if (servicingTag && servicingTag->GetParentUserTagId() == self.GetTagId()) {
                if (!deviceTagManager.RemoveTag(tag, userId, server, tx)) {
                    return false;
                }
            }
        }
    }
    return true;
}

bool TAdditionalServiceOfferHolderTag::ProvideDataOnEvolve(const TDBTag& from, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
    if (!TBase::ProvideDataOnEvolve(from, permissions, server, tx)) {
        return false;
    }
    auto fromTag = from.GetTagAs<TAdditionalServiceOfferHolderTag>();
    if (!fromTag) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::ProvideDataOnEvolve", "cannot cast source tag to TAdditionalServiceOfferHolderTag");
        return false;
    }
    if (!HasParentSessionId()) {
        SetParentSessionId(fromTag->OptionalParentSessionId());
    }
    if (!GetOffer()) {
        SetOffer(fromTag->GetOffer());
    }
    return true;
}

bool TAdditionalServiceOfferHolderTag::OnAfterEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx, const TEvolutionContext* eContext) const {
    if (!TBase::OnAfterEvolve(from, to, permissions, server, tx, eContext)) {
        return false;
    }
    auto toTag = std::dynamic_pointer_cast<TAdditionalServiceOfferHolderTag>(to);
    if (!toTag) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnAfterEvolve", "cannot cast target tag to TAdditionalServiceOfferHolderTag");
        return false;
    }
    if (!toTag->HasParentSessionId()) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnAfterEvolve", TStringBuilder() << "no parent session id " << from.GetTagId());
        return false;
    }
    const auto& database = Yensured(server)->GetDriveDatabase();
    auto optionalSession = database.GetSessionManager().GetSession(toTag->GetParentSessionIdRef(), tx);
    if (!optionalSession) {
        return false;
    }
    auto session = *optionalSession;
    if (!session) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnAfterEvolve", "cannot find session " + toTag->GetParentSessionIdRef());
        return false;
    }
    const auto& tagsManager = database.GetTagsManager();
    const auto& deviceTagsManager = tagsManager.GetDeviceTags();
    const auto& tagsMeta = tagsManager.GetTagsMeta();
    auto&& toTagName = toTag->GetName();
    if (toTagName == TAdditionalServiceOfferHolderTag::Starting) {
        auto servicingTag = tagsMeta.CreateTagAs<TServicingTag>(TServicingTag::Type());
        if (!servicingTag) {
            tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnAfterEvolve", TStringBuilder() << "cannot create tag " << TServicingTag::Type());
            return false;
        }
        servicingTag->SetSessionId(toTag->GetParentSessionIdRef());
        servicingTag->SetParentUserTagId(from.GetTagId());
        auto optionalAddedTags = deviceTagsManager.AddTag(servicingTag, permissions.GetUserId(), session->GetObjectId(), server, tx);
        if (!optionalAddedTags) {
            return false;
        }
        if (optionalAddedTags->size() != 1) {
            tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnAfterEvolve", TStringBuilder() << "added " << optionalAddedTags->size() << " tags");
            return false;
        }
    } else if (toTagName == TAdditionalServiceOfferHolderTag::Started) {
        // TODO(iudovin@): Add some implementation.
    } else {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnAfterEvolve", TStringBuilder() << "unsupported tag name " << toTagName);
        return false;
    }
    return true;
}

bool TAdditionalServiceOfferHolderTag::OnBeforeUpdate(const TDBTag& from, ITag::TPtr to, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    if (!TBase::OnBeforeUpdate(from, to, userId, server, tx)) {
        return false;
    }
    const auto& database = Yensured(server)->GetDriveDatabase();
    const auto& roleManager = database.GetUserPermissionManager();
    const auto& deviceTagManager = database.GetTagsManager().GetDeviceTags();
    auto target = std::dynamic_pointer_cast<TAdditionalServiceOfferHolderTag>(to);
    if (!target) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnBeforeUpdate", "cannot cast target tag to AdditionalServiceOfferHolderTag");
        return false;
    }
    if (!target->GetOffer()) {
        target->SetOffer(GetOffer());
    }
    target->SetParentSessionId(OptionalParentSessionId());
    auto&& tagName = target->GetName();
    auto offer = target->GetOffer();
    auto serviceOffer = std::dynamic_pointer_cast<TAdditionalServiceOffer>(offer);
    bool locationUpdated = false;
    if (serviceOffer) {
        auto offerBuilderAction = roleManager.GetAction(offer->GetBehaviourConstructorId());
        if (!offerBuilderAction) {
            tx.SetErrorInfo("TAdditionalServiceOfferHolderTag::OnBeforeUpdate", "cannot GetAction " + offer->GetBehaviourConstructorId());
            return false;
        }
        auto serviceOfferBuilder = offerBuilderAction->GetAs<TAdditionalServiceOfferBuilder>();
        if (!serviceOfferBuilder) {
            tx.SetErrorInfo("TAdditionalServiceOfferHolderTag::OnBeforeUpdate", "cannot cast to TAdditionalServiceOfferBuilder " + offer->GetBehaviourConstructorId());
            return false;
        }
        if (tagName == Planned && target->HasScheduledAt()) {
            auto schedule = serviceOfferBuilder->GetAvailableSchedule(nullptr);
            if (!schedule) {
                tx.SetErrorInfo("TAdditionalServiceOfferHolderTag::OnBeforeUpdate", "cannot GetAvailableSchedule " + offer->GetBehaviourConstructorId());
                return false;
            }
            if (!schedule->Contains(target->GetScheduledAtRef())) {
                tx.SetErrorInfo("TAdditionalServiceOfferHolderTag::OnBeforeUpdate", "non valid date", NDrive::MakeError("available_time_constraint_violated"));
                tx.SetCode(HTTP_BAD_REQUEST);
                return false;
            }
            serviceOffer->SetScheduledAt(target->GetScheduledAtRef());
        }
        if (tagName != Started && target->HasSameLocation()) {
            serviceOffer->SetSameLocation(target->GetSameLocationRef());
            if (target->GetSameLocationRef()) {
                serviceOffer->SetDeliveryLocationName(TString());
                serviceOffer->SetDeliveryLocation(Nothing());
                serviceOffer->SetDeliveryArea(Nothing());
            }
        }
        if (target->HasDeliveryArea()) {
            serviceOffer->SetDeliveryArea(target->GetDeliveryAreaRef());
        }
        if (target->HasDeliveryLocation()) {
            const auto& deliveryLocation = target->GetDeliveryLocationRef();
            if (!serviceOfferBuilder->CheckDeliveryLocation(deliveryLocation, *server)) {
                tx.SetCode(HTTP_BAD_REQUEST);
                tx.SetErrorInfo(
                    "TAdditionalServiceOfferHolderTag::OnBeforeUpdate",
                    TStringBuilder() << "cannot deliver to " << deliveryLocation,
                    NDrive::MakeError("bad_delivery_location")
                );
                return false;
            }
            serviceOffer->SetSameLocation(false);
            serviceOffer->SetDeliveryLocation(target->GetDeliveryLocationRef());
            locationUpdated = true;
        }
        if (target->HasDeliveryLocationName()) {
            serviceOffer->SetDeliveryLocationName(target->GetDeliveryLocationNameRef());
        }
    }
    if (locationUpdated) {
        auto billingSession = serviceOffer->GetBillingSession(*server);
        auto deviceTags = serviceOffer->GetSessionDeviceTags(tx, billingSession, *server);
        if (!deviceTags) {
            return false;
        }
        for (auto&& tag : *deviceTags) {
            auto deliveryTag = tag.MutableTagAs<TCarDeliveryTag>();
            if (deliveryTag && deliveryTag->GetPerformer()) {
                tx.SetErrorInfo(
                    "TAdditionalServiceOfferHolderTag::OnBeforeUpdate",
                    "delivery tag " + tag.GetTagId() + " is already performed",
                    NDrive::MakeError("delivery_in_progress")
                );
                tx.SetCode(HTTP_BAD_REQUEST);
                return false;
            }
            if (deliveryTag) {
                deliveryTag->SetLatitude(target->GetDeliveryLocationRef().Y);
                deliveryTag->SetLongitude(target->GetDeliveryLocationRef().X);
            }
            if (!deviceTagManager.UpdateTagData(tag, userId, tx)) {
                return false;
            }
        }
    }
    tx.Committed().Subscribe([offer](const NThreading::TFuture<void>& c) {
        if (c.HasValue() && offer) {
            auto serializedOffer = offer->SerializeToProto();
            NDrive::TEventLog::Log("OfferUpdated", NJson::ToJson(NJson::Proto(serializedOffer)));
        }
    });

    return true;
}

bool TAdditionalServiceOfferHolderTag::StartBillingTask(const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto api = server ? server->GetDriveAPI() : nullptr;
    if (!api || !api->HasBillingManager()) {
        session.SetErrorInfo("AdditionalServiceOfferHolderTag::StartBillingTask", "BillingManager is missing");
        return false;
    }
    const auto& billingManager = api->GetBillingManager();
    auto offer = GetOffer();
    if (!offer) {
        session.SetErrorInfo("AdditionalServiceOfferHolderTag::StartBillingTask", "null offer");
        return false;
    }
    auto optionalBillingTask = billingManager.GetActiveTasksManager().GetTask(offer->GetOfferId(), session);
    if (!optionalBillingTask) {
        return false;
    }
    auto billingTask = std::move(*optionalBillingTask);
    if (!billingTask && !billingManager.CreateBillingTask(offer->GetUserId(), offer, session)) {
        return false;
    }
    return true;
}

TDBTag TAdditionalServiceOfferHolderTag::Book(ICommonOffer::TPtr offer, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx, const TBookOptions& bookOptions) {
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "Book")
            ("offer", NJson::ToJson(offer))
            ("user_id", permissions.GetUserId())
            ("options", NJson::ToJson(NJson::Pass(bookOptions)))
        );
    }
    auto serviceOffer = std::dynamic_pointer_cast<TAdditionalServiceOffer>(offer);
    if (!serviceOffer) {
        tx.SetErrorInfo("AdditionalServiceOffer::Book", TStringBuilder() << "cannot cast offer to TAdditionalServiceOffer");
        return {};
    }
    bool checkBlocked = bookOptions.CheckBlocked;
    if (checkBlocked && !TUserProblemTag::EnsureNotBlocked(permissions.GetUserId(), server, tx)) {
        return {};
    }
    if (serviceOffer->GetUserId() != permissions.GetUserId()) {
        tx.SetErrorInfo("AdditionalServiceOffer::Book", "offer belongs to another user", EDriveSessionResult::InconsistencyOffer);
        return {};
    }

    tx.Committed().Subscribe([serviceOffer](const NThreading::TFuture<void>& c) {
        if (c.HasValue()) {
            SendGlobalMessage<TOfferBookingCompleted>(serviceOffer);
        }
    });

    const auto& database = server.GetDriveDatabase();
    const auto& tagsManager = database.GetTagsManager();
    const auto& userTagsManager = tagsManager.GetUserTags();
    const auto& tagsMeta = tagsManager.GetTagsMeta();

    const TString& userId = permissions.GetUserId();

    auto offerHolderTag = tagsMeta.CreateTagAs<TAdditionalServiceOfferHolderTag>(TAdditionalServiceOfferHolderTag::Planned);
    if (!offerHolderTag) {
        tx.SetErrorInfo("AdditionalServiceOffer::Book", TStringBuilder() << "cannot create tag " << TAdditionalServiceOfferHolderTag::Planned);
        return {};
    }
    offerHolderTag->SetOffer(serviceOffer);
    if (serviceOffer->GetSessionId()) {
        offerHolderTag->SetParentSessionId(serviceOffer->GetSessionId());
    }
    auto optionalAddedTags = userTagsManager.AddTag(offerHolderTag, userId, userId, &server, tx);
    if (!optionalAddedTags) {
        return {};
    }
    if (optionalAddedTags->size() != 1) {
        tx.SetErrorInfo("AdditionalServiceOffer::Book", TStringBuilder() << "added " << optionalAddedTags->size() << " tags");
        return {};
    }

    auto result = std::move(optionalAddedTags->front());
    return result;
}

bool TAdditionalServiceOfferHolderTag::Start(
    const TDBTag& self,
    NDrive::TEntitySession& tx,
    const TAdditionalServiceOffer& offer,
    const TUserPermissions& permissions,
    const NDrive::IServer& server
) {
    auto tag = self.GetTagAs<TAdditionalServiceOfferHolderTag>();
    if (!tag) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::Start", "cannot cast to AdditionalServiceOfferHolderTag" + self.GetTagId());
        return false;
    }
    const auto& database = server.GetDriveDatabase();
    const auto& roleManager = database.GetUserPermissionManager();
    const auto& tagsManager = database.GetTagsManager();
    auto offerBuilderAction = roleManager.GetAction(offer.GetBehaviourConstructorId());
    if (!offerBuilderAction) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::Start", "cannot GetAction " + offer.GetBehaviourConstructorId());
        return false;
    }
    auto serviceOfferBuilder = offerBuilderAction->GetAs<TAdditionalServiceOfferBuilder>();
    if (!serviceOfferBuilder) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::Start", "cannot cast to AdditionalServiceOfferBuilder " + offer.GetBehaviourConstructorId());
        return false;
    }
    auto optionalSession = database.GetSessionManager().GetSession(tag->GetParentSessionIdRef(), tx);
    if (!optionalSession) {
        return false;
    }
    auto session = *optionalSession;
    if (!session) {
        // Remove tag for dropped session.
        return tagsManager.GetUserTags().RemoveTag(self, permissions.GetUserId(), &server, tx);
    }
    auto snapshot = server.GetSnapshotsManager().GetSnapshot(session->GetObjectId());
    auto location = snapshot.GetLocation();
    if (!location) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::Start", "cannot get car snapshot " + session->GetObjectId());
        return false;
    }
    if (!serviceOfferBuilder->CheckDeliveryLocation(location->GetCoord(), server)) {
        TVector<ITag::TPtr> tags;
        for (auto&& tagName : serviceOfferBuilder->GetCancelStartUserTags()) {
            auto tag = tagsManager.GetTagsMeta().CreateTag(tagName);
            if (!tag) {
                tx.SetErrorInfo("AdditionalServiceOfferHolderTag::Start", "cannot create tag " + tagName);
                return false;
            }
            tags.push_back(std::move(tag));
        }
        if (!tagsManager.GetUserTags().AddTags(tags, permissions.GetUserId(), offer.GetUserId(), &server, tx)) {
            return false;
        }
        return tagsManager.GetUserTags().RemoveTag(self, permissions.GetUserId(), &server, tx);
    }
    auto newTag = MakeHolder<TAdditionalServiceOfferHolderTag>(TAdditionalServiceOfferHolderTag::Starting);
    NDrive::ITag::TEvolutionContext context;
    return !!tagsManager.GetUserTags().EvolveTag(self, std::move(newTag), permissions, &server, tx, &context);
}

bool TAdditionalServiceOfferHolderTag::OnStarted(
    const TDBTag& self,
    NDrive::TEntitySession& tx,
    TChargableTag& servicingTag,
    const TUserPermissions& permissions,
    const NDrive::IServer& server
) {
    auto tag = self.GetTagAs<TAdditionalServiceOfferHolderTag>();
    if (!tag) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnStarted", "cannot cast to AdditionalServiceOfferHolderTag " + self.GetTagId());
        return false;
    }
    auto offer = tag->GetOffer();
    if (!offer) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnStarted", "null offer");
        return false;
    }
    auto serviceOffer = std::dynamic_pointer_cast<TAdditionalServiceOffer>(offer);
    if (!serviceOffer) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnStarted", "cannot cast offer to AdditionalServiceOffer " + offer->GetOfferId());
        return false;
    }
    if (!tag->HasParentSessionId()) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnStarted", "tag does not have parent session " + self.GetTagId());
        return false;
    }
    const auto& database = server.GetDriveDatabase();
    const auto& roleManager = database.GetUserPermissionManager();
    const auto& tagsManager = database.GetTagsManager();
    auto optionalSession = database.GetSessionManager().GetSession(tag->GetParentSessionIdRef(), tx);
    if (!optionalSession) {
        return false;
    }
    auto session = *optionalSession;
    if (!session) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnStarted", "cannot find session " + tag->GetParentSessionIdRef());
        return false;
    }
    auto offerBuilderAction = roleManager.GetAction(offer->GetBehaviourConstructorId());
    if (!offerBuilderAction) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnStarted", "cannot GetAction " + offer->GetBehaviourConstructorId());
        return false;
    }
    auto serviceOfferBuilder = offerBuilderAction->GetAs<TAdditionalServiceOfferBuilder>();
    if (!serviceOfferBuilder) {
        tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnStarted", "cannot cast to AdditionalServiceOfferBuilder " + offer->GetBehaviourConstructorId());
        return false;
    }
    auto newTag = MakeHolder<TAdditionalServiceOfferHolderTag>(TAdditionalServiceOfferHolderTag::Started);
    if (!serviceOffer->HasDeliveryLocation()) {
        const auto& snapshotManager = server.GetSnapshotsManager();
        auto locationOptions = snapshotManager.GetLocationOptions();
        auto snapshot = snapshotManager.GetSnapshot(session->GetObjectId());
        auto location = snapshot.GetLocation(TDuration::Max(), locationOptions);
        if (!location) {
            tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnStarted", "cannot get car location " + session->GetObjectId());
            return false;
        }
        serviceOffer->SetDeliveryLocation(location->GetCoord());
        if (auto geocoded = snapshot.GetGeocoded(TDuration::Max(), locationOptions)) {
            serviceOffer->SetDeliveryLocationName(geocoded->Content);
        }
        // Rewrite tag offer.
        newTag->SetOffer(serviceOffer);
    }
    TVector<ITag::TPtr> subtags;
    for (auto&& tagName : serviceOfferBuilder->GetDeviceTags()) {
        auto tag = tagsManager.GetTagsMeta().CreateTag(tagName);
        if (!tag) {
            tx.SetErrorInfo("AdditionalServiceOfferHolderTag::OnStarted", "cannot create tag " + tagName);
            return false;
        }
        if (auto deliveryTag = std::dynamic_pointer_cast<TCarDeliveryTag>(tag)) {
            Y_ENSURE(serviceOffer->HasDeliveryLocation());
            deliveryTag->SetLatitude(serviceOffer->GetDeliveryLocationRef().Y);
            deliveryTag->SetLongitude(serviceOffer->GetDeliveryLocationRef().X);
        }
        subtags.push_back(std::move(tag));
    }
    servicingTag.SetSubtags(std::move(subtags));
    NDrive::ITag::TEvolutionContext context;
    return !!tagsManager.GetUserTags().EvolveTag(self, std::move(newTag), permissions, &server, tx, &context);
}

NDrive::NProto::TOfferHolderTag TAdditionalServiceOfferHolderTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TOfferHolderTag result = TBase::DoSerializeSpecialDataToProto();
    NDrive::NProto::TAdditionalServiceInfo additionalServiceInfo;
    if (ParentSessionId) {
        additionalServiceInfo.SetParentSessionId(*ParentSessionId);
    }
    *result.MutableAdditionalServiceInfo() = additionalServiceInfo;
    return result;
}

bool TAdditionalServiceOfferHolderTag::DoDeserializeSpecialDataFromProto(const NDrive::NProto::TOfferHolderTag& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }
    if (proto.HasAdditionalServiceInfo()) {
        const auto& additionalServiceInfo = proto.GetAdditionalServiceInfo();
        if (additionalServiceInfo.HasParentSessionId()) {
            ParentSessionId = additionalServiceInfo.GetParentSessionId();
        }
    }
    return true;
}

bool TAdditionalServiceOfferHolderTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(value, errors)) {
        return false;
    }
    return NJson::TryFieldsFromJson(value, GetFields(), errors);
}

ITag::TFactory::TRegistrator<TAdditionalServiceOfferHolderTag> TAdditionalServiceOfferHolderTag::Registrator(TAdditionalServiceOfferHolderTag::Type());

TTagDescription::TFactory::TRegistrator<TAdditionalServiceOfferHolderTag::TDescription> AdditionalServiceOfferHolderTagDescriptionRegistrator(TAdditionalServiceOfferHolderTag::Type());

template <>
NJson::TJsonValue NJson::ToJson(const TAdditionalServiceOfferHolderTag::TBookOptions& value) {
    NJson::TJsonValue result;
    result["check_blocked"] = value.CheckBlocked;
    return result;
}
