#include "alerts_info.h"

#include "tags.h"

#include <drive/backend/saas/api.h>

#include <rtline/api/search_client/client.h>
#include <rtline/api/search_client/query.h>
#include <rtline/library/graph/geometry_graph/common/types.h>
#include <rtline/util/types/messages_collector.h>

#include <util/generic/adaptor.h>

class TIncorrectMovingAlert: public IAlertInfo {
private:
    TDuration DurationForDetection;
public:
    TIncorrectMovingAlert(const TDuration duration)
        : DurationForDetection(duration)
    {

    }
    TString GetSpecialId() const override {
        return "incorrect_moving_in_current_status";
    }

    TString GetShortName() const override {
        return "Incorrect status for riding";
    }

    NJson::TJsonValue GetInfo() const override {
        NJson::TJsonValue result;
        result["minutes"] = DurationForDetection.Minutes();
        return "Incorrect status for riding";
    }

    TString GetHRReport() const override {
        return "Некорректное состояние для движения машины за последние " + ToString(DurationForDetection.Minutes()) + " минут";
    }

    IAlertTag::TPtr ConstructTag() const override {
        return new TTagIncorrectMoving(DurationForDetection);
    }
};

class TViolationAlert: public IAlertInfo {
private:
    double VDelta;
    double Length;
    TDuration Duration;
    TString UserId;
    TString DeviceId;
    TString SessionId;
public:
    TViolationAlert(const double vDelta, const double length, const TDuration d, const TString& userId, const TString& deviceId, const TString& sessionId)
        : VDelta(vDelta)
        , Length(length)
        , Duration(d)
        , UserId(userId)
        , DeviceId(deviceId)
        , SessionId(sessionId)
    {

    }

    TString GetSpecialId() const override {
        return "violation";
    }

    TString GetShortName() const override {
        return "Incorrect velocity for riding";
    }

    NJson::TJsonValue GetInfo() const override {
        NJson::TJsonValue result;
        result["v_delta"] = VDelta;
        result["length"] = Length;
        result["duration"] = (Duration == TDuration::Max()) ? "Inf" : ToString(Duration.Minutes());
        return result;
    }

    TString GetHRReport() const override {
        return "Превышение скорости более чем на " + ToString(VDelta) + "км/ч на участке " + ToString(Length) + "м за последние " + ToString(Duration.Minutes()) + " минут";
    }

    IAlertTag::TPtr ConstructTag() const override {
        return new TTagIncorrectVelocity(TSpeedWithUnit::KmH(VDelta).MS(), Length, Duration, UserId, DeviceId, SessionId);
    }
};

TAlertInfo::TAlertInfo(const TTraceInfo& trace, const TAlertsConfig& config)
    : TTraceInfo(trace)
{
    if (CarHaveFixPlace()) {
        for (auto&& sigma : Sigmas) {
            if (sigma.second > config.GetDistanceMovingCheckerForStoppedCars()) {
                INFO_LOG << "Moved car: " << sigma.second << " / " << GetTraceId() << Endl;
                AlertMessages.emplace_back(new TIncorrectMovingAlert(TDuration::Minutes(sigma.first)));
                break;
            }
        }
    } else {
        for (auto&& v : Reversed(Violations5)) {
            if (v.second > config.GetCriticalLengthForViolationDetect()) {
                INFO_LOG << "Violation: " << trace.GetUserId() << " on " << trace.GetDeviceId() << Endl;
                AlertMessages.emplace_back(new TViolationAlert(v.first, v.second, TDuration::Minutes(5), trace.GetUserId(), trace.GetDeviceId(), trace.GetSessionId()));
                break;
            }
        }
    }
}
