#pragma once
#include "config.h"
#include "tags.h"
#include "traces.h"

#include <drive/backend/database/drive_api.h>

#include <library/cpp/json/writer/json_value.h>

#include <rtline/protos/proto_helper.h>

#include <util/generic/ptr.h>

class TMessagesCollector;

class IAlertInfo {
public:

    using TPtr = TAtomicSharedPtr<IAlertInfo>;

    virtual ~IAlertInfo() = default;


    virtual TString GetSpecialId() const = 0;
    virtual TString GetShortName() const = 0;
    virtual IAlertTag::TPtr ConstructTag() const = 0;

    virtual TString GetDescription() const {
        return GetShortName();
    }

    virtual NJson::TJsonValue GetInfo() const {
        return NJson::TJsonValue(NJson::JSON_NULL);
    }

    virtual TString GetHRReport() const = 0;

    virtual NJson::TJsonValue GetReport() const {
        NJson::TJsonValue result;
        result["description"] = GetDescription();
        result["short_name"] = GetShortName();
        result["id"] = GetSpecialId();
        NJson::TJsonValue info = GetInfo();
        if (!info.IsNull()) {
            result["info"] = GetInfo();
        }
        return result;
    }
};

class TAlertInfo: public TTraceInfo {
private:
    TVector<IAlertInfo::TPtr> AlertMessages;
    R_FIELD(TString, DeviceModelCode);
public:

    TVector<IAlertTag::TPtr> ConstructTags() const {
        TVector<IAlertTag::TPtr> result;
        for (auto&& i : AlertMessages) {
            result.push_back(i->ConstructTag());
        }
        return result;
    }

    TString GetHRReport() const {
        TStringStream ss;
        ss << TTraceInfo::GetHRReport() << "\n";
        for (auto&& i : AlertMessages) {
            ss << i->GetHRReport() << "\n";
        }
        return ss.Str();
    }

    NJson::TJsonValue GetReport() const {
        NJson::TJsonValue result = TTraceInfo::GetReport();
        NJson::TJsonValue& alerts = result["alerts"];
        alerts.SetType(NJson::JSON_ARRAY);
        for (auto&& i : AlertMessages) {
            alerts.AppendValue(i->GetReport());
        }
        return result;
    }

    bool HasAlerts() const {
        return !AlertMessages.empty();
    }


    TAlertInfo(const TTraceInfo& trace, const TAlertsConfig& config);

};

class TWatchingInfo {
private:
    TVector<TAlertInfo> Alerts;
    const TTracesAccessor& Traces;
    const TAlertsConfig Config;
public:

    const TVector<TAlertInfo>& GetAlerts() const {
        return Alerts;
    }

    TVector<TAlertInfo>& MutableAlerts() {
        return Alerts;
    }

    NJson::TJsonValue GetReport() const {
        NJson::TJsonValue result;
        NJson::TJsonValue alertsJson(NJson::JSON_ARRAY);
        for (auto&& i : Alerts) {
            alertsJson.AppendValue(i.GetReport());
        }
        result["alerts"] = alertsJson;
        return result;
    }

    bool Initialize() {
        for (auto&& i : Traces.GetTraces()) {
            TAlertInfo alert(i, Config);
            if (alert.HasAlerts()) {
                Alerts.push_back(alert);
            }
        }
        return true;
    }

    TWatchingInfo(const TTracesAccessor& traces, const TAlertsConfig& config)
        : Traces(traces)
        , Config(config)
    {
    }
};

