#pragma once
#include <drive/backend/data/proto/alerts.pb.h>

#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tag.h>

#include <library/cpp/object_factory/object_factory.h>

class IAlertTag: public ITag {
public:
    using TPtr = TAtomicSharedPtr<IAlertTag>;

    using ITag::ITag;

    virtual const TVector<TString>& GetConflictingTags() const {
        return Default<TVector<TString>>();
    }

    bool HasConflictingTags() const {
        return !GetConflictingTags().empty();
    }
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& /*json*/) const override {

    }

};

class TTagLowVoltage: public IAlertTag {
private:
    static TFactory::TRegistrator<TTagLowVoltage> Registrator;
    float Voltage = -1;
protected:
    virtual TBlob DoSerializeSpecialData(NDrive::NProto::TTagHeader& /*h*/) const override {
        NDrive::NProto::TTagLowVoltage proto;
        proto.SetVoltage(Voltage);
        return TBlob::FromString(proto.SerializeAsString());
    }

    virtual bool DoDeserializeSpecialData(const NDrive::NProto::TTagHeader& /*h*/, const TBlob& data) override {
        NDrive::NProto::TTagLowVoltage proto;
        if (!proto.ParseFromArray(data.AsCharPtr(), data.Size())) {
            return false;
        }
        Voltage = proto.GetVoltage();
        return true;
    }

public:
    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }

    float GetVoltage() const {
        return Voltage;
    }

    static const TString TagName;

    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& /*json*/) const override {

    }

public:
    TTagLowVoltage()
        : IAlertTag(TagName)
    {
    }

    TTagLowVoltage(const float voltage)
        : IAlertTag(TagName)
        , Voltage(voltage)
    {
    }

    virtual TString GetHRDescription() const override;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }
};

class TTagNoSignal: public IAlertTag {
private:
    static TFactory::TRegistrator<TTagNoSignal> Registrator;
protected:
    virtual TBlob DoSerializeSpecialData(NDrive::NProto::TTagHeader& /*h*/) const override {
        NDrive::NProto::TTagNoSignal proto;
        return TBlob::FromString(proto.SerializeAsString());
    }

    virtual bool DoDeserializeSpecialData(const NDrive::NProto::TTagHeader& /*h*/, const TBlob& /*data*/) override {
        return true;
    }
public:
    static const TString TagName;

    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& /*json*/) const override {

    }

public:
    TTagNoSignal()
        : IAlertTag(TagName)
    {
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }

    virtual TString GetHRDescription() const override;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }
};

class TTraceTag: public IAlertTag {
protected:
    TString UserId;
    TString DeviceId;
    TString SessionId;

public:
    TTraceTag() = default;
    TTraceTag(const TString& userId, const TString& deviceId, const TString& sessionId, const TString& tagName)
        : IAlertTag(tagName)
        , UserId(userId)
        , DeviceId(deviceId)
        , SessionId(sessionId)
    {
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Trace };
    }

    const TString& GetSessionId() const {
        return SessionId;
    }

    const TString& GetUserId() const {
        return UserId;
    }

    const TString& GetDeviceId() const {
        return DeviceId;
    }

    template <class TProto>
    void SerializeToProto(TProto& proto) const {
        proto.SetUserId(UserId);
        proto.SetDeviceId(DeviceId);
        proto.SetSessionId(SessionId);
    }

    template <class TProto>
    void DeserializeFromProto(const TProto& proto) {
        UserId = proto.GetUserId();
        DeviceId = proto.GetDeviceId();
        SessionId = proto.GetSessionId();
    }

    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override {
        IAlertTag::SerializeSpecialDataToJson(json);
        json.InsertValue("user_id", UserId);
        json.InsertValue("car_id", DeviceId);
        json.InsertValue("session_id", SessionId);
    }
};

class TTagIncorrectRidingZone: public TTraceTag {
private:
    static TFactory::TRegistrator<TTagIncorrectRidingZone> Registrator;

protected:
    virtual TBlob DoSerializeSpecialData(NDrive::NProto::TTagHeader& /*h*/) const override {
        NDrive::NProto::TTagIncorrectRidingZone proto;
        TTraceTag::SerializeToProto(proto);
        return TBlob::FromString(proto.SerializeAsString());
    }

    virtual bool DoDeserializeSpecialData(const NDrive::NProto::TTagHeader& /*h*/, const TBlob& data) override {
        NDrive::NProto::TTagIncorrectRidingZone proto;
        if (!proto.ParseFromArray(data.AsCharPtr(), data.Size())) {
            return false;
        }
        TTraceTag::DeserializeFromProto(proto);
        return true;
    }

public:
    static const TString TagName;

public:
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& /*json*/) const override {

    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car, NEntityTagsManager::EEntityType::Trace };
    }

    TTagIncorrectRidingZone() = default;
    TTagIncorrectRidingZone(const TString& userId, const TString& deviceId, const TString& sessionId)
        : TTraceTag(userId, deviceId, sessionId, TagName)
    {
    }

    virtual TString GetHRDescription() const override;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }
};

class TTagIncorrectMoving: public IAlertTag {
private:
    static TFactory::TRegistrator<TTagIncorrectMoving> Registrator;
    TDuration Duration = TDuration::Zero();
    const TVector<TString> ConflictingTags = { "evacuation", "possible_evacuation" };

protected:
    virtual TBlob DoSerializeSpecialData(NDrive::NProto::TTagHeader& /*h*/) const override {
        NDrive::NProto::TTagIncorrectMoving proto;
        proto.SetDurationSeconds(Duration.Seconds());
        return TBlob::FromString(proto.SerializeAsString());
    }

    virtual bool DoDeserializeSpecialData(const NDrive::NProto::TTagHeader& /*h*/, const TBlob& data) override {
        NDrive::NProto::TTagIncorrectMoving proto;
        if (!proto.ParseFromArray(data.AsCharPtr(), data.Size())) {
            return false;
        }
        Duration = TDuration::Seconds(proto.GetDurationSeconds());
        return true;
    }

public:
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& /*json*/) const override {

    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car, NEntityTagsManager::EEntityType::Trace };
    }

    virtual TString GetHRDescription() const override;

    TTagIncorrectMoving() = default;
    TTagIncorrectMoving(const TDuration duration)
        : IAlertTag(TagName)
        , Duration(duration)
    {
    }

    static const TString TagName;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    virtual const TVector<TString>& GetConflictingTags() const override {
        return ConflictingTags;
    }
};

class TTagIncorrectVelocity: public TTraceTag {
private:
    double VDelta = 0;
    double Length = 0;
    TDuration Duration = TDuration::Zero();

private:
    static TFactory::TRegistrator<TTagIncorrectVelocity> Registrator;

protected:
    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Trace };
    }

    virtual TString GetHRDescription() const override;

    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;

    virtual TBlob DoSerializeSpecialData(NDrive::NProto::TTagHeader& /*h*/) const override {
        NDrive::NProto::TTagIncorrectVelocity proto;
        proto.SetVDelta(VDelta);
        proto.SetLength(Length);
        proto.SetDurationSeconds(Duration.Seconds());
        TTraceTag::SerializeToProto(proto);
        return TBlob::FromString(proto.SerializeAsString());
    }

    virtual bool DoDeserializeSpecialData(const NDrive::NProto::TTagHeader& /*h*/, const TBlob& data) override {
        NDrive::NProto::TTagIncorrectVelocity proto;
        if (!proto.ParseFromArray(data.AsCharPtr(), data.Size())) {
            return false;
        }
        VDelta = proto.GetVDelta();
        Length = proto.GetLength();
        TTraceTag::DeserializeFromProto(proto);
        Duration = TDuration::Seconds(proto.GetDurationSeconds());
        return true;
    }

public:
    TTagIncorrectVelocity() = default;
    TTagIncorrectVelocity(const double vDelta, const double length, const TDuration duration, const TString& userId, const TString& deviceId, const TString& sessionId)
        : TTraceTag(userId, deviceId, sessionId, TagName)
        , VDelta(vDelta)
        , Length(length)
        , Duration(duration)
    {
    }

    static const TString TagName;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    double GetVDelta() const {
        return VDelta;
    }

    double GetLength() const {
        return Length;
    }
};
