#include "traces.h"

#include <drive/backend/saas/api.h>

#include <drive/library/cpp/common/status.h>
#include <drive/library/cpp/tracks/proto/trace.pb.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/api/search_client/client.h>
#include <rtline/api/search_client/query.h>
#include <rtline/util/instant_model.h>
#include <rtline/util/types/messages_collector.h>

#include <util/generic/adaptor.h>

TTraceInfo::TTraceInfo(const TReadSearchProtoHelper& reader, const TVector<ui32>& velocityGaps) {
    Y_ENSURE(reader.GetProperty("s_user_id", UserId));
    Y_ENSURE(reader.GetProperty("s_device_id", DeviceId));
    Y_ENSURE(reader.GetProperty("s_trace_id", TraceId));
    if (!reader.GetProperty("s_session_id", SessionId)) {
        SessionId = "unknown";
    }

    CarStatus = NDrive::GetStatus(TraceId);

    TString violationInfoStr;
    if (reader.TryGetProperty("ViolationsInfo", violationInfoStr)) {
        Y_ENSURE(NJson::ReadJsonFastTree(violationInfoStr, &ViolationInfo));
    }

    {
        double lastSignalTimestamp;
        if (reader.TryGetProperty("__LastSignalTimestamp", lastSignalTimestamp)) {
            LastSignalInstant = TInstant::MilliSeconds(lastSignalTimestamp * 1000);
        }

        ui32 firstSignalTimestamp;
        Y_ENSURE(reader.TryGetProperty("start_time", firstSignalTimestamp));
        StartPointTime = TInstant::Seconds(firstSignalTimestamp);

        ui32 lastPointTimestamp;
        Y_ENSURE(reader.TryGetProperty("end_time", lastPointTimestamp));
        FinishPointTime = TInstant::Seconds(lastPointTimestamp);
    }

    {
        TString tagsStr;
        if (reader.GetProperty("SpecialTags", tagsStr)) {
            TVector<TString> tagsV;
            if (TryParseStringToVector(tagsStr, tagsV, ',', false)) {
                Tags.insert(tagsV.begin(), tagsV.end());
            }
        }
    }

    if (CarHaveFixPlace()) {
        double sigma;

        TVector<ui32> durations = { 5, 10, 20 };
        for (auto&& d : durations) {
            if (reader.TryGetProperty("Sigma" + ToString(d), sigma)) {
                Sigmas[d] = sigma;
            }
        }
    } else {
        for (auto&& v : velocityGaps) {
            double lengthViolation;
            if (reader.TryGetProperty("_Erf_LengthViolation_5_" + ToString(v), lengthViolation)) {
                Violations5[v] = lengthViolation;
            }
        }
    }

    TString traceData;
    if (reader.GetProperty("Trace", traceData)) {
        NRTLineProto::TAimTrace traceProto;
        Y_ENSURE(TSearchProtoHelper::UnpackObject(traceData, traceProto));
        bool atFirst = true;
        double vFlowCurrent = 0;
        double vDeviceCurrent = 0;
        double vLimitCurrent = 0;
        TGeoCoord cCurrent;
        for (auto&& i : traceProto.GetPoint()) {
            Timestamps.push_back(i.GetTimestamp());
            Y_ENSURE(!atFirst || (i.HasVFlow() && i.HasVVehicle() && i.HasVLimit() && i.HasCoord()));

            if (i.HasCoord()) {
                cCurrent.Deserialize(i.GetCoord());
            }
            if (i.HasVFlow()) {
                vFlowCurrent = i.GetVFlow();
            }
            if (i.HasVVehicle()) {
                vDeviceCurrent = i.GetVVehicle();
            }
            if (i.HasVLimit()) {
                vLimitCurrent = i.GetVLimit();
            }

            Coords.push_back(cCurrent);
            VFlow.push_back(vFlowCurrent);
            VDevice.push_back(vDeviceCurrent);
            VLimit.push_back(vLimitCurrent);
            atFirst = false;
        }
    } else {
        TString timestampCoords;
        if (reader.GetProperty("TrackCoords", traceData) && reader.GetProperty("TimestampsCoords", timestampCoords)) {
            try {
                TGeoCoord::DeserializeVector(traceData, Coords);
                StringSplitter(timestampCoords).SplitBySet(", ").SkipEmpty().ParseInto(&Timestamps);
            } catch (const std::exception& e) {
                ERROR_LOG << FormatExc(e) << Endl;
            }
        }

    }
}

bool TTraceInfo::ParseFromReply(const NRTLine::TSearchReply& reply, TVector<TTraceInfo>& traces, const TVector<ui32>& VelocityGaps, TMessagesCollector* collector) {
    if (reply.GetCode() != 200) {
        if (collector) {
            collector->AddMessage("rtline reply", "Incorrect response from rtline cluster " + ::ToString(reply.GetCode()) + ": " + reply.GetRawReport());
        }
        return false;
    }

    const NMetaProtocol::TReport& report = reply.GetReport();
    if (reply.GetReport().GetTotalDocCount(0) == 0) {
        if (collector) {
            collector->AddMessage("rtline reply", "No data in reply");
        }
        return true;
    }

    if (report.GroupingSize() != 1) {
        if (collector) {
            collector->AddMessage("watching", "Incorrect reply (grouping size != 1): " + ToString(report.GroupingSize()));
        }
        return false;
    }

    const ::NMetaProtocol::TGrouping& grouping = report.GetGrouping(0);

    if (grouping.GroupSize() == 0) {
        if (collector) {
            collector->AddMessage("watching", "Request empty");
        }
        return false;
    }

    for (auto&& group : grouping.GetGroup()) {
        if (group.DocumentSize() != 1) {
            if (collector) {
                collector->AddMessage("watching", "Incorrect watching reply structure: Documents count is incorrect in group (" + ToString(group.DocumentSize()) + ")");
            }
            return false;
        }
        const NMetaProtocol::TDocument& doc = group.GetDocument(0);
        TReadSearchProtoHelper reader(doc);
        TTraceInfo info(reader, VelocityGaps);
        traces.emplace_back(std::move(info));
    }
    return true;
}

bool TTracesAccessor::FetchTracesData(TMessagesCollector* errors) {
    TSet<TString> requestsCars;
    TSet<TString> requestsUsers;
    for (auto&& i : Traces) {
        requestsCars.emplace(i.GetDeviceId());
        if (i.GetUserId() != "unknown" && i.GetUserId() != "null") {
            requestsUsers.emplace(i.GetUserId());
        }
    }
    auto gCars = Server->GetDriveAPI()->GetCarsData()->FetchInfo(requestsCars);
    auto gUsers = Server->GetDriveAPI()->GetUsersData()->FetchInfo(requestsUsers);
    const TMap<TString, TDriveCarInfo>& carsInfo = gCars.GetResult();
    const TMap<TString, TDriveUserData>& usersInfo = gUsers.GetResult();

    for (auto&& i : Traces) {
        auto itCar = carsInfo.find(i.GetDeviceId());
        if (itCar != carsInfo.end()) {
            i.SetCarInfo(itCar->second);
        } else {
            if (errors) {
                errors->AddMessage("fetch car data", "Incorrect car id (not fetched from base): " + i.GetDeviceId());
            }
            return false;
        }

        auto itUser = usersInfo.find(i.GetUserId());
        if (itUser != usersInfo.end()) {
            i.SetUserInfo(itUser->second);
        } else {
            if (errors) {
                errors->AddMessage("fetch user data", "Incorrect user id (not fetched from base): " + i.GetUserId());
            }
        }
    }
    return true;
}

bool TTracesAccessor::Execute(TMessagesCollector& errors, const TDuration period) {
    const auto api = Server->GetRTLineAPI(RTLineAPIName);
    if (!api) {
        errors.AddMessage("TracesAccessor::Execute", "cannot get API " + RTLineAPIName);
        return false;
    }
    const auto& sClient = api->GetSearchClient();

    NRTLine::TQuery query;
    if (!OverrideText) {
        query.SetText("i_last_signal_ts:>=" + ToString((ModelingNow() - period).Seconds()));
    } else {
        query.SetText(OverrideText);
    }

    if (NeedTrace) {
        query.AddProperty("Trace");
    }

    query.AddProperty("s_session_id").AddProperty("s_user_id").AddProperty("s_device_id").AddProperty("s_trace_id").
        AddProperty("Sigma20").AddProperty("Sigma10").AddProperty("Sigma5").AddProperty("SigmaAll").AddProperty("SpecialTags").
        AddProperty("length_violation_30").AddProperty("end_time").AddProperty("start_time").AddProperty("ViolationsInfo").AddProperty("__LastSignalTimestamp").
        SetNumDoc(50000).HowBuilder().SetAttrName("docid");
    query.SetTimeout(TDuration::Seconds(20));

    for (auto&& i : VelocityGaps) {
        query.AddErfFactor("LengthViolation_5_" + ToString(i));
    }
    THolder<NRTLine::TSearchReply> reply;
    for (ui32 i = 1; i < 3; ++i) {
        NRTLine::TSearchReply replyLocal = sClient.SendQuery(query, TDuration::Seconds(30));
        reply.Reset(new NRTLine::TSearchReply(std::move(replyLocal)));
        if (reply->GetCode() == 200) {
            break;
        }

    }
    if (reply->GetCode() != 200) {
        errors.AddMessage("rtline reply", "Incorrect response from rtline cluster " + ::ToString(reply->GetCode()) + ": " + reply->GetRawReport() + "/" + query.BuildQuery());
        return false;
    }

    OriginalObjectsCount = reply->GetReport().GetTotalDocCount(0);
    const NMetaProtocol::TReport& report = reply->GetReport();
    if (OriginalObjectsCount == 0) {
        errors.AddMessage("rtline reply", "No data in reply");
        return true;
    }

    if (report.GroupingSize() != 1) {
        errors.AddMessage("watching", "Incorrect reply (grouping size != 1): " + ToString(report.GroupingSize()));
        return false;
    }

    const ::NMetaProtocol::TGrouping& grouping = report.GetGrouping(0);

    if (grouping.GroupSize() == 0) {
        errors.AddMessage("watching", "Request empty");
        return false;
    }

    for (auto&& group : grouping.GetGroup()) {
        if (group.DocumentSize() != 1) {
            errors.AddMessage("watching", "Incorrect watching reply structure: Documents count is incorrect in group (" + ToString(group.DocumentSize()) + ")");
            return false;
        }
        const NMetaProtocol::TDocument& doc = group.GetDocument(0);
        TReadSearchProtoHelper reader(doc);
        TTraceInfo info(reader, VelocityGaps);
        Traces.emplace_back(std::move(info));
    }
    return true;
}
