#pragma once

#include "config.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/url.h>

#include <drive/library/cpp/common/status.h>

#include <library/cpp/json/writer/json_value.h>

#include <rtline/protos/proto_helper.h>

#include <util/generic/ptr.h>
#include <util/string/vector.h>

class TMessagesCollector;

namespace NRTLine {
    class TSearchReply;
}

class TTraceInfo {
private:
    TString UserId;
    TString DeviceId;
    TString TraceId;
    NDrive::ECarStatus CarStatus;
    TString SessionId;
    TInstant LastSignalInstant = TInstant::Zero();
    TInstant FinishPointTime;
    TInstant StartPointTime;

    TVector<ui32> Timestamps;
    TVector<TGeoCoord> Coords;
    TVector<double> VFlow;
    TVector<double> VDevice;
    TVector<double> VLimit;

    NJson::TJsonValue ViolationInfo = NJson::JSON_ARRAY;

    TMaybe<TDriveCarInfo> CarInfo;
    TMaybe<TDriveUserData> UserInfo;

protected:
    TSet<TString> Tags;
    TMap<ui32, double> Sigmas;
    TMap<ui32, double> Violations5;
public:
    static bool ParseFromReply(const NRTLine::TSearchReply& reply, TVector<TTraceInfo>& traces, const TVector<ui32>& VelocityGaps = {30, 40, 50, 60}, TMessagesCollector* collector = nullptr);

    bool HasTraceData() const {
        return !Timestamps.empty();
    }

    const TVector<ui32>& GetTimestamps() const {
        return Timestamps;
    }

    const TVector<TGeoCoord>& GetCoords() const {
        return Coords;
    }

    const TVector<double>& GetVFlow() const {
        return VFlow;
    }

    const TVector<double>& GetVDevice() const {
        return VDevice;
    }

    const TVector<double>& GetVLimit() const {
        return VLimit;
    }

    TTraceInfo& SetCarInfo(const TDriveCarInfo& carInfo) {
        CarInfo = carInfo;
        return *this;
    }

    TTraceInfo& SetUserInfo(const TDriveUserData& carInfo) {
        UserInfo = carInfo;
        return *this;
    }

    const TString& GetDeviceId() const {
        return DeviceId;
    }

    const TString& GetTraceId() const {
        return TraceId;
    }

    const TString& GetUserId() const {
        return UserId;
    }

    const TString& GetSessionId() const {
        return SessionId;
    }

    bool IsCarTag() const {
        return CarHaveFixPlace() || CarStatus == NDrive::ECarStatus::csService;
    }

    bool CarHaveFixPlace() const {
        return CarStatus == NDrive::ECarStatus::csPost || CarStatus == NDrive::ECarStatus::csParking || CarStatus == NDrive::ECarStatus::csReservation || CarStatus == NDrive::ECarStatus::csReservationPaid || CarStatus == NDrive::ECarStatus::csFree;
    }

    TString GetHRReport() const {
        TStringStream ss;
        const TString fullName = !!UserInfo ? UserInfo->GetFullName() : UserId;
        const TString phone = !!UserInfo ? UserInfo->GetPhone() : "NOPHONE";
        if (TraceId.StartsWith("post-")) {
            ss << Sprintf("<a href=\"%s\">после %s</a> (%s)\n", TCarsharingUrl().ClientInfo(UserId).data(), fullName.data(), phone.data());
        } else {
            ss << Sprintf("<a href=\"%s\">%s</a> (%s)\n", TCarsharingUrl().ClientInfo(UserId).data(), fullName.data(), phone.data());
        }
        ss << Sprintf("<a href=\"%s\">%s</a>\n", TCarsharingUrl().CarInfo(DeviceId).data(), !!CarInfo ? CarInfo->GetNumber().data() : DeviceId.data());

        ss << Sprintf("<a href=\"%s\">order</a>\n", TCarsharingUrl().SessionPage(SessionId).data());
        ss << Sprintf("<a href=\"%s\">route</a>\n", TCarsharingUrl().TrackPage(SessionId).data());
        return ss.Str();
    }

    NJson::TJsonValue GetReport() const {
        NJson::TJsonValue result;
        result["user_id"] = UserId;
        result["device_id"] = DeviceId;
        result["trace_id"] = TraceId;
        result["session_id"] = SessionId;
        result["last_signal_ts"] = LastSignalInstant.Seconds();
        result["finish_point_signal_ts"] = FinishPointTime.Seconds();
        result["start_point_signal_ts"] = StartPointTime.Seconds();
        result["duration"] = (FinishPointTime - StartPointTime).Seconds();
        result["violation_info"] = ViolationInfo;
        if (!!CarInfo) {
            result["car_info"] = CarInfo->GetReport(DefaultLocale, NDeviceReport::EReportTraits::ReportStatus | NDeviceReport::EReportTraits::ReportCarId);
        }
        if (!!UserInfo) {
            result["user_info"] = UserInfo->GetReport();
        }
        if (HasTraceData()) {
            result["timestamps"] = JoinVectorIntoString(Timestamps, " ");
            result["coords"] = TGeoCoord::SerializeVector(Coords);
            result["v_flow"] = JoinVectorIntoString(VFlow, " ");
            result["v_limit"] = JoinVectorIntoString(VLimit, " ");
            result["v_device"] = JoinVectorIntoString(VDevice, " ");
        }
        return result;
    }

    TTraceInfo(const TReadSearchProtoHelper& reader, const TVector<ui32>& velocityGaps = Default<TVector<ui32>>());

};

class TTracesAccessor {
private:
    ui32 OriginalObjectsCount = 0;
    TString OverrideText;
    bool NeedTrace = false;
    TString RTLineAPIName;
    TVector<ui32> VelocityGaps = { 30, 40, 50, 60 };
protected:
    const NDrive::IServer* Server;
    TVector<TTraceInfo> Traces;
public:

    TTracesAccessor& SetOverrideText(const TString& overrideText) {
        OverrideText = overrideText;
        return *this;
    }

    TTracesAccessor& SetNeedTrace(const bool value) {
        NeedTrace = value;
        return *this;
    }

    TTracesAccessor(const NDrive::IServer* server, const TString& rtLineAPIName)
        : RTLineAPIName(rtLineAPIName)
        , Server(server)
    {

    }

    const NDrive::IServer* GetServer() const {
        return Server;
    }

    const TVector<TTraceInfo>& GetTraces() const {
        return Traces;
    }

    ui32 GetOriginalObjectsCount() const {
        return OriginalObjectsCount;
    }
    bool Execute(TMessagesCollector& errors, const TDuration period);
    bool FetchTracesData(TMessagesCollector* errors = nullptr);
};
