#include "area_tags.h"

#include <drive/backend/areas/areas.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/models/storage.h>
#include <drive/backend/offers/offers/abstract.h>
#include <drive/backend/offers/ranking/model.h>
#include <drive/backend/roles/manager.h>

#include <drive/library/cpp/catboost/multiclass.h>
#include <drive/library/cpp/deeplink/signature.h>
#include <drive/library/cpp/openssl/oio.h>
#include <drive/library/cpp/scheme/scheme.h>
#include <drive/library/cpp/tracks/quality.h>

#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/library/json/parse.h>

NJson::TJsonValue NDrive::GetPoiReport(TConstArrayRef<TArea> areas, const NDrive::TLocationTags& poiLocationTags, const TMaybe<TString>& beaconParkingPlaceNumber, ELocalization locale, const NDrive::IServer& server) {
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&& area : areas) {
        if (area.GetType() != "poi") {
            continue;
        }
        if (IsIntersectionEmpty(area.GetTags(), poiLocationTags)) {
            continue;
        }
        if (area.GetTooltip()) {
            result.AppendValue(area.GetTooltip()->BuildReport(beaconParkingPlaceNumber, locale, server));
        }
    }
    return result;
}

NDrive::TOptionalLocationTag NDrive::GetRegionTag(const TGeoCoord& coordinate, const NDrive::IServer& server) {
    TSet<TString> regionTags = StringSplitter(
        server.GetSettings().GetValue<TString>("areas.region.tags").GetOrElse("msc_area,spb_area,kazan_area,sochi_area")
    ).Split(',').SkipEmpty();
    auto tags = Yensured(server.GetDriveAPI())->GetTagsInPoint(coordinate);
    for (auto&& tag : tags) {
        if (regionTags.contains(tag)) {
            return tag;
        }
    }
    return {};
}

TAreaInfoForUser::TProto TAreaInfoForUser::DoSerializeSpecialDataToProto() const {
    TProto proto;
    if (!!GetComment()) {
        proto.SetComment(GetComment());
    }
    for (auto&& i : Infos) {
        NDrive::NProto::TSimpleInfoForUser* protoInfo = proto.AddInfos();
        protoInfo->SetKey(i.first);
        protoInfo->SetValue(i.second);
    }
    return proto;
}

bool TAreaInfoForUser::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    TBase::SetComment(proto.GetComment());
    for (auto&& i : proto.GetInfos()) {
        Infos.emplace(i.GetKey(), i.GetValue());
    }
    return true;
}

void TAreaInfoForUser::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::TJsonValue& infosJson = json.InsertValue("infos", NJson::JSON_ARRAY);
    for (auto&& i : Infos) {
        NJson::TJsonValue& kv = infosJson.AppendValue(NJson::JSON_MAP);
        kv.InsertValue("key", i.first);
        kv.InsertValue("value", i.second);
    }
}

bool TAreaInfoForUser::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(json, errors)) {
        return false;
    }
    if (json.Has("infos")) {
        const NJson::TJsonValue::TArray* arr;
        if (!json["infos"].GetArrayPointer(&arr)) {
            return false;
        }
        for (auto&& i : *arr) {
            TString key;
            TString value;
            if (!i["key"].GetString(&key) || !i["value"].GetString(&value)) {
                return false;
            }
            Infos.emplace(key, value);
        }
    }
    return true;
}

NDrive::TScheme TDropAreaFeatures::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSBoolean>("DefaultDropAllowance", "Разрешен сброс машины по умолчанию");
    result.Add<TFSArray>("DropPolicies", "Политики сброса в зависимости от оффера").SetElement(TAreaDropPolicyBuilder::GetScheme(server));
    return result;
}

NDrive::TScheme TAreaInfoForUser::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    NDrive::TScheme& element = result.Add<TFSArray>("infos", "Информация").SetElement<NDrive::TScheme>();
    element.Add<TFSString>("key", "Ключ");
    element.Add<TFSString>("value", "Значение");
    return result;
}

NDrive::TScheme TClusterTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("threshold", "Threshold to join cars into the cluster");
    return result;
}

bool TClusterTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    Y_UNUSED(errors);
    return NJson::ParseField(value["threshold"], Threshold, true);
}

void TClusterTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    value["threshold"] = Threshold;
}

NDrive::TScheme TFeedbackButtonTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    NDrive::TScheme& value = result.Add<TFSStructure>("value", "Свойства кнопки").SetStructure<NDrive::TScheme>();
    value.Add<TFSString>("icon", "Иконка").SetRequired(true);
    value.Add<TFSString>("title", "Заголовок").SetRequired(true);
    value.Add<TFSString>("button", "Id кнопки");
    value.Add<TFSString>("link", "Web-cсылка");
    value.Add<TFSString>("deeplink", "Universal ссылка (deeplink)");
    value.Add<TFSString>("description", "Описание");
    value.Add<TFSString>("detailed_description", "Подробное описание");
    value.Add<TFSNumeric>("priority", "Приоритет");
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const TFeedbackButtonTag::TValue& object) {
    NJson::TJsonValue result;
    result["icon"] = object.Icon;
    result["title"] = object.Title;
    if (object.Code) {
        result["button"] = object.Code;
    }
    if (object.Link) {
        result["link"] = object.Link;
    }
    if (object.Deeplink) {
        result["deeplink"] = object.Deeplink;
    }
    if (object.Description) {
        result["description"] = object.Description;
    }
    if (object.DetailedDescription) {
        result["detailed_description"] = object.DetailedDescription;
    }
    if (object.Priority) {
        result["priority"] = object.Priority;
    }
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TFeedbackButtonTag::TValue& result) {
    return
        NJson::ParseField(value["icon"], result.Icon, true) &&
        NJson::ParseField(value["title"], result.Title, true) &&
        NJson::ParseField(value["button"], result.Code) &&
        NJson::ParseField(value["link"], result.Link) &&
        NJson::ParseField(value["deeplink"], result.Deeplink) &&
        NJson::ParseField(value["description"], result.Description) &&
        NJson::ParseField(value["detailed_description"], result.DetailedDescription) &&
        NJson::ParseField(value["priority"], result.Priority);
}

bool TFeedbackButtonTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    Y_UNUSED(errors);
    return NJson::ParseField(value["value"], Value, true);
}

void TFeedbackButtonTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    value["value"] = NJson::ToJson(Value);
}

void TTransportationTag::TFirstMileReportOptions::SetBeaconsInfo(const ISettings& settings, const TRTDeviceSnapshot& snapshot) {
    auto beacon = snapshot.GetBeaconsLocation();
    if (beacon) {
        BeaconLocation = beacon->GetCoord();
    }
    TMaybe<TString> beaconLocationMethodNames = settings.GetValue<TString>("transportation.beacon_first_mile_methods_names");
    if (NJson::TJsonValue jsonInfo; beaconLocationMethodNames && NJson::ReadJsonFastTree(*beaconLocationMethodNames, &jsonInfo)) {
        BeaconLocationMethodNames = NJson::FromJsonWithDefault(jsonInfo, BeaconLocationMethodNames);
    }
}

NJson::TJsonValue TTransportationTag::GetFirstMileReport(const TFirstMileReportOptions& options, const NDrive::IServer& server) {
    const IDriveDatabase& database = server.GetDriveDatabase();
    const TAreasDB& areas = database.GetAreaManager();
    const ILocalization* localization = server.GetLocalization();
    TVector<TMethod> mm;
    {
        areas.ProcessHardTagsInPoint<TTransportationTag>(options.Location, [&mm](const TTransportationTag* tag) {
            if (tag) {
                for (auto&& method : tag->GetMethods()) {
                    mm.push_back(method);
                }
            }
            return true;
        }, TInstant::Zero());
        if (options.BeaconLocation) {
            areas.ProcessHardTagsInPoint<TTransportationTag>(*options.BeaconLocation, [&options, &mm](const TTransportationTag* tag) {
                if (tag) {
                    for (auto&& method : tag->GetMethods()) {
                        if (options.BeaconLocationMethodNames.contains(method.Id)) {
                            mm.push_back(method);
                        }
                    }
                }
                return true;
            }, TInstant::Zero());
        }
    }
    std::sort(mm.begin(), mm.end(), [](const TMethod& left, const TMethod& right) {
        return std::tie(left.Id, left.Priority) < std::tie(right.Id, right.Priority);
    });
    mm.erase(std::unique(mm.begin(), mm.end(), [](const TMethod& left, const TMethod& right) {
        return left.Id == right.Id;
    }), mm.end());
    std::sort(mm.begin(), mm.end(), [](const TMethod& left, const TMethod& right) {
        return left.Priority < right.Priority;
    });

    TSet<TString> fallbacks;
    TMap<TString, const TMethod*> methodsMap;
    for (auto&& m : mm) {
        methodsMap[m.Id] = &m;
        fallbacks.insert(m.FallbackId);
    }

    NJson::TJsonValue result;
    if (options.Description) {
        auto description = localization ? localization->ApplyResources(options.Description, options.Locale) : options.Description;
        result.InsertValue("description", std::move(description));
    }

    NJson::TJsonValue& methods = result.InsertValue("methods", NJson::JSON_ARRAY);
    for (auto&& m : mm) {
        if (fallbacks.contains(m.Id)) {
            continue;
        }
        if (m.MinimalWalkingDuration && m.MinimalWalkingDuration > options.WalkingDuration && !options.BeaconLocation) {
            continue;
        }

        NJson::TJsonValue method = GetMethodReport(options, m);
        auto p = methodsMap.find(m.FallbackId);
        if (p != methodsMap.end() && p->second) {
            method["fallback"] = GetMethodReport(options, *p->second);
        }
        methods.AppendValue(std::move(method));
    }
    if (options.Geocoded) {
        NJson::TJsonValue method;
        method["id"] = "copy";
        method["clipboard"] = options.Geocoded;
        methods.AppendValue(std::move(method));
    }
    return result;
}

NJson::TJsonValue TTransportationTag::GetMethodReport(const TFirstMileReportOptions& options, const TMethod& m) {
    NJson::TJsonValue method;
    if (m.CalculateCost) {
        method["cost"] = 4200; // PLACEHOLDER
    }

    TString link;
    TString package;
    switch (options.App) {
        case IRequestProcessor::EApp::AndroidClient:
            link = m.PlayMarketLink;
            package = m.PlayMarketPackage;
            break;
        case IRequestProcessor::EApp::iOS:
            link = m.AppStoreLink;
            break;
        default:
            break;
    }
    if (!link) {
        link = m.Link;
    }

    TString deeplink = m.DeepLinkTemplate;
    SubstGlobal(deeplink, "OBJECT_LATITUDE", ToString(options.Location.Y));
    SubstGlobal(deeplink, "OBJECT_LONGITUDE", ToString(options.Location.X));
    if (m.PrivateKey) {
        TString signature = NDrive::GetDeeplinkSignature(deeplink, m.PrivateKey);
        deeplink.append("&signature=");
        deeplink.append(CGIEscapeRet(signature));
    }

    method["id"] = m.Id;
    method["icon"] = m.Icon;
    method["title"] = m.Title;
    method["link"] = link;
    method["deeplink"] = deeplink;
    method["required"] = m.Required;
    if (package) {
        method["package"] = package;
    }
    return method;
}

NDrive::TScheme TTransportationTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("Comment", "Commentary");
    NDrive::TScheme& method = result.Add<TFSArray>("FirstMileMethod", "transportation method").SetElement<NDrive::TScheme>();

    method.Add<TFSString>("FallbackId", "fallback method identifier");
    method.Add<TFSString>("Id", "method identifier");
    method.Add<TFSString>("Icon", "url of method icon");
    method.Add<TFSString>("Title", "method title");
    method.Add<TFSString>("Link", "link for the method");
    method.Add<TFSString>("DeepLinkTemplate", "deep link template for the method");
    method.Add<TFSString>("AppStoreLink", "AppStore link for the method");
    method.Add<TFSString>("PlayMarketLink", "PlayMarket link for the method");
    method.Add<TFSString>("PlayMarketPackage", "PlayMarket package name for the method");
    method.Add<TFSNumeric>("MinimalWalkingDuration", "Minimal walking duration in seconds to display the method");
    method.Add<TFSBoolean>("CalculateCost", "should calculate cost of the method");
    method.Add<TFSBoolean>("Required", "always show the method");
    method.Add<TFSNumeric>("Priority", "method priority");
    method.Add<TFSText>("PrivateKeyData", "private key for deeplink signature in PEM format");

    return result;
}

TTransportationTag::TProto TTransportationTag::DoSerializeSpecialDataToProto() const {
    TProto result;
    result.SetComment(GetComment());
    for (auto&& method : Methods) {
        auto m = result.AddFirstMileMethod();
        m->SetFallbackId(method.FallbackId);
        m->SetId(method.Id);
        m->SetIcon(method.Icon);
        m->SetTitle(method.Title);
        m->SetLink(method.Link);
        m->SetDeepLinkTemplate(method.DeepLinkTemplate);
        m->SetAppStoreLink(method.AppStoreLink);
        m->SetPlayMarketLink(method.PlayMarketLink);
        m->SetPlayMarketPackage(method.PlayMarketPackage);
        m->SetMinimalWalkingDuration(method.MinimalWalkingDuration.Seconds());
        m->SetCalculateCost(method.CalculateCost);
        m->SetRequired(method.Required);
        m->SetPriority(method.Priority);
        m->SetPrivateKeyData(method.PrivateKeyData);
    }
    return result;
}

bool TTransportationTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    SetComment(proto.GetComment());
    for (auto&& m : proto.GetFirstMileMethod()) {
        TMethod method;
        method.FallbackId = m.GetFallbackId();
        method.Id = m.GetId();
        method.Icon = m.GetIcon();
        method.Title = m.GetTitle();
        method.Link = m.GetLink();
        method.DeepLinkTemplate = m.GetDeepLinkTemplate();
        method.AppStoreLink = m.GetAppStoreLink();
        method.PlayMarketLink = m.GetPlayMarketLink();
        method.PlayMarketPackage = m.GetPlayMarketPackage();
        method.MinimalWalkingDuration = TDuration::Seconds(m.GetMinimalWalkingDuration());
        method.CalculateCost = m.GetCalculateCost();
        method.Required = m.GetRequired();
        method.Priority = m.GetPriority();
        method.PrivateKeyData = m.GetPrivateKeyData();
        if (method.PrivateKeyData) try {
            method.PrivateKey = NOpenssl::GetPrivateKey(method.PrivateKeyData);
        } catch (const std::exception& e) {
            ERROR_LOG << "cannot read PrivateKey from " << method.PrivateKeyData << ": " << FormatExc(e) << Endl;
            return false;
        }
        Methods.push_back(std::move(method));
    }
    return true;
}

void TTransportationTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    NProtobufJson::Proto2Json(SerializeSpecialDataToProto(), json);
}

bool TTransportationTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    try {
        TProto proto;
        NProtobufJson::Json2Proto(json, proto);
        return DoDeserializeSpecialDataFromProto(proto);
    } catch (const yexception& e) {
        if (errors) {
            errors->AddMessage(TypeName(*this) + "::DoSpecialDataFromJson", e.what());
        }
        return false;
    }
}

TAreaPotentialTag::TInfo::TInfo(const TArea& area, const TAreaPotentialTag& potential, int discountPercent)
    : DiscountPercent(discountPercent)
    , Border(area.GetCoords())
{
    if (Border.Size()) {
        Center = Border.GetRectSafe().GetCenter();
    } else {
        Center = TGeoCoord(0, 0);
    }
    Name = !potential.GetPublicName().empty() ? potential.GetPublicName() : area.GetTitle();
    Priority = potential.GetPotentialPriority();
    Public = potential.GetPublic();
    HintPosition = area.GetHintPositionDef(Center);
}

NDrive::TScheme TAreaPotentialTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("public_name", "Публичное имя");
    result.Add<TFSStructure>("potential_scheme", "Потенциал (-100 -- 100)").SetStructure(TLinearScheme::GetScheme());
    result.Add<TFSNumeric>("potential_priority", "Приоритет в выдаче");
    result.Add<TFSBoolean>("public", "Показывать в выдаче");
    TVector<TDBAction> actions = server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetCachedObjectsVector();
    TSet<TString> offerConstructors;
    for (auto&& i : actions) {
        if (i->GetType() == "fixpoint_offer_builder") {
            offerConstructors.emplace(i->GetName());
        }
    }
    result.Add<TFSVariants>("offers_available", "Допустимые офферы").SetVariants(offerConstructors).SetMultiSelect(true);
    return result;
}

double TAreaPotentialTag::CalcPriceKff(const double fromPotential, const double toPotential) {
    return Max<double>(0, 1.0 + (toPotential - fromPotential) / 100.0);
}

double TAreaPotentialTag::CalcPriceKff(const NDrive::IServer& server, const TGeoCoord& from, const TGeoCoord& to, const TString& offerName, const double precision, const TInstant actuality) {
    TInternalPointContext internalContext;
    internalContext.SetPrecision(precision);

    internalContext.SetPolicy(TInternalPointContext::EInternalPolicySet::Potentially);
    const double fromPotential = CalcPotential(server, from, offerName, actuality, internalContext);

    internalContext.SetPolicy(TInternalPointContext::EInternalPolicySet::Guarantee);
    const double toPotential = CalcPotential(server, to, offerName, actuality, internalContext);
    return CalcPriceKff(fromPotential, toPotential);
}

double TAreaPotentialTag::CalcPotential(const NDrive::IServer& server, const TGeoCoord& coordinate, const TString& offerName, const TInstant actuality, const TInternalPointContext internalContext) {
    const auto driveApi = server.GetDriveAPI();
    if (!driveApi) {
        return 0;
    }
    const auto areas = driveApi->GetAreasDB();
    if (!areas) {
        return 0;
    }

    double potential = 0;
    const auto actor = [&potential, &offerName, &server](const TDBTag& dbTag) {
        const TAreaPotentialTag* tag = dbTag.GetTagAs<TAreaPotentialTag>();
        if (!tag) {
            return true;
        }
        if (tag->GetOffersAvailable().empty() || !offerName) {
            potential = tag->GetPotential(server.GetSnapshotsManager().GetSnapshots().GetAreaCarsCount(dbTag.GetObjectId()));
            return !!offerName;
        } else if (tag->GetOffersAvailable().contains(offerName)) {
            potential = tag->GetPotential(server.GetSnapshotsManager().GetSnapshots().GetAreaCarsCount(dbTag.GetObjectId()));
            return false;
        } else {
            return true;
        }
    };
    areas->ProcessHardDBTagsInPoint<TAreaPotentialTag>(coordinate, actor, actuality, internalContext);
    return potential;
}

TAreaPotentialTag::TInfos TAreaPotentialTag::CalcInfos(const NDrive::IServer& server, const TGeoCoord& from, const TString& offerName, TInstant actuality) {
    auto api = server.GetDriveAPI();
    if (!api) {
        return {};
    }
    auto areas = api->GetAreasDB();
    if (!areas) {
        return {};
    }

    auto currentPotential = CalcPotential(server, from, offerName, actuality);

    TVector<TTaggedArea> taggedAreas;
    if (!areas->GetTagsManager().GetObjectsFromCache({TypeName}, taggedAreas, actuality)) {
        return {};
    }
    TSet<TString> areaIds;
    for (auto&& taggedArea : taggedAreas) {
        areaIds.insert(taggedArea.GetId());
    }
    TMap<TString, TArea> areasData;
    if (!areas->GetCustomObjectsMap(areaIds, areasData, actuality)) {
        ERROR_LOG << "cannot fetch areas data" << Endl;
    }

    TInfos result;
    for (auto&& taggedArea : taggedAreas) {
        auto tag = taggedArea.GetTag(TypeName);
        if (!tag) {
            continue;
        }
        auto areaPotentialTag = tag->GetTagAs<TAreaPotentialTag>();
        if (!areaPotentialTag) {
            continue;
        }
        const auto& eligibleOffers = areaPotentialTag->GetOffersAvailable();
        if (offerName && !eligibleOffers.empty() && !eligibleOffers.contains(offerName)) {
            continue;
        }
        auto areaIt = areasData.find(taggedArea.GetId());
        if (areaIt == areasData.end()) {
            continue;
        }
        const int discountPercent = std::round(100 * (1 - CalcPriceKff(currentPotential, areaPotentialTag->GetPotential(server.GetSnapshotsManager().GetSnapshots().GetAreaCarsCount(tag->GetObjectId())))));
        if (discountPercent) {
            TInfo info(areaIt->second, *areaPotentialTag, discountPercent);
            server.GetSettings().GetIntervalParamValueJsonDef("offers.fix_point.discount_styles", discountPercent, NJson::JSON_NULL, info.MutableStyleInfo());
            result.emplace_back(std::move(info));
        }
    }
    return result;
}

NDrive::TScheme TFixPointCorrectionTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("LinkType", "Тип прилинкованной области");
    result.Add<TFSVariants>("LinkArea", "Область реального завершения по fix").SetVariants(server->GetDriveAPI()->GetAreasDB()->GetAreaIds()).AddVariant("");
    result.Add<TFSVariants>("LinkPublicArea", "Область завершения публичная").SetVariants(server->GetDriveAPI()->GetAreasDB()->GetAreaIds()).AddVariant("");
    result.Add<TFSString>("Area", "Область реального завершения по fix (координаты lon1 lat1 lon2 lat2 ..)");
    result.Add<TFSString>("PublicArea", "Область завершения публичная (координаты lon1 lat1 lon2 lat2 ..)");
    result.Add<TFSNumeric>("CarLimit", "Максимальное количество машин в зоне");
    result.Add<TFSNumeric>("WalkingDuration", "Длительность прогулки до границы зоны (с)");
    return result;
}

const TString TAreaInfoForUser::TypeName = "area_user_info";
ITag::TFactory::TRegistrator<TAreaInfoForUser> TAreaInfoForUser::Registrator(TAreaInfoForUser::TypeName);

const TString TAreaPotentialTag::TypeName = "area_potential";
ITag::TFactory::TRegistrator<TAreaPotentialTag> TAreaPotentialTag::Registrator(TAreaPotentialTag::TypeName);

const TString TFixPointCorrectionTag::TypeName = "fix_point_area_correction";
ITag::TFactory::TRegistrator<TFixPointCorrectionTag> TFixPointCorrectionTag::Registrator(TFixPointCorrectionTag::TypeName);

const TString TDropAreaFeatures::TypeName = "drop_area_features";

TMaybe<TOfferDropPolicy> TDropAreaFeatures::BuildFeeBySession(const IOffer* offer, const NDrive::IServer* server, const TString& areaId, const bool isInternalFlow) const {
    if (!server) {
        return {};
    }

    TMaybe<TDBAction> action;
    if (offer) {
        action = server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetObject(offer->GetBehaviourConstructorId());
        if (!action) {
            return {};
        }
    }
    auto offerTags = action ? &action.GetRef()->GetGrouppingTags() : nullptr;
    return BuildFeeBySession(offerTags, server, areaId, isInternalFlow);
}

TMaybe<TOfferDropPolicy> TDropAreaFeatures::BuildFeeBySession(const TSet<TString>* offerTags, const NDrive::IServer* server, const TString& areaId, bool isInternalFlow) const {
    if (!server) {
        return {};
    }

    for (auto&& i : DropPolicies) {
        if (i.GetIsInternalFlowCorrector() != isInternalFlow) {
            continue;
        }
        if (offerTags) {
            const auto& offerAttributesFilter = i.GetOfferAttributesFilter();
            if (offerAttributesFilter.IsEmpty() || offerAttributesFilter.IsMatching(*offerTags)) {
                return i.GetOfferDropPolicy(*server, areaId);
            }
        } else {
            return i.GetOfferDropPolicy(*server, areaId);
        }
    }
    return {};
}

TMaybe<TOfferDropPolicy> TDropAreaFeatures::GetFeeBySession(const IOffer* offer, const NDrive::IServer* server, const ui32 fee, const bool isInternalFlow) const {
    if (!offer || !server) {
        return {};
    }
    TMaybe<TDBAction> action = server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetObject(offer->GetBehaviourConstructorId());
    if (!action) {
        return {};
    }

    for (auto&& i : DropPolicies) {
        if (i.GetIsInternalFlowCorrector() != isInternalFlow) {
            continue;
        }
        if (i.GetOfferAttributesFilter().IsEmpty() || i.GetOfferAttributesFilter().IsMatching((*action)->GetGrouppingTags())) {
            return i.GetOfferDropPolicy(fee);
        }
    }
    return {};
}

ITag::TFactory::TRegistrator<TDropAreaFeatures> TDropAreaFeatures::Registrator(TDropAreaFeatures::TypeName);

const TString TClusterTag::TypeName = "cluster_tag";
ITag::TFactory::TRegistrator<TClusterTag> TClusterTag::Registrator(TClusterTag::TypeName);

const TString TFeedbackButtonTag::TypeName = "feedback_button_tag";
ITag::TFactory::TRegistrator<TFeedbackButtonTag> TFeedbackButtonTag::Registrator(TFeedbackButtonTag::TypeName);

const TString TDestinationDetectorTag::TypeName = "destination_detector_tag";
ITag::TFactory::TRegistrator<TDestinationDetectorTag> TDestinationDetectorTag::Registrator(TDestinationDetectorTag::TypeName);

NDrive::TScheme TDestinationDetectorTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    TCommonDestinationDetector::AddScheme(result, server);
    result.Add<TFSString>("StartZoneAttributesFilter", "фильтр тегов зоны старта в стандартном формате");
    return result;
}

ITag::TFactory::TRegistrator<TTransportationTag> TTransportationTag::Registrator("transportation_tag");

const TString TStandRegistrationPromoTag::TypeName = "area_stand_registration_promo";
ITag::TFactory::TRegistrator<TStandRegistrationPromoTag> TStandRegistrationPromoTag::Registrator(TStandRegistrationPromoTag::TypeName);
TStandRegistrationPromoTag::TDescription::TFactory::TRegistrator<TStandRegistrationPromoTag::TDescription> TStandRegistrationPromoTag::TDescription::Registrator(TStandRegistrationPromoTag::TypeName);

const TString TServiceDelegationPossibilityTag::TypeName = "area_service_delegation_ability";
ITag::TFactory::TRegistrator<TServiceDelegationPossibilityTag> TServiceDelegationPossibilityTag::Registrator(TServiceDelegationPossibilityTag::TypeName);
TServiceDelegationPossibilityTag::TDescription::TFactory::TRegistrator<TServiceDelegationPossibilityTag::TDescription> TServiceDelegationPossibilityTag::TDescription::Registrator(TServiceDelegationPossibilityTag::TypeName);

TFixPointCorrectionTag::TOptionalLinkedAreaInfo TFixPointCorrectionTag::GetLinkedArea(const TGeoCoord& c, const NDrive::IServer& server, const TInstant reqActuality /*= TInstant::Zero()*/) {
    TOptionalLinkedAreaInfo result;
    const auto actor = [&server, &result](const TFixPointCorrectionTag* tag) -> bool {
        if (tag->GetLinkArea() || tag->GetLinkPublicArea()) {
            auto area = server.GetDriveAPI()->GetAreasDB()->GetObject(tag->GetLinkPublicArea() ? tag->GetLinkPublicArea() : tag->GetLinkArea());
            if (area) {
                result.ConstructInPlace();
                result->Area = std::move(*area);
                result->CarLimit = tag->OptionalCarLimit();
                return false;
            }
        }
        return true;
    };
    server.GetDriveAPI()->GetAreasDB()->ProcessHardTagsInPoint<TFixPointCorrectionTag>(c, actor, reqActuality);
    return result;
}

TFixPointCorrectionTag::TProto TFixPointCorrectionTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TFixPointCorrectionTag proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetLinkType(LinkType);
    proto.SetArea(TGeoCoord::SerializeVector(Area));
    proto.SetPublicArea(TGeoCoord::SerializeVector(PublicArea));
    proto.SetLinkArea(LinkArea);
    proto.SetLinkPublicArea(LinkPublicArea);
    if (CarLimit) {
        proto.SetCarLimit(*CarLimit);
    }
    if (WalkingDuration) {
        proto.SetWalkingDuration(WalkingDuration->Seconds());
    }
    return proto;
}

bool TFixPointCorrectionTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }
    if (proto.HasCarLimit()) {
        CarLimit = proto.GetCarLimit();
    }
    if (proto.HasWalkingDuration()) {
        WalkingDuration = TDuration::Seconds(proto.GetWalkingDuration());
    }
    LinkType = proto.GetLinkType();
    LinkArea = proto.GetLinkArea();
    LinkPublicArea = proto.GetLinkPublicArea();
    if (!TGeoCoord::DeserializeVector(proto.GetArea(), Area)) {
        return false;
    }
    if (!TGeoCoord::DeserializeVector(proto.GetPublicArea(), PublicArea)) {
        return false;
    }
    return true;
}

void TCommonDestinationDetector::AddScheme(NDrive::TScheme& result, const NDrive::IServer* server) {
    TCommonCategoryScoreCalcer::AddScheme(result, server);
    TCommonCategoryScoreThreshold::AddScheme(result, server);
}

void TCommonDestinationDetector::SerializeToJson(NJson::TJsonValue& result) const {
    TCommonCategoryScoreCalcer::SerializeToJson(result);
    TCommonCategoryScoreThreshold::SerializeToJson(result);
}

bool TCommonDestinationDetector::DeserializeFromJson(const NJson::TJsonValue& value) {
    return TCommonCategoryScoreCalcer::DeserializeFromJson(value) && TCommonCategoryScoreThreshold::DeserializeFromJson(value);
}

void TCommonCategoryScoreThreshold::AddScheme(NDrive::TScheme& result, const NDrive::IServer* server) {
    if (!result.HasField("ModelName")) {
        const TVector<TString> models = (server && server->GetModelsStorage()) ? server->GetModelsStorage()->ListOfferModels() : TVector<TString>();
        result.Add<TFSVariants>("ModelName", "Название модели").SetVariants(models).SetMultiSelect(false);
    }
    TSimpleCategoryScoreThreshold::AddScheme(result);
}

void TSimpleCategoryScoreThreshold::AddScheme(NDrive::TScheme& result) {
    result.Add<TFSNumeric>("MinValue", "Минимальное значение (может отсутствовать)").SetPrecision(10);
    result.Add<TFSNumeric>("MaxValue", "Максимальное значение (может отсутствовать)").SetPrecision(10);
}

void TCommonCategoryScoreCalcer::AddScheme(NDrive::TScheme& result, const NDrive::IServer* server) {
    if (!result.HasField("ModelName")) {
        const TVector<TString> models = (server && server->GetModelsStorage()) ? server->GetModelsStorage()->ListOfferModels() : TVector<TString>();
        result.Add<TFSVariants>("ModelName", "Название модели").SetVariants(models).SetMultiSelect(false);
    }
    result.Add<TFSNumeric>("Dimension", "Измерение").SetMin(0);
    result.Add<TFSNumeric>("FixValue", "Фиксированное значение (для тестирования)").SetPrecision(10);
}

TMaybe<double> TCommonCategoryScoreCalcer::CalcScore(const NDrive::TOfferFeatures& features, TCategoryScoringContext& context, const NDrive::IServer* server, const TCommonCategoryScoreThreshold* threshold) const {
    auto it = context.GetCachedScores().find(ModelName);
    TVector<double> scores;
    if (FixValue) {
        scores.resize(Dimension + 1, *FixValue);
        features.StoreDebugInfo(ModelName + "-FixValue-" + ::ToString(*FixValue), scores);
    } else if (it != context.GetCachedScores().end()) {
        scores = it->second;
        features.StoreDebugInfo(ModelName, scores);
    } else {
        NDrive::TOfferModelConstPtr model = (server && server->GetModelsStorage()) ? server->GetModelsStorage()->GetOfferModel(ModelName) : nullptr;
        auto cmm = std::dynamic_pointer_cast<const NDrive::IMulticlassModel>(model);
        if (!cmm) {
            return {};
        }
        scores = cmm->Predict(features);
        context.MutableCachedScores().emplace(ModelName, scores);
        features.StoreDebugInfo(ModelName, scores);
    }
    if (scores.size() <= Dimension) {
        return {};
    }
    const double val = scores[Dimension];
    return threshold ? threshold->GetValue(val, ModelName) : val;
}

const TString TSpeedLimitCorrectionAreaTag::TypeName = "speed_limit_correction_area_tag";
ITag::TFactory::TRegistrator<TSpeedLimitCorrectionAreaTag> TSpeedLimitCorrectionAreaTag::Registrator(TSpeedLimitCorrectionAreaTag::TypeName);

NDrive::TScheme TSpeedLimitCorrectionAreaTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("custom_speed_limit", "Новая максимально допустимая скорость на участке в м/с");
    return result;
}

TSpeedLimitCorrectionAreaTag::TProto TSpeedLimitCorrectionAreaTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetCustomSpeedLimit(CustomSpeedLimit);
    return proto;
}

bool TSpeedLimitCorrectionAreaTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }
    CustomSpeedLimit = proto.GetCustomSpeedLimit();
    return true;
}

void TSpeedLimitCorrectionAreaTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    json.InsertValue("custom_speed_limit", CustomSpeedLimit);
}

bool TSpeedLimitCorrectionAreaTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(json, errors)) {
        return false;
    }
    return NJson::ParseField(json["custom_speed_limit"], CustomSpeedLimit, true);
}

TMaybe<double> FetchCustomSpeedLimit(const TGeoCoord& c, const NDrive::IServer& server) {
    TMaybe<double> customSpeedLimit;
    auto actor = [&customSpeedLimit](const TSpeedLimitCorrectionAreaTag* tag) -> bool {
        if (!tag) {
            return true;
        }
        if (!tag->GetCustomSpeedLimit()) {
            return true;
        }
        customSpeedLimit = std::max(customSpeedLimit.GetOrElse(0), tag->GetCustomSpeedLimit());
        return false;
    };
    server.GetDriveAPI()->GetAreasDB()->ProcessHardTagsInPoint<TSpeedLimitCorrectionAreaTag>(c, actor, TInstant::Zero());
    return customSpeedLimit;
}

void CorrectSpeedLimitDuration(TSpeedLimitRange& range, const NDrive::IServer& server) {
    if (!range.IsSpeedLimitExceeded() || range.Points.empty()) {
        return;
    }

    auto beginRangePoint = range.Points.front();
    auto endRangePoint = range.Points.back();

    auto beginSpeedLimit = FetchCustomSpeedLimit(beginRangePoint, server);
    auto endSpeedLimit = FetchCustomSpeedLimit(endRangePoint, server);

    double customSpeedLimit = std::max(beginSpeedLimit.GetOrElse(0), endSpeedLimit.GetOrElse(0));

    if (customSpeedLimit == 0) {
        return;
    }

    range.SpeedLimitDuration = range.Length / customSpeedLimit;
}

void TSpeedLimitCorrectionAreaTag::CorrectSpeedLimitRange(NDrive::TTracksLinker::TResult& result, const NDrive::IServer& server) {
    for (auto& segment : result.Segments) {
        for (auto& range : segment.Processed) {
            CorrectSpeedLimitDuration(range, server);
        }
    }
}

void TSpeedLimitCorrectionAreaTag::CorrectSpeedLimitRanges(NDrive::TTracksLinker::TResults& results, const NDrive::IServer& server) {
    for (auto& result : results) {
        CorrectSpeedLimitRange(result, server);
    }
}

const TString TStandardWithDiscountAreaOfferAreaTag::TypeName = "standard_with_discount_area_offer_area_tag";
ITag::TFactory::TRegistrator<TStandardWithDiscountAreaOfferAreaTag> TStandardWithDiscountAreaOfferAreaTag::Registrator(TStandardWithDiscountAreaOfferAreaTag::TypeName);

NDrive::TScheme TStandardWithDiscountAreaOfferAreaTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("discount", "Скидка в полигоне, от 0 до 1 (20.11% == 0.2011)");
    return result;
}

TStandardWithDiscountAreaOfferAreaTag::TProto TStandardWithDiscountAreaOfferAreaTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetDiscount(Discount);
    return proto;
}

bool TStandardWithDiscountAreaOfferAreaTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }
    Discount = proto.GetDiscount();
    return true;
}

void TStandardWithDiscountAreaOfferAreaTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    json.InsertValue("discount", Discount);
}

bool TStandardWithDiscountAreaOfferAreaTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(json, errors)) {
        return false;
    }
    return NJson::ParseField(json["discount"], Discount, true);
}
