#pragma once

#include <drive/backend/areas/areas.h>
#include <drive/backend/areas/drop_object_features.h>
#include <drive/backend/data/common/serializable.h>
#include <drive/backend/data/proto/tags.pb.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/price/price.h>
#include <drive/backend/processor/processor.h>
#include <drive/backend/proto/tags.pb.h>

#include <drive/library/cpp/openssl/types.h>
#include <drive/library/cpp/tracks/client.h>

#include <rtline/library/geometry/polyline.h>

namespace NDrive {
    struct TOfferFeatures;

    NJson::TJsonValue GetPoiReport(
        TConstArrayRef<TArea> areas,
        const TLocationTags& poiLocationTags,
        const TMaybe<TString>& beaconParkingPlaceNumber,
        ELocalization locale,
        const NDrive::IServer& server
    );
    TOptionalLocationTag GetRegionTag(const TGeoCoord& coordinate, const NDrive::IServer& server);
}

class TCategoryScoringContext {
private:
    using TCachedScores = TMap<TString, TVector<double>>;

public:
    R_FIELD(TCachedScores, CachedScores);
};

class TSimpleCategoryScoreThreshold {
private:
    R_OPTIONAL(double, MinValue);
    R_OPTIONAL(double, MaxValue);

public:
    TMaybe<double> GetValue(const double val) const {
        if (MaxValue && *MaxValue <= val) {
            return {};
        }
        if (MinValue && *MinValue > val) {
            return {};
        }
        return val;
    }
    static void AddScheme(NDrive::TScheme& result);

    void SerializeToJson(NJson::TJsonValue& result) const {
        TJsonProcessor::Write(result, "MinValue", MinValue);
        TJsonProcessor::Write(result, "MaxValue", MaxValue);
    }

    bool DeserializeFromJson(const NJson::TJsonValue& value) {
        if (!TJsonProcessor::Read(value, "MinValue", MinValue)) {
            return false;
        }
        if (!TJsonProcessor::Read(value, "MaxValue", MaxValue)) {
            return false;
        }
        return true;
    }

    template <class TProto>
    void SerializeToProto(TProto& proto) const {
        if (MinValue) {
            proto.SetMinValue(*MinValue);
        }
        if (MaxValue) {
            proto.SetMaxValue(*MaxValue);
        }
    }

    template <class TProto>
    bool DeserializeFromProto(const TProto& proto) {
        if (proto.HasMaxValue()) {
            MaxValue = proto.GetMaxValue();
        }
        if (proto.HasMinValue()) {
            MinValue = proto.GetMinValue();
        }
        return true;
    }
};

class TCommonCategoryScoreThreshold: public TSimpleCategoryScoreThreshold {
private:
    R_FIELD(TString, ModelName);

public:
    TMaybe<double> GetValue(const double val, const TString& modelName) const {
        TMaybe<double> result = TSimpleCategoryScoreThreshold::GetValue(val);
        if (!result || ModelName != modelName) {
            return {};
        }

        return result;
    }
    static void AddScheme(NDrive::TScheme& result, const NDrive::IServer* server);

    void SerializeToJson(NJson::TJsonValue& result) const {
        if (!result.Has("ModelName")) {
            TJsonProcessor::Write(result, "ModelName", ModelName);
        }
        TSimpleCategoryScoreThreshold::SerializeToJson(result);
    }

    bool DeserializeFromJson(const NJson::TJsonValue& value) {
        NJson::TJsonValue result(NJson::JSON_MAP);
        if (!TJsonProcessor::Read(value, "ModelName", ModelName)) {
            return false;
        }
        return TSimpleCategoryScoreThreshold::DeserializeFromJson(value);
    }

    template <class TProto>
    void SerializeToProto(TProto& proto) const {
        proto.SetModelName(ModelName);
        TSimpleCategoryScoreThreshold::SerializeToProto(proto);
    }

    template <class TProto>
    bool DeserializeFromProto(const TProto& proto) {
        ModelName = proto.GetModelName();
        return TSimpleCategoryScoreThreshold::DeserializeFromProto(proto);
    }
};

class TCommonCategoryScoreCalcer {
private:
    R_FIELD(TString, ModelName);
    R_FIELD(ui32, Dimension, 0);
    R_OPTIONAL(double, FixValue);

public:
    static void AddScheme(NDrive::TScheme& result, const NDrive::IServer* server);

    TMaybe<double> CalcScore(const NDrive::TOfferFeatures& features, TCategoryScoringContext& context, const NDrive::IServer* server, const TCommonCategoryScoreThreshold* threshold) const;

    void SerializeToJson(NJson::TJsonValue& result) const {
        if (!result.Has("ModelName")) {
            TJsonProcessor::Write(result, "ModelName", ModelName);
        }
        TJsonProcessor::Write(result, "Dimension", Dimension);
        TJsonProcessor::Write(result, "FixValue", FixValue);
    }

    bool DeserializeFromJson(const NJson::TJsonValue& value) {
        NJson::TJsonValue result(NJson::JSON_MAP);
        if (!TJsonProcessor::Read(value, "ModelName", ModelName)) {
            return false;
        }
        if (!TJsonProcessor::Read(value, "Dimension", Dimension)) {
            return false;
        }
        if (!TJsonProcessor::Read(value, "FixValue", FixValue)) {
            return false;
        }
        return true;
    }

    template <class TProto>
    void SerializeToProto(TProto& proto) const {
        proto.SetModelName(ModelName);
        proto.SetDimension(Dimension);
        if (FixValue) {
            proto.SetFixValue(*FixValue);
        }
    }

    template <class TProto>
    bool DeserializeFromProto(const TProto& proto) {
        ModelName = proto.GetModelName();
        Dimension = proto.GetDimension();
        if (proto.HasFixValue()) {
            FixValue = proto.GetFixValue();
        }
        return true;
    }
};

class TCommonDestinationDetector: public TCommonCategoryScoreCalcer, public TCommonCategoryScoreThreshold {
public:
    using TContext = TCategoryScoringContext;

public:
    const TString& GetModelName() const {
        return TCommonCategoryScoreCalcer::GetModelName();
    }

    static void AddScheme(NDrive::TScheme& result, const NDrive::IServer* server);

    bool DeserializeFromJson(const NJson::TJsonValue& value);
    void SerializeToJson(NJson::TJsonValue& result) const;

    template <class TProto>
    void SerializeToProto(TProto& proto) const {
        TCommonCategoryScoreCalcer::SerializeToProto(proto);
        TCommonCategoryScoreThreshold::SerializeToProto(proto);
    }

    template <class TProto>
    bool DeserializeFromProto(const TProto& proto) {
        return TCommonCategoryScoreCalcer::DeserializeFromProto(proto) && TCommonCategoryScoreThreshold::DeserializeFromProto(proto);
    }
};

class TDestinationDetectorTag: public INativeSerializationTag<NDrive::NProto::TDestinationDetectorTag>, public TCommonDestinationDetector {
private:
    using TBase = INativeSerializationTag<NDrive::NProto::TDestinationDetectorTag>;

public:
    using TContext = TCommonDestinationDetector::TContext;

private:
    static TFactory::TRegistrator<TDestinationDetectorTag> Registrator;

public:
    R_FIELD(TTagsFilter, StartZoneAttributesFilter);

public:
    using TBase::TBase;

    static const TString TypeName;

    TDestinationDetectorTag() = default;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Area};
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual TProto DoSerializeSpecialDataToProto() const override {
        NDrive::NProto::TDestinationDetectorTag proto;
        TCommonDestinationDetector::SerializeToProto(proto);
        if (!StartZoneAttributesFilter.IsEmpty()) {
            proto.SetStartZoneAttributesFilter(StartZoneAttributesFilter.ToString());
        }
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
            return false;
        }
        if (!TCommonDestinationDetector::DeserializeFromProto(proto)) {
            return false;
        }
        if (proto.HasStartZoneAttributesFilter() && !StartZoneAttributesFilter.DeserializeFromString(proto.GetStartZoneAttributesFilter())) {
            return false;
        }
        return true;
    }

    NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
};

class TSurgeAreaTag: public INativeSerializationTag<NDrive::NProto::TSurgeAreaTag> {
private:
    using TBase = INativeSerializationTag<NDrive::NProto::TSurgeAreaTag>;
    static TFactory::TRegistrator<TSurgeAreaTag> Registrator;

public:
    virtual TProto DoSerializeSpecialDataToProto() const override {
        NDrive::NProto::TSurgeAreaTag proto;
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
            return false;
        }
        return true;
    }

    NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        return result;
    }
};

class TDropAreaFeatures: public INativeSerializationTag<NDrive::NProto::TDropAreaFeaturesTag> {
private:
    using TBase = INativeSerializationTag<NDrive::NProto::TDropAreaFeaturesTag>;

private:
    static TFactory::TRegistrator<TDropAreaFeatures> Registrator;

public:
    R_FIELD(bool, DefaultDropAllowance, false);
    R_FIELD(TVector<TAreaDropPolicyBuilder>, DropPolicies);

public:
    static const TString TypeName;

    using TBase::TBase;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Area};
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    TMaybe<TOfferDropPolicy> BuildFeeBySession(const IOffer* offer, const NDrive::IServer* server, const TString& areaId, const bool isInternalFlow) const;
    TMaybe<TOfferDropPolicy> BuildFeeBySession(const TSet<TString>* offerTags, const NDrive::IServer* server, const TString& areaId, bool isInternalFlow) const;
    TMaybe<TOfferDropPolicy> GetFeeBySession(const IOffer* offer, const NDrive::IServer* server, const ui32 fee, const bool isInternalFlow) const;

    virtual TProto DoSerializeSpecialDataToProto() const override {
        NDrive::NProto::TDropAreaFeaturesTag proto = TBase::DoSerializeSpecialDataToProto();
        for (auto&& i : DropPolicies) {
            *proto.AddDropPolicies() = i.SerializeToProto();
        }
        proto.SetDefaultDropAllowance(DefaultDropAllowance);
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
            return false;
        }
        DefaultDropAllowance = proto.GetDefaultDropAllowance();
        for (auto&& i : proto.GetDropPolicies()) {
            TAreaDropPolicyBuilder dropPolicyBuilder;
            if (!dropPolicyBuilder.DeserializeFromProto(i)) {
                return false;
            }
            DropPolicies.emplace_back(dropPolicyBuilder);
        }
        return true;
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
};

class TFixPointCorrectionTag: public INativeSerializationTag<NDrive::NProto::TFixPointCorrectionTag> {
private:
    using TBase = INativeSerializationTag<NDrive::NProto::TFixPointCorrectionTag>;

public:
    struct TLinkedAreaInfo {
        TArea Area;
        TMaybe<ui64> CarLimit;
    };
    using TOptionalLinkedAreaInfo = TMaybe<TLinkedAreaInfo>;

public:
    R_FIELD(TString, LinkType);
    R_FIELD(TString, LinkArea);
    R_FIELD(TString, LinkPublicArea);
    R_FIELD(TVector<TGeoCoord>, Area);
    R_FIELD(TVector<TGeoCoord>, PublicArea);
    R_OPTIONAL(TDuration, WalkingDuration);
    R_OPTIONAL(ui64, CarLimit);

private:
    static TFactory::TRegistrator<TFixPointCorrectionTag> Registrator;

public:
    static const TString TypeName;

public:
    static TOptionalLinkedAreaInfo GetLinkedArea(const TGeoCoord& c, const NDrive::IServer& server, const TInstant actuality = TInstant::Zero());

public:
    using TBase::TBase;

    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Area};
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }
};

class TAreaPotentialTag: public ISerializableTag<NDrive::NProto::TAreaPotentialTag> {
public:
    class TInfo {
    public:
        R_READONLY(int, DiscountPercent, 0);
        R_FIELD(NJson::TJsonValue, StyleInfo, NJson::JSON_NULL);
        R_READONLY(TPolyLine<TGeoCoord>, Border);
        R_READONLY(TGeoCoord, Center);
        R_READONLY(TString, Name);
        R_READONLY(ui32, Priority, 0);
        R_READONLY(bool, Public, false);
        R_READONLY(TGeoCoord, HintPosition);
        mutable TMaybe<double> AreaSize;

    public:
        double GetAreaSize() const {
            if (!AreaSize) {
                AreaSize = Border.GetArea();
            }
            return *AreaSize;
        }

        TString GetDiscountInfo() const {
            if (DiscountPercent > 0) {
                return "-" + ::ToString(DiscountPercent) + "%";
            } else if (DiscountPercent < 0) {
                return "+" + ::ToString(-DiscountPercent) + "%";
            } else {
                return "0%";
            }
        }

        TInfo(const TArea& area, const TAreaPotentialTag& potential, int discountPercent);
    };
    using TInfos = TVector<TInfo>;

private:
    using TBase = ISerializableTag<NDrive::NProto::TAreaPotentialTag>;

private:
    R_FIELD(TString, PublicName);
    R_FIELD(TSet<TString>, OffersAvailable);
    R_FIELD(ui32, PotentialPriority, 0);
    R_FIELD(bool, Public, false);
    THolder<ILinearScheme> PotentialScheme = MakeHolder<TLinearScheme>();

private:
    virtual TProto DoSerializeSpecialDataToProto() const override {
        NDrive::NProto::TAreaPotentialTag proto = TBase::DoSerializeSpecialDataToProto();
        proto.SetPublicName(PublicName);
        proto.SetPotential(GetPotential(0));
        *proto.MutablePotentialLinearScheme() = PotentialScheme->SerializeToProto();
        proto.SetPotentialPriority(PotentialPriority);
        proto.SetPublic(Public);
        for (auto&& i : OffersAvailable) {
            proto.AddOffersAvailable(i);
        }
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
            return false;
        }
        PublicName = proto.GetPublicName();
        PotentialPriority = proto.GetPotentialPriority();
        if (proto.HasPotentialLinearScheme()) {
            TLinearScheme ls;
            if (!ls.DeserializeFromProto(proto.GetPotentialLinearScheme())) {
                return false;
            }
            PotentialScheme.Reset(new TLinearScheme(std::move(ls)));
        } else {
            PotentialScheme.Reset(new TLinearScheme(proto.GetPotential()));
        }
        Public = proto.GetPublic();
        for (auto&& i : proto.GetOffersAvailable()) {
            OffersAvailable.emplace(i);
        }
        if (PotentialScheme->GetMinValue().GetOrElse(0) < -100) {
            return false;
        }
        return true;
    }

    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override {
        TBase::SerializeSpecialDataToJson(json);
        JWRITE(json, "public_name", PublicName);
        JWRITE(json, "potential", GetPotential(0));
        JWRITE(json, "potential_scheme", PotentialScheme->SerializeToJson());
        JWRITE(json, "potential_priority", PotentialPriority);
        JWRITE(json, "public", Public);
        TJsonProcessor::WriteContainerArray(json, "offers_available", OffersAvailable);
    }

    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
        if (!TBase::DoSpecialDataFromJson(json, errors)) {
            return false;
        }
        JREAD_STRING_OPT(json, "public_name", PublicName);
        if (json.Has("potential_scheme")) {
            TLinearScheme ls;
            if (!ls.DeserializeFromJson(json["potential_scheme"])) {
                return false;
            }
            PotentialScheme.Reset(new TLinearScheme(std::move(ls)));
        } else {
            double p = 0;
            JREAD_DOUBLE_OPT(json, "potential", p);
            PotentialScheme.Reset(new TLinearScheme(p));
        }
        TJsonProcessor::ReadContainer(json, "offers_available", OffersAvailable);
        JREAD_UINT_OPT(json, "potential_priority", PotentialPriority);
        JREAD_BOOL_OPT(json, "public", Public);
        if (PotentialScheme->GetMinValue().GetOrElse(0) < -100) {
            return false;
        }
        return true;
    }

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TAreaPotentialTag> Registrator;

public:
    using TBase::TBase;

    TLinearScheme& MutablePotentialSchemeTestOnly() {
        TLinearScheme* potential = dynamic_cast<TLinearScheme*>(PotentialScheme.Get());
        CHECK_WITH_LOG(potential);
        return *potential;
    }

    double GetPotential(const ui32 carsCount) const {
        return PotentialScheme ? PotentialScheme->GetValue(carsCount).GetOrElse(0) : 0;
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Area};
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

public:
    static double CalcPriceKff(const double fromPotential, const double toPotential);
    static double CalcPriceKff(const NDrive::IServer& server, const TGeoCoord& from, const TGeoCoord& to, const TString& offerName, const double precision, TInstant actuality = TInstant::Zero());
    static double CalcPotential(const NDrive::IServer& server, const TGeoCoord& coordinate, const TString& offerName, const TInstant actuality = TInstant::Zero(), const TInternalPointContext internalContext = TInternalPointContext());
    static TInfos CalcInfos(const NDrive::IServer& server, const TGeoCoord& from, const TString& offerName, TInstant actuality = TInstant::Zero());
};

class TAreaInfoForUser: public ISerializableTag<NDrive::NProto::TAreaInfoForUser> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TAreaInfoForUser>;
    using TKVMap = TMap<TString, TString>;

private:
    R_FIELD(TKVMap, Infos);

private:
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TAreaInfoForUser> Registrator;

public:
    using TBase::TBase;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Area};
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }
};

class TClusterTag: public IJsonSerializableTag {
public:
    using TBase = IJsonSerializableTag;

public:
    static const TString TypeName;

public:
    using TBase::TBase;

    inline ui64 GetThreshold() const {
        return Threshold;
    }
    inline void SetThreshold(ui64 value) {
        Threshold = value;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Area};
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

private:
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& value) const override;

private:
    ui64 Threshold = 0;

private:
    static TFactory::TRegistrator<TClusterTag> Registrator;
};

class TFeedbackButtonTag: public IJsonSerializableTag {
public:
    using TBase = IJsonSerializableTag;

    struct TValue {
        TString Code;
        TString Icon;
        TString Title;
        TString Link;
        TString Deeplink;
        TString Description;
        TString DetailedDescription;
        i32 Priority = 0;
    };

public:
    static const TString TypeName;

public:
    using TBase::TBase;

    const TValue& GetValue() const {
        return Value;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Area};
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

private:
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& value) const override;

private:
    TValue Value;

private:
    static TFactory::TRegistrator<TFeedbackButtonTag> Registrator;
};

class TServiceDelegationPossibilityTag: public IJsonSerializableTag {
public:
    using TBase = IJsonSerializableTag;

public:
    static const TString TypeName;

public:
    using TBase::TBase;

    class TDescription: public TTagDescription {
    private:
        static TFactory::TRegistrator<TDescription> Registrator;
    };

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Area};
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

private:
    static TFactory::TRegistrator<TServiceDelegationPossibilityTag> Registrator;
};

class TTransportationTag: public ISerializableTag<NDrive::NProto::TTransportationTag> {
public:
    struct TFirstMileReportOptions {
    public:
        void SetBeaconsInfo(const ISettings& settings, const TRTDeviceSnapshot& snapshot);

    public:
        TGeoCoord Location;
        TString Description;
        TString Geocoded;
        TDuration WalkingDuration = TDuration::Zero();
        IRequestProcessor::EApp App = IRequestProcessor::EApp::Unknown;
        ELocalization Locale = DefaultLocale;
        TMaybe<TGeoCoord> BeaconLocation;
        TSet<TString> BeaconLocationMethodNames{"oko_parking"};
    };
    struct TMethod {
        TString FallbackId;
        TString Id;
        TString Icon;
        TString Title;
        TString Link;
        TString AppStoreLink;
        TString PlayMarketLink;
        TString PlayMarketPackage;
        TString DeepLinkTemplate;
        TDuration MinimalWalkingDuration = TDuration::Zero();
        ui32 Priority = Max<ui16>();
        bool CalculateCost = false;
        bool Required = false;

        TString PrivateKeyData;
        NOpenssl::TEVP_PKEYPtr PrivateKey;
    };

    using TBase = ISerializableTag<NDrive::NProto::TTransportationTag>;

public:
    static NJson::TJsonValue GetFirstMileReport(const TFirstMileReportOptions& options, const NDrive::IServer& server);
    static NJson::TJsonValue GetMethodReport(const TFirstMileReportOptions& options, const TMethod& method);

public:
    using TBase::TBase;

    const TVector<TMethod>& GetMethods() const {
        return Methods;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Area};
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

private:
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;

private:
    TVector<TMethod> Methods;

private:
    static TFactory::TRegistrator<TTransportationTag> Registrator;
};

class TStandRegistrationPromoTag: public ISerializableTag<NDrive::NProto::TStandRegistrationPromoTag> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TStandRegistrationPromoTag>;

private:
    R_FIELD(TInstant, StartInstant);
    R_FIELD(TInstant, FinishInstant);

private:
    virtual TProto DoSerializeSpecialDataToProto() const override {
        NDrive::NProto::TStandRegistrationPromoTag proto = TBase::DoSerializeSpecialDataToProto();
        proto.SetStartInstant(StartInstant.Seconds());
        proto.SetFinishInstant(FinishInstant.Seconds());
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
            return false;
        }
        StartInstant = TInstant::Seconds(proto.GetStartInstant());
        FinishInstant = TInstant::Seconds(proto.GetFinishInstant());
        return true;
    }

    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override {
        TBase::SerializeSpecialDataToJson(json);
        JWRITE_INSTANT(json, "start_instant", StartInstant);
        JWRITE_INSTANT(json, "finish_instant", FinishInstant);
    }

    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
        if (!TBase::DoSpecialDataFromJson(json, errors)) {
            return false;
        }
        JREAD_INSTANT(json, "start_instant", StartInstant);
        JREAD_INSTANT(json, "finish_instant", FinishInstant);
        return true;
    }

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TStandRegistrationPromoTag> Registrator;

public:
    class TDescription: public TTagDescription {
    private:
        using TBase = TTagDescription;

    private:
        R_FIELD(TString, MarkupTagName);
        R_FIELD(TString, BonusTagName);
        R_FIELD(ui32, BonusAmount, 0);

    private:
        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
            NDrive::TScheme result = TTagDescription::GetScheme(server);
            result.Add<TFSString>("markup_tag_name", "Тег для помечания поучаствовавших").SetRequired(true);
            result.Add<TFSString>("bonus_tag_name", "Тег для начисления бонусов").SetRequired(true);
            result.Add<TFSNumeric>("bonus_amount", "Кол-во бонусов в копейках").SetRequired(true);
            return result;
        }

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override {
            NJson::TJsonValue jsonMeta = TBase::DoSerializeMetaToJson();
            JWRITE(jsonMeta, "markup_tag_name", MarkupTagName);
            JWRITE(jsonMeta, "bonus_tag_name", BonusTagName);
            JWRITE(jsonMeta, "bonus_amount", BonusAmount);
            return jsonMeta;
        }

        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override {
            JREAD_STRING(jsonMeta, "markup_tag_name", MarkupTagName);
            JREAD_STRING(jsonMeta, "bonus_tag_name", BonusTagName);
            JREAD_UINT(jsonMeta, "bonus_amount", BonusAmount);
            return TBase::DoDeserializeMetaFromJson(jsonMeta);
        }
    };

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        result.Add<TFSNumeric>("start_instant", "Начало акции, UTC Timestamp").SetRequired(true);
        result.Add<TFSNumeric>("finish_instant", "Конец акции, UTC Timestamp").SetRequired(true);
        return result;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Area};
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    const TStandRegistrationPromoTag::TDescription* GetDescription(const ITagsMeta& tagsMeta) const {
        return tagsMeta.GetDescriptionByName(GetName())->GetAs<TStandRegistrationPromoTag::TDescription>();
    }

    TString GetBonusTagName(const ITagsMeta& tagsMeta) const {
        auto description = GetDescription(tagsMeta);
        return !!description ? description->GetBonusTagName() : "";
    }

    TString GetMarkupTagName(const ITagsMeta& tagsMeta) const {
        auto description = GetDescription(tagsMeta);
        return !!description ? description->GetMarkupTagName() : "";
    }

    ui32 GetBonusAmount(const ITagsMeta& tagsMeta) const {
        auto description = GetDescription(tagsMeta);
        return !!description ? description->GetBonusAmount() : 0;
    }
};

class TSimpleAreaTag: public IJsonSerializableTag {
public:
    using TBase = IJsonSerializableTag;

public:
    static inline const TString TypeName{"simple_area_tag"};

public:
    using TBase::TBase;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Area };
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

private:
    static inline TFactory::TRegistrator<TSimpleAreaTag> Registrator{TypeName};
};

class TSpeedLimitCorrectionAreaTag: public ISerializableTag<NDrive::NProto::TSpeedLimitCorrectionAreaTag> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TSpeedLimitCorrectionAreaTag> ;

public:
    R_FIELD(double, CustomSpeedLimit, 0); // m/s

public:
    static TFactory::TRegistrator<TSpeedLimitCorrectionAreaTag> Registrator;

public:
    using TBase::TBase;

    static const TString TypeName;

    TProto DoSerializeSpecialDataToProto() const override;
    bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;

    NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Area};
    }

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    static void CorrectSpeedLimitRanges(NDrive::TTracksLinker::TResults& results, const NDrive::IServer& server);
    static void CorrectSpeedLimitRange(NDrive::TTracksLinker::TResult& result, const NDrive::IServer& server);
};

class TStandardWithDiscountAreaOfferAreaTag: public ISerializableTag<NDrive::NProto::TStandardWithDiscountAreaOfferAreaTag> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TStandardWithDiscountAreaOfferAreaTag> ;

public:
    R_FIELD(double, Discount, 0);

public:
    static TFactory::TRegistrator<TStandardWithDiscountAreaOfferAreaTag> Registrator;

public:
    using TBase::TBase;

    static const TString TypeName;

    TProto DoSerializeSpecialDataToProto() const override;
    bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;

    NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Area};
    }

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }
};
