#include "billing_tags.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/types/uuid.h>

NDrive::TScheme TBillingTagDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSArray>("user_states", "Допустимые статусы пользователя").SetElement<TFSString>();
    result.Add<TFSVariants>("action", "Действие").InitVariants<EAction>();
    result.Add<TFSVariants>("terminal", "Тип списания (или терминал в Trust)").InitVariants<EBillingType>();
    result.Add<TFSNumeric>("limit", "Ограничение на размер транзакции (для тегов с произвольной суммой)").SetVisual(TFSNumeric::EVisualType::Money);
    result.Add<TFSVariants>("billing_queue", "Окружение для обработки тега").InitVariants<EBillingQueue>();
    result.Add<TFSDuration>("amount_lifetime", "Время жизни суммы");
    result.Add<TFSVariants>("lifetime_policy", "Политика сгорания").InitVariants<EDeadlinePolicy>().SetDefault(::ToString(EDeadlinePolicy::Day));
    result.Add<TFSNumeric>("amount_deadline", "Время сгорания").SetVisual(TFSNumeric::EVisualType::DateTime);
    result.Add<TFSDuration>("timezone", "Сдвиг временной зоны пользователя");
    result.Add<TFSString>("amount_description", "Цель выдачи");

    result.Add<TFSVariants>("accounts", "Доступные для списания аккаунты").SetMultiSelect(true).SetReference("accounts");
    result.Add<TFSBoolean>("remove_on_error", "Удалять при невозможности операции").SetDefault(false);
    result.Add<TFSString>("topic_link", "Идентификатор чата");
    result.Add<TFSVariants>("message_type", "Тип сообщения").InitVariants<NDrive::NChat::IMessage::EMessageType>();
    result.Add<TFSText>("remove_message_template", "Шаблон сообщения об удалении");
    NDrive::TScheme resolutionScheme;
    resolutionScheme.Add<TFSString>("resolution", "Резолюция");
    resolutionScheme.Add<TFSString>("node", "Нода");
    result.Add<TFSArray>("move_to_node", "Перевести в ноду").SetElement(std::move(resolutionScheme));

    result.Add<TFSBoolean>("is_realtime", "Обработка при навешивании");
    result.Add<TFSBoolean>("session_required", "Не позволять навешивать без прикрепленной сессии").SetDefault(false);
    result.Add<TFSBoolean>("use_only_selected_card", "Использовать только прикрепленную карту").SetDefault(false);
    result.Add<TFSJson>("payload", "payload для траста");
    return result;
}

NJson::TJsonValue TBillingTagDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta = TBase::DoSerializeMetaToJson();
    JWRITE(jsonMeta, "action", ::ToString(Action));
    JWRITE(jsonMeta, "terminal", ::ToString(Terminal));
    JWRITE_DEF(jsonMeta, "billing_queue", ::ToString(BillingQueue), ::ToString(EBillingQueue::Active));
    JWRITE(jsonMeta, "limit", Limit);
    JWRITE_DEF(jsonMeta, "remove_on_error", RemoveOnProcessingError, false);
    JWRITE_DEF(jsonMeta, "session_required", SessionRequired, false);

    NJson::TJsonValue accountsJson(NJson::JSON_ARRAY);
    for (auto&& i : AvailableAccounts) {
        accountsJson.AppendValue(i);
    }
    jsonMeta["accounts"] = accountsJson;

    NJson::TJsonValue statesJson(NJson::JSON_ARRAY);
    for (auto&& i : UserStates) {
        statesJson.AppendValue(i);
    }
    jsonMeta["user_states"] = statesJson;

    TJsonProcessor::WriteDurationString(jsonMeta, "amount_lifetime", AmountLifetime, TDuration::Max());
    JWRITE_ENUM(jsonMeta, "lifetime_policy", LifetimePolicy);
    TJsonProcessor::WriteDurationString(jsonMeta, "timezone", Timezone);
    JWRITE(jsonMeta, "amount_description", AmountDescription);
    TJsonProcessor::WriteInstant(jsonMeta, "amount_deadline", AmountDeadline, TInstant::Max());
    JWRITE(jsonMeta, "topic_link", TopicLink);
    JWRITE_ENUM(jsonMeta, "message_type", MessageType);
    JWRITE(jsonMeta, "remove_message_template", RemoveMessageTemplate);
    NJson::InsertField(jsonMeta, "move_to_node", NJson::KeyValue(MoveToStepByResolution, "resolution", "node"));
    JWRITE(jsonMeta, "is_realtime", Realtime);
    JWRITE(jsonMeta, "use_only_selected_card", UseOnlySelectedCard);
    JWRITE(jsonMeta, "payload", Payload);
    return jsonMeta;
}

bool TBillingTagDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    JREAD_FROM_STRING(jsonMeta, "action", Action);
    JREAD_FROM_STRING_OPT(jsonMeta, "terminal", Terminal);
    JREAD_FROM_STRING_OPT(jsonMeta, "billing_queue", BillingQueue);
    JREAD_BOOL_OPT(jsonMeta, "remove_on_error", RemoveOnProcessingError);
    JREAD_BOOL_OPT(jsonMeta, "session_required", SessionRequired);


    if (!jsonMeta.Has("accounts") || !jsonMeta["accounts"].IsArray()) {
        return false;
    }

    for (auto&& account : jsonMeta["accounts"].GetArraySafe()) {
        AvailableAccounts.insert(account.GetString());
    }

    if (jsonMeta.Has("user_states")) {
        if (!jsonMeta["user_states"].IsArray()) {
            return false;
        }
        for (auto&& i : jsonMeta["user_states"].GetArraySafe()) {
            UserStates.insert(i.GetStringRobust());
        }
    }

    JREAD_UINT_OPT(jsonMeta, "limit", Limit);
    if (Limit == 0) {
        return false;
    }

    JREAD_DURATION_OPT(jsonMeta, "amount_lifetime", AmountLifetime);
    if (AmountLifetime == TDuration::Zero()) {
        return false;
    }
    JREAD_FROM_STRING_OPT(jsonMeta, "lifetime_policy", LifetimePolicy);
    JREAD_DURATION_OPT(jsonMeta, "timezone", Timezone);
    JREAD_FROM_STRING_OPT(jsonMeta, "amount_description", AmountDescription);
    JREAD_INSTANT_OPT(jsonMeta, "amount_deadline", AmountDeadline);
    if (AmountDeadline == TInstant::Zero()) {
        return false;
    }

    JREAD_STRING_OPT(jsonMeta, "topic_link", TopicLink);
    JREAD_FROM_STRING_OPT(jsonMeta, "message_type", MessageType);
    JREAD_STRING_OPT(jsonMeta, "remove_message_template", RemoveMessageTemplate);
    if (!NJson::ParseField(jsonMeta, "move_to_node", NJson::KeyValue(MoveToStepByResolution, "resolution", "node"))) {
        return false;
    }
    JREAD_BOOL_OPT(jsonMeta, "is_realtime", Realtime);
    JREAD_BOOL_OPT(jsonMeta, "use_only_selected_card", UseOnlySelectedCard);

    if (GetAction() == TBillingTagDescription::EAction::Cancel) {
        SessionRequired = true;
    }

    if (jsonMeta.Has("payload")) {
        Payload = jsonMeta["payload"];
    }

    return TBase::DoDeserializeMetaFromJson(jsonMeta);
}

bool TBillingTagDescription::ProcessBillingOperation(const TString& userId, const TString& robotId, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const {
    if (AvailableAccounts.size() != 1) {
        session.SetErrorInfo("ProcessBillingOperation", "incorrect billing description " + GetName(), EDriveSessionResult::InternalError);
        return false;
    }

    auto accountDescription = server.GetDriveAPI()->GetBillingManager().GetAccountsManager().GetDescriptionByName(*AvailableAccounts.begin());
    if (!accountDescription.Defined()) {
        session.SetErrorInfo("ProcessBillingOperation", "incorrect billing account for " + GetName(), EDriveSessionResult::InternalError);
        return false;
    }
    auto account = server.GetDriveAPI()->GetBillingManager().GetAccountsManager().GetOrCreateAccount(userId, *accountDescription, robotId, session);
    if (!account) {
        session.SetErrorInfo("ProcessBillingOperation", "cannot create account for " + GetName(), EDriveSessionResult::InternalError);
        return false;
    }

    ITag::TPtr tag = ITag::TFactory::Construct(GetType());
    auto tagImpl = dynamic_cast<TBillingTag*>(tag.Get());
    if (!tagImpl) {
        session.SetErrorInfo("ProcessBillingOperation", "incorrect billing tag " + GetName(), EDriveSessionResult::InternalError);
        return false;
    }

    tag->SetName(GetName());
    tag->SetTagPriority(GetDefaultPriority());
    if (!tag->OnBeforeAdd(userId, userId, &server, session)) {
        return false;
    }
    TString operationId = NUtil::CreateUUID();
    TDBTag dbTag;
    dbTag.SetData(tag).SetObjectId(userId).SetTagId(operationId);
    if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().GetHistoryManager().AddHistory({ dbTag }, userId, EObjectHistoryAction::Add, session)) {
        return false;
    }

    switch (tagImpl->ProcessOperation(operationId, account, robotId, server, session, &chatSession)) {
    case TBillingTag::EOperationAction::Error:
        return false;
    case TBillingTag::EOperationAction::Remove:
    case TBillingTag::EOperationAction::RemoveWithMessage:
        if (tagImpl->GetResolution()) {
            if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().GetHistoryManager().AddHistory({ dbTag }, robotId, EObjectHistoryAction::UpdateData, session)) {
                return false;
            }
        }
        if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().GetHistoryManager().AddHistory({ dbTag }, robotId, EObjectHistoryAction::Remove, session)) {
            return false;
        }
        break;
    default:
        session.SetErrorInfo("ProcessBillingOperation", "incorrect operation for " + GetName(), EDriveSessionResult::InternalError);
        return false;
    }
    return true;
}

NDrive::TScheme TFixedBillingTagDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("amount", "Сумма списания/начисления (копейки)").SetVisual(TFSNumeric::EVisualType::Money);
    return result;
}

NJson::TJsonValue TFixedBillingTagDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta = TBase::DoSerializeMetaToJson();
    JWRITE(jsonMeta, "amount", Amount);
    return jsonMeta;
}

bool TFixedBillingTagDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    JREAD_UINT(jsonMeta, "amount", Amount);
    if (Amount == 0) {
        return false;
    }
    return TBase::DoDeserializeMetaFromJson(jsonMeta);
}

NDrive::TScheme TBillingTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("active_since", "Время активации").SetVisual(TFSNumeric::EVisualType::DateTime);
    result.Add<TFSNumeric>("operation_ts", "Время операции для биллинга").SetVisual(TFSNumeric::EVisualType::DateTime);
    result.Add<TFSString>("preferred_card", "Предпочитаемая карта для списания (только для инициаторов списаний)");
    result.Add<TFSNumeric>("amount_deadline", "Время сгорания").SetVisual(TFSNumeric::EVisualType::DateTime).SetReadOnly(true);
    result.Add<TFSVariants>("additional_accounts", "Дополнительные аккаунты для списания").SetMultiSelect(true).SetReference("accounts");
    result.Add<TFSBoolean>("is_plus_user", "Плюсовик");
    return result;
}


void TBillingTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    JWRITE_DEF(json, "resolution", Resolution, "");
    JWRITE_DEF(json, "preferred_card", PreferredCard, "");
    JWRITE_DEF(json, "amount", Amount, 0);
    TJsonProcessor::WriteInstant(json, "operation_ts", OperationTs, TInstant::Zero());
    TJsonProcessor::WriteInstant(json, "active_since", ActivateSince, TInstant::Zero());
    TJsonProcessor::WriteInstant(json, "amount_deadline", AmountDeadline, TInstant::Max());
    TJsonProcessor::WriteContainerArray(json, "additional_accounts", AdditionalAccounts);
    auto& operations = json.InsertValue("operations", NJson::JSON_ARRAY);
    for (auto&& item : Operations) {
        operations.AppendValue(item.GetReport());
    }
    if (PlusUser) {
        json["is_plus_user"] = *PlusUser;
    }
}

bool TBillingTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    JREAD_UINT_OPT(json, "amount", Amount);
    JREAD_STRING_OPT(json, "resolution", Resolution);
    JREAD_STRING_OPT(json, "preferred_card", PreferredCard);
    JREAD_INSTANT_OPT(json, "active_since", ActivateSince);
    JREAD_INSTANT_OPT(json, "operation_ts", OperationTs);
    JREAD_INSTANT_OPT(json, "amount_deadline", AmountDeadline);
    if (!TJsonProcessor::ReadContainer(json, "additional_accounts", AdditionalAccounts)) {
        return false;
    }
    if (json.Has("operations") && Operations.empty()) {
        for (auto&& item : json["operations"].GetArray()) {
            Operations.emplace_back(item["payment_id"].GetString(), item["amount"].GetUInteger());
        }
    }
    if (json.Has("is_plus_user")) {
        PlusUser = json["is_plus_user"].GetBoolean();
    }
    return TBase::DoSpecialDataFromJson(json, errors);
}

TBillingTag::TProto TBillingTag::DoSerializeSpecialDataToProto() const {
    TBillingTag::TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.MutableBillingData()->SetActiveSince(ActivateSince.Seconds());
    proto.MutableBillingData()->SetOperationTs(OperationTs.Seconds());
    proto.MutableBillingData()->SetAmount(Amount);
    if (Resolution) {
        proto.MutableBillingData()->SetResolution(Resolution);
    }
    if (AmountDeadline != TInstant::Max()) {
        proto.MutableBillingData()->SetAmountDeadline(AmountDeadline.Seconds());
    }
    if (PreferredCard) {
        proto.MutableBillingData()->SetPreferredCard(PreferredCard);
    }
    for(const auto& account : AdditionalAccounts) {
        proto.MutableBillingData()->AddAdditionalAccounts(account);
    }
    if (PlusUser) {
        proto.MutableBillingData()->SetPlusUser(*PlusUser);
    }
    OperationsToProto(proto);
    return proto;
}

bool TBillingTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    ActivateSince = TInstant::Seconds(proto.GetBillingData().GetActiveSince());
    OperationTs = TInstant::Seconds(proto.GetBillingData().GetOperationTs());
    Amount = proto.GetBillingData().GetAmount();
    Resolution = proto.GetBillingData().GetResolution();
    if (proto.GetBillingData().HasAmountDeadline()) {
        AmountDeadline = TInstant::Seconds(proto.GetBillingData().GetAmountDeadline());
    }
    PreferredCard = proto.GetBillingData().GetPreferredCard();
    for(auto&& account : proto.GetBillingData().GetAdditionalAccounts()) {
        AdditionalAccounts.insert(account);
    }
    if (proto.GetBillingData().HasPlusUser()) {
        PlusUser = proto.GetBillingData().GetPlusUser();
    }
    return OperationsFromProto(proto) && TBase::DoDeserializeSpecialDataFromProto(proto);
}

TString TBillingTag::GetTopicLink(const NDrive::IServer& server) const {
    if (GetChatTopic()) {
        return GetChatTopic();
    }
    const TDriveAPI*  driveApi = server.GetDriveAPI();
    if (!driveApi) {
        return TString();
    }

    auto descriptionImpl = dynamic_cast<const TBillingTagDescription*>(driveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName()).Get());
    return !!descriptionImpl ? descriptionImpl->GetTopicLink() : TString();
}

bool TBillingTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const TDriveAPI* driveApi = server->GetDriveAPI();
    if (!driveApi) {
        session.SetErrorInfo(GetName(), "cannot find drive api", EDriveSessionResult::InternalError);
        return false;
    }

    auto description = dynamic_cast<const TBillingTagDescription*>(driveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName()).Get());
    if (!description || description->GetAvailableAccounts().size() != 1) {
        session.SetErrorInfo(GetName(), "Incorrect description type", EDriveSessionResult::InconsistencySystem);
        return false;
    }

    if (!description->IsRealtime()) {
        return true;
    }

    auto accountDescription = driveApi->GetBillingManager().GetAccountsManager().GetDescriptionByName(*description->GetAvailableAccounts().begin());
    if (!accountDescription.Defined()) {
        session.SetErrorInfo(GetName(), "incorrect billing account for " + GetName(), EDriveSessionResult::InternalError);
        return false;
    }

    auto account = driveApi->GetBillingManager().GetAccountsManager().GetOrCreateAccount(self.GetObjectId(), *accountDescription, userId, session);
    if (!account) {
        session.SetErrorInfo(GetName(), "cannot create account for " + GetName(), EDriveSessionResult::InternalError);
        return false;
    }

    auto optionalTag = driveApi->GetTagsManager().GetUserTags().RestoreTag(self.GetTagId(), session);
    if (!optionalTag) {
        return false;
    }

    auto billingTag = optionalTag->MutableTagAs<TBillingTag>();
    if (!billingTag) {
        session.SetErrorInfo(GetName(), "incorrect tag " + GetName(), EDriveSessionResult::InternalError);
        return false;
    }

    const TString lastResolution = billingTag->GetResolution();
    switch (billingTag->ProcessOperation(self.GetTagId(), account, userId, *server, session, nullptr)) {
    case TBillingTag::EOperationAction::Error:
        return false;
    case TBillingTag::EOperationAction::Remove:
    case TBillingTag::EOperationAction::RemoveWithMessage:
        if (billingTag->GetResolution() != lastResolution) {
            if (!driveApi->GetTagsManager().GetUserTags().UpdateTagData(*optionalTag, userId, session)) {
                return false;
            }
        }
        if (!driveApi->GetTagsManager().GetUserTags().RemoveTag(*optionalTag, userId, server, session, true)) {
            return false;
        }
        TUnistatSignalsCache::SignalAdd("realtime-remove-billing-tags", "count", 1);
        break;
    case TBillingTag::EOperationAction::Update:
        if (!driveApi->GetTagsManager().GetUserTags().UpdateTagData(*optionalTag, userId, session)) {
            return false;
        }
        TUnistatSignalsCache::SignalAdd("realtime-update-billing-tags", "count", 1);
        break;
    case TBillingTag::EOperationAction::Nothing:
        break;
    default:
        session.SetErrorInfo("ProcessBillingOperation", "incorrect operation for " + GetName(), EDriveSessionResult::InternalError);
        TUnistatSignalsCache::SignalAdd("realtime-error-billing-tags", "count", 1);
        return false;
    }
    return true;
}

bool TBillingTag::OnBeforeUpdate(const TDBTag& self, ITag::TPtr to, const TString& /*userId*/, const NDrive::IServer* /*server*/, NDrive::TEntitySession& session) const  {
    auto selfBillingTag = self.GetTagAs<TBillingTag>();
    auto toBillingTag = dynamic_cast<TBillingTag*>(to.Get());
    if (!selfBillingTag || !toBillingTag) {
        session.SetErrorInfo("OnBeforeUpdate", "incorrect tags for updating");
        return false;
    }
    for (const auto& beforeOp : selfBillingTag->Operations) {
        if (beforeOp.HasPaid()) {
            for (auto& afterOp : toBillingTag->Operations) {
                if (beforeOp.GetPaymentId() != afterOp.GetPaymentId()) {
                    continue;
                }
                if (afterOp.HasPaid() && afterOp.GetPaidRef() > beforeOp.GetPaidRef()) {
                    afterOp.MutableDiff() = afterOp.GetPaidRef() - beforeOp.GetPaidRef();
                } else {
                    afterOp.MutableDiff() = TMaybe<ui64>();
                }
            }
        } else {
            for (auto& afterOp : toBillingTag->Operations) {
                if (beforeOp.GetPaymentId() == afterOp.GetPaymentId()) {
                    afterOp.MutableDiff() = afterOp.OptionalPaid();
                }
            }
        }
    }
    return true;
}

bool TBillingTag::SendMessageAfterRemove(const TString& userId, const TString& robotId, const NDrive::IServer& server, NDrive::TEntitySession* chatSession) const {
    const TDriveAPI*  driveApi = server.GetDriveAPI();
    if (!driveApi) {
        return false;
    }
    auto descriptionImpl = dynamic_cast<const TBillingTagDescription*>(driveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName()).Get());
    if (!descriptionImpl) {
        if (chatSession) {
            chatSession->SetErrorInfo(GetName(), "Incorrect description type", EDriveSessionResult::IncorrectRequest);
        }
        return false;
    }

    auto locale = ELocalization::Rus;
    TString text = server.GetLocalization()->ApplyResources(descriptionImpl->GetRemoveMessageTemplate(), locale);

    SubstGlobal(text, "AMOUNT_DEADLINE", server.GetLocalization()->FormatInstantWithYear(locale, GetAmountDeadline() + descriptionImpl->GetTimezone()));
    SubstGlobal(text, "AMOUNT", server.GetLocalization()->FormatPrice(locale, GetAmount()));

    if (text && !SendMessage(userId, text, descriptionImpl->GetMessageType(), robotId, server, chatSession)) {
        return false;
    }
    auto nextStep = descriptionImpl->GetMoveToStepByResolution().find(GetResolution());
    if (nextStep != descriptionImpl->GetMoveToStepByResolution().end() && !MoveToStep(userId, nextStep->second, robotId, server, chatSession)) {
        return false;
    }
    return true;
}

TBillingTag::EOperationAction TBillingTag::ProcessOperation(const TString& operationId, NDrive::NBilling::IBillingAccount::TPtr account, const TString& robotId, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession* chatSession) {
    EOperationAction result = ProcessOperationImpl(operationId, account, robotId, server, session);
    if (account && result == EOperationAction::RemoveWithMessage && !SendMessageAfterRemove(account->GetUserId(), robotId, server, chatSession)) {
        return EOperationAction::Error;
    }
    return result;
}

TBillingTag::EOperationAction TBillingTag::ProcessOperationImpl(const TString& operationId, NDrive::NBilling::IBillingAccount::TPtr account, const TString& robotId, const NDrive::IServer& server, NDrive::TEntitySession& session) {
    const TDriveAPI*  driveApi = server.GetDriveAPI();
    if (!driveApi) {
        return EOperationAction::Error;
    }
    const auto& manager = driveApi->GetBillingManager();

    auto description = dynamic_cast<const TBillingTagDescription*>(driveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName()).Get());
    if (!description) {
        session.SetErrorInfo(GetName(), "Incorrect description type", EDriveSessionResult::IncorrectRequest);
        return EOperationAction::Error;
    }

    ui32 amount = GetAmount();

    if (description->GetAction() == TBillingTagDescription::EAction::Cancel) {
        if (GetResolution()) {
            return EOperationAction::RemoveWithMessage;
        }
        if (amount != 0) {
            return EOperationAction::Nothing;
        }

        auto task = manager.GetActiveTasksManager().GetTask(GetSessionId(), session);
        if (!task) {
            return EOperationAction::Nothing;
        }
        if (!task.GetRef()) {
            SetResolution(ToString(EOperationResolution::NoTask));
            return EOperationAction::Update;
        }
        if (!task->IsFinished()) {
            SetResolution(ToString(EOperationResolution::NotFinished));
            return EOperationAction::Update;
        }

        if (manager.GetActiveTasksManager().CancelTask(*task, robotId, session, amount)) {
            SetResolution(ToString(EOperationResolution::Canceled));
            return EOperationAction::Update;
        }

        return EOperationAction::Nothing;
    }

    if (!account) {
        return EOperationAction::Error;
    }

    bool isTransactionOperation = (account->GetType() == NDrive::NBilling::EAccount::Trust ||
                                  (account->GetType() == NDrive::NBilling::EAccount::Wallet && account->IsSelectable()));

    if (!isTransactionOperation) {
        // Skip empty bonuses (old)
        if (account->GetId() == 0 && account->GetBalance() == 0) {
            WARNING_LOG << "Skip old for " << account->GetUserId() << Endl;
            return EOperationAction::Nothing;
        }

        EDriveOpResult result = EDriveOpResult::TransactionError;
        switch (description->GetAction()) {
        case TBillingTagDescription::EAction::Debit:
            result = account->Remove(amount, session);
            break;
        case TBillingTagDescription::EAction::Credit:
            if (account->IsLimitedPolicy()) {
                result = account->Add(amount, session, GetAmountDeadline(), description->GetName());
            } else {
                result = account->Add(amount, session);
            }
            break;
        default:
            break;
        }
        if (result == EDriveOpResult::Ok) {
            SetResolution(ToString(EOperationResolution::Finished));
            return EOperationAction::RemoveWithMessage;
        }
        if (description->GetRemoveOnProcessingError() && result == EDriveOpResult::LogicError) {
            return EOperationAction::Remove;
        }
    } else {
        if (GetOperations().empty()) {
            TBillingTask billingTask;
            billingTask.SetId(operationId).SetUserId(account->GetUserId()).SetBill(0);
            TSet<TString> accounts = AdditionalAccounts;
            accounts.insert(description->GetAvailableAccounts().begin(), description->GetAvailableAccounts().end());
            billingTask.SetChargableAccounts(accounts);
            billingTask.SetQueue(description->GetBillingQueue());
            billingTask.SetNextQueue(description->GetBillingQueue());
            billingTask.SetRealSessionId(GetSessionId());
            billingTask.SetComment(GetComment());
            if (PlusUser) {
                billingTask.SetIsPlusUser(*PlusUser);
            }
            if (description->GetAction() == TBillingTagDescription::EAction::Debit) {
                billingTask.SetBill(amount);
                billingTask.SetBillingType(description->GetTerminal());
                if (PreferredCard) {
                    TBillingTask::TExecContext context;
                    context.SetPaymethod(PreferredCard);
                    billingTask.SetExecContext(context);
                }
                billingTask.SetUseOnlySelectedPaymethod(description->GetUseOnlySelectedCard());

                if (!manager.CreateBillingTask(billingTask, robotId, session)) {
                    return EOperationAction::Nothing;
                }
                if (!manager.FinishingBillingTask(billingTask.GetId(), session)) {
                    return EOperationAction::Nothing;
                }
            } else if (description->GetAction() == TBillingTagDescription::EAction::Credit) {
                billingTask.SetCashback(amount);
                billingTask.SetBillingType(EBillingType::CarUsage);
                if (!TBillingGlobals::CashbackBillingTypes.contains(description->GetTerminal())) {
                    return EOperationAction::Error;
                }
                billingTask.AddCashbackInfo({ amount, description->GetPayload(), description->GetTerminal() });

                if (!manager.CreateBillingTask(billingTask, robotId, session)) {
                    return EOperationAction::Nothing;
                }
                if (!manager.FinishingBillingTask(billingTask.GetId(), session)) {
                    return EOperationAction::Nothing;
                }
                if (!manager.AddClosedBillingInfo(billingTask.GetId(), robotId, session)) {
                    return EOperationAction::Nothing;
                }
            } else {
                return EOperationAction::Error;
            }
            TBillingTag::TOperation opItem(operationId, amount);
            SetOperations({ opItem });
            return EOperationAction::Update;
        } else {
            if (GetResolution()) {
                return EOperationAction::RemoveWithMessage;
            }

            auto optionalUserTask = manager.GetActiveTasksManager().GetTask(operationId, session);
            if (!optionalUserTask) {
                return EOperationAction::Error;
            }
            if (optionalUserTask.GetRef() && optionalUserTask->GetState() == "finishing") {
                Y_UNUSED(manager.AddClosedBillingInfo(optionalUserTask->GetId(), robotId));
                return EOperationAction::Nothing;
            }

            TCachedPayments snapshot;
            if (!manager.GetPaymentsManager().GetPayments(snapshot, operationId, session)) {
                return EOperationAction::Error;
            }

            if (snapshot.GetHeldSum() > snapshot.GetCanceledSum()) {
                bool update = false;
                for (auto&& op : Operations) {
                    if (op.GetPaymentId() != operationId) {
                        continue;
                    }
                    if (!op.HasPaid() || op.GetPaidRef() < snapshot.GetHeldSum() - snapshot.GetCanceledSum()) {
                        op.MutablePaid() = snapshot.GetHeldSum() - snapshot.GetCanceledSum();
                        update = true;
                    }
                }
                if (update) {
                    return EOperationAction::Update;
                }
            }

            if (optionalUserTask.GetRef()) {
                return EOperationAction::Nothing;
            }

            auto tasks = manager.GetHistoryManager().GetSessionHistory(operationId, {}, session);
            if (!tasks) {
                return EOperationAction::Error;
            }
            for (auto&& historyTask : Reversed(*tasks)) {
                if (historyTask.GetHistoryAction() == EObjectHistoryAction::Remove) {
                    SetResolution(historyTask.GetState());
                    return EOperationAction::Update;
                }
            }
        }
    }
    return EOperationAction::Nothing;
}

bool TBillingTag::OnBeforeAdd(const TString& objectId, const TString& /* userId */, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    const TDriveAPI* driveApi = server->GetDriveAPI();
    if (!driveApi || !driveApi->HasBillingManager()) {
        session.SetErrorInfo(GetName(), "No billing configured", EDriveSessionResult::InternalError);
        return false;
    }

    if (auto isDeleted = driveApi->GetUsersData()->IsDeletedByTags(objectId, session)) {
        if (*isDeleted) {
            session.SetErrorInfo(GetName(), "User already is deleted", EDriveSessionResult::IncorrectRequest);
            return false;
        }
    } else {
        return false;
    }

    auto description = driveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
    if (!description) {
        session.SetErrorInfo(GetName(), "Unknown type", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    const TBillingTagDescription* descriptionImpl = dynamic_cast<const TBillingTagDescription*>(description.Get());
    if (!descriptionImpl) {
        session.SetErrorInfo(GetName(), "Incorrect description type", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    if (descriptionImpl->GetSessionRequired() && !GetSessionId()) {
        session.SetErrorInfo(GetName(), "session_id required", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    if (descriptionImpl->GetAction() != TBillingTagDescription::EAction::Debit && !AdditionalAccounts.empty()) {
        session.SetErrorInfo(GetName(), "There are additional accounts for action " + ::ToString(descriptionImpl->GetAction()), EDriveSessionResult::IncorrectRequest);
        return false;
    }

    if (descriptionImpl->GetUseOnlySelectedCard() && !PreferredCard) {
        session.SetErrorInfo(GetName(), "Absent card", EDriveSessionResult::IncorrectRequest);
        return false;
    }
    return OnBeforeAddImpl(objectId, *descriptionImpl, session);
}

const TString TOperationTag::TypeName = "operation_tag";
ITag::TFactory::TRegistrator<TOperationTag> TOperationTag::Registrator(TOperationTag::TypeName);
TTagDescription::TFactory::TRegistrator<TBillingTagDescription> TOperationTag::RegistratorDescription(TOperationTag::TypeName);

bool TOperationTag::OnBeforeAddImpl(const TString& /*objectId*/, const TBillingTagDescription& description, NDrive::TEntitySession& session) {
    if (description.GetAction() == TBillingTagDescription::EAction::Cancel) {
        SetAmount(0);
    } else if (GetAmount() == 0) {
        session.SetErrorInfo(GetName(), "Incorrect sum", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    if (description.GetLimit() < GetAmount()) {
        session.SetErrorInfo(GetName(), "Incorrect sum", EDriveSessionResult::IncorrectRequest);
        return false;
    }
    AmountDeadline = description.GetAmountDeadline();
    return true;
}

NDrive::TScheme TOperationTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("amount", "Сумма для списания/начисления").SetVisual(TFSNumeric::EVisualType::Money);
    return result;
}

bool TOperationTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(json, errors)) {
        return false;
    }
    return true;
}

const TString TFixedSumTag::TypeName = "fixed_sum_tag";

bool TFixedSumTag::OnBeforeAddImpl(const TString& /*objectId*/, const TBillingTagDescription& description, NDrive::TEntitySession& session) {
    const TFixedBillingTagDescription* descriptionImpl = dynamic_cast<const TFixedBillingTagDescription*>(&description);
    if (!descriptionImpl) {
        session.SetErrorInfo(GetName(), "Incorrect description type", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    Amount = descriptionImpl->GetAmount();
    AmountDeadline = descriptionImpl->GetAmountDeadline();
    return true;
}

ITag::TFactory::TRegistrator<TFixedSumTag> TFixedSumTag::Registrator(TFixedSumTag::TypeName);
TTagDescription::TFactory::TRegistrator<TFixedBillingTagDescription> TFixedSumTag::RegistratorDescription(TFixedSumTag::TypeName);


const TString TRefundTag::TypeName = "refund_tag";
ITag::TFactory::TRegistrator<TRefundTag> TRefundTag::Registrator(TRefundTag::TypeName);
TTagDescription::TFactory::TRegistrator<TRefundTagDescription> TRefundTag::RegistratorDescription(TRefundTag::TypeName);

NDrive::TScheme TRefundTagDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("limit", "Limit").SetVisual(TFSNumeric::EVisualType::Money);
    return result;
}

NJson::TJsonValue TRefundTagDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta = TBase::DoSerializeMetaToJson();
    JWRITE(jsonMeta, "limit", Limit);
    return jsonMeta;
}

bool TRefundTagDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    JREAD_UINT_OPT(jsonMeta, "limit", Limit);
    return TBase::DoDeserializeMetaFromJson(jsonMeta);
}


NDrive::TScheme TRefundTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("compensation", "Размер компенсации бонусами").SetDefault(0);
    result.Add<TFSNumeric>("amount", "Сумма к возврату").SetVisual(TFSNumeric::EVisualType::Money);
    return result;
}

void TRefundTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    JWRITE(json, "compensation", BonusCompensation);
    JWRITE(json, "amount", Amount);
    NJson::TJsonValue& refunds = json.InsertValue("refunds", NJson::JSON_ARRAY);
    for (auto&& item : Operations) {
        refunds.AppendValue(item.GetReport());
    }
    TBase::SerializeSpecialDataToJson(json);
}

bool TRefundTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    JREAD_UINT(json, "compensation", BonusCompensation);
    JREAD_UINT(json, "amount", Amount);
    if (Amount == 0) {
        errors->AddMessage("opration_tag", "Invalid amount");
        return false;
    }
    return TBase::DoSpecialDataFromJson(json, errors);
}

TRefundTag::TProto TRefundTag::DoSerializeSpecialDataToProto() const {
    auto proto = TBase::DoSerializeSpecialDataToProto();
    proto.MutableBillingData()->MutableRefundData()->SetCompensation(BonusCompensation);
    proto.MutableBillingData()->SetAmount(Amount);
    OperationsToProto(proto);
    return proto;
}

bool TRefundTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (!proto.GetBillingData().HasRefundData()) {
        return false;
    }
    Amount = proto.GetBillingData().GetAmount();
    BonusCompensation = proto.GetBillingData().GetRefundData().GetCompensation();
    return OperationsFromProto(proto) && TBase::DoDeserializeSpecialDataFromProto(proto);
}

bool TRefundTag::OnBeforeAdd(const TString& objectId, const TString& /*userId*/, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (!GetSessionId()) {
        session.SetErrorInfo(GetName(), "session_id required", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    if (!GetComment()) {
        session.SetErrorInfo(GetName(), "comment required", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    const TDriveAPI* driveApi = server->GetDriveAPI();
    if (!driveApi || !driveApi->HasBillingManager()) {
        session.SetErrorInfo(GetName(), "No billing configured", EDriveSessionResult::InternalError);
        return false;
    }

    auto description = driveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
    if (!description) {
        session.SetErrorInfo(GetName(), "Unknown type", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    const TBillingManager& billingManager = driveApi->GetBillingManager();
    auto activeTask = billingManager.GetActiveTasksManager().GetTask(GetSessionId(), session);
    if (!activeTask) {
        return false;
    }
    if (activeTask.GetRef()) {
        session.SetErrorInfo(GetName(), "Can't refund not closed task", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    const TRefundTagDescription* descriptionImpl = dynamic_cast<const TRefundTagDescription*>(description.Get());
    if (!description) {
        session.SetErrorInfo(GetName(), "Incorrect description type", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    if (descriptionImpl->GetLimit() < BonusCompensation) {
        session.SetErrorInfo(GetName(), "Incorrect sum", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    const TPaymentsManager& paymentsManager = billingManager.GetPaymentsManager();
    TCachedPayments snapshot;
    if (!paymentsManager.GetPayments(snapshot, GetSessionId(), session)) {
        session.SetErrorInfo(GetName(), "Can't restore payments", EDriveSessionResult::InternalError);
        return false;
    }

    TVector<TRefundTask> refunds;
    if (!paymentsManager.GetRefundsDB().GetSessionRefunds(GetSessionId(), refunds, session)) {
        session.SetErrorInfo(GetName(), "Can't restore refunds", EDriveSessionResult::InternalError);
        return false;
    }

    auto issues =  snapshot.CalculateRefunds(GetAmount(), refunds, billingManager.GetAccountsManager().GetSettings().GetValue<ui64>("billing.card_pay_min_sum").GetOrElse(0));
    if (!issues) {
        session.SetErrorInfo(GetName(), "Incorrect refund sum: " + issues.GetError(), EDriveSessionResult::IncorrectRequest);
        return false;
    }

    if (issues.GetRef().empty()) {
        session.SetErrorInfo(GetName(), "Nothing to refund", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    for (auto issue : issues.GetRef()) {
        if (!driveApi->GetBillingManager().ProcessFiscalRefundIssue(issue.GetPayment(), objectId, issue.GetSum(), true, session)) {
            session.SetErrorInfo(GetName(), "Can't process refund", EDriveSessionResult::InternalError);
            return false;
        }
        TOperation op(issue.GetPayment().GetPaymentId(), issue.GetSum());
        Operations.push_back(op);
    }

    return true;
}

bool TDeferredSessionRefundTag::Exercise(const TDBTag& tag, const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& session) {
    const auto api = Checked(server.GetDriveAPI());
    const TBillingManager& billingManager = api->GetBillingManager();
    const TTraceTagsManager& traceTagsManager = api->GetTagsManager().GetTraceTags();
    const TUserTagsManager& userTagsManager = api->GetTagsManager().GetUserTags();

    const auto& sessionId = tag.GetObjectId();
    auto bill = billingManager.GetCompiledBills().GetFullBillFromDB(sessionId, session);
    if (!bill || !bill->GetFinal()) {
        session.AddErrorMessage("DeferredSessionRefundTag::Execute", "cannot get CompiledBill for " + sessionId);
        return false;
    }

    auto refundTag = MakeHolder<TRefundTag>();
    refundTag->SetName(TRefundTag::TypeName);
    refundTag->SetAmount(bill->GetBill());
    refundTag->SetComment(TStringBuilder() << tag->GetName() << ':' << tag.GetTagId() << ':' << userId);
    refundTag->SetSessionId(sessionId);
    if (!userTagsManager.AddTag(std::move(refundTag), userId, bill->GetUserId(), &server, session)) {
        return false;
    }
    if (!traceTagsManager.RemoveTag(tag, userId, &server, session)) {
        return false;
    }
    return true;
}

ITag::TFactory::TRegistrator<TDeferredSessionRefundTag> TDeferredSessionRefundTag::Registrator(TDeferredSessionRefundTag::Type());

const TString TWalletAccessTag::TypeName = "wallet_access_tag";
ITag::TFactory::TRegistrator<TWalletAccessTag> TWalletAccessTag::Registrator(TWalletAccessTag::TypeName);
TTagDescription::TFactory::TRegistrator<TWalletAccessTag::TDescription> TWalletAccessTag::TDescription::Registrator(TWalletAccessTag::TypeName);

void TWalletAccessTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    IOrganizationTag::SerializeToJson(json);
    TBase::SerializeSpecialDataToJson(json);
}

bool TWalletAccessTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    return IOrganizationTag::DeserializeFromJson(json, errors) && TBase::DoSpecialDataFromJson(json, errors);
}

NDrive::NProto::TWalletAccessTagData TWalletAccessTag::DoSerializeSpecialDataToProto() const {
    auto proto = TBase::DoSerializeSpecialDataToProto();
    IOrganizationTag::SerializeToProto(proto);
    return proto;
}

bool TWalletAccessTag::DoDeserializeSpecialDataFromProto(const NDrive::NProto::TWalletAccessTagData& proto) {
    return IOrganizationTag::DeserializeFromProto(proto) && TBase::DoDeserializeSpecialDataFromProto(proto);
}

NDrive::TScheme TWalletAccessTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    IOrganizationTag::AddScheme(result);
    return result;
}

TDBActions TWalletAccessTag::GetActions(const TConstDBTag& self, const IDriveTagsManager& tagsManager, const TRolesManager& /*rolesManager*/, bool /*getPotential*/) const {
    auto description = tagsManager.GetTagsMeta().GetDescriptionByName(GetName());
    const TWalletAccessTag::TDescription* descriptionImpl = dynamic_cast<const TWalletAccessTag::TDescription*>(description.Get());
    if (!descriptionImpl) {
        return {};
    }

    auto action = MakeHolder<TAdministrativeAction>();
    action->MutableActions() = descriptionImpl->GetActions();
    action->MutableEntities().insert(TAdministrativeAction::EEntity::B2BOrganization);
    action->MutableB2BOrganizations().insert(::ToString(GetParentId()));
    action->MutableSourceContext().SetTagId(self.GetTagId());

    TDBActions result;
    result.emplace_back(std::move(action));
    return result;
}

NDrive::TScheme TWalletAccessTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSVariants>("wallet_actions", "Действия над кошельком").InitVariants<TAdministrativeAction::EAction>().SetMultiSelect(true);
    return result;
}

NJson::TJsonValue TWalletAccessTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta(NJson::JSON_MAP);
    NJson::InsertField(jsonMeta, "wallet_actions", Actions);
    return jsonMeta;
}

bool TWalletAccessTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    return NJson::ParseField(jsonMeta, "wallet_actions", Actions);
}
