#pragma once

#include "user_tags.h"
#include "organization_tag.h"

#include <drive/backend/data/proto/tags.pb.h>

#include <drive/backend/billing/interfaces/account.h>
#include <drive/backend/billing/interfaces/entities.h>
#include <drive/backend/chat/message.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/proto/tags.pb.h>

class TBillingTagDescription: public TTagDescription {
public:
    enum EAction {
        Credit /* "credit" */ = 0,
        Debit /* "debit" */ = 1,
        Cancel /* "cancel" */ = 2,
    };

    enum EDeadlinePolicy {
        None,
        Day,
        Week,
        Month,
    };

private:
    using TFinalSteps = TMap<TString, TString>;
    R_FIELD(EAction, Action, EAction::Debit);
    R_FIELD(TSet<TString>, AvailableAccounts, {});
    R_FIELD(EBillingType, Terminal, {});
    R_READONLY(TSet<TString>, UserStates, {});
    R_READONLY(ui32, Limit, 500000);
    R_READONLY(EBillingQueue, BillingQueue, EBillingQueue::Active);
    R_READONLY(TDuration, Timezone, TDuration::Hours(3));
    R_READONLY(TString, AmountDescription);
    R_READONLY(TString, TopicLink);
    R_READONLY(NDrive::NChat::IMessage::EMessageType, MessageType, NDrive::NChat::IMessage::EMessageType::Plaintext);
    R_READONLY(TString, RemoveMessageTemplate, "AMOUNT");
    R_READONLY(TFinalSteps, MoveToStepByResolution);
    R_READONLY(bool, SessionRequired, false);
    R_FIELD(bool, RemoveOnProcessingError, false);
    R_FIELD(bool, Realtime, false);
    R_FIELD(bool, UseOnlySelectedCard, false);
    R_FIELD(NJson::TJsonValue, Payload);
    using TBase = TTagDescription;

private:
    TInstant AmountDeadline = TInstant::Max();
    TDuration AmountLifetime = TDuration::Max();
    EDeadlinePolicy LifetimePolicy = EDeadlinePolicy::None;

public:
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
    virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;

    TInstant GetAmountDeadline() const {
        if (AmountDeadline != TInstant::Max() && AmountDeadline != TInstant::Zero()) {
            return AmountDeadline;
        }
        if (AmountLifetime != TDuration::Max() && AmountLifetime != TDuration::Zero()) {
            TInstant minimalTime = ModelingNow() + AmountLifetime;
            struct tm tmNow;
            minimalTime.LocalTime(&tmNow);
            TDateTimeFields timeParts;
            switch (LifetimePolicy) {
            case EDeadlinePolicy::Month:
                timeParts.Hour = 0;
                timeParts.Day = 1;
                timeParts.Month = (tmNow.tm_mon + 1) % 12 + 1;
                timeParts.SetLooseYear((tmNow.tm_mon + 1) > 11 ? tmNow.tm_year + 1 : tmNow.tm_year);
                return timeParts.ToInstant(TInstant::Max()) - Timezone;
            case EDeadlinePolicy::Day:
                return TInstant::Days(minimalTime.Days() + 1) - Timezone;
            case EDeadlinePolicy::Week:
                return TInstant::Days(minimalTime.Days()) + TDuration::Days(7 - tmNow.tm_wday) + Timezone;
            case EDeadlinePolicy::None:
                return minimalTime;
            }
        }
        return TInstant::Max();
    }

    bool ProcessBillingOperation(const TString& userId, const TString& robotId, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const;
};

class TFixedBillingTagDescription: public TBillingTagDescription {
    R_FIELD(ui32, Amount, 0);
    using TBase = TBillingTagDescription;

public:
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
    virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
};

class TOperationsHandler {
public:
    class TOperation {
    public:
        R_READONLY(ui32, Amount, 0);
        R_READONLY(TString, PaymentId);
        R_OPTIONAL(ui64, Paid);
        R_OPTIONAL(ui64, Diff); // For tickets

    public:
        TOperation(const TString& payment, ui32 amount)
            : Amount(amount)
            , PaymentId(payment) {}

        NJson::TJsonValue GetReport() const {
            NJson::TJsonValue result;
            result["amount"] = Amount;
            result["payment_id"] = PaymentId;
            if (Paid) {
                result["paid"] = *Paid;
            }
            if (Diff) {
                result["diff"] = *Diff;
            }
            return result;
        }
    };

protected:
    mutable TVector<TOperation> Operations;

public:
    const TVector<TOperation>& GetOperations() const {
        return Operations;
    }

    void SetOperations(const TVector<TOperation>& other) const {
        Operations = other;
    }

protected:
    void OperationsToProto(TUserProblemTag::TProto& proto) const {
        for (auto&& op : Operations) {
            NDrive::NProto::TRefundTagData::TItem* item = proto.MutableBillingData()->MutableRefundData()->AddRefunds();
            item->SetPaymentId(op.GetPaymentId());
            item->SetAmount(op.GetAmount());
            if (op.HasPaid()) {
                item->SetPaid(op.GetPaidRef());
            }
            if (op.HasDiff()) {
                item->SetDiff(op.GetDiffRef());
            }
        }
    }

    bool OperationsFromProto(const TUserProblemTag::TProto& proto) {
        for (auto&& item : proto.GetBillingData().GetRefundData().GetRefunds()) {
            TOperation op(item.GetPaymentId(), item.GetAmount());
            if (item.HasPaid()) {
                op.SetPaid(item.GetPaid());
            }
            if (item.HasDiff()) {
                op.SetDiff(item.GetDiff());
            }
            Operations.push_back(op);
        }
        return true;
    }
};


class TBillingTag: public TUserProblemTag, public TOperationsHandler {
public:
    enum class EOperationAction {
        Nothing,
        Update,
        Remove,
        RemoveWithMessage,
        Error,
    };

    enum class EOperationResolution {
        NoTask          /* "no_task" */,
        NotFinished     /* "not_finished" */,
        Canceled        /* "canceled" */,
        Finished        /* "finished" */,
    };

private:
    using TBase = TUserProblemTag;
    R_FIELD(TInstant, ActivateSince, TInstant::Zero());
    R_FIELD(TInstant, OperationTs, TInstant::Zero());
    R_FIELD(TString, Resolution);
    R_FIELD(TString, PreferredCard);
    R_FIELD(TSet<TString>, AdditionalAccounts);
    R_OPTIONAL(bool, PlusUser);
protected:
    mutable ui32 Amount = 0;
    mutable TInstant AmountDeadline = TInstant::Max();

public:
    using TBase::TBase;

    ui32 GetAmount() const {
        return Amount;
    }

    TBillingTag& SetAmount(ui32 amount) {
        Amount = amount;
        return *this;
    }

    TInstant GetAmountDeadline() const {
        return AmountDeadline;
    }

    TBillingTag& SetAmountDeadline(const TInstant deadline) {
        AmountDeadline = deadline;
        return *this;
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;
    virtual bool OnBeforeAdd(const TString& /*objectId*/, const TString& /*userId*/, const NDrive::IServer* /*server*/, NDrive::TEntitySession& /*session*/) override final;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual TString GetTopicLink(const NDrive::IServer& server) const override;
    EOperationAction ProcessOperation(const TString& operationId, NDrive::NBilling::IBillingAccount::TPtr account, const TString& robotId, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession* chatSession);

private:
    virtual bool OnBeforeAddImpl(const TString& /*objectId*/, const TBillingTagDescription& description, NDrive::TEntitySession& /*session*/) = 0;
    virtual bool OnAfterAdd(const TDBTag& DBTag, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual bool OnBeforeUpdate(const TDBTag& self, ITag::TPtr to, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    bool SendMessageAfterRemove(const TString& userId, const TString& robotId, const NDrive::IServer& server, NDrive::TEntitySession* chatSession) const;
    EOperationAction ProcessOperationImpl(const TString& operationId, NDrive::NBilling::IBillingAccount::TPtr account, const TString& robotId, const NDrive::IServer& server, NDrive::TEntitySession& session);
};


class TOperationTag: public TBillingTag {
private:
    using TBase = TBillingTag;
public:
    static const TString TypeName;
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    virtual bool OnBeforeAddImpl(const TString& /*objectId*/, const TBillingTagDescription& description, NDrive::TEntitySession& /*session*/) override;

private:
    static TFactory::TRegistrator<TOperationTag> Registrator;
    static TTagDescription::TFactory::TRegistrator<TBillingTagDescription> RegistratorDescription;
};

class TFixedSumTag: public TBillingTag {
private:
    using TBase = TBillingTag;
public:
    static const TString TypeName;

    virtual bool OnBeforeAddImpl(const TString& /*objectId*/, const TBillingTagDescription& description, NDrive::TEntitySession& /*session*/) override;

    virtual TTagDescription::TPtr GetMetaDescription(const TString& /*type*/) const override {
        return nullptr;
    }
private:
    static TFactory::TRegistrator<TFixedSumTag> Registrator;
    static TTagDescription::TFactory::TRegistrator<TFixedBillingTagDescription> RegistratorDescription;
};

class TRefundTagDescription : public TTagDescription {
private:
    R_READONLY(ui32, Limit, 0);
    using TBase = TTagDescription;

public:
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
    virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
};

class TRefundTag: public TUserProblemTag, public TOperationsHandler {
private:
    using TBase = TUserProblemTag;
    R_FIELD(ui32, Amount, 0);
    R_READONLY(ui32, BonusCompensation, 0);

public:
    static const TString TypeName;
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;
    virtual bool OnBeforeAdd(const TString& /*objectId*/, const TString& /*userId*/, const NDrive::IServer* /*server*/, NDrive::TEntitySession& /*session*/) override;

public:
    static TFactory::TRegistrator<TRefundTag> Registrator;
    static TTagDescription::TFactory::TRegistrator<TRefundTagDescription> RegistratorDescription;
};

class TDeferredSessionRefundTag: public IJsonSerializableTag {
public:
    using TBase = IJsonSerializableTag;

public:
    static bool Exercise(const TDBTag& tag, const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& session);

public:
    TDeferredSessionRefundTag()
        : TBase(Type())
    {
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Trace };
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    static TString Type() {
        return "deferred_session_refund_tag";
    }

private:
    static TFactory::TRegistrator<TDeferredSessionRefundTag> Registrator;
};

class TWalletAccessTag: public ISerializableTag<NDrive::NProto::TWalletAccessTagData>, public IUserActionTag, public IOrganizationTag {
private:
    using TBase = ISerializableTag<NDrive::NProto::TWalletAccessTagData>;

public:
    static const TString TypeName;

public:
    class TDescription: public TTagDescription {
    private:
        using TBase = TTagDescription;
        R_FIELD(TSet<TAdministrativeAction::EAction>, Actions);

    public:
        using TBase::TBase;

        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;

    private:
        static TFactory::TRegistrator<TDescription> Registrator;
    };

public:
    using TBase::TBase;

    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    virtual NDrive::NProto::TWalletAccessTagData DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const NDrive::NProto::TWalletAccessTagData& proto) override;
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::User };
    }

    TDBActions GetActions(const TConstDBTag& self, const IDriveTagsManager& tagsManager, const TRolesManager& /*rolesManager*/, bool /*getPotential*/) const override;

private:
    static TFactory::TRegistrator<TWalletAccessTag> Registrator;
};
