#include "chargable.h"

#include "additional_service.h"
#include "area_tags.h"
#include "delegation.h"
#include "evolution_policy.h"
#include "offer.h"
#include "sharing.h"
#include "transformation.h"
#include "user_tags.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/database/drive/landing.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/head/head_account.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/logging/evlog.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/sessions/manager/billing.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/user_devices/manager.h>
#include <drive/backend/data/radar.h>

#include <drive/library/cpp/datasync/client.h>
#include <drive/library/cpp/threading/future.h>
#include <drive/telematics/server/location/location.h>

#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <rtline/library/geometry/polyline.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/algorithm/type_traits.h>
#include <rtline/library/json/proto/adapter.h>

namespace {
    TString DatasyncEnableOption = "datasync.maps_common_cars.enable";
    TString DatasyncCollectionOption = "datasync.maps_common_cars.collection";
    TString DatasyncKeyOption = "datasync.maps_common_cars.key";
    TString DatasyncDefaultCollection = "v1/personality/profile/maps_common/cars";
    TString DatasyncDefaultKey = "drive";
}

NDrive::NProto::TTagReservationFutures TTagReservationFutures::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    if (!!Offer) {
        *proto.MutableOffer() = Offer->SerializeToProto();
    }
    if (!!ExpectedCoord) {
        *proto.MutableExpectedCoord() = ExpectedCoord->SerializeLP();
    }
    proto.SetErrorMessage(ErrorMessage);
    proto.SetFailed(Failed);
    return proto;
}

bool TTagReservationFutures::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }
    ErrorMessage = proto.GetErrorMessage();
    if (proto.HasOffer()) {
        auto offer = ICommonOffer::ConstructFromProto<IOffer>(proto.GetOffer());
        if (!offer) {
            return false;
        }
        SetOffer(offer);
    }
    if (proto.HasFailed()) {
        Failed = proto.GetFailed();
    }
    if (proto.HasExpectedCoord()) {
        TGeoCoord c;
        if (!c.Deserialize(proto.GetExpectedCoord())) {
            return false;
        }
        ExpectedCoord = c;
    }
    return true;
}

bool TTagReservationFutures::OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const TString& offerUserId = Offer ? Offer->GetUserId() : Default<TString>();
    const TString& eventName = (offerUserId == userId) ? "DropFuturesOffer" : "RemoveFuturesOfferTag";
    NJson::TJsonValue event = NJson::TMapBuilder
        ("expected_coordinate", NJson::ToJson(ExpectedCoord))
        ("failed", Failed)
        ("object_id", self.GetObjectId())
        ("tag_id", self.GetTagId())
        ("user_id", userId)
        ("offer", (server && Offer) ? Offer->BuildJsonReport(ICommonOffer::TReportOptions{}, *server) : NJson::JSON_NULL)
    ;
    session.Committed().Subscribe([eventName, event = std::move(event)](const NThreading::TFuture<void>& w) {
        if (w.HasValue()) {
            NDrive::TEventLog::Log(eventName, event);
        }
    });
    return true;
}

const TString TTagReservationFutures::TypeName = "tag_reservation_features";
const TString TTagReservationFutures::TypeName1 = "tag_reservation_futures";
TTagReservationFutures::TFactory::TRegistrator<TTagReservationFutures> TTagReservationFutures::Registrator(TTagReservationFutures::TypeName);
TTagReservationFutures::TFactory::TRegistrator<TTagReservationFutures> TTagReservationFutures::Registrator1(TTagReservationFutures::TypeName1);

ITag::TPtr TTagReservationFutures::BuildFailed(const TString& errorMessage) const {
    auto result = MakeHolder<TTagReservationFutures>();
    result->SetOffer(Offer);
    result->SetFailed(true);
    result->SetErrorMessage(errorMessage);
    return result;
}

bool TTagReservationFutures::IsCorrectPosition(const TGeoCoord& c) const {
    if (!Offer->HasAvailableStartArea()) {
        return true;
    }
    TPolyLine<TGeoCoord> line(Offer->GetAvailableStartAreaUnsafe());
    return line.IsPointInternal(c);
}

const TString TChargableTag::Prereservation = "prereservation";
const TString TChargableTag::Reservation = "old_state_reservation";
const TString TChargableTag::Acceptance = "old_state_acceptance";
const TString TChargableTag::Riding = "old_state_riding";
const TString TChargableTag::Parking = "old_state_parking";
const TString TChargableTag::Servicing = "servicing";
const TString TChargableTag::Sharing = "sharing";

const TString TChargableTag::ServicingDoneStage = "servicing_done";
const TString TChargableTag::ServicingExpectedStage = "servicing_expected";
const TString TChargableTag::ServicingInProgressStage = "servicing_in_progress";

const TString TChargableTag::TypeName = "chargable_tag";
ITag::TFactory::TRegistrator<TChargableTag> TChargableTag::Registrator(TChargableTag::TypeName);
TChargableTag::TDescription::TFactory::TRegistrator<TChargableTag::TDescription> TChargableTag::TDescription::Registrator(TChargableTag::TypeName);

TString TChargableTag::GetPerformGroup() const {
    const auto& name = GetName();
    if (name != Reservation) {
        return name;
    } else {
        return TBase::GetPerformGroup();
    }
}

const TString& TChargableTag::GetStage(const TTaggedObject* object) const {
    const TString& name = GetName();
    if (object && name == Servicing) {
        auto evlog = NDrive::GetThreadEventLogger();
        ui64 subtagsCount = 0;
        ui64 subtagsEvolved = 0;
        ui64 subtagsPerformed = 0;
        for (auto&& tag : object->GetTags()) {
            if (!tag) {
                continue;
            }
            if (!HasSubtagId(tag.GetTagId())) {
                continue;
            }
            if (tag->GetPerformer()) {
                if (evlog) {
                    evlog->AddEvent(NJson::TMapBuilder
                        ("event", "GetServicingStageDiscoverPerformer")
                        ("tag", NJson::ToJson(tag))
                    );
                }
                subtagsPerformed += 1;
                break;
            }
            if (tag.Is<TChargableTag>()) {
                continue;
            }
            auto name = GetSubtagName(tag.GetTagId());
            if (name && name != tag->GetName()) {
                if (evlog) {
                    evlog->AddEvent(NJson::TMapBuilder
                        ("event", "GetServicingStageDiscoverEvolved")
                        ("expected_name", name)
                        ("tag", NJson::ToJson(tag))
                    );
                }
                subtagsEvolved += 1;
                continue;
            }
            subtagsCount += 1;
        }
        if (subtagsPerformed) {
            return ServicingInProgressStage;
        }
        if (subtagsEvolved) {
            return ServicingInProgressStage;
        }
        if (subtagsCount) {
            return ServicingExpectedStage;
        } else {
            return ServicingDoneStage;
        }
    }
    return name;
}

NJson::TJsonValue TChargableTag::GetStageReport(const TTaggedObject* object, const NDrive::IServer& server) const {
    Y_UNUSED(object);
    if (GetName() == Sharing) {
        auto api = Yensured(server.GetDriveAPI());
        auto sessionBuilder = api->GetTagsManager().GetDeviceTags().GetSessionsBuilder("billing");
        auto session = sessionBuilder ? sessionBuilder->GetSession(GetSharedSessionId()) : nullptr;

        bool busy = false;
        bool finished = session ? session->GetClosed() : true;
        auto lastEvent = session ? session->GetLastEvent() : Nothing();
        if (lastEvent && lastEvent.GetRef()->GetName() == Riding) {
            busy = true;
        }

        auto usersData = Yensured(api->GetUsersData());
        auto user = session ? usersData->GetCachedObject(session->GetUserId()) : Nothing();
        NJson::TJsonValue result;
        result["busy"] = busy;
        result["finished"] = finished;
        result["user"] = user ? user->GetPublicReport() : NJson::JSON_NULL;
        return result;
    }
    return {};
}

IOffer::TPtr TChargableTag::GetVehicleOffer() const {
    if (Context) {
        return Context->Offer;
    } else {
        return nullptr;
    }
}

ICommonOffer::TPtr TChargableTag::GetOffer() const {
    return GetVehicleOffer();
}

void TChargableTag::SetOffer(IOffer::TPtr offer) {
    MutableContext().Offer = offer;
}

const TString& TChargableTag::GetSharedSessionId() const {
    if (Context) {
        return Context->SharedSessionId;
    } else {
        return Default<TString>();
    }
}

void TChargableTag::SetSharedSessionId(const TString& value) {
    MutableContext().SharedSessionId = value;
}

const TServicingInfo* TChargableTag::GetServicingInfo() const {
    if (Context) {
        return Context->ServicingInfo.Get();
    } else {
        return nullptr;
    }
}

void TChargableTag::SetServicingInfo(TServicingInfo&& result) {
    MutableContext().ServicingInfo = std::move(result);
}

TConstArrayRef<ITag::TPtr> TChargableTag::GetSubtags() const {
    if (Context) {
        return Context->Subtags;
    } else {
        return {};
    }
}

void TChargableTag::SetSubtags(TVector<ITag::TPtr>&& value) {
    MutableContext().Subtags = std::move(value);
}

TSet<TString> TChargableTag::GetSubtagIds() const {
    if (Context) {
        return MakeSet(NContainer::Keys(Context->SubtagNames));
    } else {
        return {};
    }
}

TStringBuf TChargableTag::GetSubtagName(TStringBuf tagId) const {
    auto name = Context ? Context->SubtagNames.FindPtr(tagId) : nullptr;
    if (name) {
        return *name;
    } else {
        return {};
    }
}

void TChargableTag::AddSubtag(const TString& tagId, const TString& name) {
    MutableContext().SubtagNames.emplace(tagId, name);
}

bool TChargableTag::HasSubtagId(const TString& tagId) const {
    if (Context) {
        return Context->SubtagNames.contains(tagId);
    } else {
        return false;
    }
}

bool TChargableTag::IsServicingCancellable() const {
    auto value = Context ? Context->ServicingCancellable : Nothing();
    return value.GetOrElse(true);
}

void TChargableTag::SetServicingCancellable(bool value) {
    MutableContext().ServicingCancellable = value;
}

TChargableTag::TContext& TChargableTag::MutableContext() {
    if (!Context) {
        Context = MakeHolder<TContext>();
    }
    return *Context;
}

void TChargableTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    NProtobufJson::Proto2Json(TBase::SerializeSpecialDataToProto(), json);
}

bool TChargableTag::DoSpecialDataFromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* errors) {
    const auto& server = NDrive::GetServerAs<NDrive::IServer>();
    const auto& tagsManager = Yensured(server.GetDriveAPI())->GetTagsManager();
    const auto& subtags = jsonValue["subtags"];
    if (subtags.IsDefined() && !subtags.IsArray()) {
        if (errors) {
            errors->AddMessage("ChargableTag::SpecialDataFromJson", "subtags is not an array");
        }
        return false;
    }
    for (auto&& subtag : subtags.GetArray()) {
        auto tag = IJsonSerializableTag::BuildFromJson(tagsManager, subtag, errors);
        if (!tag) {
            return false;
        }
        auto chargableTag = std::dynamic_pointer_cast<TChargableTag>(tag);
        if (chargableTag) {
            if (errors) {
                errors->AddMessage("ChargableTag::SpecialDataFromJson", "subtags cannot contain chargable_tags");
            }
            return false;
        }
        MutableContext().Subtags.push_back(std::move(tag));
    }
    return true;
}

NDrive::NProto::TChargableTag TChargableTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetIsSwitching(IsSwitching);
    proto.SetIsReplacing(Replacing);
    TDropPolicyProvider::SerializeToProto(proto);
    if (HasDelegationType()) {
        proto.SetDelegationType((ui32)GetDelegationTypeUnsafe());
    }
    if (TransformationSkippedByExternalCommand) {
        proto.SetTransformationSkippedByExternalCommand(TransformationSkippedByExternalCommand);
    }
    if (auto offer = GetOffer()) {
        *proto.MutableOffer() = offer->SerializeToProto();
    }
    if (Context) {
        if (Context->ServicingCancellable) {
            proto.SetServicingCancellable(*Context->ServicingCancellable);
        }
        if (Context->SharedSessionId) {
            proto.SetSharedSessionId(Context->SharedSessionId);
        }
        for (auto&& [subtagId, subtagName] : Context->SubtagNames) {
            proto.AddSubtagId(subtagId);
            proto.AddSubtagName(subtagName);
        }
    }
    if (auto servicingInfo = GetServicingInfo()) {
        auto servicingInfoProto = proto.MutableServicingInfo();
        servicingInfoProto->SetStart(servicingInfo->Start.Seconds());
        servicingInfoProto->SetFinish(servicingInfo->Finish.Seconds());
        servicingInfoProto->SetMileage(servicingInfo->Mileage);
    }
    return proto;
}

bool TChargableTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }
    if (proto.HasServicingCancellable()) {
        SetServicingCancellable(proto.GetServicingCancellable());
    }
    if (proto.HasServicingInfo()) {
        const auto& servicingInfoProto = proto.GetServicingInfo();
        TServicingInfo servicingInfo;
        servicingInfo.Start = TInstant::Seconds(servicingInfoProto.GetStart());
        servicingInfo.Finish = TInstant::Seconds(servicingInfoProto.GetFinish());
        servicingInfo.Mileage = servicingInfoProto.GetMileage();
        SetServicingInfo(std::move(servicingInfo));
    }
    if (proto.HasOffer()) {
        auto offer = ICommonOffer::ConstructFromProto<IOffer>(proto.GetOffer());
        if (!offer) {
            return false;
        }
        SetOffer(offer);
    }
    if (proto.HasSharedSessionId()) {
        SetSharedSessionId(proto.GetSharedSessionId());
    }
    for (size_t i = 0; i < proto.SubtagIdSize(); ++i) {
        const TString& tagId = proto.GetSubtagId(i);
        const TString& tagName = i < proto.SubtagNameSize() ? proto.GetSubtagName(i) : Default<TString>();
        AddSubtag(tagId, tagName);
    }
    if (!TDropPolicyProvider::DeserializeFromProto(proto)) {
        return false;
    }
    IsSwitching = proto.GetIsSwitching();
    Replacing = proto.GetIsReplacing();
    TransformationSkippedByExternalCommand = proto.GetTransformationSkippedByExternalCommand();
    if (proto.HasDelegationType()) {
        DelegationType = static_cast<ECarDelegationType>(proto.GetDelegationType());
    }
    return true;
}

bool TChargableTag::OnBeforeEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const {
    const auto api = Yensured(server)->GetDriveAPI();
    const auto& tagsMeta = Yensured(api)->GetTagsManager().GetTagsMeta();
    const auto& deviceTagManager = Yensured(api)->GetTagsManager().GetDeviceTags();
    const auto& userTagManager = Yensured(api)->GetTagsManager().GetUserTags();
    const auto evlog = NDrive::GetThreadEventLogger();

    auto transformation = dynamic_cast<TTransformationTag*>(toTag.Get());
    bool rollout = transformation && fromTag->GetName() == transformation->GetFrom();
    TChargableTag* toChargable = dynamic_cast<TChargableTag*>(toTag.Get());
    const TChargableTag* fromChargable = fromTag.GetTagAs<TChargableTag>();
    TString tagNameFrom;
    if (fromChargable) {
        tagNameFrom = fromChargable->GetName();
    } else if (transformation) {
        tagNameFrom = transformation->GetFrom();
    }
    TString tagNameTo;
    if (toChargable) {
        rollout = true;
        tagNameTo = toChargable->GetName();
    } else if (transformation) {
        tagNameTo = transformation->GetTo();
    }
    bool zerodiff = tagNameFrom == tagNameTo;

    if (rollout && ((tagNameTo == Acceptance && tagNameFrom != Sharing) || tagNameTo == Riding) && !zerodiff && (!eContext || !eContext->IsSwitching())) {
        auto optionalSession = api->GetSessionManager().GetTagSession(fromTag.GetTagId(), session);
        if (!optionalSession) {
            return false;
        } else {
            auto sessionTrace = *optionalSession;
            if (sessionTrace) {
                const TBillingSession* bSession = dynamic_cast<const TBillingSession*>(sessionTrace.Get());
                if (bSession && !bSession->GetClosed()) {
                    auto currentOffer = bSession->GetCurrentOffer();
                    if (!!currentOffer) {
                        if (currentOffer->CheckSession(permissions, session, server, false) != EDriveSessionResult::Success) {
                            return false;
                        }
                    }
                } else {
                    WARNING_LOG << "Incorrect billing session context for " << permissions.GetUserId() << Endl;
                }
            }
        }
    }

    if (rollout && tagNameFrom == Servicing) {
        auto subtagIds = GetSubtagIds();
        auto subtags = TVector<TDBTag>();
        auto optionalObject = deviceTagManager.RestoreObject(fromTag.GetObjectId(), session);
        if (!optionalObject) {
            return false;
        }
        for (auto&& subtag : optionalObject->GetTags()) {
            if (subtag.GetTagId() == fromTag.GetTagId()) {
                continue;
            }
            if (!subtagIds.contains(subtag.GetTagId()) && !subtag.Is<TChargableTag>()) {
                continue;
            }
            if (subtag && subtag->GetPerformer()) {
                session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "tag " + subtag.GetTagId() + " is still being performed");
                return false;
            }
            subtags.push_back(subtag);
        }
        if (!subtags.empty()) {
            bool cancellable = IsServicingCancellable();
            if (!cancellable) {
                session.SetCode(HTTP_BAD_REQUEST);
                session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "servicing is non-cancellable", NDrive::MakeError("servicing_non_cancellable"));
                return false;
            }
        }
        if (!deviceTagManager.RemoveTags(subtags, permissions.GetUserId(), server, session)) {
            return false;
        }
        auto servicingRequested = TInstant::Max();
        auto servicingStarted = TInstant::Max();
        auto servicingFinished = TInstant::Zero();
        auto startMileage = std::numeric_limits<double>::max();
        auto finishMileage = double(0);
        for (auto&& subtagId : subtagIds) {
            auto events = deviceTagManager.GetEventsByTag(subtagId, session);
            if (!events) {
                return false;
            }
            for (auto&& ev : *events) {
                auto timestamp = ev.GetHistoryInstant();
                auto snapshot = ev->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
                auto mileage = snapshot ? snapshot->GetMileage() : Nothing();
                switch (ev.GetHistoryAction()) {
                case EObjectHistoryAction::Add:
                    servicingRequested = std::min(servicingRequested, timestamp);
                    break;
                case EObjectHistoryAction::SetTagPerformer:
                    if (mileage) {
                        startMileage = std::min(startMileage, *mileage);
                    }
                    servicingStarted = std::min(servicingStarted, timestamp);
                    break;
                case EObjectHistoryAction::Remove:
                    if (mileage) {
                        finishMileage = std::max(finishMileage, *mileage);
                    }
                    servicingStarted = std::min(servicingStarted, timestamp);
                    servicingFinished = std::max(servicingFinished, timestamp);
                    break;
                default:
                    break;
                }
            }
        }
        TServicingInfo result;
        result.Start = servicingStarted;
        result.Finish = servicingFinished;
        result.Mileage = finishMileage > startMileage ? finishMileage - startMileage : 0;

        auto servicingInfoHolder = std::dynamic_pointer_cast<IServicingInfoHolder>(toTag);
        Yensured(servicingInfoHolder)->SetServicingInfo(std::move(result));
        auto optionalSession = api->GetSessionManager().GetTagSession(fromTag.GetTagId(), session);
        if (!optionalSession) {
            return false;
        }
        auto currentSession = *optionalSession;
        if (!currentSession) {
            session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "cannot get session for tag " + fromTag.GetTagId());
            return false;
        }
        auto user = userTagManager.RestoreObject(currentSession->GetUserId(), session);
        if (!user) {
            return false;
        }
        for (auto&& tag : user->GetTags()) {
            if (auto serviceTag = tag.GetTagAs<TAdditionalServiceOfferHolderTag>()) {
                if (serviceTag->HasParentSessionId() &&
                    serviceTag->GetParentSessionIdRef() == currentSession->GetSessionId() &&
                    serviceTag->GetName() == TAdditionalServiceOfferHolderTag::Started) {
                    if (!userTagManager.RemoveTag(tag, permissions.GetUserId(), server, session, /*force=*/true)) {
                        return false;
                    }
                }
            }
        }
    }

    if (rollout && tagNameTo == Servicing) {
        if (!toChargable) {
            session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "cannot evolve to servicing through transformation");
            return false;
        }
        auto addableTags = permissions.GetTagNamesByAction(NTagActions::ETagAction::Add);
        auto subtags = MakeVector(toChargable->GetSubtags());
        if (subtags.empty()) {
            auto d = tagsMeta.GetDescriptionByName(tagNameTo);
            auto description = std::dynamic_pointer_cast<const TDescription>(d);
            if (!description) {
                session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "cannot get description for " + tagNameTo);
                return false;
            }
            for (auto&& subtagName : description->GetDefaultSubtags()) {
                auto subtag = tagsMeta.CreateTag(subtagName);
                if (!subtag) {
                    session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "cannot get create tag " + subtagName);
                    return false;
                }
                subtags.push_back(std::move(subtag));
            }
        }
        for (auto&& subtag : subtags) {
            if (!addableTags.contains(subtag->GetName())) {
                session.SetCode(HTTP_FORBIDDEN);
                session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "no permissions to add tag " + subtag->GetName(), NDrive::MakeError("no_permissions_for_servicing"));
                return false;
            }
        }
        if (!subtags.empty()) {
            auto reservation = MakeAtomicShared<TChargableTag>(Reservation);
            subtags.push_back(reservation);

            auto optionalObject = deviceTagManager.RestoreObject(fromTag.GetObjectId(), session);
            if (!optionalObject) {
                session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "cannot get tags for object " + fromTag.GetObjectId());
                return false;
            }
            auto enableServicingTag = optionalObject->GetTag("enable_servicing");
            if (enableServicingTag) {
                for (auto& tag : subtags) {
                    tag->SetComment((*enableServicingTag)->GetComment());
                }
            }

            auto added = deviceTagManager.AddTags(subtags, permissions.GetUserId(), fromTag.GetObjectId(), server, session);
            if (!added) {
                return false;
            }
            for (auto&& tag : *added) {
                toChargable->AddSubtag(tag.GetTagId(), tag->GetName());
            }
        }
    }

    if (rollout && tagNameFrom == Sharing) {
        const auto& sharedSessionId = Yensured(fromChargable)->GetSharedSessionId();
        auto optionalSharedSession = api->GetSessionManager().GetSession(sharedSessionId, session);
        if (!optionalSharedSession) {
            return false;
        }
        auto sharedSession = *optionalSharedSession;
        if (sharedSession && !sharedSession->GetClosed()) {
            const auto& sharedSessionTagId = sharedSession->GetInstanceId();
            auto optionalSharedSessionTag = deviceTagManager.RestoreTag(sharedSessionTagId, session);
            if (!optionalSharedSessionTag) {
                return false;
            }
            auto sharedSessionTag = std::move(*optionalSharedSessionTag);
            if (!sharedSessionTag) {
                session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "cannot restore tag " + sharedSessionTagId);
                return false;
            }
            if (sharedSessionTag->GetName() == Riding) {
                session.SetCode(HTTP_BAD_REQUEST);
                session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "shared session " + sharedSessionId + " is in riding");
                return false;
            }
            if (auto transformation = sharedSessionTag.GetTagAs<TTransformationTag>()) {
                session.SetCode(HTTP_BAD_REQUEST);
                session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "shared session " + sharedSessionId + " is in transformation");
                return false;
            }
            auto userPermissions = api->GetUserPermissions(sharedSession->GetUserId());
            if (!userPermissions) {
                session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "cannot get UserPermissions for shared session " + sharedSessionId);
                return false;
            }
            auto previousOriginatorId = session.GetOriginatorId();
            session.SetOriginatorId(permissions.GetUserId());
            if (!DirectEvolve(sharedSessionTag, Reservation, *userPermissions, *server, session, eContext)) {
                return false;
            }
            session.SetOriginatorId(previousOriginatorId);

            auto d = tagsMeta.GetDescriptionByName(TSessionSharingTag::Type());
            auto description = std::dynamic_pointer_cast<const TSessionSharingTag::TDescription>(d);
            if (description && description->GetInterruptionTag()) {
                auto interruptionTag = tagsMeta.CreateTag(description->GetInterruptionTag());
                if (interruptionTag) {
                    auto genericUserState = std::dynamic_pointer_cast<TGenericUserStateTag>(interruptionTag);
                    if (genericUserState) {
                        genericUserState->SetCarId(fromTag.GetObjectId());
                        genericUserState->SetUserId(permissions.GetUserId());
                    }
                    auto added = userTagManager.AddTag(interruptionTag, permissions.GetUserId(), sharedSession->GetUserId(), server, session);
                    if (!added) {
                        return false;
                    }
                } else if (evlog) {
                    evlog->AddEvent(NJson::TMapBuilder
                        ("event", "create_interruption_tag_error")
                        ("name", description->GetInterruptionTag())
                    );
                }
            }
        }
    }

    if (rollout && tagNameTo == Sharing) {
        if (!toChargable) {
            session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "cannot evolve to " + Sharing + " through transformation");
            return false;
        }
        if (tagNameFrom != Parking) {
            session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "cannot evolve to " + Sharing + " from " + tagNameFrom);
            return false;
        }
        const auto& sharedSessionId = toChargable->GetSharedSessionId();
        if (!sharedSessionId) {
            session.SetErrorInfo("ChargableTag::OnBeforeEvolve", "SharedSessionId is not set");
            return false;
        }
    }

    if (rollout && eContext) {
        for (auto&& policy : eContext->GetPolicies()) {
            if (!policy->ExecuteBeforeEvolution(fromTag, toTag, server, permissions, eContext, session)) {
                return false;
            }
        }
    }
    return true;
}

bool TChargableTag::OnAfterEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const {
    const auto driveApi = server ? server->GetDriveAPI() : nullptr;
    const auto& deviceTagManager = Yensured(driveApi)->GetTagsManager().GetDeviceTags();

    const auto& objectId = fromTag.TConstDBTag::GetObjectId();
    const auto& userId = permissions.GetUserId();

    const auto transformation = fromTag.GetTagAs<TTransformationTag>();
    const bool rollout = transformation && toTag->GetName() == transformation->GetTo();
    if (toTag->GetName() == Reservation) {
        const bool resetFlag = !transformation || rollout;
        if (resetFlag) {
            const auto& tagId = fromTag.GetTagId();
            if (!deviceTagManager.DropPerformer({}, {tagId}, permissions, server, session)) {
                return false;
            }
        }
    }

    if (toTag->GetName() == Reservation) {
        auto deviceTags = deviceTagManager.RestoreObject(objectId, session);
        if (!deviceTags) {
            session.AddErrorMessage("ChargableTag::OnAfterEvolve", TStringBuilder() << "cannot restore object " << objectId);
            return false;
        }
        auto replacingTag = deviceTags->GetTag(TReplaceCarTag::TypeName);
        if (replacingTag) {
            bool removed = deviceTagManager.RemoveTag(*replacingTag, userId, server, session);
            if (!removed) {
                session.AddErrorMessage("ChargableTag::OnAfterEvolve", TStringBuilder() << "cannot remove car replacing tag " << replacingTag->GetTagId());
                return false;
            }
        }
    }

    TSessionManager::TOptionalConstSession optionalSession;
    {
        TTaggedObject obj;
        if (!deviceTagManager.RestoreObject(objectId, obj, session)) {
            return false;
        }
        TDBTag dbTag = obj.GetFirstTagByClass<IDelegationTag>();
        const IDelegationTag* delegationTag = obj.GetFirstTagByClass<IDelegationTag>().GetTagAs<IDelegationTag>();
        if (delegationTag && (dbTag->GetPerformer() == "" || dbTag->GetPerformer() == delegationTag->GetBaseUserId())) {
            if (!deviceTagManager.RemoveTag(dbTag, userId, server, session)) {
                return false;
            }
        }

        if (rollout && toTag->GetName() == Riding) {
            if (!optionalSession) {
                optionalSession = driveApi->GetSessionManager().GetTagSession(fromTag.GetTagId(), session);
            }
            if (!optionalSession) {
                return false;
            }
            NotifyPassport(*optionalSession, permissions, *server, session, false);
        }
    }

    if (toTag->GetName() == Parking) {
        if (!optionalSession) {
            optionalSession = driveApi->GetSessionManager().GetTagSession(fromTag.GetTagId(), session);
        }
        if (!optionalSession) {
            return false;
        }
        NotifyPassport(*optionalSession, permissions, *server, session, true);
    }

    if (rollout && transformation->GetFrom() == Reservation && transformation->GetTo() == Acceptance) {
        TUnistatSignalsCache::SignalAdd("frontend", "evolve-reservation-to-acceptance", 1);
        auto snapshot = server->GetSnapshotsManager().GetSnapshot(objectId);
        const TSet<TString> objectTags = snapshot.GetLocationTagsPotentially();

        const auto signAreaTags = permissions.GetSetting<TString>("signals.offers.significant_area_tags", "msc_area,spb_area,kazan_area");
        TSet<TString> areaTags;
        StringSplitter(signAreaTags).SplitBySet(", ").SkipEmpty().Collect(&areaTags);

        auto itAreas = areaTags.begin();
        for (auto&& objectTag : objectTags) {
            if (AdvanceSimple(itAreas, areaTags.end(), objectTag)) {
                TUnistatSignalsCache::SignalAdd("frontend", "evolve-reservation-to-acceptance-" + objectTag, 1);
            }
        }
        bool launchAppFlag = driveApi->CheckCrossloginFlag(permissions, *server, session);
        if (!driveApi->GetHeadAccountManager().CreateSession(userId, objectId, launchAppFlag, session)) {
            return false;
        }
    }

    bool restoreChargableSessionTag = permissions.GetSetting<bool>("book.restore_chargable_session_tag").GetOrElse(false);
    if (restoreChargableSessionTag) {
        if (!optionalSession) {
            optionalSession = driveApi->GetSessionManager().GetTagSession(fromTag.GetTagId(), session);
        }
        if (!optionalSession) {
            return false;
        }
        if (*optionalSession) {
            const auto& sessionId = optionalSession.GetRef()->GetSessionId();
            auto optionalChargableSessionTag = TChargableSessionTag::Get(sessionId, driveApi->GetTagsManager(), session);
            if (!optionalChargableSessionTag) {
                return false;
            }
            auto chargableSessionTag = std::move(*optionalChargableSessionTag);
            if (!chargableSessionTag && !TChargableSessionTag::Register(sessionId, fromTag, NEntityTagsManager::EEntityType::Car, permissions, *server, session)) {
                return false;
            }
        }
    }

    if ((rollout || !transformation) && eContext) {
        for (auto&& policy : eContext->GetPolicies()) {
            if (!policy->ExecuteAfterEvolution(fromTag, toTag, server, permissions, eContext, session)) {
                return false;
            }
        }
    }

    return true;
}

void TChargableTag::NotifyPassport(ISession::TConstPtr session, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx, bool stopFlag) const {
    R_ENSURE(session, HTTP_INTERNAL_SERVER_ERROR, "null session in NotifyPassport", tx);
    if (permissions.GetSetting<bool>("yaauto.crosslogin.enable").GetOrElse(false)) {
        if (stopFlag) {
            server.GetDriveAPI()->GetHeadAccountManager().StopPassportSession(permissions, session->GetSessionId(), session->GetObjectId());
        } else if (server.GetDriveAPI()->CheckCrossloginFlag(permissions, server, tx)) {
            server.GetDriveAPI()->GetHeadAccountManager().StartPassportSession(permissions, session->GetSessionId(), session->GetObjectId());
        }
    } else {
        NDrive::TEventLog::Log("NotifyPassportSkip", NJson::TMapBuilder
            ("type", "CrossloginDisabled")
        );
    }
}

bool TChargableTag::OnBeforeDropPerform(const TDBTag& self, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    SetOffer(nullptr);
    const auto& objectId = self.GetObjectId();
    const ISettings& settings = Yensured(server)->GetSettings();
    bool restrictDropPerformer = settings.GetValue<bool>("tag.chargable_tag.restrict_drop_performer").GetOrElse(true);
    if (restrictDropPerformer && GetName() != Reservation) {
        session.SetErrorInfo(
            "ChargableTag::OnBeforeDropPerform",
            TStringBuilder() << "dropping " << GetName() << " is restricted",
            NDrive::MakeError("restrict_drop_performer")
        );
        return false;
    }

    auto api = Yensured(server->GetDriveAPI());
    auto datasync = server->GetDatasyncClient();
    auto enable = settings.GetValue<bool>(DatasyncEnableOption).GetOrElse(false);
    if (datasync && enable) {
        const TString& userId = GetPerformer();
        auto fetchResult = api->GetUsersData()->FetchInfo(userId, session);
        auto userData = fetchResult.GetResultPtr(userId);
        ui64 uid = userData ? userData->GetPassportUid() : 0;
        if (uid) {
            auto collection = settings.GetValue<TString>(DatasyncCollectionOption).GetOrElse(DatasyncDefaultCollection);
            auto key = settings.GetValue<TString>(DatasyncKeyOption).GetOrElse(DatasyncDefaultKey);
            auto asyncResponse = datasync->Delete(collection, key, uid);
            auto eventLogState = NDrive::TEventLog::CaptureState();
            asyncResponse.Subscribe([collection, key, objectId, eventLogState = std::move(eventLogState)](const NThreading::TFuture<TDatasyncClient::TResponse>& r) {
                NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
                NJson::TJsonValue response;
                ui16 code = 0;
                if (r.HasValue()) {
                    const TDatasyncClient::TResponse& dsr = r.GetValue();
                    response = dsr.GetValue();
                    code = dsr.GetCode();
                } else {
                    response = NJson::TMapBuilder("exception", NThreading::GetExceptionInfo(r));
                }
                NDrive::TEventLog::Log("DatasyncDelete", NJson::TMapBuilder
                    ("collection", collection)
                    ("key", key)
                    ("object_id", objectId)
                    ("code", code)
                    ("response", std::move(response))
                );
            });
        }
    }

    auto optionalSession = api->GetSessionManager().GetTagSession(self.GetTagId(), session);
    if (!optionalSession) {
        return false;
    }
    auto currentSession = *optionalSession;
    if (!currentSession) {
        session.SetErrorInfo("ChargableTag::OnAfterEvolve", "cannot get session for tag " + self.GetTagId());
        return false;
    }
    MutableContext().CurrentSession = currentSession;

    auto finishedSessionId = currentSession->GetSessionId();
    if (api->HasBillingManager() && !api->GetBillingManager().FinishingBillingTask(finishedSessionId, session)) {
        return false;
    }

    return true;
}

bool TChargableTag::OnAfterDropPerform(const TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    const auto& objectId = self.GetObjectId();
    const auto& tagId = self.GetTagId();
    const auto api = Yensured(server)->GetDriveAPI();
    const auto& deviceTagManager = Yensured(api)->GetTagsManager().GetDeviceTags();
    const auto& userTagManager = Yensured(api)->GetTagsManager().GetUserTags();
    if (!api->GetHeadAccountManager().FinishSession(objectId, tx)) {
        return false;
    }
    if (auto currentSession = Context ? Context->CurrentSession : nullptr) {
        auto finishedSessionId = currentSession->GetSessionId();
        auto object = deviceTagManager.RestoreObject(objectId, tx);
        if (!object) {
            tx.AddErrorMessage("ChargableTag::OnAfterEvolve", TStringBuilder() << "cannot restore object " << objectId);
            return false;
        }
        bool readdTag = true;
        auto restoredTag = TDBTag();
        for (auto&& tag : object->GetTags()) {
            if (tag.GetTagId() == tagId) {
                restoredTag = tag;
                continue;
            }
            auto anotherChargable = tag.GetTagAs<TChargableTag>();
            if (anotherChargable && anotherChargable->GetName() == Sharing && anotherChargable->GetSharedSessionId() == finishedSessionId) {
                readdTag = false;
                continue;
            }
        }
        if (!restoredTag) {
            tx.AddErrorMessage("ChargableTag::OnAfterEvolve", TStringBuilder() << "cannot restore tag " << tagId);
            return false;
        }
        if (!deviceTagManager.RemoveTag(restoredTag, permissions.GetUserId(), server, tx)) {
            return false;
        }
        if (readdTag) {
            auto newTag = MakeAtomicShared<TChargableTag>(Reservation);
            if (!deviceTagManager.AddTag(newTag, permissions.GetUserId(), objectId, server, tx)) {
                return false;
            }
        }

        auto user = userTagManager.RestoreObject(currentSession->GetUserId(), tx);
        if (!user) {
            return false;
        }
        for (auto&& tag : user->GetTags()) {
            if (auto sharingTag = tag.GetTagAs<TSessionSharingTag>()) {
                if (sharingTag->GetSessionId() == finishedSessionId) {
                    if (!userTagManager.RemoveTag(tag, permissions.GetUserId(), server, tx, /*force=*/true)) {
                        return false;
                    }
                }
            } else if (auto serviceTag = tag.GetTagAs<TAdditionalServiceOfferHolderTag>()) {
                if (serviceTag->HasParentSessionId() && serviceTag->GetParentSessionIdRef() == finishedSessionId) {
                    if (!userTagManager.RemoveTag(tag, permissions.GetUserId(), server, tx, /*force=*/true)) {
                        return false;
                    }
                }
            }
        }
        tx.Committed().Subscribe([
            finishedSessionId = std::move(finishedSessionId),
            permissions = permissions.Self(),
            api,
            server
        ](const NThreading::TFuture<void>& w) {
            if (!w.HasValue()) {
                return;
            }
            auto tx = api->template BuildTx<NSQL::Writable>();
            auto optionalSession = api->GetSessionManager().GetSession(finishedSessionId, tx);
            R_ENSURE(optionalSession, {}, "cannot GetSession " << finishedSessionId, tx);
            auto finishedSession = *optionalSession;
            auto closed = api->CloseSession(finishedSession, *permissions, *server, true);
            if (!closed) {
                NDrive::TEventLog::Log("CloseSessionError", NJson::TMapBuilder
                    ("session_id", finishedSessionId)
                    ("session", finishedSession ? finishedSession->GetDebugInfo() : NJson::TJsonValue())
                    ("error", NJson::GetExceptionInfo(closed.GetError()))
                );
            }
        });
        NotifyPassport(currentSession, permissions, *server, tx, true);
    }
    return true;
}

bool TChargableTag::OnAfterPerform(TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    Y_UNUSED(session);
    const ISettings& settings = server->GetSettings();
    const TString& objectId = self.GetObjectId();
    auto datasync = server->GetDatasyncClient();
    auto enable = settings.GetValue<bool>(DatasyncEnableOption).GetOrElse(false);
    if (datasync && enable) {
        auto collection = settings.GetValue<TString>(DatasyncCollectionOption).GetOrElse(DatasyncDefaultCollection);
        auto key = settings.GetValue<TString>(DatasyncKeyOption).GetOrElse(DatasyncDefaultKey);

        auto snapshot = server->GetSnapshotsManager().GetSnapshot(objectId);
        auto location = snapshot.GetLocation();
        ui64 uid = permissions.GetPassportUid();
        if (location && uid) {
            NJson::TJsonValue data;
            data["latitude"] = location->Latitude;
            data["longitude"] = location->Longitude;
            data["modified"] = location->Timestamp.ToStringUpToSeconds();
            auto asyncResponse = datasync->Update(collection, key, uid, data);
            auto eventLogState = NDrive::TEventLog::CaptureState();
            asyncResponse.Subscribe([collection, key, objectId, eventLogState = std::move(eventLogState), data = std::move(data)](const NThreading::TFuture<TDatasyncClient::TResponse>& r) {
                NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
                NJson::TJsonValue response;
                ui16 code = 0;
                if (r.HasValue()) {
                    const TDatasyncClient::TResponse& dsr = r.GetValue();
                    response = dsr.GetValue();
                    code = dsr.GetCode();
                } else {
                    response = NJson::TMapBuilder("exception", NThreading::GetExceptionInfo(r));
                }
                NDrive::TEventLog::Log("DatasyncUpdate", NJson::TMapBuilder
                    ("collection", collection)
                    ("key", key)
                    ("object_id", objectId)
                    ("data", data)
                    ("code", code)
                    ("response", std::move(response))
                );
            });
        }
    }
    return true;
}

bool TChargableTag::OnBeforePerform(TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (!!GetPerformer()) {
        session.SetErrorInfo("incorrect logic", "Cannot perform this tag type with performer", EDriveSessionResult::IncorrectRequest);
        return false;
    }
    auto offer = GetVehicleOffer();
    if (!offer) {
        session.SetErrorInfo("incorrect logic", "cannot perform tag without vehicle offer", EDriveSessionResult::InconsistencyOffer);
        return false;
    }
    if (offer->GetObjectId() != self.GetObjectId()) {
        session.SetErrorInfo("inconsistency_car_id", "Incorrect car id for performing with offer", EDriveSessionResult::InconsistencyOffer);
        return false;
    }

    if (offer->CheckSession(permissions, session, server, true) != EDriveSessionResult::Success) {
        return false;
    }

    auto api = server ? server->GetDriveAPI() : nullptr;
    if (api && api->HasBillingManager()) {
        const auto& billingManager = server->GetDriveAPI()->GetBillingManager();
        auto optionalBillingTask = billingManager.GetActiveTasksManager().GetTask(offer->GetOfferId(), session);
        if (!optionalBillingTask) {
            return false;
        }
        auto billingTask = std::move(*optionalBillingTask);
        if (!billingTask) {
            if (!billingManager.CreateBillingTask(permissions.GetUserId(), offer, session)) {
                return false;
            }
        }
    }
    Drop();
    return true;
}

bool TChargableTag::OnBeforeRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    Y_UNUSED(self);
    Y_UNUSED(server);
    Y_UNUSED(userId);
    if (!GetPerformer().empty()) {
        session.SetErrorInfo("ChargableTag::OnBeforeRemove", "cannot remove chargable_tag with performer");
        return false;
    }
    return true;
}

TVector<TString> TChargableTag::GetTagNames(const ITagsMeta& meta) {
    auto descriptions = meta.GetRegisteredTags();

    TVector<TString> result;
    for (auto&&[name, description] : descriptions) {
        if (description && description->GetType() == TChargableTag::TypeName) {
            result.push_back(name);
        }
    }
    result.push_back(TTransformationTag::TypeName);
    return result;
}

TVector<TString> TChargableTag::GetTagNamesWithTransformations(const ITagsMeta& meta) {
    auto result = GetTagNames(meta);
    result.push_back(TTransformationTag::TypeName);
    return result;
}

TMaybe<TChargableTag::TActiveSessionTags> TChargableTag::GetActiveSessionTagIds(const TString& objectId, const ITagsMeta& meta, const ui64 historyEventIdMaxWithFinishInstant, NDrive::TEntitySession& session) {
    TSet<TString> tagIds;
    {
        auto tagNames = GetTagNamesWithTransformations(meta);
        auto query = TStringBuilder()
            << "SELECT DISTINCT(tag_id) FROM car_tags "
            << "WHERE performer != '' "
            << "AND tag IN (" << session->Quote(tagNames) << ") ";
        if (objectId) {
            query << "AND object_id = " << session->Quote(objectId);
        }

        TRecordsSet records;
        auto queryResult = session->Exec(query, &records);
        if (!ParseQueryResult(queryResult, session)) {
            return {};
        }

        for (auto&& i : records) {
            tagIds.emplace(i.Get("tag_id"));
        }
    }

    TSet<ui64> historyEventIds;

    if (historyEventIdMaxWithFinishInstant != Max<ui64>()) {
        auto transaction = session.GetTransaction();
        auto tagNames = GetTagNamesWithTransformations(meta);
        auto query = TStringBuilder()
            << "SELECT tag_id, history_event_id FROM car_tags_history "
            << "WHERE history_action = 'set_performer' "
            << "AND tag IN (" << transaction->Quote(tagNames) << ") AND history_event_id > " << historyEventIdMaxWithFinishInstant;
        if (objectId) {
            query << "AND object_id = " << transaction->Quote(objectId);
        }

        TRecordsSet records;
        auto queryResult = transaction->Exec(query, &records);
        if (!ParseQueryResult(queryResult, session)) {
            return {};
        }

        for (auto&& i : records) {
            tagIds.erase(i.Get("tag_id"));
        }

        for (auto&& i : records) {
            ui64 hei;
            Y_ENSURE_BT(i.TryGet<ui64>("history_event_id", hei));
            historyEventIds.erase(hei);
        }
    }

    return TActiveSessionTags(tagIds, historyEventIds);
}

TDBTag TChargableTag::Book(IOffer::TPtr offer, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session, const TBookOptions& bookOptions) {
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "Book")
            ("offer", NJson::ToJson(offer))
            ("user_id", permissions.GetUserId())
            ("options", NJson::ToJson(NJson::Pass(bookOptions)))
        );
    }
    bool checkBlocked = bookOptions.CheckBlocked;
    if (checkBlocked && !TUserProblemTag::EnsureNotBlocked(permissions.GetUserId(), server, session)) {
        return {};
    }
    if (!offer) {
        return {};
    }
    if (offer->GetUserId() != permissions.GetUserId()) {
        session.SetErrorInfo("ChargableTag::Book", "offer belongs to another user", EDriveSessionResult::InconsistencyOffer);
        return {};
    }

    session.Committed().Subscribe([offer](const NThreading::TFuture<void>& c) {
        if (c.HasValue()) {
            SendGlobalMessage<TOfferBookingCompleted>(offer);
        }
    });

    const auto driveApi = Checked(server.GetDriveAPI());
    const auto& database = server.GetDriveDatabase();
    const auto& tagsManager = database.GetTagsManager();
    const auto& deviceTagsManager = tagsManager.GetDeviceTags();
    const auto& userTagsManager = tagsManager.GetUserTags();
    const auto& accountTagsManager = tagsManager.GetAccountTags();
    const auto& tagsMeta = tagsManager.GetTagsMeta();
    const auto& settings = server.GetSettings();

    bool multiRent = bookOptions.MultiRent;
    if (!multiRent) {
        TVector<TDBTag> tags;
        if (!deviceTagsManager.RestorePerformerTags({Reservation, Acceptance, TTransformationTag::TypeName, Parking, Riding}, {permissions.GetUserId()}, tags, session)) {
            return {};
        }
        if (!tags.empty()) {
            const auto& objectId = tags.front().GetObjectId();
            const auto carFetchResult = database.GetCarManager().FetchInfo(objectId, session);
            if (!carFetchResult) {
                return {};
            }
            const auto carInfo = carFetchResult.GetResultPtr(objectId);
            if (carInfo) {
                session.SetErrorInfo("ChargableTag::Book", "user_have_rented_car (" + carInfo->GetNumber() + ")", EDriveSessionResult::UserHaveRentedCar);
            } else {
                session.SetErrorInfo("ChargableTag::Book", "user_have_rented_car (" + objectId + ")", EDriveSessionResult::UserHaveRentedCar);
            }
            return {};
        }
    }

    const TString& objectId = offer->GetObjectId();
    const TString& offerId = offer->GetOfferId();
    const TString& userId = permissions.GetUserId();
    offer->OnBooking(bookOptions.DeviceId, bookOptions.MobilePaymethodId);

    if (!offer->IsBookable()) {
        session.SetCode(HTTP_BAD_REQUEST);
        session.SetErrorInfo(
            "ChargableTag::Book",
            TStringBuilder() << "offer is not bookable: " << offerId,
            NDrive::MakeError("booking_unbookable_offer")
        );
        return {};
    }

    TTaggedObject taggedDevice;
    if (objectId) {
        auto optionalTaggedDevice = deviceTagsManager.RestoreObject(objectId, session);
        if (!optionalTaggedDevice) {
            return {};
        }
        taggedDevice = std::move(*optionalTaggedDevice);
    }

    bool shouldAddChargableSessionTag = permissions.GetSetting<bool>(settings, "book.add_chargable_session_tag").GetOrElse(true);
    if (!offer->GetTargetHolderTag().empty()) {
        auto tag = tagsMeta.CreateTag(offer->GetTargetHolderTag());
        if (!tag) {
            session.SetErrorInfo("ChargableTag::Book", TStringBuilder() << "cannot create tag " << offer->GetTargetHolderTag());
            return {};
        }
        auto offerHolderTag = std::dynamic_pointer_cast<TOfferHolderTag>(tag);
        if (!offerHolderTag) {
            session.SetErrorInfo("ChargableTag::Book", TStringBuilder() << "cannot cast tag " << offer->GetTargetHolderTag());
            return {};
        }
        offerHolderTag->SetOffer(offer);

        auto optionalAddedTags = userTagsManager.AddTag(tag, userId, userId, &server, session);
        if (!optionalAddedTags) {
            return {};
        }
        if (optionalAddedTags->size() != 1) {
            session.SetErrorInfo("ChargableTag::Book", TStringBuilder() << "added " << optionalAddedTags->size() << " tags");
            return {};
        }

        auto result = std::move(optionalAddedTags->front());
        if (shouldAddChargableSessionTag) {
            if (!TChargableSessionTag::Register(offerId, result, NEntityTagsManager::EEntityType::User, permissions, server, session)) {
                return {};
            }
        }
        return result;
    }

    auto previousEvents = deviceTagsManager.GetEventsByObject(offer->GetObjectId(), session, 0, offer->GetTimestamp());
    if (!previousEvents) {
        return {};
    }
    for (auto&& ev : *previousEvents) {
        auto previousOfferContainer = ev.GetTagAs<TChargableTag>();
        auto previousOffer = previousOfferContainer ? previousOfferContainer->GetOffer() : nullptr;
        if (previousOffer && previousOffer->GetOfferId() == offer->GetOfferId()) {
            session.SetCode(HTTP_BAD_REQUEST);
            session.SetErrorInfo("ChargableTag::Book", TStringBuilder() << "offer " << offer->GetOfferId() << " has already been in use");
            return {};
        }
    }

    auto chargableTags = taggedDevice.GetTagsByClass<TChargableTag>();
    if (chargableTags.empty()) {
        session.SetErrorInfo("ChargableTag::Book", "no chargable tags");
        return {};
    }
    for (auto&& tag : chargableTags) {
        if (tag && tag->GetPerformer() == permissions.GetUserId()) {
            session.SetErrorInfo("ChargableTag::Book", "chargable tag is already performed");
            return {};
        }
    }
    std::sort(chargableTags.begin(), chargableTags.end(), [](const TDBTag& left, const TDBTag& right) {
        return left->GetPerformer() < right->GetPerformer();
    });
    auto chargableTag = std::move(chargableTags[0]);

    auto delegationTags = taggedDevice.GetTagsByClass<IDelegationTag>();
    if (delegationTags.size() > 1) {
        session.SetErrorInfo("ChargableTag::Book", TStringBuilder() << "multiple delegation tags: " << delegationTags.size(), EDriveSessionResult::IncorrectCarTags);
        return {};
    }
    auto delegationTag = !delegationTags.empty() ? std::move(delegationTags[0]) : TDBTag();

    bool futuresFlag = false;
    if (chargableTag->GetPerformer()) {
        if (delegationTag) {
            if (delegationTag->IsBusyFor(permissions.GetUserId())) {
                session.SetErrorInfo("ChargableTag::Book", "delegation tag is busy", EDriveSessionResult::CarIsBusy);
                return {};
            }
            if (!delegationTag.GetTagAs<IDelegationTag>()->IsAutoChargable()) {
                session.SetErrorInfo("delegation", "delegation.auto_chargable.disabled", EDriveSessionResult::InconsistencyUser);
                return {};
            }
            delegationTag.template MutableTagAs<IDelegationTag>()->SetOffer(offer);
            if (!deviceTagsManager.InitPerformer({delegationTag}, permissions, &server, session)) {
                return {};
            }
            return delegationTag;
        }
        if (!permissions.GetSetting<bool>(server.GetSettings(), "offers.futures.enabled", true)) {
            session.SetErrorInfo("ChargableTag::Book", "futures are disabled", EDriveSessionResult::CarIsBusy);
            return {};
        }
        if (!offer->HasCarWaitingDuration()) {
            session.SetErrorInfo("ChargableTag::Book", "waiting duration is missing", EDriveSessionResult::CarIsBusy);
            return {};
        }
        futuresFlag = true;
        if (bookOptions.Futures && bookOptions.Futures != true) {
            session.SetErrorInfo("ChargableTag::Book", TStringBuilder() << "unexpected futures", EDriveSessionResult::CarIsBusy);
            return {};
        }
        TVector<TDBTag> futuresTagsChecker;
        if (!deviceTagsManager.RestoreEntityTags(objectId, {TTagReservationFutures::TypeName, TTagReservationFutures::TypeName1}, futuresTagsChecker, session)) {
            return {};
        } else if (futuresTagsChecker.size()) {
            session.SetErrorInfo("ChargableTag::Book", "futures tags is already present", EDriveSessionResult::CarIsBusy);
            return {};
        }

        auto futuresTag = MakeAtomicShared<TTagReservationFutures>(offer);
        auto sb = deviceTagsManager.GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
        auto currentSession = sb ? sb->GetLastObjectSession(objectId) : nullptr;
        auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(currentSession);
        auto currentOffer = billingSession ? billingSession->GetCurrentOffer() : nullptr;
        auto fixPointOffer = std::dynamic_pointer_cast<const TFixPointOffer>(currentOffer);
        if (fixPointOffer) {
            futuresTag->SetExpectedCoord(fixPointOffer->GetFinish());
        }
        futuresTag->SetPerformer(permissions.GetUserId());

        bool shouldLockFutures = permissions.GetSetting<bool>(settings, "book.lock_futures").GetOrElse(true);
        if (shouldLockFutures) {
            auto futuresLockId = "futures:" + objectId;
            auto optionalFuturesLocked = session.TryLock(futuresLockId);
            if (!optionalFuturesLocked) {
                return {};
            }
            auto futuresLocked = *optionalFuturesLocked;
            if (!futuresLocked) {
                session.SetErrorInfo("ChargableTag::Book", "cannot lock " + futuresLockId, EDriveSessionResult::CarIsBusy);
                return {};
            }
        }

        auto optionalAddedTags = deviceTagsManager.AddTag(futuresTag, permissions.GetUserId(), objectId, &server, session, EUniquePolicy::SkipIfExists);
        if (!optionalAddedTags) {
            session.AddErrorMessage("ChargableTag::Book", "cannot add futures tag");
            return {};
        }
        if (optionalAddedTags->size() != 1) {
            session.SetErrorInfo("ChargableTag::Book", TStringBuilder() << "added " << optionalAddedTags->size() << " tags");
            return {};
        }
        session.Committed().Subscribe([offer, &server](const NThreading::TFuture<void>& w) {
            if (w.HasValue()) {
                NDrive::TEventLog::Log("BookFuturesOffer", offer->BuildJsonReport(ICommonOffer::TReportOptions{}, server));
            }
        });
        return std::move(optionalAddedTags->front());
    }

    if (!futuresFlag) {
        if (!offer->GetProfitableTags().empty()) {
            TVector<TDBTag> tags;
            if (!userTagsManager.RestoreTags(offer->GetProfitableTags(), tags, session)) {
                return {};
            }
            TVector<TDBTag> accountTags;
            if (!accountTagsManager.RestoreTags(offer->GetProfitableTags(), accountTags, session)) {
                return {};
            }
            tags.insert(tags.end(), accountTags.begin(), accountTags.end());
            if (tags.size() != offer->GetProfitableTags().size()) {
                session.SetError(NDrive::MakeError("offer_expired"));
                session.SetErrorInfo("ChargableTag::Book", "incorrect profitable tags", EDriveSessionResult::OfferExpired);
                return {};
            }
            for (auto&& i : tags) {
                if (!i.HasData() || offer->GetAlreadyUsedProfitableTags().contains(i.GetTagId())) {
                    continue;
                }
                auto tagImpl = i.GetTagAs<ITemporaryActionTag>();
                if (tagImpl && !tagImpl->Process(std::move(i), permissions.GetUserId(), server, session)) {
                    return {};
                }
            }
            offer->MutableAlreadyUsedProfitableTags().insert(offer->GetProfitableTags().begin(), offer->GetProfitableTags().end());
        }
    }

    // Promo landing
    auto modelPromoAction = server.GetSettings().GetValueDef<TString>("model_promo.flag_action", "");
    if (modelPromoAction && permissions.HasAction(modelPromoAction) && !offer->GetFromScanner()) {
        auto carObject = database.GetCarManager().GetObject(objectId);
        bool hasLandingToShow = false;
        TString landingId;
        TString carModelName;
        if (carObject) {
            carModelName = carObject->GetModel();
            auto settingName = "model_promo." + carModelName + ".landing_id";
            landingId = server.GetSettings().GetValueDef<TString>(settingName, "");
            if (landingId) {
                auto gAcceptances = database.GetUserLandingData()->FetchInfo(permissions.GetUserId(), session);
                if (!gAcceptances) {
                    return {};
                }
                auto result = gAcceptances.GetResultPtr(permissions.GetUserId());
                if (!result) {
                    hasLandingToShow = true;
                } else {
                    hasLandingToShow = true;
                    for (auto&& acceptance : result->GetLandings()) {
                        if (acceptance.GetId() == landingId && acceptance.GetComment()) {
                            hasLandingToShow = false;
                            break;
                        }
                    }
                }
            }
        }
        if (hasLandingToShow) {
            const TString& choice = session.GetRequestContext().GetUserChoice();
            if (choice) {
                TLandingAcceptance acceptance;
                acceptance.SetUserId(permissions.GetUserId()).SetId(landingId).SetComment(choice);
                INFO_LOG << "On-book landing answered with comment: " << choice << Endl;
                if (!driveApi->AcceptLandings(permissions, { acceptance }, Now(), session)) {
                    ALERT_LOG << "Could not accept promo landing: " << landingId << Endl;
                }
                if (choice == "accept") {
                    auto tagSettingName = "model_promo." + carModelName + ".accept_tag_id";
                    auto tagName = server.GetSettings().GetValueDef<TString>(tagSettingName, "");
                    if (tagName) {
                        auto tag = database.GetTagsManager().GetTagsMeta().CreateTag(tagName);
                        if (tag && !userTagsManager.AddTag(tag, permissions.GetUserId(), permissions.GetUserId(), &server, session)) {
                            ERROR_LOG << "Could not add acceptance tag" << Endl;
                            return {};
                        }
                    }
                }
            } else {
                if (!database.GetLandingsDB()->InitLandingUserRequest({ landingId }, "book", session)) {
                    ALERT_LOG << "No promo landing configured: " << landingId << Endl;
                } else {
                    return {};
                }
            }
        }
    }

    const bool isFastOffer = session.GetComment() == "metric:fast_button_tapped";
    if (session.GetRequestContext().GetUserChoice() != "accept") {
        auto action = server.GetDriveAPI()->GetRolesManager()->GetAction(offer->GetBehaviourConstructorId());
        auto offerBuilder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
        if (offerBuilder) {
            const auto& tags = offerBuilder->GetGrouppingTags();
            for (auto&& tag : tags) {
                auto key = isFastOffer ? "fast_offers." + tag + ".acceptance_landing_id" : "offers." + tag + ".acceptance_landing_id";
                auto landingId = server.GetSettings().GetValueDef<TString>(key, "");
                if (landingId) {
                    TLandingContext lContext(&server);
                    lContext.SetOffer(offer);
                    if (!database.GetLandingsDB()->InitLandingUserRequest({ landingId }, "book", session, &lContext)) {
                        ALERT_LOG << "No fast offer acceptance landing configured: " << landingId << Endl;
                    } else {
                        return {};
                    }
                }
            }
        }
    }

    auto tag = chargableTag.MutableTagAs<TChargableTag>();
    if (!tag) {
        session.SetErrorInfo("ChargableTag::Book", "internal_error", EDriveSessionResult::InternalError);
        return {};
    }
    tag->ClearDelegationType();
    tag->SetTransformationSkippedByExternalCommand(false);
    tag->SetOffer(offer);

    bool shouldValidateSnapshot = permissions.GetSetting<bool>(settings, "book.validate_snapshot").GetOrElse(true);
    auto validateSnapshot = [&](TAtomicSharedPtr<TRTDeviceSnapshot> snapshot) {
        if (!snapshot) {
            return false;
        }
        auto location = snapshot->GetLocation();
        auto mileage = snapshot->GetMileage();
        return location && mileage;
    };

    auto snapshot = server.GetSnapshotsManager().GetSnapshotPtr(objectId);
    if (shouldValidateSnapshot) {
        if (!validateSnapshot(snapshot)) {
            auto now = Now();
            auto deadline = now + TDuration::MilliSeconds(250);
            snapshot = server.GetSnapshotsManager().FetchSnapshot(objectId, deadline);
        }
        if (!snapshot) {
            session.SetErrorInfo("ChargableTag::Book", "cannot get device snapshot for " + objectId, NDrive::MakeError("device_snapshot_missing"));
            return {};
        }
        if (snapshot->GetImei() && !validateSnapshot(snapshot)) {
            session.SetErrorInfo("ChargableTag::Book", "cannot get mileage or location in device snapshot", NDrive::MakeError("device_snapshot_incomplete"));
            return {};
        }
    }
    tag->SetObjectSnapshot(snapshot);

    if (!deviceTagsManager.InitPerformer({chargableTag}, permissions, &server, session, /*replaceSnapshots=*/false)) {
        return {};
    }
    for (auto&& userTagName : offer->GetTargetUserTags()) {
        auto userTag = tagsMeta.CreateTag(userTagName);
        if (!userTag) {
            session.SetErrorInfo("ChargableTag::Book", "cannot create user tag " + userTagName);
            return {};
        }
        auto added = userTagsManager.AddTag(userTag, permissions.GetUserId(), offer->GetUserId(), &server, session);
        if (!added) {
            return {};
        }
    }

    if (shouldAddChargableSessionTag) {
        if (!TChargableSessionTag::Register(offerId, chargableTag, NEntityTagsManager::EEntityType::Car, permissions, server, session)) {
            return {};
        }
    }

    session.Committed().Subscribe([offer, &server, userId](const NThreading::TFuture<void>& w) {
        if (w.HasValue()) {
            NDrive::TEventLog::Log("BookOffer", offer->BuildJsonReport(ICommonOffer::TReportOptions{}, server));
        }
    });

    return chargableTag;
}

void TChargableTag::CloseSession(IEventsSession<TCarTagHistoryEvent>::TConstPtr session, const TUserPermissions& permissions, const NDrive::IServer& server, bool strict) {
    R_ENSURE(session, HTTP_INTERNAL_SERVER_ERROR, "null session");
    auto api = Yensured(server.GetDriveAPI());
    auto closedSession = session;
    if (!closedSession->GetClosed()) {
        auto firstEvent = session->GetFirstEvent();
        R_ENSURE(firstEvent, HTTP_INTERNAL_SERVER_ERROR, "cannot GetFirstEvent from session " << session->GetSessionId());

        auto tx = api->template BuildTx<NSQL::Writable>();
        auto sessionTags = api->GetTagsManager().GetDeviceTags().GetHistoryManager().GetEventsByTag(firstEvent->GetTagId(), tx, firstEvent->GetHistoryEventId());
        R_ENSURE(sessionTags, {}, "cannot GetEventsByTag " << firstEvent->GetTagId(), tx);

        auto closingSession = MakeAtomicShared<TBillingSession>();
        for (auto&& localEvent : *sessionTags) {
            if (closingSession->GetClosed()) {
                break;
            }
            auto cat = TBillingSessionSelector::AcceptImpl(localEvent);
            if (cat == NEventsSession::EEventCategory::IgnoreExternal || cat == NEventsSession::EEventCategory::IgnoreInternal) {
                continue;
            }
            closingSession->AddEvent(new TCarTagHistoryEvent(localEvent), cat);
        }
        closedSession = closingSession;
    }
    R_ENSURE(closedSession, HTTP_INTERNAL_SERVER_ERROR, "null closed session");
    R_ENSURE(closedSession->GetClosed(), HTTP_INTERNAL_SERVER_ERROR, "non-closed session " << closedSession->GetSessionId());

    const TString& userId = closedSession->GetUserId() ? closedSession->GetUserId() : permissions.GetUserId();
    NDrive::TEventLog::TUserIdGuard userIdGuard(userId);

    TBillingSession::TBillingCompilation compilation;
    R_ENSURE(closedSession->FillCompilation(compilation), HTTP_INTERNAL_SERVER_ERROR, "cannot FillCompilation");
    const auto& sessionId = compilation.GetSessionId();
    R_ENSURE(sessionId, HTTP_INTERNAL_SERVER_ERROR, "null session_id in compilation");

    auto finishedTask = api->HasBillingManager() ? api->AddClosedBillingInfo(compilation, permissions.GetUserId(), server, strict) : MakeUnexpected<TString>("no_billing_manager");
    R_ENSURE(finishedTask || !api->HasBillingManager(), HTTP_INTERNAL_SERVER_ERROR, "cannot AddClosedBillingInfo: " << finishedTask.GetError());

    auto billingSession = dynamic_cast<const TBillingSession*>(closedSession.Get());
    auto compiledRiding = billingSession ? billingSession->BuildCompiledRiding(&server, compilation, finishedTask.Get()) : nullptr;
    R_ENSURE(compiledRiding, HTTP_INTERNAL_SERVER_ERROR, "cannot construct compiled riding");

    {
        auto tx = api->template BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
        R_ENSURE(
            TChargableSessionTag::Deregister(sessionId, strict, permissions, server, tx),
            {},
            "cannot Deregister chargable_session_tag for " << sessionId,
            tx
        );
        R_ENSURE(
            api->GetMinimalCompiledRides().CreateCompiledRide(*compiledRiding, server, permissions.GetUserId(), tx),
            {},
            "cannot CreateCompiledRide",
            tx
        );
        R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    }

    TUnistatSignalsCache::SignalAdd("billing", "", compilation.GetBillingSumPrice());

    {
        auto optionalEvent = closedSession->GetLastEvent();
        if (optionalEvent) {
            auto ds = optionalEvent.GetRef()->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
            if (ds) {
                NJson::TJsonValue locationInfo = NJson::TMapBuilder("snapshot", ds->SerializeToJson());
                auto& areasList = locationInfo["areas"];
                if (auto areasDb = api->GetAreasDB()) {
                    auto actor = [&areasList](const TFullAreaInfo& areaInfo) {
                        areasList.AppendValue(areaInfo.GetArea().GetIdentifier());
                        return true;
                    };
                    areasDb->ProcessAreasInPoint(ds->GetSimpleLocation().GetCoord(), actor);
                }
                NDrive::TEventLog::Log("CloseSessionLastLocation", std::move(locationInfo));
            }
        }
    }

    if (permissions.IsFirstRiding() && compilation.GetBillingSumPrice() > 0) {
        bool invalidated = api->InvalidateFirstRiding(permissions.GetUserId(), *compiledRiding, &server);
        if (!invalidated) {
            NDrive::TEventLog::Log("CloseSessionError", NJson::TMapBuilder
                ("type", "InvalidateFirstRiding")
                ("session_debug_info", closedSession->GetDebugInfo())
            );
        }
    }

    if (auto userDevicesManager = server.GetUserDevicesManager()) {
        auto offer = compiledRiding->GetOffer().Get();
        if (offer) {
            Y_UNUSED(userDevicesManager->RegisterEvent(IUserDevicesManager::EEventGlobalTypes::TripOfferType, ::ToString(IUserDevicesManager::EOfferTypeEvents::Success), permissions.GetUserId(), compiledRiding->GetFinishInstant()));
            if (dynamic_cast<const TFixPointOffer*>(offer)) {
                Y_UNUSED(userDevicesManager->RegisterEvent(IUserDevicesManager::EEventGlobalTypes::TripOfferType, ::ToString(IUserDevicesManager::EOfferTypeEvents::SuccessFix), permissions.GetUserId(), compiledRiding->GetFinishInstant()));
            } else if (dynamic_cast<const TPackOffer*>(offer)) {
                Y_UNUSED(userDevicesManager->RegisterEvent(IUserDevicesManager::EEventGlobalTypes::TripOfferType, ::ToString(IUserDevicesManager::EOfferTypeEvents::SuccessPack), permissions.GetUserId(), compiledRiding->GetFinishInstant()));
            } else if (dynamic_cast<const TStandartOffer*>(offer)) {
                Y_UNUSED(userDevicesManager->RegisterEvent(IUserDevicesManager::EEventGlobalTypes::TripOfferType, ::ToString(IUserDevicesManager::EOfferTypeEvents::SuccessStandart), permissions.GetUserId(), compiledRiding->GetFinishInstant()));
            }

            auto action = api->GetRolesManager()->GetAction(offer->GetBehaviourConstructorId());
            if (!!action && !!action->GetAs<IOfferBuilderAction>()) {
                for (const auto& tag : (*action)->GetGrouppingTags()) {
                    Y_UNUSED(userDevicesManager->RegisterEvent(IUserDevicesManager::EEventGlobalTypes::TripTag, tag, permissions.GetUserId(), compiledRiding->GetFinishInstant()));
                }

                // deprecated
                const auto& tags = (*action)->GetGrouppingTags();
                if (tags.contains("cargo")) {
                    Y_UNUSED(userDevicesManager->RegisterEvent(IUserDevicesManager::EEventType::SuccessCargoTrip, permissions.GetUserId(), compiledRiding->GetFinishInstant()));
                }
                if (tags.contains("shuttle")) {
                    Y_UNUSED(userDevicesManager->RegisterEvent(IUserDevicesManager::EEventType::SuccessShuttleTrip, permissions.GetUserId(), compiledRiding->GetFinishInstant()));
                }
            }
        }
    }

    if (compiledRiding->HasBill() && permissions.GetUserFeatures().GetIsPlusUser()) {
        auto cashback = compiledRiding->GetBillUnsafe().GetCashbackSum();
        if (cashback > 0) {
            auto tx = api->template BuildTx<NSQL::Writable>();
            if (!api->IncrementCashbackSessions(permissions, &tx, server) || !tx.Commit()) {
                NDrive::TEventLog::Log("CloseSessionError", NJson::TMapBuilder
                    ("type", "IncrementCashbackSessions")
                    ("session_debug_info", closedSession->GetDebugInfo())
                    ("errors", tx.GetReport())
                );
            }
        }
        if (cashback > 0) {
            if (auto offer = compiledRiding->GetOffer()) {
                if (offer->HasIncreasedCashbackPercent() && offer->GetCashbackPercent() >= offer->GetIncreasedCashbackPercentRef()) {
                    auto increasedCashbackPushTag = permissions.GetSetting<TString>(server.GetSettings(), "tag.chargable_tag.increased_cashback_push_tag").GetOrElse("");
                    if (increasedCashbackPushTag) {
                        const auto& tagsMeta = server.GetDriveDatabase().GetTagsManager().GetTagsMeta();
                        const auto& userTags = server.GetDriveDatabase().GetTagsManager().GetUserTags();
                        if (auto tag = tagsMeta.CreateTagAs<TUserPushTag>(increasedCashbackPushTag)) {
                            auto locale = offer->GetLocale();
                            tag->AddMacro("_CashbackValue_", server.GetLocalization()->FormatPrice(locale, cashback));
                            auto tx = userTags.BuildTx<NSQL::Writable>();
                            auto added = userTags.AddTag(tag, permissions.GetUserId(), offer->GetUserId(), &server, tx);
                            if (!added || !tx.Commit()) {
                                NDrive::TEventLog::Log("CloseSessionError", NJson::TMapBuilder
                                    ("type", "AddIncreasedCashbackPushTag")
                                    ("session_debug_info", closedSession->GetDebugInfo())
                                    ("errors", tx.GetReport())
                                );
                                TUnistatSignalsCache::SignalAdd("orders", "add-increased_cashback_push_tag-error", 1);
                            }
                        } else {
                            ALERT_LOG << "Cannot create tag: " << increasedCashbackPushTag << Endl;
                        }
                    }
                }
            }
        }
    }
    if (server.GetSettings().GetValue<bool>("radar.close_session.enabled").GetOrElse(false)) {
        auto maybeEvent = session->GetLastEvent();
        if (!maybeEvent) {
            return;
        }
        auto devSnapshot = (*maybeEvent)->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
        if (!devSnapshot) {
            return;
        }
        const auto objectId = session->GetObjectId();
        auto coord = devSnapshot->GetSimpleLocation().GetCoord();
        auto tx = api->template BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
        bool applied = TRadarUserTag::ApplyRadarByCar(server, objectId, coord, tx);
        if (!applied) {
            NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
                ("stage", "ApplyRadarByCar")
                ("error", tx.GetStringReport())
            );
        }
        if (!tx.Commit()) {
            NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
                ("stage", "sessionCommit")
                ("error", tx.GetStringReport())
            );
        }
    }
}

bool TChargableTag::DropSession(const TString& sessionId, const TString& userId, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session) {
    const auto& database = server.GetDriveDatabase();
    const auto& deviceTagsManager = database.GetTagsManager().GetDeviceTags();
    const auto& userTagsManager = database.GetTagsManager().GetUserTags();

    TVector<TDBTag> userTags;
    if (!userTagsManager.RestoreEntityTags(userId, {}, userTags, session)) {
        return false;
    }

    for (auto&& tag : userTags) {
        auto offerContainer = tag.GetTagAs<IOfferContainer>();
        auto offer = offerContainer ? offerContainer->GetOffer() : nullptr;
        if (offer && offer->GetOfferId() == sessionId) {
            if (userTagsManager.RemoveTag(tag, permissions.GetUserId(), &server, session)) {
                session.Committed().Subscribe([offer, &server](const NThreading::TFuture<void>& w) {
                    if (w.HasValue()) {
                        NDrive::TEventLog::Log("DropOffer", offer->BuildJsonReport(ICommonOffer::TReportOptions{}, server));
                    }
                });
                return true;
            } else {
                return false;
            }
        }
    }

    TVector<TDBTag> preformedObjectTags;
    if (!deviceTagsManager.RestorePerformerTags({ userId }, preformedObjectTags, session)) {
        return false;
    }

    for (auto&& tag : preformedObjectTags) {
        auto futures = tag.GetTagAs<TTagReservationFutures>();
        auto offer = futures ? futures->GetOffer() : nullptr;
        if (offer && offer->GetOfferId() == sessionId) {
            if (deviceTagsManager.RemoveTag(tag, permissions.GetUserId(), &server, session)) {
                session.Committed().Subscribe([offer, &server](const NThreading::TFuture<void>& w) {
                    if (w.HasValue()) {
                        NDrive::TEventLog::Log("DropOffer", offer->BuildJsonReport(ICommonOffer::TReportOptions{}, server));
                    }
                });
                return true;
            } else {
                return false;
            }
        }
    }

    session.SetErrorInfo("ChargableTag::Drop", "cannot find session " + sessionId);
    return false;
}

bool TChargableTag::DirectEvolve(ISession::TPtr current, const TString& to, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session, const TEvolutionContext* context) {
    auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(current);
    if (!billingSession) {
        session.SetErrorInfo(
            "ChargableTag::Evolve",
            TStringBuilder() << "cannot cast session " << (current ? current->GetSessionId() : "null") << " to BillingSession",
            EDriveSessionResult::InternalError
        );
        return false;
    }
    auto lastEvent = billingSession->GetLastEvent();
    if (!lastEvent) {
        session.SetErrorInfo(
            "ChargableTag::Evolve",
            TStringBuilder() << "no last event in " << (current ? current->GetSessionId() : "null"),
            EDriveSessionResult::InternalError
        );
        return false;
    }
    if (billingSession->GetClosed()) {
        session.SetErrorInfo(
            "ChargableTag::Evolve",
            TStringBuilder() << "session " << (current ? current->GetSessionId() : "null") << " is closed",
            EDriveSessionResult::InternalError
        );
        return false;
    }
    return DirectEvolve(lastEvent->GetTagId(), to, permissions, server, session, context);
}

bool TChargableTag::DirectEvolve(const TString& tagId, const TString& to, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session, const TEvolutionContext* context) {
    auto expectedTag = Checked(server.GetDriveAPI())->GetTagsManager().GetDeviceTags().RestoreTag(tagId, session);
    if (!expectedTag) {
        auto where = "ChargableTag::DirectEvolve";
        session.AddErrorMessage(where, "cannot restore tag " + tagId);
        return false;
    }
    const auto& tag = *expectedTag;
    if (!tag) {
        session.SetErrorInfo("ChargableTag::DirectEvolve", "tag " + tagId + " is missing");
        return false;
    }
    return DirectEvolve(*expectedTag, to, permissions, server, session, context);
}

bool TChargableTag::DirectEvolve(const TDBTag& tag, const TString& to, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session, const TEvolutionContext* context) {
    auto next = MakeAtomicShared<TChargableTag>(to);
    return DirectEvolve(tag, next, permissions, server, session, context);
}

bool TChargableTag::DirectEvolve(const TDBTag& tag, TAtomicSharedPtr<TChargableTag> next, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session, const TEvolutionContext* context) {
    if (!tag) {
        session.SetErrorInfo("ChargableTag::Evolve", "null tag passed", EDriveSessionResult::InternalError);
        return false;
    }
    if (!next) {
        session.SetErrorInfo("ChargableTag::Evolve", "null target tag passed", EDriveSessionResult::InternalError);
        return false;
    }

    if (!next->CopyOnEvolve(*tag, nullptr, server)) {
        session.SetErrorInfo(
            "ChargableTag::Evolve",
            TStringBuilder() << "cannot CopyOnEvolve tag " << next->GetName() << " from " << tag.GetTagId()
        );
    }
    if (context && context->GetSwitching()) {
        next->SetIsSwitching(true);
    }

    if (context && context->IsReplacing()) {
        next->SetReplacing(true);
    }

    auto evolved = server.GetDriveDatabase().GetTagsManager().GetDeviceTags().EvolveTag(tag, next, permissions, &server, session, context);
    if (!evolved) {
        session.AddErrorMessage("ChargableTag::Evolve", TStringBuilder() << "cannot evolve tag " << tag.GetTagId() << " to " << next->GetName());
        return false;
    }
    return true;
}

bool TChargableTag::Switch(ISession::TConstPtr current, IOffer::TPtr offer, TUserPermissions::TPtr permissions, const NDrive::IServer& server, NDrive::TEntitySession& session, const TSwitchOptions& switchOptions) {
    if (!permissions) {
        session.SetErrorInfo("ChargableTag::Switch", "null permissions", EDriveSessionResult::InternalError);
        return false;
    }
    auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(current);
    if (!billingSession) {
        session.SetErrorInfo(
            "ChargableTag::Switch",
            TStringBuilder() << "cannot cast session " << (current ? current->GetSessionId() : "null") << " to BillingSession",
            EDriveSessionResult::InternalError
        );
        return false;
    }
    auto currentOffer = billingSession->GetCurrentOffer();
    if (!currentOffer) {
        session.SetErrorInfo(
            "ChargableTag::Switch",
            TStringBuilder() << "cannot get current offer from session " << current->GetSessionId(),
            EDriveSessionResult::InternalError
        );
        return false;
    }
    if (!currentOffer->GetSwitchable() && !switchOptions.IgnoreNonSwitchable) {
        session.SetErrorInfo(
            "ChargableTag::Switch",
            TStringBuilder() << "session " << current->GetSessionId() << " is not switchable",
            EDriveSessionResult::InternalError
        );
        return false;
    }
    if (currentOffer->CheckSession(*permissions, session, &server, true) != EDriveSessionResult::Success) {
        return false;
    }
    auto lastEvent = billingSession->GetLastEvent();
    if (!lastEvent) {
        session.SetErrorInfo(
            "ChargableTag::Switch",
            TStringBuilder() << "cannot get last event from session " << current->GetSessionId(),
            EDriveSessionResult::InternalError
        );
        return false;
    }
    const TString& lastEventTagName = lastEvent->GetData()->GetName();
    if (lastEventTagName == TTransformationTag::TypeName) {
        session.SetErrorInfo("ChargableTag::Switch", "cannot switch offer during transformation", EDriveSessionResult::CarIsBusy);
        return false;
    }
    const TString& targetTagName = switchOptions.TargetTagName ? switchOptions.TargetTagName : lastEventTagName;
    {
        const TString& tagId = lastEvent->GetTagId();
        TEvolutionContext forceEvolveContext;
        forceEvolveContext.SetMode(EEvolutionMode::Force);
        forceEvolveContext.SetSwitching(true);
        forceEvolveContext.SetReplacing(switchOptions.Replacing);
        if (!TChargableTag::DirectEvolve(tagId, Reservation, *permissions, server, session, &forceEvolveContext)) {
            return false;
        }
    }
    if (!offer) {
        offer = currentOffer->Clone();
        offer->SetOfferId(switchOptions.CopiedOfferId ? switchOptions.CopiedOfferId : ICommonOffer::CreateOfferId());
    } else {
        if (offer->GetOfferId() == currentOffer->GetOfferId()) {
            session.SetErrorInfo("ChargableTag::Switch", "cannot self-switch", EDriveSessionResult::InconsistencyUser);
            return false;
        }
    }
    offer->SetParentId(currentOffer->GetOfferId());
    offer->SetAlreadyUsedProfitableTags(currentOffer->GetAlreadyUsedProfitableTags());
    //offer->SetSelectedCharge(currentOffer->GetSelectedCharge());

    TChargableTag::TBookOptions bookOptions;
    bookOptions.DeviceId = switchOptions.DeviceId;
    bookOptions.MultiRent = switchOptions.MultiRent;
    auto activeSessionTag = TChargableTag::Book(offer, *permissions, server, session, bookOptions);
    if (!activeSessionTag) {
        return false;
    }
    {
        const auto& tagId = activeSessionTag.GetTagId();
        TEvolutionContext forceEvolveContext;
        forceEvolveContext.SetMode(EEvolutionMode::Force);
        forceEvolveContext.SetSwitching(true);
        if (targetTagName != Reservation) {
            if (!TChargableTag::DirectEvolve(activeSessionTag, Acceptance, *permissions, server, session, &forceEvolveContext)) {
                return false;
            }
            if (targetTagName != Acceptance) {
                if (!TChargableTag::DirectEvolve(tagId, Riding, *permissions, server, session, &forceEvolveContext)) {
                    return false;
                }
                if (targetTagName != Riding) {
                    if (!TChargableTag::DirectEvolve(tagId, Parking, *permissions, server, session, &forceEvolveContext)) {
                        return false;
                    }
                }
            }
        }
    }
    return true;
}

TMaybe<TDBTag> TChargableSessionTag::Get(const TString& sessionId, const IDriveTagsManager& driveTagsManagers, NDrive::TEntitySession& tx) {
    const auto& traceTagManager = driveTagsManagers.GetTraceTags();
    auto taggedSession = traceTagManager.RestoreObject(sessionId, tx);
    if (!taggedSession) {
        return {};
    }
    auto existing = taggedSession->GetTag(TChargableSessionTag::Type());
    if (!existing) {
        return TDBTag();
    }
    return existing;
}

bool TChargableSessionTag::Register(const TString& sessionId, const TDBTag& sessionTag, NEntityTagsManager::EEntityType type, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
    const auto eg = NDrive::BuildEventGuard("ChargableSessionTag::Register");
    const auto& tagsMeta = server.GetDriveDatabase().GetTagsManager().GetTagsMeta();
    const auto& traceTagManager = server.GetDriveDatabase().GetTagsManager().GetTraceTags();
    auto optionalEvents = traceTagManager.GetEventsByObject(sessionId, tx);
    if (!optionalEvents) {
        return false;
    }
    for (auto&& ev : *optionalEvents) {
        auto impl = ev.GetTagAs<TChargableSessionTag>();
        if (impl && impl->GetSessionTagEntityType() == type) {
            tx.SetCode(HTTP_BAD_REQUEST);
            tx.SetErrorInfo("ChargableSessionTag::Register", TChargableSessionTag::Type() + " has already been in use: " + ev.GetTagId());
            return false;
        }
    }

    auto tag = tagsMeta.CreateTagAs<TChargableSessionTag>(TChargableSessionTag::Type());
    if (!tag) {
        tx.SetErrorInfo("ChargableSessionTag::Register", "cannot create tag " + TChargableSessionTag::Type());
        return false;
    }
    tag->SetSessionTagId(TUuid::Parse(sessionTag.GetTagId()));
    tag->SetSessionTagEntityType(type);

    auto added = traceTagManager.AddTag(tag, permissions.GetUserId(), sessionId, &server, tx);
    if (!added) {
        return false;
    }
    if (added->size() != 1) {
        tx.SetErrorInfo("ChargableSessionTag::Register", TStringBuilder() << "bad number of chargable_session tags added: " << added->size());
        return false;
    }
    return true;
}

bool TChargableSessionTag::Deregister(const TString& sessionId, bool strict, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
    const auto eg = NDrive::BuildEventGuard("ChargableSessionTag::Deregister");
    const auto& traceTagManager = server.GetDriveDatabase().GetTagsManager().GetTraceTags();
    auto restoredTrace = traceTagManager.RestoreObject(sessionId, tx);
    if (!restoredTrace) {
        return false;
    }

    auto chargableSessionTag = restoredTrace->GetTag(TChargableSessionTag::Type());
    if (chargableSessionTag) {
        if (!traceTagManager.RemoveTag(*chargableSessionTag, permissions.GetUserId(), &server, tx)) {
            return false;
        }
    } else if (strict) {
        tx.SetErrorInfo("ChargableSessionTag::Deregister", "cannot find tag for session " + sessionId);
        return false;
    }
    return true;
}

bool TChargableSessionTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(value, errors)) {
        return false;
    }
    return NJson::TryFieldsFromJson(value, GetFields(), errors);
}

void TChargableSessionTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    NJson::FieldsToJson(value, GetFields());
}

TChargableSessionTag::TProto TChargableSessionTag::DoSerializeSpecialDataToProto() const {
    auto result = TBase::DoSerializeSpecialDataToProto();
    SessionTagId.Serialize(*result.MutableSessionTagId());
    result.SetSessionTagEntityType(static_cast<ui32>(SessionTagEntityType));
    return result;
}

bool TChargableSessionTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }
    if (proto.HasSessionTagId() && !SessionTagId.Deserialize(proto.GetSessionTagId())) {
        return false;
    }
    if (proto.HasSessionTagEntityType()) {
        SessionTagEntityType = static_cast<NEntityTagsManager::EEntityType>(proto.GetSessionTagEntityType());
    }
    return true;
}

ITag::TFactory::TRegistrator<TChargableSessionTag> TChargableSessionTag::Registrator(TChargableSessionTag::Type());

bool TBillingCompilation::Fill(const TVector<TBillingCompilation::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events) {
    OfferSegments.clear();
    OfferStates.clear();
    LocalEvents.clear();
    ReportSumPrice = 0;
    ReportSumOriginalPrice = 0;
    BillingSumPrice = 0;
    BillingSumOriginalPrice = 0;
    DepositSum = 0;
    Revenue = 0;
    IsFinished = true;
    if (timeline.size()) {
        const TChargableTag* cTag = events[timeline.front().GetEventIndex()]->GetTagAs<TChargableTag>();
        if (cTag) {
            const THistoryDeviceSnapshot* snapshot = cTag->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
            if (snapshot) {
                StartGeoTags = snapshot->GetLocationTagsArray();
            }
        }
    }
    {
        TString currentTagName;
        TOfferPricing pricesByOffer(nullptr);

        IOffer::TPtr offerCurrent = nullptr;
        TVector<TTimeEvent> timelineOffer;
        ui32 acceptanceCount = 0;
        ui32 idx = 0;
        for (auto&& tl : timeline) {
            bool offersIsEqual = true;
            const TChargableTag* chargableTag = nullptr;
            EObjectHistoryAction historyAction = EObjectHistoryAction::Unknown;
            IOffer::TPtr newOffer;
            if (tl.GetTimeEvent() != EEvent::CurrentFinish) {
                const auto& ev = events[tl.GetEventIndex()];
                LocalEvents.emplace_back(ev);
                chargableTag = ev->GetTagAs<TChargableTag>();
                currentTagName = (*ev)->GetName();
                historyAction = ev->GetHistoryAction();
                newOffer = chargableTag ? chargableTag->GetVehicleOffer() : std::dynamic_pointer_cast<IOffer>(TChargableTag::RestoreOffer(*ev));
                bool transformationSkippedByExternalCommand = chargableTag ? chargableTag->IsTransformationSkippedByExternalCommand() : false;
                if (transformationSkippedByExternalCommand) {
                    LocalEvents.back().SetTransformationSkippedByExternalCommand(true);
                }
            } else {
                IsFinished = false;
            }
            if (newOffer && offerCurrent) {
                offersIsEqual = offerCurrent->GetOfferId() == newOffer->GetOfferId() &&
                                offerCurrent->GetOfferRevision() == newOffer->GetOfferRevision();
            }
            if (++idx == timeline.size() || (!!newOffer && (!offerCurrent || !offersIsEqual))) {
                if (!!offerCurrent) {
                    timelineOffer.push_back(tl);
                    TOfferPricing pricesByOffer(offerCurrent);
                    TOfferStatePtr state = offerCurrent->Calculate(timelineOffer, events, Until, pricesByOffer);
                    if (!state) {
                        return false;
                    }
                    OfferSegments.push_back(pricesByOffer);
                    OfferStates.push_back(std::move(state));
                } else if (!!newOffer) {
                    SessionId = newOffer->GetOfferId();
                }
                offerCurrent = newOffer;
                timelineOffer.clear();
                timelineOffer.push_back(tl);
            } else {
                timelineOffer.push_back(tl);
            }
            if (currentTagName == TChargableTag::Acceptance && historyAction == EObjectHistoryAction::TagEvolve) {
                acceptanceCount++;
                HasAcceptance = true;
            }
            if (currentTagName == TChargableTag::Riding || currentTagName == TChargableTag::Parking) {
                Accepted = true;
            }
            if (acceptanceCount > 1) {
                Accepted = true;
            }
            if (chargableTag) {
                DelegationType = chargableTag->OptionalDelegationType();
            }
        }

        for (auto&& i : OfferSegments) {
            BillingSumPrice += i.GetBillingSumPrice();
            BillingSumOriginalPrice += i.GetBillingSumOriginalPrice();
            ReportSumPrice += i.GetReportSumPrice();
            ReportSumOriginalPrice += i.GetReportSumOriginalPrice();
            Revenue += i.GetBillingRevenue();
            DepositSum = Max<double>(DepositSum, i.GetDeposit());
        }
        if (OfferSegments.size() && timeline.size()) {
            auto* priceInfo = OfferSegments.back().GetPrices(currentTagName);
            const TDuration stateDuration = priceInfo ? priceInfo->GetDuration() : TDuration::Zero();
            FreeTime = OfferSegments.back().GetOffer()->GetFreeTime(stateDuration, currentTagName);
        }
    }
    auto servicingResults = TBillingSession::CalcServicingResults(timeline, events, TInstant::Max());
    auto servicingInfos = TVector<TServicingInfo>();
    for (auto&& servicingResult : servicingResults) {
        if (!servicingResult.Info) {
            continue;
        }
        servicingInfos.push_back(*servicingResult.Info);
    }
    ServicingInfos = std::move(servicingInfos);
    return true;
}

TBill TBillingCompilation::GetBill(ELocalization locale, const NDrive::IServer* server, const TPaymentsData* task) const {
    TBill result;
    Y_ENSURE_BT(OfferSegments.size() == OfferStates.size(), OfferSegments.size() << " / " << OfferStates.size());
    if (task) {
        Y_ENSURE_BT(task->GetBillingTask().GetId() == GetSessionId(), task->GetBillingTask().GetId() << " / " << GetSessionId());
    }
    for (ui32 segmentIdx = 0; segmentIdx < OfferSegments.size(); ++segmentIdx) {
        const TOfferPricing& offerSegment = OfferSegments[segmentIdx];
        if (!offerSegment.GetOffer()) {
            ERROR_LOG << "Incorrect offer in session" << Endl;
            continue;
        }
        offerSegment.GetOffer()->FillBill(result, offerSegment, OfferStates[segmentIdx], locale, server, offerSegment.GetOffer()->GetCashbackPercent());
    }

    ui32 bonuses = 0;
    if (server->GetDriveAPI()->HasBillingManager() && task) {
        const TBillingManager& billingManager = server->GetDriveAPI()->GetBillingManager();
        auto billingReport = billingManager.GetSessionReport(*task);
        {
            TMap<ui32, NDrive::NBilling::IBillingAccount::TPtr> accounts;
            for (auto&& account : billingManager.GetAccountsManager().GetUserAccounts(billingReport.GetUserId())) {
                accounts[account->GetId()] = account;
            }

            ui32 paymethodsCount = 0;
            ui32 noCardSum = 0;
            for (auto&& accountInfo : billingReport.SumByAccount) {
                auto it = accounts.find(accountInfo.first);
                if (it == accounts.end()) {
                    continue;;
                }
                TBillRecord billRecord;
                billRecord.SetCost(accountInfo.second);
                if (it->second->GetType() == NDrive::NBilling::EAccount::Bonus || it->second->GetType() == NDrive::NBilling::EAccount::Coins) {
                    paymethodsCount++;
                    bonuses += accountInfo.second;
                    noCardSum += accountInfo.second;
                    billRecord
                        .SetType(TBillRecord::BillingTypePrefix + "bonus")
                        .SetTitle(NDrive::TLocalization::BonusBillingHeader())
                        .SetDetails(NDrive::TLocalization::BonusBillingDetails());
                    result.MutableRecords().emplace_back(std::move(billRecord));
                } else if (it->second->GetType() == NDrive::NBilling::EAccount::Wallet) {
                    paymethodsCount++;
                    noCardSum += accountInfo.second;
                    billRecord
                        .SetType(TBillRecord::BillingTypePrefix + "wallet")
                        .SetTitle(it->second->GetName());
                    result.MutableRecords().emplace_back(std::move(billRecord));
                } else if (it->second->GetType() == NDrive::NBilling::EAccount::YAccount) {
                    paymethodsCount++;
                    bonuses += accountInfo.second;
                    noCardSum += accountInfo.second;
                    billRecord
                        .SetType(TBillRecord::BillingTypePrefix + "yaccount")
                        .SetTitle(NDrive::TLocalization::BillPlus());
                    result.MutableRecords().emplace_back(std::move(billRecord));
                }
            }
            if (paymethodsCount && ReportSumPrice > noCardSum) {
                TBillRecord billRecord;
                billRecord
                    .SetCost(ReportSumPrice - noCardSum)
                    .SetType(TBillRecord::BillingTypePrefix + "card")
                    .SetTitle(NDrive::TLocalization::CardBillingHeader());
                result.MutableRecords().emplace_back(std::move(billRecord));
            }
            ui32 cashbackSum = ICommonOffer::RoundCashback(billingReport.GetCashbackBaseSum() * result.GetCashbackPercent() / 100., 100);
            if (cashbackSum > 0) {
                TBillRecord billRecord;
                billRecord
                    .SetCost(cashbackSum)
                    .SetType(TBillRecord::CashbackType)
                    .SetIcon(server->GetSettings().GetValueDef<TString>("plus_coin_icon", ""))
                    .SetTitle(NDrive::TLocalization::CashbackBillingHeader());
                result.MutableRecords().emplace_back(std::move(billRecord));
            }
        }
    }

    TBillRecord billRecord;
    billRecord
        .SetCost(ReportSumPrice - bonuses)
        .SetType(TBillRecord::TotalType)
        .SetTitle(NDrive::TLocalization::TotalBillHeader());
    result.MutableRecords().emplace_back(std::move(billRecord));

    TDuration freeDuration;
    TDuration pricedDuration;
    if (GetFreeAndPricedDurations(freeDuration, pricedDuration)) {
        result.SetFreeDuration(freeDuration);
        result.SetPricedDuration(pricedDuration);
    }
    return result;
}

TVector<TString> TBillingCompilation::GetBillingSessionSignals(const NDrive::IServer& server) const {
    TVector<TString> result;
    if (!GetCurrentOffer()) {
        return TVector<TString>();
    } else {
        result.emplace_back(GetCurrentOffer()->GetTypeName());
        result.emplace_back(GetCurrentOffer()->GetPriceConstructorId());
        result.emplace_back(GetCurrentOffer()->GetBehaviourConstructorId());
        TMaybe<TDriveCarInfo> carData = server.GetDriveAPI()->GetCarsData()->GetObject(GetCurrentOffer()->GetObjectId());
        if (carData) {
            result.emplace_back(carData->GetModel());
        }
    }
    if (StartGeoTags.empty()) {
        result.emplace_back("other");
    }
    result.insert(result.end(), StartGeoTags.begin(), StartGeoTags.end());
    return result;
}

const NUnistat::TIntervals TBillingCompilation::IntervalCompiledBills = {0, 3000, 5000, 7500, 10000, 20000, 30000, 40000, 50000, 75000, 100000, 200000, 500000, 1000000};
const NUnistat::TIntervals TBillingCompilation::IntervalCompiledDurations = {0, 1 * 60, 2 * 60, 3 * 60, 5 * 60, 7 * 60, 10 * 60, 13 * 60, 15 * 60, 17 * 60, 20 * 60, 25 * 60, 30 * 60, 40 * 60, 50 * 60, 60 * 60, 120 * 60};
const NUnistat::TIntervals TBillingCompilation::IntervalCompiledFPFraction = {0, 0.1, 0.2, 0.3, 0.4, 0.5, 0.5, 0.6, 0.7, 0.8, 0.9, 1, 2, 5, 10, 15, 20, 100};

TMaybe<TOfferPricing> TBillingCompilation::GetCurrentOfferPricing() const {
    if (OfferSegments.empty()) {
        WARNING_LOG << "Try to use empty session: " << SessionId << Endl;
        return {};
    }
    return OfferSegments.back();
}

IOffer::TPtr TBillingCompilation::GetCurrentOffer() const {
    auto pricing = GetCurrentOfferPricing();
    if (!pricing) {
        return nullptr;
    }
    return pricing->GetOffer();
}

TOfferStatePtr TBillingCompilation::GetCurrentOfferState() const {
    if (OfferStates.empty()) {
        return nullptr;
    }
    return OfferStates.back();
}

NJson::TJsonValue TBillingCompilation::GetReport(ELocalization locale, const NDrive::IServer* server, ISessionReportCustomization::TPtr customization) const {
    auto billingCustomization = std::dynamic_pointer_cast<const TBillingReportCustomization>(customization);
    NJson::TJsonValue result;
    TMap<TString, TDuration> durationByTags;
    TMap<TString, double> priceByTags;
    TDuration totalDuration = GetSumDuration();
    const bool isSimpleReport = !!billingCustomization && billingCustomization->GetIsSimpleReport();
    {
        for (auto&& offerPrices : OfferSegments) {
            for (auto&& i : offerPrices.GetPrices()) {
                durationByTags[i.first] += i.second.GetDuration();

                if (billingCustomization && billingCustomization->GetNeedPriceByTags()) {
                    priceByTags[i.first] += i.second.GetPrice();
                }
            }
        }
    }
    auto currentOffer = GetCurrentOffer();
    auto currentState = GetCurrentOfferState();
    if (!isSimpleReport && currentOffer && billingCustomization) {
        ICommonOffer::TReportOptions options;
        options.Agreement = billingCustomization->GetAgreement();
        options.Traits = billingCustomization->GetReportTraits();
        options.Locale = locale;
        options.DurationByTags = durationByTags;
        result.InsertValue("current_offer", currentOffer->BuildJsonReport(options, *server));
    }
    if (!isSimpleReport && currentState) {
        result.InsertValue("current_offer_state", currentState->GetReport(locale, *server));
    }
    auto currency = currentOffer ? ("units.short." + currentOffer->GetCurrency()) : "";

    result.InsertValue("total_price_hr", server->GetLocalization()->FormatPrice(locale, TStandartOffer::RoundPrice(int(ReportSumPrice)), {currency}));
    result.InsertValue("total_price", int(ReportSumPrice));
    result.InsertValue("total_duration", totalDuration.Seconds());
    {
        NJson::TJsonValue jsonInfo(NJson::JSON_MAP);
        for (auto&& i : durationByTags) {
            jsonInfo.InsertValue(i.first, i.second.Seconds());
        }
        result.InsertValue("durations_by_tags", jsonInfo);
    }
    if (billingCustomization && billingCustomization->GetNeedPriceByTags()) {
        NJson::TJsonValue jsonInfo(NJson::JSON_MAP);
        for (auto&& i : priceByTags) {
            jsonInfo.InsertValue(i.first, i.second);
        }
        result.InsertValue("price_by_tags", jsonInfo);
    }

    if ((durationByTags.size() == 1) && ReportSumOriginalPrice < 1e-5 && IsFinished) {
        result.InsertValue("is_cancelled", true);
    } else {
        result.InsertValue("is_cancelled", false);
    }

    if (!billingCustomization || billingCustomization->GetNeedBill()) {
        auto bill = GetBill(locale, server, billingCustomization ? billingCustomization->GetPaymentsDataPtr() : nullptr);
        auto cashback = bill.GetCashbackSum();
        if (cashback > 0) {
            result.InsertValue("total_cashback", cashback);
            result.InsertValue("total_cashback_hr", server->GetLocalization()->FormatPrice(locale, TStandartOffer::RoundCashback(cashback, 100)));
        }
        result.InsertValue("bill", bill.GetReport(locale, billingCustomization ? billingCustomization->GetReportTraits() : NDriveSession::ReportAll, *server));
    }
    if (!isSimpleReport) {
        result.InsertValue("free_time", FreeTime.Seconds());
    }
    if (DelegationType) {
        result.InsertValue("delegation_type", ToString(DelegationType));
    }
    if (!ServicingInfos.empty()) {
        result.InsertValue("servicing_info", NJson::ToJson(ServicingInfos));
    }
    auto standardOffer = std::dynamic_pointer_cast<TStandartOffer>(currentOffer);
    if (standardOffer) {
        result.InsertValue(InsuranceTypeId, standardOffer->GetInsuranceType());
    }
    auto currentPricing = GetCurrentOfferPricing();
    if (currentPricing) {
        result.InsertValue("price_details", currentPricing->GetPublicReport(*Yensured(server->GetLocalization()), locale, currency));
    }
    return result;
}

bool TBillingCompilation::GetFreeAndPricedDurations(TDuration& freeDuration, TDuration& pricedDuration) const {
    freeDuration = TDuration::Zero();
    pricedDuration = TDuration::Zero();
    for (auto&& i : OfferSegments) {
        TDuration freeDurationOffer;
        TDuration pricedDurationOffer;
        if (i.GetOffer()->GetFreeAndPricedDurations(freeDuration, pricedDuration, i.GetPrices())) {
            freeDuration += freeDurationOffer;
            pricedDuration += pricedDurationOffer;
        }
    }
    return true;
}

TDuration TBillingCompilation::GetSumDuration() const {
    return LocalEvents.empty() ? TDuration::Zero() : (LocalEvents.back().GetInstant() - LocalEvents.front().GetInstant());
}

TDuration TBillingCompilation::GetRideDuration() const {
    TDuration result = TDuration::Zero();
    for (auto&& i : OfferSegments) {
        for (auto&& s : i.GetPrices()) {
            if (s.first == TChargableTag::Riding) {
                result += s.second.GetDuration();
            }
        }
    }
    return result;
}

TDuration TBillingCompilation::GetActiveDuration() const {
    TDuration result = TDuration::Zero();
    for (auto&& i : OfferSegments) {
        for (auto&& s : i.GetPrices()) {
            if (s.first != TChargableTag::Reservation) {
                result += s.second.GetDuration();
            }
        }
    }
    return result;
}

IOffer::TPtr TBillingSession::GetCurrentOffer() const {
    TGuard<TMutex> g(Mutex);
    TMaybe<TBillingCompilation> compilation = GetCompilationAs<TBillingCompilation>();
    if (!compilation) {
        return nullptr;
    }
    return compilation->GetCurrentOffer();
}

THolder<TBillingSession::ICompilation> TBillingSession::BuildDefaultCompilation() const {
    return MakeHolder<TBillingCompilation>();
}

bool TBillingSession::GetFreeAndPricedDurations(TDuration& freeDuration, TDuration& pricedDuration) const {
    TGuard<TMutex> g(Mutex);
    TMaybe<TBillingCompilation> compilation = GetCompilationAs<TBillingCompilation>();
    if (!compilation || !Compile()) {
        return false;
    }
    return compilation->GetFreeAndPricedDurations(freeDuration, pricedDuration);
}

THolder<TFullCompiledRiding> TBillingSession::BuildCompiledRiding(const NDrive::IServer* server, const TPaymentsData* task) const {
    TBillingCompilation compilation;
    if (!FillCompilation(compilation)) {
        return nullptr;
    }
    return BuildCompiledRiding(server, compilation, task);
}

THolder<TFullCompiledRiding> TBillingSession::BuildCompiledRiding(const NDrive::IServer* server, const TBillingCompilation& compilation, const TPaymentsData* task) const {
    auto offer = compilation.GetCurrentOffer();
    auto locale = offer ? offer->GetLocale() : DefaultLocale;
    auto result = MakeHolder<TFullCompiledRiding>();
    result
        ->SetOffer(offer)
        .SetBill(compilation.GetBill(locale, server, task))
        .SetLocalEvents(compilation.GetLocalEvents())
        .SetServicingInfos(compilation.GetServicingInfos())
        .SetDelegationType(compilation.OptionalDelegationType())
        .SetOfferName(offer ? offer->GetName() : TString{})
        .SetFinishInstant(GetLastTS())
        .SetSumPrice(compilation.GetReportSumPrice())
        .SetStartInstant(GetStartTS())
        .SetObjectId(GetObjectId())
        .SetSessionId(GetSessionId());

    const auto& objectId = GetObjectId();
    if (objectId && server) {
        auto object = server->GetDriveDatabase().GetCarManager().GetObject(objectId);
        R_ENSURE(object, HTTP_INTERNAL_SERVER_ERROR, "cannot find object " << objectId);
        result->SetIMEI(object->GetIMEI());
    }

    TSnapshotsDiffCompilation snapshotsDiff;
    if (FillCompilation(snapshotsDiff)) {
        result->SetSnapshotsDiff(snapshotsDiff.GetSnapshotsDiff(server));
    }
    return result;
}

TAtomicSharedPtr<TBillingSession> TBillingSession::BuildFromCompiled(const NDrive::IServer* server, const TObjectEvent<TMinimalCompiledRiding>& compiledRide, TMessagesCollector& errors) {
    const auto& compiledRides = server->GetDriveAPI()->GetMinimalCompiledRides();
    auto tx = compiledRides.BuildSession(true);
    auto optionalCompiledSession = compiledRides.GetEvent(compiledRide.GetHistoryEventId(), tx);
    if (!optionalCompiledSession) {
        errors.MergeMessages(tx.GetMessages());
        return nullptr;
    }

    auto compiledSession = *optionalCompiledSession;
    if (!compiledSession) {
        errors.AddMessage("BuildFromCompiled", "cannot find session " + compiledRide.GetSessionId());
        return nullptr;
    }

    TVector<TString> localEventsIds;
    for (auto&& event : compiledSession->GetLocalEvents()) {
        localEventsIds.push_back(ToString(event.GetHistoryEventId()));
    }

    NStorage::TObjectRecordsSet<TCarTagHistoryEvent, IHistoryContext> historyEvents(&(server->GetDriveAPI()->GetTagsManager().GetContext()));
    {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        auto result = session->Exec("SELECT * FROM car_tags_history WHERE history_event_id IN (" + JoinVectorIntoString(localEventsIds, ",") + ")", &historyEvents);

        if (!result->IsSucceed()) {
            errors.AddMessage("BuildFromCompiled", "Cannot restore tags history");
            return nullptr;
        }
        if (historyEvents.size() != localEventsIds.size()) {
            if (!result->IsSucceed()) {
                errors.AddMessage("BuildFromCompiled", "Inconsistency in tags history");
                return nullptr;
            }
        }
    }

    auto result = MakeAtomicShared<TBillingSession>();
    for (auto&& localEvent : historyEvents) {
        result->AddEvent(new TCarTagHistoryEvent(localEvent), TBillingSessionSelector::AcceptImpl(localEvent));
    }
    return result;
}

TBillingSession::TServicingResults TBillingSession::CalcServicingResults(const TVector<TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, TInstant until) {
    TServicingResults result;
    TMaybe<TServicingResult> segment;
    TString objectId;
    Y_ASSERT(std::is_sorted(timeline.begin(), timeline.end()));
    for (auto&& tl : timeline) {
        if (tl.GetEventInstant() > until && !segment) {
            break;
        }
        if (tl.GetTimeEvent() == IEventsSession<TCarTagHistoryEvent>::EEvent::CurrentFinish) {
            break;
        }
        Y_ASSERT(tl.GetEventIndex() < events.size());
        const auto& ev = *Yensured(events[tl.GetEventIndex()]);
        if (!segment && ev->GetName() == TChargableTag::Servicing) {
            segment.ConstructInPlace();
            segment->Start = tl.GetEventInstant();
            auto snapshot = ev->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
            if (snapshot) {
                segment->StartMileage = snapshot->GetMileage();
                segment->LocationTags = MakeSet(snapshot->GetLocationTagsArray());
            }
            if (!objectId) {
                objectId = ev.GetObjectId();
            }
        }
        if (segment && ev->GetName() != TChargableTag::Servicing) {
            segment->Finish = std::min(tl.GetEventInstant(), until);
            auto chargableTag = ev.GetTagAs<TChargableTag>();
            auto servicingInfo = chargableTag ? chargableTag->GetServicingInfo() : nullptr;
            if (servicingInfo) {
                segment->Info = *servicingInfo;
                segment->Info->Finish = std::min(segment->Info->Finish, until);
            }
            auto snapshot = ev->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
            if (snapshot) {
                segment->FinishMileage = snapshot->GetMileage();
            }
            result.push_back(std::move(*segment));
            segment.Clear();
        }
    }
    if (segment) {
        segment->Finish = until;
        auto snapshot = NDrive::HasServer() ? NDrive::GetServerAs<NDrive::IServer>().GetSnapshotsManager().GetSnapshotPtr(objectId) : nullptr;
        if (snapshot) {
            segment->FinishMileage = snapshot->GetMileage();
        }
        segment->Info.ConstructInPlace();
        segment->Info->Start = segment->Start;
        segment->Info->Finish = segment->Finish;
        segment->Info->Mileage = segment->StartMileage && segment->FinishMileage
            ? *segment->FinishMileage - *segment->StartMileage
            : 0
        ;
        result.push_back(std::move(*segment));
    }
    return result;
}

TMaybe<TSnapshotsDiff> TSnapshotsDiffCompilation::GetSnapshotsDiff(const NDrive::IServer* server) const {
    auto deviceSnapshotStart = std::dynamic_pointer_cast<const THistoryDeviceSnapshot>(GetSnapshotStart());
    auto deviceSnapshotFinish = std::dynamic_pointer_cast<const THistoryDeviceSnapshot>(GetSnapshotFinish());

    if (!deviceSnapshotStart) {
        return {};
    }
    if (!deviceSnapshotFinish) {
        deviceSnapshotFinish = server ? server->GetSnapshotsManager().GetSnapshotPtr(ObjectId) : nullptr;
    }
    if (!deviceSnapshotFinish) {
        return {};
    }

    TSnapshotsDiff result;
    bool accepted = false;
    auto startMileage = deviceSnapshotStart->GetMileage();
    auto startFuelLevel = deviceSnapshotStart->GetFuelLevel();
    auto startFuelVolume = deviceSnapshotStart->GetFuelVolume();
    auto startFuelLevelTimestamp = deviceSnapshotStart->GetSnapshot().FuelLevelTimestamp;
    auto startFuelVolumeTimestamp = deviceSnapshotStart->GetSnapshot().FuelVolumeTimestamp;
    for (auto&& [tag, snapshot] : NamedSnapshots) {
        auto historyDeviceSnapshot = std::dynamic_pointer_cast<const THistoryDeviceSnapshot>(snapshot);
        if (!startMileage && historyDeviceSnapshot) {
            startMileage = historyDeviceSnapshot->GetMileage();
        }
        if (!startFuelLevel && historyDeviceSnapshot) {
            startFuelLevel = historyDeviceSnapshot->GetFuelLevel();
            startFuelLevelTimestamp = historyDeviceSnapshot->GetSnapshot().FuelLevelTimestamp;
        }
        if (!startFuelVolume && historyDeviceSnapshot) {
            startFuelVolume = historyDeviceSnapshot->GetFuelVolume();
            startFuelVolumeTimestamp = historyDeviceSnapshot->GetSnapshot().FuelVolumeTimestamp;
        }
        if (!accepted && (tag == TChargableTag::Riding || tag == TChargableTag::Parking)) {
            accepted = true;
            auto ridingMileage = historyDeviceSnapshot ? historyDeviceSnapshot->GetMileage() : Nothing();
            if (ridingMileage) {
                startMileage = ridingMileage;
            }
            auto ridingFuelLevel = historyDeviceSnapshot ? historyDeviceSnapshot->GetFuelLevel() : Nothing();
            if (ridingFuelLevel) {
                startFuelLevel = ridingFuelLevel;
                startFuelLevelTimestamp = historyDeviceSnapshot->GetSnapshot().FuelLevelTimestamp;
            }
            auto ridingFuelVolume = historyDeviceSnapshot ? historyDeviceSnapshot->GetFuelVolume() : Nothing();
            if (ridingFuelVolume) {
                startFuelVolume = ridingFuelVolume;
                startFuelVolumeTimestamp = historyDeviceSnapshot->GetSnapshot().FuelDistanceTimestamp;
            }
        }
    }

    auto finishFuelVolumeTimestamp = deviceSnapshotFinish->GetSnapshot().FuelVolumeTimestamp;
    auto finishFuelLevelTimestamp = deviceSnapshotFinish->GetSnapshot().FuelLevelTimestamp;
    auto finishFuelVolume = deviceSnapshotFinish->GetFuelVolume();
    auto finishFuelLevel = deviceSnapshotFinish->GetFuelLevel();
    auto finishMileage = deviceSnapshotFinish->GetMileage();
    if (accepted && startMileage && finishMileage) {
        auto mileageStart = *startMileage;
        auto mileageFinish = *finishMileage;
        if (mileageFinish >= mileageStart) {
            result.SetMileage(mileageFinish - mileageStart);
        }
    }
    if (!accepted && startMileage && finishMileage) {
        result.SetMileage(0);
    }
    if (result.HasMileage() && ServicingMileage > 0) {
        auto mileage = result.GetMileageRef();
        if (mileage > ServicingMileage) {
            mileage -= ServicingMileage;
        }
        result.SetMileage(mileage);
    }
    result.SetFinished(Finished);
    result.SetStart(deviceSnapshotStart->GetHistoryLocation());
    result.SetLast(deviceSnapshotFinish->GetHistoryLocation());
    result.SetStartFuelVolumeTimestamp(startFuelVolumeTimestamp);
    result.SetLastFuelVolumeTimestamp(finishFuelVolumeTimestamp);
    result.SetStartFuelLevelTimestamp(startFuelLevelTimestamp);
    result.SetLastFuelLevelTimestamp(finishFuelLevelTimestamp);
    result.SetStartFuelVolume(startFuelVolume);
    result.SetLastFuelVolume(finishFuelVolume);
    result.SetStartFuelLevel(startFuelLevel);
    result.SetLastFuelLevel(finishFuelLevel);
    result.SetStartMileage(startMileage);
    result.SetLastMileage(finishMileage);
    return result;
}

NJson::TJsonValue TSnapshotsDiffCompilation::GetReport(ELocalization locale, const NDrive::IServer* server, ISessionReportCustomization::TPtr /*customization*/) const {
    auto diff = GetSnapshotsDiff(server);
    return (diff && server) ? diff->GetReport(locale, *server) : NJson::JSON_NULL;
}

bool TSnapshotsDiffCompilation::Fill(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events) {
    if (timeline.empty()) {
        return false;
    }
    Started = true;
    for (auto&& element : timeline) {
        if (element.GetTimeEvent() != IEventsSession<TCarTagHistoryEvent>::EEvent::Tag) {
            continue;
        }
        Y_ASSERT(element.GetEventIndex() < events.size());
        auto ev = events[element.GetEventIndex()].Get();
        Y_ASSERT(ev);
        auto tag = ev ? ev->GetData().Get() : nullptr;
        Y_ASSERT(tag);
        auto snapshot = tag ? tag->GetObjectSnapshot() : nullptr;
        if (snapshot) {
            NamedSnapshots.emplace_back(tag->GetName(), std::move(snapshot));
        }
        auto offerContainer = ev ? ev->GetTagAs<IOfferContainer>() : nullptr;
        auto offer = offerContainer ? offerContainer->GetOffer() : nullptr;
        if (offer) {
            SessionId = offer->GetOfferId();
        }
        if (!ObjectId) {
            ObjectId = ev->TConstDBTag::GetObjectId();
        }
    }
    if (timeline.back().GetTimeEvent() == IEventsSession<TCarTagHistoryEvent>::EEvent::Tag) {
        Finished = true;
    }
    auto servicingResults = TBillingSession::CalcServicingResults(timeline, events, TInstant::Max());
    for (auto&& servicingResult : servicingResults) {
        auto info = servicingResult.Info.Get();
        if (info) {
            ServicingMileage += info->Mileage;
        }
    }
    return true;
}

NJson::TJsonValue TBillingEventsCompilation::TShortEvent::SerializeToJson() const {
    NJson::TJsonValue result;
    result.InsertValue("name", Name);
    result.InsertValue("action", ToString(Action));
    result.InsertValue("instant", Instant.Seconds());
    return result;
}

std::pair<TBillingEventsCompilation::AcceptanceSnapshot, TBillingEventsCompilation::AcceptanceSnapshot> TBillingEventsCompilation::GetAcceptanceSnapshots() const {
    AcceptanceSnapshot acceptanceFinishedSnapshot, acceptanceStartedSnapshot;
    for (auto&& i : Events) {
        if (i.GetName() == TChargableTag::Riding) {
            acceptanceFinishedSnapshot.SetDate(i.GetInstant());
            acceptanceFinishedSnapshot.SetMileage(i.GetMileage());
            acceptanceFinishedSnapshot.SetFuelLevel(i.GetFuelLevel());
            break;
        }
        if (i.GetName() == TChargableTag::Acceptance) {
            acceptanceStartedSnapshot.SetDate(i.GetInstant());
            acceptanceStartedSnapshot.SetMileage(i.GetMileage());
            acceptanceStartedSnapshot.SetFuelLevel(i.GetFuelLevel());
        }
    }
    return { acceptanceStartedSnapshot, acceptanceFinishedSnapshot };
}

bool TBillingEventsCompilation::GetCurrentPhaseInfo(TString& name, TInstant& instantStart, TInstant& instantCurrent, TSet<TString>& tagsInPoint) const {
    if (Events.back().GetName() == "$$current_state") {
        if (Events.size() > 1) {
            TShortEvent predEvent = Events[Events.size() - 2];
            for (i32 i = Events.size() - 3; i >= 0; --i) {
                if (Events[i].GetName() == predEvent.GetName()) {
                    predEvent = Events[i];
                } else {
                    break;
                }
            }
            name = predEvent.GetName();
            tagsInPoint = predEvent.GetTagsInPoint();
            instantStart = predEvent.GetInstant();
            instantCurrent = Events.back().GetInstant();
            return true;
        }
        return false;
    } else {
        return false;
    }
}

double TBillingEventsCompilation::GetMileageOnStart() const {
    if (Events.empty()) {
        return 0;
    }
    return Events.front().GetMileage();
}

double TBillingEventsCompilation::GetMileageMax() const {
    double result = 0;
    for (auto&& i : Events) {
        result = Max(result, i.GetMileage());
    }
    return result;
}

NJson::TJsonValue TBillingEventsCompilation::GetReport(ELocalization /*locale*/, const NDrive::IServer* /*server*/, ISessionReportCustomization::TPtr /*customization*/) const {
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&& ev : Events) {
        result.AppendValue(ev.SerializeToJson());
    }
    return result;
}

bool TBillingEventsCompilation::Fill(const TVector<TBillingEventsCompilation::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events) {
    Events.clear();
    TString objectId;
    for (auto&& i : timeline) {
        TShortEvent shortEvent;
        if (i.GetTimeEvent() == EEvent::Tag) {
            objectId = events[i.GetEventIndex()]->TConstDBTag::GetObjectId();
            shortEvent.SetName((*events[i.GetEventIndex()])->GetName());
            shortEvent.SetAction(events[i.GetEventIndex()]->GetHistoryAction());
            const THistoryDeviceSnapshot* ds = (*events[i.GetEventIndex()])->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
            if (ds) {
                const NDrive::TLocationTagsArray& locationTags = ds->GetLocationTagsArray();
                shortEvent.SetTagsInPoint({ locationTags.begin(), locationTags.end() });
                double mileCurrent = 0;
                double fuelLevelCurrent = 0;
                if (ds->GetMileage(mileCurrent)) {
                    shortEvent.SetMileage(mileCurrent);
                }
                if (ds->GetFuelLevel(fuelLevelCurrent)) {
                    shortEvent.SetFuelLevel(fuelLevelCurrent);
                }
            }
        } else {
            if (!!objectId && NDrive::HasServer()) {
                TRTDeviceSnapshot snapshot = NDrive::GetServerAs<NDrive::IServer>().GetSnapshotsManager().GetSnapshot(objectId);
                shortEvent.SetMileage(snapshot.GetMileage().GetOrElse(0));
                shortEvent.SetFuelLevel(snapshot.GetFuelLevel().GetOrElse(0));
            }
            shortEvent.SetName("$$current_state");
        }
        shortEvent.SetInstant(i.GetEventInstant());
        Events.emplace_back(shortEvent);
    }
    return true;
}

NDrive::TScheme TChargableTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSVariants>("default_subtags", "Default service subtags").SetMultiSelect(true).SetVariants(
        NContainer::Keys(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car))
    );
    return result;
}

NJson::TJsonValue TChargableTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeMetaToJson();
    NJson::InsertField(result, "default_subtags", DefaultSubtags);
    return result;
}

bool TChargableTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    if (!TBase::DoDeserializeMetaFromJson(jsonMeta)) {
        return false;
    }
    if (OptionalCleanupPolicy() == ECleanupPolicy::Ignore) {
        ClearCleanupPolicy();
    }
    SetEnableSessions(false);
    return
        NJson::ParseField(jsonMeta["default_subtags"], DefaultSubtags);
}

void TBillingSessionSelector::FillAdditionalEventsConditions(TMap<TString, TSet<TString>>& result, const std::deque<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const ui64 historyEventIdMaxWithFinishInstant) const {
    if (!ExpectedTagIds) {
        auto tx = TagsManager.GetDeviceTags().BuildTx<NSQL::ReadOnly>();
        ExpectedTagIds = TChargableTag::GetActiveSessionTagIds({}, TagsManager.GetTagsMeta(), historyEventIdMaxWithFinishInstant, tx);
        R_ENSURE(ExpectedTagIds, {}, "cannot GetActiveSessionTagIds", tx);
    }

    auto tagIds = ExpectedTagIds->GetTagIds();
    auto historyEventIds = ExpectedTagIds->GetHistoryEventIds();
    for (auto&& i : events) {
        if ((i->GetHistoryAction() == EObjectHistoryAction::SetTagPerformer) && tagIds.contains(i->GetTagId())) {
            tagIds.erase(i->GetTagId());
        }
        historyEventIds.erase(i->GetHistoryEventId());
    }
    if (historyEventIds.size()) {
        for (auto&& i : historyEventIds) {
            result["history_event_id"].emplace(::ToString(i));
        }
    }
    if (tagIds.size()) {
        result["tag_id"].insert(tagIds.begin(), tagIds.end());
    }
}

bool TBillingSession::TestEvent(const TCarTagHistoryEvent& histEvent) const {
    if (histEvent.GetObjectId() != GetObjectId()) {
        return false;
    }
    if (histEvent.GetHistoryAction() == EObjectHistoryAction::DropTagPerformer && !histEvent->GetPerformer()) {
        return histEvent.GetHistoryUserId() == GetUserId();
    } else if (histEvent.GetHistoryAction() == EObjectHistoryAction::TagEvolve && !histEvent->GetPerformer()) {
        return true;
    } else {
        return histEvent->GetPerformer() == GetUserId();
    }
}

NEventsSession::EEventCategory TBillingSessionSelector::AcceptImpl(const TCarTagHistoryEvent& e) {
    if (!!e->GetPerformer() && e.GetTagAs<TTransformationTag>()) {
        return NEventsSession::EEventCategory::IgnoreInternal;
    }
    if (!e.GetTagAs<IOfferContainer>()) {
        return NEventsSession::EEventCategory::IgnoreExternal;
    }
    if (!e->GetPerformer() && e.GetHistoryAction() != EObjectHistoryAction::DropTagPerformer) {
        return NEventsSession::EEventCategory::IgnoreExternal;
    }
    if (e.GetHistoryAction() == EObjectHistoryAction::DropTagPerformer) {
        return NEventsSession::EEventCategory::End;
    }
    if (e.GetHistoryAction() == EObjectHistoryAction::SetTagPerformer) {
        return NEventsSession::EEventCategory::Begin;
    }
    return NEventsSession::EEventCategory::Internal;
}

NEventsSession::EEventCategory TBillingSessionSelector::Accept(const TCarTagHistoryEvent& e) const {
    return AcceptImpl(e);
}

template struct TExpectedSizeOf<TChargableTag, 128>;

template <>
NJson::TJsonValue NJson::ToJson(const TChargableTag::TBookOptions& value) {
    NJson::TJsonValue result;
    result["check_blocked"] = value.CheckBlocked;
    result["futures"] = NJson::ToJson(value.Futures);
    result["device_id"] = value.DeviceId;
    result["mobile_paymethod_id"] = value.MobilePaymethodId;
    result["multi_rent"] = value.MultiRent;
    return result;
}
