#pragma once

#include "delegation.h"
#include "offer_container.h"
#include "servicing.h"

#include <drive/backend/areas/drop_object_features.h>
#include <drive/backend/billing/interfaces/payments.h>
#include <drive/backend/compiled_riding/compiled_riding.h>
#include <drive/backend/data/common/serializable.h>
#include <drive/backend/database/history/session.h>
#include <drive/backend/offers/abstract.h>
#include <drive/backend/proto/offer.pb.h>
#include <drive/backend/roles/permissions.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/util/types/uuid.h>

class TRadarUserTag;
class TDriveCarInfo;

class TTagReservationFutures
    : public INativeSerializationTag<NDrive::NProto::TTagReservationFutures>
    , public IOfferContainer
{
private:
    using TBase = INativeSerializationTag<NDrive::NProto::TTagReservationFutures>;

private:
    mutable IOffer::TPtr Offer;

    R_OPTIONAL(TGeoCoord, ExpectedCoord);
    R_FIELD(TString, ErrorMessage);
    R_FIELD(bool, Failed, false);

protected:
    virtual NDrive::NProto::TTagReservationFutures DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

public:
    static const TString TypeName;
    static const TString TypeName1;
    static TFactory::TRegistrator<TTagReservationFutures> Registrator;
    static TFactory::TRegistrator<TTagReservationFutures> Registrator1;

public:
    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car, NEntityTagsManager::EEntityType::User };
    }

    virtual TString GetPerformGroup() const override {
        return "futures";
    }

    TTagReservationFutures()
        : TBase(TypeName)
    {
    }

    TTagReservationFutures(IOffer::TPtr offer)
        : TBase(TypeName)
        , Offer(offer)
    {
    }

    ITag::TPtr BuildFailed(const TString& errorMessage) const;

    bool IsCorrectPosition(const TGeoCoord& c) const;
    bool IsExpired() const {
        return !!Offer && (Offer->GetDeadline() < ModelingNow());
    }

    TTagReservationFutures& SetOffer(IOffer::TPtr offer) {
        Offer = offer;
        return *this;
    }

    ICommonOffer::TPtr GetOffer() const override {
        return GetVehicleOffer();
    }

    IOffer::TPtr GetVehicleOffer() const {
        return Offer;
    }

    virtual const TString& GetStage(const TTaggedObject* /*object*/) const override {
        return GetName();
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    virtual bool OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
};

class TOfferBookingCompleted : public IMessage {
public:
    TOfferBookingCompleted(const ICommonOffer::TPtr offer)
        : Offer(offer)
    {
    }

private:
    R_READONLY(ICommonOffer::TPtr, Offer);
};

class TChargableTag
    : public ISerializableTag<NDrive::NProto::TChargableTag>
    , public IOfferContainer
    , public IServicingInfoHolder
    , public TDropPolicyProvider
{
private:
    using TBase = ISerializableTag<NDrive::NProto::TChargableTag>;

    struct TContext {
        // persistent
        IOffer::TPtr Offer;
        TString SharedSessionId;
        TMap<TString, TString> SubtagNames;
        TMaybe<TServicingInfo> ServicingInfo;
        TMaybe<bool> ServicingCancellable;

        // runtime only
        TAtomicSharedPtr<const IEventsSession<TTagHistoryEvent>> CurrentSession;
        TVector<ITag::TPtr> Subtags;
    };

private:
    THolder<TContext> Context;

public:
    R_OPTIONAL(ECarDelegationType, DelegationType);
    R_FIELD(bool, IsSwitching, false);
    R_FIELD(bool, Replacing, false);
    R_FIELD(bool, TransformationSkippedByExternalCommand, false);

private:
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* errors) override;

    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    TContext& MutableContext();
    void NotifyPassport(ISession::TConstPtr session, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx, bool stopFlag) const;

public:
    static const TString Prereservation;
    static const TString Reservation;
    static const TString Acceptance;
    static const TString Riding;
    static const TString Parking;
    static const TString Servicing;
    static const TString Sharing;

    static const TString ServicingDoneStage;
    static const TString ServicingExpectedStage;
    static const TString ServicingInProgressStage;

    static const TString TypeName;
    static TFactory::TRegistrator<TChargableTag> Registrator;

public:
    class TDescription: public TTagDescription {
    private:
        using TBase = TTagDescription;

    private:
        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        R_FIELD(TSet<TString>, DefaultSubtags);

    public:
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    protected:
        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
    };

    struct TBookOptions {
        TMaybe<bool> Futures;
        bool CheckBlocked = true;
        bool MultiRent = false;
        TString DeviceId;
        TString MobilePaymethodId;
    };

    struct TSwitchOptions {
        TString CopiedOfferId;
        TString DeviceId;
        TString TargetTagName;
        bool IgnoreNonSwitchable = false;
        bool MultiRent = false;
        bool Replacing = false;
    };

public:
    TChargableTag() = default;
    TChargableTag(const TString& name)
        : TBase(name)
    {
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    TString GetPerformGroup() const override;

    const TString& GetStage(const TTaggedObject* object) const override;
    NJson::TJsonValue GetStageReport(const TTaggedObject* object, const NDrive::IServer& server) const override;

    ICommonOffer::TPtr GetOffer() const override;
    IOffer::TPtr GetVehicleOffer() const;
    void SetOffer(IOffer::TPtr offer);

    const TString& GetSharedSessionId() const;
    void SetSharedSessionId(const TString& value);

    const TServicingInfo* GetServicingInfo() const override;
    void SetServicingInfo(TServicingInfo&& value) override;

    TConstArrayRef<ITag::TPtr> GetSubtags() const;
    void SetSubtags(TVector<ITag::TPtr>&& value);

    TSet<TString> GetSubtagIds() const;
    TStringBuf GetSubtagName(TStringBuf tagId) const;
    void AddSubtag(const TString& tagId, const TString& name);
    bool HasSubtagId(const TString& tagId) const;

    bool IsServicingCancellable() const;
    void SetServicingCancellable(bool value);

    virtual bool OnAfterDropPerform(const TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual bool OnBeforeDropPerform(const TDBTag& self, const NDrive::IServer* server, NDrive::TEntitySession& session) override;

    virtual bool OnAfterPerform(TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual bool OnBeforePerform(TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) override;

    virtual bool OnAfterEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const override;
    virtual bool OnBeforeEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const override;

    virtual bool OnBeforeRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) override;

public:
    static TVector<TString> GetTagNames(const ITagsMeta& meta);
    static TVector<TString> GetTagNamesWithTransformations(const ITagsMeta& meta);

    class TActiveSessionTags {
    private:
        R_FIELD(TSet<TString>, TagIds);
        R_FIELD(TSet<ui64>, HistoryEventIds);

    public:
        TActiveSessionTags(const TSet<TString>& tagIds, const TSet<ui64>& historyEventIds)
            : TagIds(tagIds)
            , HistoryEventIds(historyEventIds)
        {
        }

        bool IsEmpty() const {
            return TagIds.empty() && HistoryEventIds.empty();
        }
    };

    static TMaybe<TActiveSessionTags> GetActiveSessionTagIds(const TString& objectId, const ITagsMeta& meta, const ui64 historyEventIdMaxWithFinishInstant, NDrive::TEntitySession& session);

    static TDBTag Book(IOffer::TPtr offer, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session, const TBookOptions& bookOptions);
    static void CloseSession(IEventsSession<TCarTagHistoryEvent>::TConstPtr session, const TUserPermissions& permissions, const NDrive::IServer& server, bool strict = false);
    static bool DropSession(const TString& sessionId, const TString& userId, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session);
    static bool DirectEvolve(ISession::TPtr current, const TString& to, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session, const TEvolutionContext* context);
    static bool DirectEvolve(const TString& tagId, const TString& to, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session, const TEvolutionContext* context);
    static bool DirectEvolve(const TDBTag& tag, const TString& to, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session, const TEvolutionContext* context);
    static bool DirectEvolve(const TDBTag& tag, TAtomicSharedPtr<TChargableTag> next, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session, const TEvolutionContext* context);
    static bool Switch(ISession::TConstPtr current, IOffer::TPtr offer, TUserPermissions::TPtr permissions, const NDrive::IServer& server, NDrive::TEntitySession& session, const TSwitchOptions& switchOptions);
};

class TChargableSessionTag: public ISerializableTag<NDrive::NProto::TChargableSessionTag> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TChargableSessionTag>;

public:
    static TString Type() {
        return "chargable_session_tag";
    }

    static TMaybe<TDBTag> Get(const TString& sessionId, const IDriveTagsManager& driveTagsManagers, NDrive::TEntitySession& tx);
    static bool Register(const TString& sessionId, const TDBTag& sessionTag, NEntityTagsManager::EEntityType type, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx);
    static bool Deregister(const TString& sessionId, bool strict, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx);

public:
    using TBase::TBase;

    TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Trace };
    }

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

public:
    R_FIELD(TUuid, SessionTagId);
    R_FIELD(NEntityTagsManager::EEntityType, SessionTagEntityType, NEntityTagsManager::EEntityType::Car);

    DECLARE_FIELDS(
        Field(SessionTagId, "session_tag_id"),
        Field(SessionTagEntityType, "session_tag_entity_type")
    );

private:
    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* errors) override;

    TProto DoSerializeSpecialDataToProto() const override;
    bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

private:
    static TFactory::TRegistrator<TChargableSessionTag> Registrator;
};

class TBillingReportCustomization: public ISessionReportCustomization {
    R_FIELD(TString, Agreement);
    R_FIELD(bool, NeedBill, true);
    R_FIELD(bool, IsSimpleReport, false);
    R_FIELD(bool, NeedPriceByTags, false);
    R_OPTIONAL(TPaymentsData, PaymentsData);
    R_FIELD(NDriveSession::TReportTraits, ReportTraits, NDriveSession::ReportAll);
};

class TSnapshotsDiffCompilation: public IEventsSession<TCarTagHistoryEvent>::ICompilation {
private:
    using TNamedSnapshot = std::pair<TString, NDrive::IObjectSnapshot::TPtr>;
    using TNamedSnapshots = TVector<TNamedSnapshot>;

private:
    TNamedSnapshots NamedSnapshots;
    TString ObjectId;
    TString SessionId;
    double ServicingMileage = 0;
    bool Started = false;
    bool Finished = false;

public:
    const TNamedSnapshots& GetSnapshots() const {
        return NamedSnapshots;
    }

    NDrive::IObjectSnapshot::TPtr GetSnapshotStart() const {
        return (Started && !NamedSnapshots.empty()) ? Yasserted(NamedSnapshots.front().second) : nullptr;
    }

    NDrive::IObjectSnapshot::TPtr GetSnapshotFinish() const {
        return (Finished && !NamedSnapshots.empty()) ? Yasserted(NamedSnapshots.back().second) : nullptr;
    }

    TMaybe<TSnapshotsDiff> GetSnapshotsDiff(const NDrive::IServer* server) const;

    virtual NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer* server, ISessionReportCustomization::TPtr /*customization*/) const override;
    virtual const TString& GetSessionId() const override {
        return SessionId;
    }

    double GetServicingMileage() const {
        return ServicingMileage;
    }

    virtual bool Fill(const TVector<IEventsSession<TCarTagHistoryEvent>::TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events) override;
};

class TBillingCompilation;
class TBillingEventsCompilation;
class TBillingSessionSelector;

class TBillingSession: public IEventsSession<TCarTagHistoryEvent> {
private:
    using TBase = IEventsSession<TCarTagHistoryEvent>;

protected:
    virtual bool TestEvent(const TCarTagHistoryEvent& histEvent) const override;

public:
    using TTimeEvent = typename TBase::TTimeEvent;
    using ICompilation = typename TBase::ICompilation;

    using TBillingCompilation = TBillingCompilation;
    using TBillingEventsCompilation = TBillingEventsCompilation;

    using TSelector = TBillingSessionSelector;

    struct TServicingResult {
        TOptionalServicingInfo Info;
        NDrive::TLocationTags LocationTags;
        TInstant Start;
        TInstant Finish;
        TMaybe<double> StartMileage;
        TMaybe<double> FinishMileage;
    };
    using TServicingResults = TVector<TServicingResult>;

public:
    TBillingSession() = default;

    static TAtomicSharedPtr<TBillingSession> BuildFromCompiled(const NDrive::IServer* server, const TObjectEvent<TMinimalCompiledRiding>& compiledRide, TMessagesCollector& errors);
    static TServicingResults CalcServicingResults(const TVector<TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, TInstant until);

    THolder<ICompilation> BuildDefaultCompilation() const override;
    THolder<TFullCompiledRiding> BuildCompiledRiding(const NDrive::IServer* server, const TPaymentsData* payments) const;
    THolder<TFullCompiledRiding> BuildCompiledRiding(const NDrive::IServer* server, const TBillingCompilation& compilation, const TPaymentsData* payments) const;

    IOffer::TPtr GetCurrentOffer() const;

    bool GetFreeAndPricedDurations(TDuration& freeDuration, TDuration& pricedDuration) const;

protected:
    virtual TUniquePtr DoClone() const override {
        return MakeHolder<TBillingSession>(*this);
    }
};

class TBillingCompilation: public TBillingSession::ICompilation {
private:
    using EEvent = TBillingSession::EEvent;
    using TTimeEvent = TBillingSession::TTimeEvent;

private:
    R_FIELD(TInstant, Until, TInstant::Max());
    R_OPTIONAL(ECarDelegationType, DelegationType);
    R_READONLY(bool, Accepted, false);
    R_READONLY(bool, HasAcceptance, false);
    R_READONLY(double, ReportSumPrice, 0);
    R_READONLY(double, ReportSumOriginalPrice, 0);
    R_READONLY(double, BillingSumPrice, 0);
    R_READONLY(double, Revenue, 0);
    R_READONLY(double, DepositSum, 0);
    R_READONLY(double, BillingSumOriginalPrice, 0);
    R_READONLY(bool, IsFinished, true);
    R_READONLY(TDuration, FreeTime, TDuration::Zero());
    R_READONLY(TVector<TCompiledLocalEvent>, LocalEvents);
    R_READONLY(TVector<TServicingInfo>, ServicingInfos);
    R_READONLY(TVector<TString>, StartGeoTags);

private:
    TString SessionId;
    TVector<TOfferPricing> OfferSegments;
    TVector<TOfferStatePtr> OfferStates;

public:
    TBillingCompilation() = default;

    virtual const TString& GetSessionId() const override {
        return SessionId;
    }

    TVector<TString> GetBillingSessionSignals(const NDrive::IServer& server) const;

    static const NUnistat::TIntervals IntervalCompiledBills;
    static const NUnistat::TIntervals IntervalCompiledDurations;
    static const NUnistat::TIntervals IntervalCompiledFPFraction;

    void Signals(const TString& eventId) const;

    virtual bool Fill(const TVector<TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events) override;
    virtual NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer* server, ISessionReportCustomization::TPtr customization) const override;

    TBill GetBill(ELocalization locale, const NDrive::IServer* server, const TPaymentsData* task) const;

    TMaybe<TOfferPricing> GetCurrentOfferPricing() const;
    IOffer::TPtr GetCurrentOffer() const;
    TOfferStatePtr GetCurrentOfferState() const;

    template <class T>
    const T* GetCurrentOfferStateAs() const {
        auto current = GetCurrentOfferState();
        return dynamic_cast<const T*>(current.Get());
    }

    bool GetFreeAndPricedDurations(TDuration& freeDuration, TDuration& pricedDuration) const;
    TDuration GetSumDuration() const;
    TDuration GetRideDuration() const;
    TDuration GetActiveDuration() const;
};

class TBillingEventsCompilation: public TBillingSession::ICompilation {
public:
    struct AcceptanceSnapshot {
        R_OPTIONAL(TInstant, Date);
        R_OPTIONAL(double, Mileage);
        R_OPTIONAL(double, FuelLevel);
    };
private:
    using EEvent = TBillingSession::EEvent;
    using TTimeEvent = TBillingSession::TTimeEvent;

private:
    class TShortEvent {
    public:
        R_FIELD(TString, Name);
        R_FIELD(EObjectHistoryAction, Action, EObjectHistoryAction::TagEvolve);
        R_FIELD(TInstant, Instant, TInstant::Zero());
        R_FIELD(TSet<TString>, TagsInPoint);
        R_FIELD(double, Mileage, 0);
        R_FIELD(double, FuelLevel, 0);

    public:
        NJson::TJsonValue SerializeToJson() const;
    };

private:
    R_READONLY(TVector<TShortEvent>, Events);

public:
    TBillingEventsCompilation() = default;

    virtual const TString& GetSessionId() const override {
        ythrow TWithBackTrace<yexception>() << "unreachable called";
    }

    std::pair<AcceptanceSnapshot, AcceptanceSnapshot> GetAcceptanceSnapshots() const;
    bool GetCurrentPhaseInfo(TString& name, TInstant& instantStart, TInstant& instantCurrent, TSet<TString>& tagsInPoint) const;
    double GetMileageOnStart() const;
    double GetMileageMax() const;

    virtual NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer* server, ISessionReportCustomization::TPtr customization) const override;
    virtual bool Fill(const TVector<TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events) override;
};

class TBillingSessionSelector: public ISessionSelector<TCarTagHistoryEvent> {
private:
    const IDriveTagsManager& TagsManager;

    mutable TMaybe<TChargableTag::TActiveSessionTags> ExpectedTagIds;

public:
    TBillingSessionSelector(const IDriveTagsManager& tagsManager)
        : TagsManager(tagsManager)
    {
    }

    virtual TString GetName() const override {
        return "billing";
    }

    virtual IEventsSession<TCarTagHistoryEvent>::TPtr BuildSession() const override {
        return new TBillingSession;
    }

    virtual void FillAdditionalEventsConditions(TMap<TString, TSet<TString>>& result, const std::deque<TAtomicSharedPtr<TCarTagHistoryEvent>>& events, const ui64 historyEventIdMaxWithFinishInstant) const override;

    static NEventsSession::EEventCategory AcceptImpl(const TCarTagHistoryEvent& e);
    virtual NEventsSession::EEventCategory Accept(const TCarTagHistoryEvent& e) const override;
};
