#include "common.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/actions/evolution.h>
#include <drive/backend/actions/evolution_policy.h>
#include <drive/backend/roles/permissions.h>

NDrive::ITag::TAggregateEvolutionPolicy TCommonTag::BuildCustomEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const {
    TAggregateEvolutionPolicy result = TBase::BuildCustomEvolutionPolicy(dbTag, newTag, server, permissions, requestData, eMode, session);
    if (!result) {
        return result;
    }
    for (auto&& action : permissions.GetActionsActual()) {
        auto epAction = action.GetAs<IEvolutionPolicyAction>();
        if (!epAction) {
            continue;
        }
        result += epAction->BuildEvolutionPolicy(dbTag, newTag, server, permissions, requestData, eMode, session);
        if (!result) {
            break;
        }
    }
    return result;
}

bool TCommonTag::CopyOnEvolve(const ITag& source, const TTagEvolutionAction* evolution, const NDrive::IServer& server) {
    if (evolution) {
        auto td = server.GetDriveDatabase().GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
        switch (evolution->GetPriorityPolicy()) {
            case TTagEvolutionAction::EPriorityPolicy::Default:
                if (!!td) {
                    SetTagPriority(td->GetDefaultPriority());
                } else {
                    return false;
                }
                break;
            case TTagEvolutionAction::EPriorityPolicy::External:
                if (evolution->HasExternalPriority()) {
                    SetTagPriority(evolution->GetExternalPriorityUnsafe());
                } else {
                    return false;
                }
                break;
            case TTagEvolutionAction::EPriorityPolicy::New:
                if (!HasTagPriority()) {
                    return false;
                }
                break;
            case TTagEvolutionAction::EPriorityPolicy::Provide:
                SetTagPriority(source.OptionalTagPriority());
                break;
            case TTagEvolutionAction::EPriorityPolicy::NPD:
                if (!HasTagPriority()) {
                    SetTagPriority(source.OptionalTagPriority());
                    if (!HasTagPriority()) {
                        SetTagPriority(td->GetDefaultPriority());
                    }
                }
                break;
        }
        switch (evolution->GetCommentPolicy()) {
            case TTagEvolutionAction::ECommentPolicy::Empty:
                MutableComment().clear();
                break;
            case TTagEvolutionAction::ECommentPolicy::New:
                if (!GetComment()) {
                    return false;
                }
                break;
            case TTagEvolutionAction::ECommentPolicy::Provide:
                SetComment(source.GetComment());
                break;
            case TTagEvolutionAction::ECommentPolicy::NP:
                if (!GetComment()) {
                    SetComment(source.GetComment());
                }
                break;
        }
    }
    SetPerformer(source.GetPerformer());
    return true;
}

TAtomicSharedPtr<const TTagDescription> TCommonTag::GetDescription(const NDrive::IServer& server) const {
    return server.GetDriveDatabase().GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
}

TAtomicSharedPtr<const TTagDescription> TCommonTag::GetDescription(const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    const auto& name = GetName();
    auto result = GetDescription(server);
    if (!result) {
        session.SetErrorInfo("GetDescription", "cannot get description for " + name);
        return nullptr;
    }
    return result;
}
